"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PersistedLog = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const defaultIsDuplicate = (oldItem, newItem) => {
  return (0, _lodash.isEqual)(oldItem, newItem);
};
class PersistedLog {
  constructor(name, options = {}, storage) {
    (0, _defineProperty2.default)(this, "name", void 0);
    (0, _defineProperty2.default)(this, "maxLength", void 0);
    (0, _defineProperty2.default)(this, "filterDuplicates", void 0);
    (0, _defineProperty2.default)(this, "isDuplicate", void 0);
    (0, _defineProperty2.default)(this, "storage", void 0);
    (0, _defineProperty2.default)(this, "items", void 0);
    (0, _defineProperty2.default)(this, "update$", new _rxjs.BehaviorSubject(undefined));
    (0, _defineProperty2.default)(this, "storageEventListener", void 0);
    (0, _defineProperty2.default)(this, "enableBrowserTabsSync", void 0);
    (0, _defineProperty2.default)(this, "subscriberCount", 0);
    this.name = name;
    this.maxLength = typeof options.maxLength === 'string' ? this.maxLength = parseInt(options.maxLength, 10) : options.maxLength;
    this.filterDuplicates = options.filterDuplicates || false;
    this.isDuplicate = options.isDuplicate || defaultIsDuplicate;
    this.storage = storage;
    this.items = this.storage.get(this.name) || [];
    this.enableBrowserTabsSync = (options === null || options === void 0 ? void 0 : options.enableBrowserTabsSync) || false;
    if (this.maxLength !== undefined && !isNaN(this.maxLength)) {
      this.items = (0, _lodash.take)(this.items, this.maxLength);
    }
  }

  /** Keeps browser tabs in sync. */
  addStorageEventListener() {
    if (typeof window !== 'undefined' && !this.storageEventListener) {
      this.storageEventListener = event => {
        // Only handle events for this specific storage key
        if (event.key === this.name && event.newValue !== null) {
          try {
            const newItems = JSON.parse(event.newValue);
            // Only update if the items have actually changed
            if (!(0, _lodash.isEqual)(this.items, newItems)) {
              this.items = newItems;
              this.update$.next(undefined);
            }
          } catch (error) {
            return;
          }
        }
      };
      window.addEventListener('storage', this.storageEventListener);
    }
  }
  removeStorageEventListener() {
    if (typeof window !== 'undefined' && this.storageEventListener) {
      window.removeEventListener('storage', this.storageEventListener);
      this.storageEventListener = undefined;
    }
  }
  add(val) {
    if (val == null) {
      return this.items;
    }

    // remove any matching items from the stack if option is set
    if (this.filterDuplicates) {
      (0, _lodash.remove)(this.items, item => {
        return this.isDuplicate(item, val);
      });
    }
    this.items.unshift(val);

    // if maxLength is set, truncate the stack
    if (this.maxLength && !isNaN(this.maxLength)) {
      this.items = (0, _lodash.take)(this.items, this.maxLength);
    }

    // persist the stack
    this.storage.set(this.name, this.items);
    this.update$.next(undefined);
    return this.items;
  }
  get() {
    return (0, _lodash.cloneDeep)(this.items);
  }
  get$() {
    return (0, _rxjs.defer)(() => {
      this.subscriberCount++;
      // Add storage event listener when the first subscriber subscribes
      if (this.subscriberCount === 1 && this.enableBrowserTabsSync) {
        this.addStorageEventListener();
      }
      return this.update$.pipe((0, _rxjs.map)(() => this.get()), (0, _rxjs.finalize)(() => {
        this.subscriberCount--;
        // Remove storage event listener when the last subscriber unsubscribes
        if (this.subscriberCount === 0 && this.enableBrowserTabsSync) {
          this.removeStorageEventListener();
        }
      }));
    });
  }
}
exports.PersistedLog = PersistedLog;