"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getReadonlyMessage = exports.getProjectRoutingAccess = exports.DEFAULT_ACCESS_CONTROL_CONFIG = void 0;
var _cpsUtils = require("@kbn/cps-utils");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Rule for determining access based on route pattern
 */

/**
 * Configuration for a single app's access control
 */

/**
 * Complete access control configuration
 * Maps app IDs to their access configurations
 */

/**
 * Default access control configuration
 *
 * Access Levels:
 * - EDITABLE: Full functionality - users can change project scope
 * - READONLY: View-only mode - shows current scope but prevents changes
 * - DISABLED: Picker is completely disabled
 *
 * Default Configuration:
 *
 * | App        | Route                  | Access Level | Notes                              |
 * |------------|------------------------|--------------|-------------------------------------|
 * | dashboards | all routes except list | EDITABLE     | All dashboard pages except listing  |
 * | dashboards | #/list                 | DISABLED     | List page only                     |
 * | discover   | all routes             | EDITABLE     | All discover pages                 |
 * | visualize  | type:vega in route     | EDITABLE     | Vega visualizations only           |
 * | visualize  | other routes           | DISABLED     | Other visualization types          |
 * | lens       | all routes             | READONLY     | All lens pages (read-only)         |
 * | maps       | all routes             | EDITABLE     | All maps pages (read-only)         |
 * | all others | all routes             | DISABLED     | Default behavior for unknown apps  |
 */
const DEFAULT_ACCESS_CONTROL_CONFIG = exports.DEFAULT_ACCESS_CONTROL_CONFIG = {
  dashboards: {
    defaultAccess: _cpsUtils.ProjectRoutingAccess.EDITABLE,
    routeRules: [{
      pattern: /#\/list/,
      access: _cpsUtils.ProjectRoutingAccess.DISABLED
    }]
  },
  discover: {
    defaultAccess: _cpsUtils.ProjectRoutingAccess.EDITABLE
  },
  visualize: {
    defaultAccess: _cpsUtils.ProjectRoutingAccess.DISABLED,
    routeRules: [{
      pattern: /type:vega/,
      // Restrict to only Vega visualizations
      access: _cpsUtils.ProjectRoutingAccess.EDITABLE
    }]
  },
  lens: {
    defaultAccess: _cpsUtils.ProjectRoutingAccess.READONLY,
    readonlyMessage: _i18n.i18n.translate('cps.accessControl.lensReadonlyMessage', {
      defaultMessage: 'Please adjust project scope for each layer in the Lens editor.'
    })
  },
  maps: {
    defaultAccess: _cpsUtils.ProjectRoutingAccess.EDITABLE
  }
};

/**
 * Determines project routing access level based on app and route
 */
const getProjectRoutingAccess = (currentAppId, hash, config = DEFAULT_ACCESS_CONTROL_CONFIG) => {
  const appConfig = config[currentAppId];
  if (!appConfig) {
    return _cpsUtils.ProjectRoutingAccess.DISABLED;
  }
  if (appConfig.routeRules) {
    for (const rule of appConfig.routeRules) {
      if (rule.pattern.test(hash)) {
        return rule.access;
      }
    }
  }
  return appConfig.defaultAccess;
};
exports.getProjectRoutingAccess = getProjectRoutingAccess;
const getReadonlyMessage = (currentAppId, config = DEFAULT_ACCESS_CONTROL_CONFIG) => {
  var _config$currentAppId;
  return currentAppId ? (_config$currentAppId = config[currentAppId]) === null || _config$currentAppId === void 0 ? void 0 : _config$currentAppId.readonlyMessage : undefined;
};
exports.getReadonlyMessage = getReadonlyMessage;