"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processGapsBatch = void 0;
var _lodash = require("lodash");
var _interval_utils = require("../../../../lib/rule_gaps/gap/interval_utils");
var _schedule = require("../../../backfill/methods/schedule");
var _utils = require("./utils");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const processGapsBatch = async (context, {
  range,
  gapsBatch,
  maxGapsCountToProcess,
  initiator,
  initiatorId
}) => {
  const {
    start,
    end
  } = range;

  // Group gaps by rule ID
  const gapsByRuleId = (0, _lodash.groupBy)(gapsBatch, 'ruleId');
  const schedulingPayloads = [];
  let totalProcessedGapsCount = 0;
  const gapsForScheduling = [];
  const results = [];
  const processedGapsByRuleId = new Map();
  const startDate = new Date(start);
  const endDate = new Date(end);

  // Prepare all scheduling payloads from all rules
  for (const [ruleId, gapBatchForRule] of Object.entries(gapsByRuleId)) {
    let ruleGapsClampedIntervals = gapBatchForRule.map(gap => ({
      gap,
      clampedIntervals: (0, _interval_utils.clampIntervals)(gap.unfilledIntervals, {
        gte: startDate,
        lte: endDate
      })
    })).filter(({
      clampedIntervals
    }) => clampedIntervals.length > 0);
    if (maxGapsCountToProcess && totalProcessedGapsCount + ruleGapsClampedIntervals.length > maxGapsCountToProcess) {
      const remainingSlots = maxGapsCountToProcess - totalProcessedGapsCount;
      ruleGapsClampedIntervals = ruleGapsClampedIntervals.slice(0, remainingSlots);
    }
    if (ruleGapsClampedIntervals.length === 0) {
      continue;
    }
    const ruleProcessedCount = ruleGapsClampedIntervals.length;
    totalProcessedGapsCount += ruleProcessedCount;
    processedGapsByRuleId.set(ruleId, ruleProcessedCount);
    gapsForScheduling.push(...ruleGapsClampedIntervals.map(({
      gap
    }) => gap));
    const gapRanges = ruleGapsClampedIntervals.flatMap(({
      clampedIntervals
    }) => clampedIntervals.map(({
      gte,
      lte
    }) => ({
      start: gte.toISOString(),
      end: lte.toISOString()
    })));
    schedulingPayloads.push({
      ruleId,
      ranges: gapRanges,
      initiator,
      ...(initiatorId ? {
        initiatorId
      } : {})
    });

    // Stop if we've reached the max gaps count limit
    if (maxGapsCountToProcess && totalProcessedGapsCount >= maxGapsCountToProcess) {
      break;
    }
  }

  // Rules might have gaps within the range that don't yield any schedulingPayload
  // This can happen when they have gaps that are in an "in progress" state.
  if (schedulingPayloads.length === 0) {
    return {
      processedGapsCount: 0,
      results: [],
      hasErrors: false
    };
  }

  // Schedule all backfills in a single bulk operation
  const scheduleResults = await (0, _schedule.scheduleBackfill)(context, schedulingPayloads, gapsForScheduling);
  let hasErrors = false;
  for (let i = 0; i < scheduleResults.length; i++) {
    const result = scheduleResults[i];
    const ruleId = schedulingPayloads[i].ruleId;
    const processedGaps = processedGapsByRuleId.get(ruleId) || 0;
    if ('error' in result) {
      var _result$error;
      hasErrors = true;
      results.push({
        ruleId,
        processedGaps,
        status: _types.GapFillSchedulePerRuleStatus.ERROR,
        error: ((_result$error = result.error) === null || _result$error === void 0 ? void 0 : _result$error.message) || 'Unknown error'
      });
    } else {
      (0, _utils.logProcessedAsAuditEvent)(context, {
        id: ruleId,
        name: ruleId
      });
      results.push({
        ruleId,
        processedGaps,
        status: _types.GapFillSchedulePerRuleStatus.SUCCESS
      });
    }
  }
  return {
    processedGapsCount: totalProcessedGapsCount,
    results,
    hasErrors
  };
};
exports.processGapsBatch = processGapsBatch;