"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleResponseSchema = exports.scheduleRequestSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _validation = require("../validation");
var _v = require("../validation/validate_integer/v1");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const scheduleRequestSchema = exports.scheduleRequestSchema = _configSchema.schema.object({
  start: _configSchema.schema.string({
    validate: _validation.validateStartDateV1,
    meta: {
      description: 'The start date and time of the schedule, provided in ISO 8601 format and set to the UTC timezone. For example: `2025-03-12T12:00:00.000Z`.'
    }
  }),
  duration: _configSchema.schema.string({
    validate: _validation.validateDurationV1,
    meta: {
      description: 'The duration of the schedule. It allows values in `<integer><unit>` format. `<unit>` is one of `d`, `h`, `m`, or `s` for hours, minutes, seconds. For example: `1d`, `5h`, `30m`, `5000s`.'
    }
  }),
  timezone: _configSchema.schema.maybe(_configSchema.schema.string({
    validate: _validation.validateTimezoneV1,
    meta: {
      description: 'The timezone of the schedule. The default timezone is UTC.'
    }
  })),
  recurring: _configSchema.schema.maybe(_configSchema.schema.object({
    end: _configSchema.schema.maybe(_configSchema.schema.string({
      validate: _validation.validateEndDateV1,
      meta: {
        description: 'The end date of a recurring schedule, provided in ISO 8601 format and set to the UTC timezone. For example: `2025-04-01T00:00:00.000Z`.'
      }
    })),
    every: _configSchema.schema.maybe(_configSchema.schema.string({
      validate: _validation.validateIntervalAndFrequencyV1,
      meta: {
        description: 'The interval and frequency of a recurring schedule. It allows values in `<integer><unit>` format. `<unit>` is one of `d`, `w`, `M`, or `y` for days, weeks, months, years. For example: `15d`, `2w`, `3m`, `1y`.'
      }
    })),
    onWeekDay: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
      minSize: 1,
      validate: _validation.validateOnWeekDayV1,
      meta: {
        description: 'The specific days of the week (`[MO,TU,WE,TH,FR,SA,SU]`) or nth day of month (`[+1MO, -3FR, +2WE, -4SA, -5SU]`) for a recurring schedule.'
      }
    })),
    onMonthDay: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.number({
      min: 1,
      max: 31,
      validate: value => (0, _v.validateInteger)(value, 'onMonthDay')
    }), {
      minSize: 1,
      meta: {
        description: 'The specific days of the month for a recurring schedule. Valid values are 1-31.'
      }
    })),
    onMonth: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.number({
      min: 1,
      max: 12,
      validate: value => (0, _v.validateInteger)(value, 'onMonth')
    }), {
      minSize: 1,
      meta: {
        description: 'The specific months for a recurring schedule. Valid values are 1-12.'
      }
    })),
    occurrences: _configSchema.schema.maybe(_configSchema.schema.number({
      validate: occurrences => (0, _v.validateInteger)(occurrences, 'occurrences'),
      min: 1,
      meta: {
        description: 'The total number of recurrences of the schedule.'
      }
    }))
  }))
}, {
  validate: _validation.validateScheduleV1
});

// The response schema has the same fields but without validation
const scheduleResponseSchema = exports.scheduleResponseSchema = _configSchema.schema.object({
  start: _configSchema.schema.string({
    meta: {
      description: 'The start date and time of the schedule, provided in ISO 8601 format and set to the UTC timezone. For example: `2025-03-12T12:00:00.000Z`.'
    }
  }),
  duration: _configSchema.schema.string({
    meta: {
      description: 'The duration of the schedule. It allows values in `<integer><unit>` format. `<unit>` is one of `d`, `h`, `m`, or `s` for hours, minutes, seconds. For example: `1d`, `5h`, `30m`, `5000s`.'
    }
  }),
  timezone: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'The timezone of the schedule. The default timezone is UTC.'
    }
  })),
  recurring: _configSchema.schema.maybe(_configSchema.schema.object({
    end: _configSchema.schema.maybe(_configSchema.schema.string({
      meta: {
        description: 'The end date of a recurring schedule, provided in ISO 8601 format and set to the UTC timezone. For example: `2025-04-01T00:00:00.000Z`.'
      }
    })),
    every: _configSchema.schema.maybe(_configSchema.schema.string({
      meta: {
        description: 'The interval and frequency of a recurring schedule. It allows values in `<integer><unit>` format. `<unit>` is one of `d`, `w`, `M`, or `y` for days, weeks, months, years. For example: `15d`, `2w`, `3m`, `1y`.'
      }
    })),
    onWeekDay: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
      meta: {
        description: 'The specific days of the week (`[MO,TU,WE,TH,FR,SA,SU]`) or nth day of month (`[+1MO, -3FR, +2WE, -4SA, -5SU]`) for a recurring schedule.'
      }
    })),
    onMonthDay: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.number(), {
      meta: {
        description: 'The specific days of the month for a recurring schedule. Valid values are 1-31.'
      }
    })),
    onMonth: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.number(), {
      meta: {
        description: 'The specific months for a recurring schedule. Valid values are 1-12.'
      }
    })),
    occurrences: _configSchema.schema.maybe(_configSchema.schema.number({
      meta: {
        description: 'The total number of recurrences of the schedule.'
      }
    }))
  }))
});