"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ingestPipelineValidatorTool = ingestPipelineValidatorTool;
var _tools = require("@langchain/core/tools");
var _langgraph = require("@langchain/langgraph");
var _messages = require("@langchain/core/messages");
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function formatSample(sample) {
  return {
    _index: 'index',
    _id: 'id',
    _source: {
      message: sample
    }
  };
}
/**
 * Creates a standalone langgraph tool that validates an ingest pipeline against samples.
 * This function can be used independently or through the IngestPipelineValidatorTool class.
 *
 * @param esClient - Elasticsearch client for simulating the pipeline
 * @param samples - Array of log samples to validate the pipeline against
 * @returns DynamicStructuredTool instance for use in langgraph agents
 */
function ingestPipelineValidatorTool(esClient, samples) {
  const validatorSchema = _zod.z.object({
    generated_pipeline: _zod.z.object({
      processors: _zod.z.array(_zod.z.any()).describe('The processors in the pipeline'),
      on_failure: _zod.z.array(_zod.z.any()).optional().describe('Optional failure handlers for the pipeline')
    }).describe('The generated ingest pipeline to validate')
  });
  return new _tools.DynamicStructuredTool({
    name: 'validate_ingest_pipeline',
    description: 'Validates a generated ingest pipeline by testing it against log samples. ' + 'Simulates the pipeline to verify it works correctly. ' + 'Returns validation results including success rate, failed samples, and error details.',
    schema: _zod.z.object({
      generated_pipeline: _zod.z.object({
        processors: _zod.z.array(_zod.z.any()).describe('The processors in the pipeline'),
        on_failure: _zod.z.array(_zod.z.any()).optional().describe('Optional failure handlers for the pipeline')
      }).describe('The generated ingest pipeline to validate')
    }),
    func: async (input, _runManager, config) => {
      const {
        generated_pipeline: generatedPipeline
      } = input;
      try {
        var _config$toolCall3;
        if (!samples || samples.length === 0) {
          var _config$toolCall;
          const message = `No samples available for validation`;
          return new _langgraph.Command({
            update: {
              pipeline_generation_results: [],
              failure_count: 0,
              pipeline_validation_results: {
                success_rate: 0,
                successful_samples: 0,
                failed_samples: 0,
                total_samples: 0,
                failure_details: []
              },
              messages: [new _messages.ToolMessage({
                content: message,
                tool_call_id: config === null || config === void 0 ? void 0 : (_config$toolCall = config.toolCall) === null || _config$toolCall === void 0 ? void 0 : _config$toolCall.id
              })]
            }
          });
        }

        // Format samples for pipeline simulation
        const docs = samples.map(sample => formatSample(sample));

        // Simulate the pipeline
        let response;
        try {
          response = await esClient.ingest.simulate({
            docs,
            pipeline: generatedPipeline
          });
        } catch (simulateError) {
          var _config$toolCall2;
          const errorMessage = `Pipeline simulation failed: ${simulateError.message}`;
          return new _langgraph.Command({
            update: {
              pipeline_generation_results: [],
              failure_count: samples.length,
              pipeline_validation_results: {
                success_rate: 0,
                successful_samples: 0,
                failed_samples: samples.length,
                total_samples: samples.length,
                failure_details: [{
                  error: errorMessage,
                  sample: 'Pipeline validation error'
                }]
              },
              messages: [new _messages.ToolMessage({
                content: errorMessage,
                tool_call_id: config === null || config === void 0 ? void 0 : (_config$toolCall2 = config.toolCall) === null || _config$toolCall2 === void 0 ? void 0 : _config$toolCall2.id
              })]
            }
          });
        }

        // Process simulation results
        const failedSamples = [];
        const successfulDocuments = [];
        let successfulCount = 0;
        response.docs.forEach((doc, index) => {
          var _doc$doc, _doc$doc$_source, _doc$doc2;
          if (!doc) {
            // Document was dropped
            failedSamples.push({
              sample: samples[index],
              error: 'Document was dropped by the pipeline'
            });
          } else if ((_doc$doc = doc.doc) !== null && _doc$doc !== void 0 && (_doc$doc$_source = _doc$doc._source) !== null && _doc$doc$_source !== void 0 && _doc$doc$_source.error) {
            // Document has an error
            const errorDetail = typeof doc.doc._source.error === 'string' ? doc.doc._source.error : JSON.stringify(doc.doc._source.error);
            failedSamples.push({
              sample: samples[index],
              error: errorDetail
            });
          } else if ((_doc$doc2 = doc.doc) !== null && _doc$doc2 !== void 0 && _doc$doc2._source) {
            // Document processed successfully
            successfulCount++;
            successfulDocuments.push(doc.doc._source);
          }
        });
        const failedCount = failedSamples.length;
        const totalSamples = samples.length;
        const successRate = totalSamples > 0 ? successfulCount / totalSamples * 100 : 0;
        const message = failedCount === 0 ? `Pipeline validation successful! All ${totalSamples} samples processed correctly.` : `Pipeline validation completed with ${successfulCount}/${totalSamples} samples successful (${successRate.toFixed(1)}% success rate). ${failedCount} samples failed.`;
        return new _langgraph.Command({
          update: {
            current_pipeline: generatedPipeline,
            pipeline_generation_results: successfulDocuments,
            failure_count: failedCount,
            pipeline_validation_results: {
              success_rate: successRate,
              successful_samples: successfulCount,
              failed_samples: failedCount,
              total_samples: totalSamples,
              failure_details: failedSamples.slice(0, 100).map(f => ({
                error: f.error,
                sample: f.sample
              }))
            },
            messages: [new _messages.ToolMessage({
              content: message,
              tool_call_id: config === null || config === void 0 ? void 0 : (_config$toolCall3 = config.toolCall) === null || _config$toolCall3 === void 0 ? void 0 : _config$toolCall3.id
            })]
          }
        });
      } catch (error) {
        var _config$toolCall4;
        const errorMessage = `Validation tool error: ${error.message}`;
        return new _langgraph.Command({
          update: {
            pipeline_generation_results: [],
            failure_count: 1,
            pipeline_validation_results: {
              success_rate: 0,
              successful_samples: 0,
              failed_samples: 1,
              total_samples: 1,
              failure_details: [{
                error: errorMessage,
                sample: 'Tool execution error'
              }]
            },
            messages: [new _messages.ToolMessage({
              content: errorMessage,
              tool_call_id: config === null || config === void 0 ? void 0 : (_config$toolCall4 = config.toolCall) === null || _config$toolCall4 === void 0 ? void 0 : _config$toolCall4.id
            })]
          }
        });
      }
    }
  });
}