"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCelInputRoutes = registerCelInputRoutes;
var _server = require("@kbn/data-plugin/server");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _common = require("../../common");
var _constants = require("../constants");
var _cel = require("../graphs/cel");
var _route_validation = require("../util/route_validation");
var _with_availability = require("./with_availability");
var _errors = require("../lib/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerCelInputRoutes(router) {
  router.versioned.post({
    path: _common.CEL_INPUT_GRAPH_PATH,
    access: 'internal',
    options: {
      timeout: {
        idleSocket: _constants.ROUTE_HANDLER_TIMEOUT
      }
    },
    security: {
      authz: {
        requiredPrivileges: [_constants.FLEET_ALL_ROLE, _constants.INTEGRATIONS_ALL_ROLE]
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidationWithZod)(_common.CelInputRequestBody)
      }
    }
  }, (0, _with_availability.withAvailability)(async (context, request, res) => {
    const {
      dataStreamTitle,
      celDetails,
      langSmithOptions
    } = request.body;
    const {
      getStartServices,
      logger
    } = await context.automaticImport;
    const [, startPlugins] = await getStartServices();
    try {
      var _celDetails$openApiDe, _celDetails$openApiDe2, _celDetails$openApiDe3, _celDetails$openApiDe4, _celDetails$openApiDe5, _celDetails$openApiDe6, _langSmithOptions$pro;
      const inference = await startPlugins.inference;
      const abortSignal = (0, _server.getRequestAbortedSignal)(request.events.aborted$);
      const connectorId = request.body.connectorId;
      const model = await inference.getChatModel({
        request,
        connectorId,
        chatModelOptions: {
          // not passing specific `model`, we'll always use the connector default model
          // temperature may need to be parametrized in the future
          temperature: 0.05,
          // Only retry once inside the model call, we already handle backoff retries in the task runner for the entire task
          maxRetries: 1,
          // Disable streaming explicitly
          disableStreaming: true,
          // Set a hard limit of 50 concurrent requests
          maxConcurrency: 50,
          telemetryMetadata: {
            pluginId: 'automatic_import'
          },
          signal: abortSignal
        }
      });
      const parameters = {
        dataStreamName: dataStreamTitle,
        path: celDetails.path,
        authType: celDetails.auth,
        openApiPathDetails: JSON.parse((_celDetails$openApiDe = (_celDetails$openApiDe2 = celDetails.openApiDetails) === null || _celDetails$openApiDe2 === void 0 ? void 0 : _celDetails$openApiDe2.operation) !== null && _celDetails$openApiDe !== void 0 ? _celDetails$openApiDe : ''),
        openApiSchemas: JSON.parse((_celDetails$openApiDe3 = (_celDetails$openApiDe4 = celDetails.openApiDetails) === null || _celDetails$openApiDe4 === void 0 ? void 0 : _celDetails$openApiDe4.schemas) !== null && _celDetails$openApiDe3 !== void 0 ? _celDetails$openApiDe3 : ''),
        openApiAuthSchema: (_celDetails$openApiDe5 = celDetails.openApiDetails) !== null && _celDetails$openApiDe5 !== void 0 && _celDetails$openApiDe5.auth ? JSON.parse((_celDetails$openApiDe6 = celDetails.openApiDetails) === null || _celDetails$openApiDe6 === void 0 ? void 0 : _celDetails$openApiDe6.auth) : undefined
      };
      const options = {
        callbacks: [new _apm.APMTracer({
          projectName: (_langSmithOptions$pro = langSmithOptions === null || langSmithOptions === void 0 ? void 0 : langSmithOptions.projectName) !== null && _langSmithOptions$pro !== void 0 ? _langSmithOptions$pro : 'default'
        }, logger), ...(0, _langsmith.getLangSmithTracer)({
          ...langSmithOptions,
          logger
        })]
      };
      const graph = await (0, _cel.getCelGraph)({
        model
      });
      const results = await graph.withConfig({
        runName: 'CEL'
      }).invoke(parameters, options);
      return res.ok({
        body: _common.CelInputResponse.parse(results)
      });
    } catch (e) {
      if ((0, _errors.isErrorThatHandlesItsOwnResponse)(e)) {
        return e.sendResponse(res);
      }
      return res.badRequest({
        body: e
      });
    }
  }));
}