"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AlertingAuthorization = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _alerting_authorization_kuery = require("./alerting_authorization_kuery");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AlertingAuthorization {
  constructor({
    ruleTypeRegistry,
    request,
    authorization,
    getSpaceId,
    allRegisteredConsumers,
    ruleTypesConsumersMap
  }) {
    (0, _defineProperty2.default)(this, "ruleTypeRegistry", void 0);
    (0, _defineProperty2.default)(this, "request", void 0);
    (0, _defineProperty2.default)(this, "authorization", void 0);
    (0, _defineProperty2.default)(this, "allRegisteredConsumers", void 0);
    (0, _defineProperty2.default)(this, "ruleTypesConsumersMap", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    this.request = request;
    this.authorization = authorization;
    this.ruleTypeRegistry = ruleTypeRegistry;
    this.allRegisteredConsumers = allRegisteredConsumers;
    this.ruleTypesConsumersMap = ruleTypesConsumersMap;
    this.spaceId = getSpaceId(request);
  }

  /**
   * Creates an AlertingAuthorization object.
   */
  static async create({
    request,
    features,
    getSpace,
    getSpaceId,
    authorization,
    ruleTypeRegistry
  }) {
    var _maybeSpace$disabledF, _maybeSpace;
    const allRegisteredConsumers = new Set();
    const ruleTypesConsumersMap = new Map();
    let maybeSpace;
    try {
      maybeSpace = await getSpace(request);
    } catch (error) {
      /**
       * Failing to fetch the space with 403 or 404 means the user is likely not privileged in the active space at all
       * which means that `allRegisteredConsumers` and `ruleTypesConsumersMap` should be empty. By initializing the alerting authorization
       * with empty data structures we ensure that the user will not have access to any rule types in the active space.
       * All other errors are treated as server errors and they are surfaced to the upper level.
       */
      if (_boom.default.isBoom(error) && [403, 404].includes(error.output.statusCode)) {
        return new AlertingAuthorization({
          request,
          authorization,
          getSpaceId,
          ruleTypeRegistry,
          allRegisteredConsumers,
          ruleTypesConsumersMap
        });
      }
      if (_boom.default.isBoom(error)) {
        throw error;
      }
      throw new Error(`Failed to create AlertingAuthorization class: ${error}`);
    }
    const disabledFeatures = new Set((_maybeSpace$disabledF = (_maybeSpace = maybeSpace) === null || _maybeSpace === void 0 ? void 0 : _maybeSpace.disabledFeatures) !== null && _maybeSpace$disabledF !== void 0 ? _maybeSpace$disabledF : []);
    const featuresWithAlertingConfigured = features.getKibanaFeatures().filter(({
      id,
      alerting
    }) =>
    // ignore features which are disabled in the user's space
    !disabledFeatures.has(id) &&
    // ignore features which don't grant privileges to alerting
    Boolean(alerting === null || alerting === void 0 ? void 0 : alerting.length));

    /**
     * Each feature configures a set of rule types. Each
     * rule type configures its valid consumers. For example,
     *
     * { id: 'my-feature-id-1', alerting: [{ ruleTypeId: 'my-rule-type', consumers: ['consumer-a', 'consumer-b'] }] }
     * { id: 'my-feature-id-2', alerting: [{ ruleTypeId: 'my-rule-type-2', consumers: ['consumer-a', 'consumer-d'] }] }
     *
     * In this loop we iterate over all features and we construct:
     * a) a set that contains all registered consumers and
     * b) a map that contains all valid consumers per rule type.
     * We remove duplicates in the process. For example,
     *
     * allRegisteredConsumers: Set(1) { 'consumer-a', 'consumer-b', 'consumer-d' }
     * ruleTypesConsumersMap: Map(1) {
     *  'my-rule-type' => Set(1) { 'consumer-a', 'consumer-b' }
     *  'my-rule-type-2' => Set(1) { 'consumer-a', 'consumer-d' }
     * }
     */
    for (const feature of featuresWithAlertingConfigured) {
      if (feature.alerting) {
        for (const entry of feature.alerting) {
          var _ruleTypesConsumersMa;
          const consumers = (_ruleTypesConsumersMa = ruleTypesConsumersMap.get(entry.ruleTypeId)) !== null && _ruleTypesConsumersMa !== void 0 ? _ruleTypesConsumersMa : new Set();
          entry.consumers.forEach(consumer => {
            consumers.add(consumer);
            allRegisteredConsumers.add(consumer);
          });
          ruleTypesConsumersMap.set(entry.ruleTypeId, consumers);
        }
      }
    }
    return new AlertingAuthorization({
      request,
      authorization,
      getSpaceId,
      ruleTypeRegistry,
      allRegisteredConsumers,
      ruleTypesConsumersMap
    });
  }
  shouldCheckAuthorization() {
    var _this$authorization$m, _this$authorization, _this$authorization$m2;
    return (_this$authorization$m = (_this$authorization = this.authorization) === null || _this$authorization === void 0 ? void 0 : (_this$authorization$m2 = _this$authorization.mode) === null || _this$authorization$m2 === void 0 ? void 0 : _this$authorization$m2.useRbacForRequest(this.request)) !== null && _this$authorization$m !== void 0 ? _this$authorization$m : false;
  }
  getSpaceId() {
    return this.spaceId;
  }

  /*
   * This method exposes the private '_getAuthorizedRuleTypesWithAuthorizedConsumers' to be
   * used by the RAC/Alerts client
   */
  async getAllAuthorizedRuleTypes(params) {
    return this._getAuthorizedRuleTypesWithAuthorizedConsumers({
      operations: params.operations,
      authorizationEntity: params.authorizationEntity
    });
  }
  async ensureAuthorized({
    ruleTypeId,
    consumer,
    operation,
    entity,
    additionalPrivileges = []
  }) {
    return this._ensureAuthorized({
      ruleTypeIdConsumersPairs: [{
        ruleTypeId,
        consumers: [consumer]
      }],
      operation,
      entity,
      additionalPrivileges
    });
  }
  async bulkEnsureAuthorized({
    ruleTypeIdConsumersPairs,
    operation,
    entity,
    additionalPrivileges = []
  }) {
    return this._ensureAuthorized({
      ruleTypeIdConsumersPairs,
      operation,
      entity,
      additionalPrivileges
    });
  }
  async _ensureAuthorized({
    ruleTypeIdConsumersPairs,
    operation,
    entity,
    additionalPrivileges = []
  }) {
    const {
      authorization
    } = this;
    const areAllConsumersAvailable = ruleTypeIdConsumersPairs.every(({
      consumers
    }) => consumers.every(consumer => this.allRegisteredConsumers.has(consumer)));
    if (authorization && this.shouldCheckAuthorization()) {
      const checkPrivileges = authorization.checkPrivilegesDynamicallyWithRequest(this.request);
      const privileges = ruleTypeIdConsumersPairs.flatMap(({
        ruleTypeId,
        consumers
      }) => consumers.map(consumer => authorization.actions.alerting.get(ruleTypeId, consumer, entity, operation)));
      const res = await checkPrivileges({
        kibana: [...privileges, ...additionalPrivileges]
      });
      const {
        hasAllRequested
      } = res;
      if (!areAllConsumersAvailable) {
        /**
         * Under most circumstances this would have been caught by `checkPrivileges` as
         * a user can't have Privileges to an unknown consumer, but super users
         * don't actually get "privilege checked" so the made up consumer *will* return
         * as Privileged.
         * This check will ensure we don't accidentally let these through
         */
        throw _boom.default.forbidden(getUnauthorizedMessage(ruleTypeIdConsumersPairs, operation, entity));
      }
      if (!hasAllRequested) {
        throw _boom.default.forbidden(getUnauthorizedMessage(ruleTypeIdConsumersPairs, operation, entity));
      }
    } else if (!areAllConsumersAvailable) {
      throw _boom.default.forbidden(getUnauthorizedMessage(ruleTypeIdConsumersPairs, operation, entity));
    }
  }
  async getFindAuthorizationFilter(params) {
    return this.getAuthorizationFilter({
      operation: _types.ReadOperations.Find,
      authorizationEntity: params.authorizationEntity,
      filterOpts: params.filterOpts
    });
  }
  async getAllAuthorizedRuleTypesFindOperation(params) {
    const {
      authorizedRuleTypes
    } = await this._getAuthorizedRuleTypesWithAuthorizedConsumers({
      ruleTypeIds: params.ruleTypeIds,
      operations: [_types.ReadOperations.Find],
      authorizationEntity: params.authorizationEntity
    });
    return authorizedRuleTypes;
  }
  async getAuthorizationFilter(params) {
    if (this.authorization && this.shouldCheckAuthorization()) {
      const {
        authorizedRuleTypes
      } = await this._getAuthorizedRuleTypesWithAuthorizedConsumers({
        operations: [params.operation],
        authorizationEntity: params.authorizationEntity
      });
      if (!authorizedRuleTypes.size) {
        throw _boom.default.forbidden(`Unauthorized to ${params.operation} ${params.authorizationEntity}s for any rule types`);
      }
      return {
        filter: (0, _alerting_authorization_kuery.asFiltersByRuleTypeAndConsumer)(authorizedRuleTypes, params.filterOpts, this.spaceId),
        ensureRuleTypeIsAuthorized: (ruleTypeId, consumer, authType) => {
          if (!authorizedRuleTypes.has(ruleTypeId) || authType !== params.authorizationEntity) {
            throw _boom.default.forbidden(getUnauthorizedMessage([{
              ruleTypeId,
              consumers: [consumer]
            }], params.operation, authType));
          }
          const authorizedRuleType = authorizedRuleTypes.get(ruleTypeId);
          const authorizedConsumers = authorizedRuleType.authorizedConsumers;
          if (!authorizedConsumers[consumer]) {
            throw _boom.default.forbidden(getUnauthorizedMessage([{
              ruleTypeId,
              consumers: [consumer]
            }], params.operation, params.authorizationEntity));
          }
        }
      };
    }
    return {
      filter: (0, _alerting_authorization_kuery.asFiltersBySpaceId)(params.filterOpts, this.spaceId),
      ensureRuleTypeIsAuthorized: (ruleTypeId, consumer, authType) => {}
    };
  }
  async getAuthorizedRuleTypes(params) {
    const {
      authorizedRuleTypes
    } = await this._getAuthorizedRuleTypesWithAuthorizedConsumers({
      ruleTypeIds: params.ruleTypeIds,
      operations: params.operations,
      authorizationEntity: params.authorizationEntity
    });
    return authorizedRuleTypes;
  }
  async _getAuthorizedRuleTypesWithAuthorizedConsumers(params) {
    const {
      operations,
      authorizationEntity
    } = params;
    const ruleTypeIds = params.ruleTypeIds ? new Set(params.ruleTypeIds) : new Set(this.ruleTypeRegistry.getAllTypes());
    const requiredPrivileges = new Map();
    if (this.authorization && this.shouldCheckAuthorization()) {
      const authorizedRuleTypes = new Map();
      const checkPrivileges = this.authorization.checkPrivilegesDynamicallyWithRequest(this.request);
      for (const ruleTypeId of ruleTypeIds) {
        var _this$ruleTypesConsum;
        /**
         * Skip if the ruleTypeId is not configured in any feature
         * or it is not set in the rule type registry.
         */
        if (!this.ruleTypesConsumersMap.has(ruleTypeId) || !this.ruleTypeRegistry.has(ruleTypeId)) {
          continue;
        }
        const ruleType = this.ruleTypeRegistry.get(ruleTypeId);
        const ruleTypeConsumers = (_this$ruleTypesConsum = this.ruleTypesConsumersMap.get(ruleTypeId)) !== null && _this$ruleTypesConsum !== void 0 ? _this$ruleTypesConsum : new Set();
        for (const consumerToAuthorize of ruleTypeConsumers) {
          for (const operation of operations) {
            requiredPrivileges.set(this.authorization.actions.alerting.get(ruleTypeId, consumerToAuthorize, authorizationEntity, operation), {
              ruleTypeId: ruleType.id,
              consumer: consumerToAuthorize,
              operation
            });
          }
        }
      }
      const {
        username,
        hasAllRequested,
        privileges
      } = await checkPrivileges({
        kibana: [...requiredPrivileges.keys()]
      });
      for (const {
        authorized,
        privilege
      } of privileges.kibana) {
        if (authorized && requiredPrivileges.has(privilege)) {
          var _authorizedRuleTypes$;
          const {
            ruleTypeId,
            consumer,
            operation
          } = requiredPrivileges.get(privilege);
          const authorizedRuleType = (_authorizedRuleTypes$ = authorizedRuleTypes.get(ruleTypeId)) !== null && _authorizedRuleTypes$ !== void 0 ? _authorizedRuleTypes$ : {
            authorizedConsumers: {}
          };
          const authorizedConsumers = authorizedRuleType.authorizedConsumers;
          const mergedOperations = mergeHasPrivileges(getPrivilegesFromOperation(operation), authorizedConsumers[consumer]);
          authorizedRuleTypes.set(ruleTypeId, {
            authorizedConsumers: {
              ...authorizedConsumers,
              [consumer]: mergedOperations
            }
          });
        }
      }
      return {
        username,
        hasAllRequested,
        authorizedRuleTypes
      };
    } else {
      return {
        hasAllRequested: true,
        authorizedRuleTypes: this.getRegisteredRuleTypesWithAllRegisteredConsumers(ruleTypeIds)
      };
    }
  }
  getRegisteredRuleTypesWithAllRegisteredConsumers(ruleTypeIds) {
    const authorizedRuleTypes = new Map();
    const authorizedConsumers = getConsumersWithPrivileges(this.allRegisteredConsumers, {
      all: true,
      read: true
    });
    Array.from(this.ruleTypesConsumersMap.keys()).filter(ruleTypeId => ruleTypeIds.has(ruleTypeId)).forEach(ruleTypeId => {
      authorizedRuleTypes.set(ruleTypeId, {
        authorizedConsumers
      });
    });
    return authorizedRuleTypes;
  }
}
exports.AlertingAuthorization = AlertingAuthorization;
function mergeHasPrivileges(left, right) {
  var _ref, _ref2;
  return {
    read: (_ref = left.read || (right === null || right === void 0 ? void 0 : right.read)) !== null && _ref !== void 0 ? _ref : false,
    all: (_ref2 = left.all || (right === null || right === void 0 ? void 0 : right.all)) !== null && _ref2 !== void 0 ? _ref2 : false
  };
}
function getPrivilegesFromOperation(operation) {
  const read = Object.values(_types.ReadOperations).includes(operation);
  const all = Object.values(_types.WriteOperations).includes(operation);
  return {
    read: read || all,
    all
  };
}
function getConsumersWithPrivileges(consumers, hasPrivileges) {
  return Array.from(consumers).reduce((acc, feature) => {
    acc[feature] = hasPrivileges;
    return acc;
  }, {});
}
function getUnauthorizedMessage(ruleTypeIdConsumersPairs, operation, entity) {
  const allConsumers = ruleTypeIdConsumersPairs.flatMap(({
    consumers
  }) => consumers);
  const allRuleTypeIds = ruleTypeIdConsumersPairs.map(({
    ruleTypeId
  }) => ruleTypeId);
  const ruleTypeIdsMessage = allRuleTypeIds.length <= 0 ? 'any' : `${allRuleTypeIds.join(', ')}`;
  const consumersMessage = allConsumers.length <= 0 ? 'any consumer' : `${allConsumers.join(', ')}`;
  return `Unauthorized by "${consumersMessage}" to ${operation} "${ruleTypeIdsMessage}" ${entity}`;
}