"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVisualizationGraph = void 0;
var _langgraph = require("@langchain/langgraph");
var _metric = require("@kbn/lens-embeddable-utils/config_builder/schema/charts/metric");
var _gauge = require("@kbn/lens-embeddable-utils/config_builder/schema/charts/gauge");
var _tagcloud = require("@kbn/lens-embeddable-utils/config_builder/schema/charts/tagcloud");
var _xy = require("@kbn/lens-embeddable-utils/config_builder/schema/charts/xy");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _actions_lens = require("./actions_lens");
var _prompts = require("./prompts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Regex to extract JSON from markdown code blocks
const INLINE_JSON_REGEX = /```(?:json)?\s*([\s\S]*?)\s*```/gm;

/**
 * Helper to extract ESQL queries from a visualization config.
 * Handles both single-dataset configs (metric, gauge, tagcloud) and layers-based configs (XY).
 * For XY charts with multiple layers, returns all unique ESQL queries.
 */
function getExistingEsqlQueries(config) {
  if (!config) return [];
  const queries = [];

  // For XY charts, check all layers' datasets
  if ('layers' in config && Array.isArray(config.layers)) {
    for (const layer of config.layers) {
      if (layer && 'dataset' in layer && layer.dataset) {
        const dataset = layer.dataset;
        if (dataset.type === 'esql' && dataset.query && !queries.includes(dataset.query)) {
          queries.push(dataset.query);
        }
      }
    }
    return queries;
  }

  // For single-dataset configs (metric, gauge, tagcloud)
  if ('dataset' in config && config.dataset) {
    const dataset = config.dataset;
    if (dataset.type === 'esql' && dataset.query) {
      queries.push(dataset.query);
    }
  }
  return queries;
}

/**
 * Helper to get a single existing ESQL query (for backward compatibility).
 * Returns the first query if multiple exist.
 */
function getExistingEsqlQuery(config) {
  const queries = getExistingEsqlQueries(config);
  return queries.length > 0 ? queries[0] : null;
}
const VisualizationStateAnnotation = _langgraph.Annotation.Root({
  // inputs
  nlQuery: (0, _langgraph.Annotation)(),
  chartType: (0, _langgraph.Annotation)(),
  schema: (0, _langgraph.Annotation)(),
  existingConfig: (0, _langgraph.Annotation)(),
  parsedExistingConfig: (0, _langgraph.Annotation)(),
  // internal
  esqlQuery: (0, _langgraph.Annotation)(),
  currentAttempt: (0, _langgraph.Annotation)({
    reducer: (_, newValue) => newValue,
    default: () => 0
  }),
  actions: (0, _langgraph.Annotation)({
    reducer: (a, b) => [...a, ...b],
    default: () => []
  }),
  // outputs
  validatedConfig: (0, _langgraph.Annotation)(),
  error: (0, _langgraph.Annotation)()
});
const createVisualizationGraph = (model, logger, events, esClient) => {
  // Node: Generate ES|QL query
  const generateESQLNode = async state => {
    logger.debug('Generating ES|QL query for visualization');
    let action;
    try {
      const existingQueries = getExistingEsqlQueries(state.parsedExistingConfig);
      let nlQueryWithContext = state.nlQuery;
      if (existingQueries.length > 0) {
        if (existingQueries.length === 1) {
          nlQueryWithContext = `Existing esql query to modify: "${existingQueries[0]}"\n\nUser query: ${state.nlQuery}`;
        } else {
          const queriesContext = existingQueries.map((q, i) => `Layer ${i + 1}: "${q}"`).join('\n');
          nlQueryWithContext = `Existing esql queries from multiple layers:\n${queriesContext}\n\nUser query: ${state.nlQuery}`;
        }
      }
      const generateEsqlResponse = await (0, _onechatGenaiUtils.generateEsql)({
        nlQuery: nlQueryWithContext,
        model,
        events,
        logger,
        esClient: esClient.asCurrentUser
      });
      if (!generateEsqlResponse.query) {
        action = {
          type: 'generate_esql',
          success: false,
          error: 'No queries generated'
        };
      } else {
        const esqlQuery = generateEsqlResponse.query;
        logger.debug(`Generated ES|QL query: ${esqlQuery}`);
        action = {
          type: 'generate_esql',
          success: true,
          query: esqlQuery
        };
      }
    } catch (error) {
      logger.error(`Failed to generate ES|QL query: ${error.message}`);
      action = {
        type: 'generate_esql',
        success: false,
        error: error.message
      };
    }
    return {
      actions: [action]
    };
  };

  // Node: Generate configuration
  const generateConfigNode = async state => {
    const attempt = state.currentAttempt + 1;
    logger.debug(`Generating visualization configuration (attempt ${attempt}/${_actions_lens.MAX_RETRY_ATTEMPTS})`);

    // Extract ES|QL query from previous actions
    const lastGenerateEsqlAction = state.actions.filter(action => action.type === 'generate_esql').filter(action => action.success && action.query).pop();
    const esqlQuery = (lastGenerateEsqlAction === null || lastGenerateEsqlAction === void 0 ? void 0 : lastGenerateEsqlAction.query) || state.esqlQuery;

    // Build context from previous actions for retry attempts
    const previousActionContext = state.actions.filter(action => (0, _actions_lens.isGenerateConfigAction)(action) || (0, _actions_lens.isValidateConfigAction)(action)).map(action => {
      if ((0, _actions_lens.isGenerateConfigAction)(action)) {
        return `Previous generation attempt ${action.attempt}: ${action.success ? 'SUCCESS' : `FAILED - ${action.error}`}`;
      }
      if ((0, _actions_lens.isValidateConfigAction)(action)) {
        return `Validation attempt ${action.attempt}: ${action.success ? 'SUCCESS' : `FAILED - ${action.error}`}`;
      }
      return '';
    }).filter(Boolean).join('\n');
    const additionalInstructions = `IMPORTANT RULES:
1. The 'dataset' field must contain: { type: "esql", query: "${esqlQuery}" }
2. Always use { operation: 'value', column: '<esql column name>', ...other options } for operations
3. All field names must match those available in the ES|QL query result
4. Follow the schema definition strictly`;
    const additionalContext = previousActionContext ? `Previous attempts:\n${previousActionContext}\n\nPlease fix the issues mentioned above.` : undefined;
    const prompt = (0, _prompts.createGenerateConfigPrompt)({
      nlQuery: state.nlQuery,
      chartType: state.chartType,
      schema: state.schema,
      existingConfig: state.existingConfig,
      additionalInstructions,
      additionalContext
    });
    let action;
    try {
      // Invoke model without schema validation
      const response = await model.chatModel.invoke(prompt);
      const responseText = (0, _langchain.extractTextContent)(response);

      // Try to extract JSON from markdown code blocks
      const jsonMatches = Array.from(responseText.matchAll(INLINE_JSON_REGEX));
      let configResponse;
      if (jsonMatches.length > 0) {
        const jsonText = jsonMatches[0][1].trim();
        configResponse = JSON.parse(jsonText);
      } else {
        configResponse = JSON.parse(responseText);
      }

      // Verify it's a valid object
      if (!configResponse || typeof configResponse !== 'object') {
        throw new Error('Response is not a valid JSON object');
      }
      action = {
        type: 'generate_config',
        success: true,
        config: configResponse,
        attempt
      };
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : String(error);
      logger.warn(`Config generation failed (attempt ${attempt}/${_actions_lens.MAX_RETRY_ATTEMPTS}): ${errorMessage}`);
      logger.debug(`Full error details: ${JSON.stringify(error, null, 2)}`);
      action = {
        type: 'generate_config',
        success: false,
        attempt,
        error: errorMessage
      };
    }
    return {
      currentAttempt: attempt,
      actions: [action]
    };
  };

  // Node: Validate configuration
  const validateConfigNode = async state => {
    const attempt = state.currentAttempt;
    logger.debug(`Validating configuration (attempt ${attempt}/${_actions_lens.MAX_RETRY_ATTEMPTS})`);

    // Get the last generate_config action
    const lastGenerateAction = [...state.actions].reverse().find(_actions_lens.isGenerateConfigAction);
    if (!lastGenerateAction || !lastGenerateAction.config) {
      const action = {
        type: 'validate_config',
        success: false,
        attempt,
        error: 'No configuration found to validate'
      };
      return {
        actions: [action]
      };
    }
    let action;
    try {
      const config = lastGenerateAction.config;

      // Check if the generation itself failed
      if ('error' in config && typeof config.error === 'string') {
        logger.warn(`Configuration generation reported error: ${config.error}`);
        action = {
          type: 'validate_config',
          success: false,
          attempt,
          error: config.error
        };
      } else {
        // Validate configuration based on chart type
        let validatedConfig = null;
        if (state.chartType === _tool_result.SupportedChartType.Metric) {
          validatedConfig = _metric.esqlMetricState.validate(config);
        } else if (state.chartType === _tool_result.SupportedChartType.Gauge) {
          validatedConfig = _gauge.gaugeStateSchemaESQL.validate(config);
        } else if (state.chartType === _tool_result.SupportedChartType.Tagcloud) {
          validatedConfig = _tagcloud.tagcloudStateSchemaESQL.validate(config);
        } else if (state.chartType === _tool_result.SupportedChartType.XY) {
          validatedConfig = _xy.xyStateSchema.validate(config);
        } else {
          throw new Error(`Unsupported chart type: ${state.chartType}`);
        }
        logger.debug('Configuration validated successfully');
        action = {
          type: 'validate_config',
          success: true,
          config: validatedConfig,
          attempt
        };
      }
    } catch (error) {
      const errorMessage = error.message;
      logger.warn(`Configuration validation failed: ${errorMessage}`);
      action = {
        type: 'validate_config',
        success: false,
        attempt,
        error: errorMessage
      };
    }
    return {
      actions: [action]
    };
  };

  // Node: Finalize - extract outputs from actions
  const finalizeNode = async state => {
    const lastValidateAction = [...state.actions].reverse().find(_actions_lens.isValidateConfigAction);
    const lastGenerateEsqlAction = [...state.actions].reverse().find(action => action.type === 'generate_esql');
    return {
      validatedConfig: lastValidateAction !== null && lastValidateAction !== void 0 && lastValidateAction.success ? lastValidateAction.config : null,
      error: lastValidateAction !== null && lastValidateAction !== void 0 && lastValidateAction.success ? null : (lastValidateAction === null || lastValidateAction === void 0 ? void 0 : lastValidateAction.error) || null,
      esqlQuery: lastGenerateEsqlAction === null || lastGenerateEsqlAction === void 0 ? void 0 : lastGenerateEsqlAction.query
    };
  };

  // Router: Check if we should retry or end after validation
  const shouldRetryRouter = state => {
    const lastValidateAction = [...state.actions].reverse().find(_actions_lens.isValidateConfigAction);

    // Success case - configuration is valid
    if (lastValidateAction !== null && lastValidateAction !== void 0 && lastValidateAction.success) {
      logger.debug('Configuration validated successfully, finalizing');
      return 'finalize';
    }

    // Failure case - max attempts reached
    if (state.currentAttempt >= _actions_lens.MAX_RETRY_ATTEMPTS) {
      logger.warn(`Max retry attempts (${_actions_lens.MAX_RETRY_ATTEMPTS}) reached, finalizing`);
      return 'finalize';
    }

    // Retry case - loop back to generate with previous actions providing context
    logger.debug(`Retry ${state.currentAttempt}/${_actions_lens.MAX_RETRY_ATTEMPTS}, generating again with action context`);
    return _actions_lens.GENERATE_CONFIG_NODE;
  };

  // Router: Decide whether to generate ESQL or use existing
  const shouldGenerateESQLRouter = state => {
    // If we have existing config with a query, skip ES|QL generation
    const existingQuery = getExistingEsqlQuery(state.parsedExistingConfig);
    if (existingQuery) {
      logger.debug('Using existing ES|QL query from parsed config');
      return _actions_lens.GENERATE_CONFIG_NODE;
    }
    logger.debug('No existing query found, generating new ES|QL query');
    return _actions_lens.GENERATE_ESQL_NODE;
  };

  // Build and compile the graph
  const graph = new _langgraph.StateGraph(VisualizationStateAnnotation)
  // Add nodes
  .addNode(_actions_lens.GENERATE_ESQL_NODE, generateESQLNode).addNode(_actions_lens.GENERATE_CONFIG_NODE, generateConfigNode).addNode(_actions_lens.VALIDATE_CONFIG_NODE, validateConfigNode).addNode('finalize', finalizeNode)
  // Add edges
  .addConditionalEdges('__start__', shouldGenerateESQLRouter, {
    [_actions_lens.GENERATE_CONFIG_NODE]: _actions_lens.GENERATE_CONFIG_NODE,
    [_actions_lens.GENERATE_ESQL_NODE]: _actions_lens.GENERATE_ESQL_NODE
  }).addEdge(_actions_lens.GENERATE_ESQL_NODE, _actions_lens.GENERATE_CONFIG_NODE).addEdge(_actions_lens.GENERATE_CONFIG_NODE, _actions_lens.VALIDATE_CONFIG_NODE).addConditionalEdges(_actions_lens.VALIDATE_CONFIG_NODE, shouldRetryRouter, {
    [_actions_lens.GENERATE_CONFIG_NODE]: _actions_lens.GENERATE_CONFIG_NODE,
    finalize: 'finalize'
  }).addEdge('finalize', '__end__').compile();
  return graph;
};
exports.createVisualizationGraph = createVisualizationGraph;