"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeDataStream = initializeDataStream;
var _nodeAssert = _interopRequireDefault(require("node:assert"));
var _elasticsearch = require("@elastic/elasticsearch");
var _retry_es = require("../retry_es");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * https://www.elastic.co/docs/manage-data/data-store/data-streams/set-up-data-stream
 *
 * Endeavour to be idempotent and race-condition safe.
 */
async function initializeDataStream({
  logger,
  dataStream,
  elasticsearchClient,
  existingDataStream,
  existingIndexTemplate,
  skipCreation = true
}) {
  const version = dataStream.version;
  logger.debug(`Setting up data stream: ${dataStream.name} v${version}`);
  if (skipCreation && !existingDataStream) {
    // data stream does not exist and we will not create it.
    logger.debug(`Skipping data stream creation during lazy initialization: ${dataStream.name}.`);
    return {
      uptoDate: false
    };
  }
  if (existingIndexTemplate) {
    var _existingIndexTemplat, _existingIndexTemplat2;
    const deployedVersion = (_existingIndexTemplat = existingIndexTemplate.index_template) === null || _existingIndexTemplat === void 0 ? void 0 : (_existingIndexTemplat2 = _existingIndexTemplat._meta) === null || _existingIndexTemplat2 === void 0 ? void 0 : _existingIndexTemplat2.version;
    (0, _nodeAssert.default)(typeof deployedVersion === 'number' && deployedVersion > 0, `Datastream ${dataStream.name} metadata is in an unexpected state, expected version to be a number but got ${deployedVersion}`);
    if (deployedVersion >= version) {
      // index already applied and updated.
      logger.debug(`Deployed ${dataStream.name} v${deployedVersion} already applied and updated.`);
      return {
        uptoDate: true
      };
    }
  }
  if (existingDataStream) {
    logger.debug(`Data stream already exists: ${dataStream.name}, applying mappings to write index`);

    // https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-indices-get-data-stream#operation-indices-get-data-stream-200-body-application-json-data_streams-indices
    // The last item in this array contains information about the stream’s current write index.
    const {
      indices
    } = existingDataStream;
    const writeIndex = indices[indices.length - 1];
    if (!writeIndex) {
      logger.debug(`Data stream ${dataStream.name} has no write index yet, cannot apply mappings or settings.`);
      // data stream has no write index yet, cannot apply mappings or settings.
      return {
        uptoDate: false
      };
    } else {
      const {
        template: {
          mappings
        }
      } = await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.simulateIndexTemplate({
        name: dataStream.name
      }));
      logger.debug(`Applying mappings to write index: ${writeIndex.index_name}`);
      await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.putMapping({
        index: writeIndex.index_name,
        ...mappings
      }));
    }

    // data stream updated successfully
    return {
      uptoDate: true
    };
  }
  logger.debug(`Creating data stream: ${dataStream.name}.`);
  try {
    await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.createDataStream({
      name: dataStream.name
    }));
  } catch (error) {
    var _error$body;
    if (error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 400 && ((_error$body = error.body) === null || _error$body === void 0 ? void 0 : _error$body.error.type) === 'resource_already_exists_exception') {
      // Data stream already exists, we can ignore this error, probably racing another create call
      logger.debug(`Data stream already exists: ${dataStream.name}`);
    } else {
      throw error;
    }
  }

  // data stream created and updated successfully
  return {
    uptoDate: true
  };
}