"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRequestHashForClientCache = exports.createRequestHashForBackgroundSearches = void 0;
var _cryptoBrowser = require("@kbn/crypto-browser");
var _jsonStableStringify = _interopRequireDefault(require("json-stable-stringify"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Generate the hash for this request.
 *
 * Ignores the `preference` parameter since it generally won't
 * match from one request to another identical request.
 *
 * (Preference is used to ensure all queries go to the same set of shards and it doesn't need to be hashed
 * https://www.elastic.co/guide/en/elasticsearch/reference/current/search-shard-routing.html#shard-and-node-preference)
 */
function createRequestHash(keys) {
  const {
    preference,
    ...rest
  } = keys;
  const hash = new _cryptoBrowser.Sha256().update((0, _jsonStableStringify.default)(rest), 'utf8').digest('hex');
  return hash;
}

/**
 * Generates the hash used as a key in the client-side request cache.
 */
const createRequestHashForClientCache = exports.createRequestHashForClientCache = createRequestHash;

/**
 * Generates the hash for associating requests with background searches stored on the server.
 *
 * Ignores sessionId for compatibility with background searches created before https://github.com/elastic/kibana/pull/237191
 */
const createRequestHashForBackgroundSearches = keys => {
  // exclude additional parameters that are not relevant for server-side caching
  const {
    sessionId,
    ...rest
  } = keys;
  return createRequestHash(rest);
};
exports.createRequestHashForBackgroundSearches = createRequestHashForBackgroundSearches;