"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkBackfillCapacity = checkBackfillCapacity;
exports.filterGapsWithOverlappingBackfills = filterGapsWithOverlappingBackfills;
exports.formatConsolidatedSummary = formatConsolidatedSummary;
exports.getGapAutoFillRunOutcome = getGapAutoFillRunOutcome;
exports.initRun = initRun;
exports.isCancelled = isCancelled;
exports.resultsFromMap = resultsFromMap;
var _lodash = require("lodash");
var _constants = require("../../../../common/constants");
var _saved_objects = require("../../../saved_objects");
var _gap_auto_fill_scheduler_event_log = require("./gap_auto_fill_scheduler_event_log");
var _scheduler = require("../../../application/gaps/types/scheduler");
var _interval_utils = require("../gap/interval_utils");
var _types = require("../../../application/gaps/methods/bulk_fill_gaps_by_rule_ids/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function resultsFromMap(aggregatedByRule) {
  return Array.from(aggregatedByRule.values());
}
function formatConsolidatedSummary(consolidated) {
  const rulesCount = consolidated.length;
  const gapsProcessed = consolidated.reduce((sum, r) => {
    var _r$processedGaps;
    return sum + ((_r$processedGaps = r.processedGaps) !== null && _r$processedGaps !== void 0 ? _r$processedGaps : 0);
  }, 0);
  const errorEntries = consolidated.filter(r => r.status === _types.GapFillSchedulePerRuleStatus.ERROR);
  const errorCount = errorEntries.length;
  const errorsList = errorEntries.map(r => r.error ? `${r.ruleId} (${r.error})` : r.ruleId).join('\n ');
  const successEntries = consolidated.filter(r => r.status === _types.GapFillSchedulePerRuleStatus.SUCCESS);
  const successCount = successEntries.length;
  const successList = successEntries.map(r => `${r.ruleId} (${r.processedGaps} gaps processed)`).join('\n ');
  const parts = [];
  parts.push(`processed ${gapsProcessed} gap${gapsProcessed === 1 ? '' : 's'} across ${rulesCount} rule${rulesCount === 1 ? '' : 's'}`);
  if (successCount > 0) {
    parts.push(`${successCount} success${successCount === 1 ? '' : 'es'}: ${successList}`);
  }
  if (errorCount > 0) {
    parts.push(`${errorCount} error${errorCount === 1 ? '' : 's'}: ${errorsList}`);
  }
  return `\n${parts.join(' \n')}`;
}
function getGapAutoFillRunOutcome(consolidated) {
  const anySuccess = consolidated.some(r => r.status === _types.GapFillSchedulePerRuleStatus.SUCCESS);
  const anyError = consolidated.some(r => r.status === _types.GapFillSchedulePerRuleStatus.ERROR);
  if (consolidated.length === 0) {
    return {
      status: _scheduler.GAP_AUTO_FILL_STATUS.SKIPPED,
      message: "Skipped execution: can't schedule gap fills for any enabled rule"
    };
  }
  if (anySuccess && !anyError) {
    return {
      status: _scheduler.GAP_AUTO_FILL_STATUS.SUCCESS,
      message: 'All rules successfully scheduled gap fills'
    };
  }
  if (!anySuccess && anyError) {
    return {
      status: _scheduler.GAP_AUTO_FILL_STATUS.ERROR,
      message: 'All rules failed to schedule gap fills'
    };
  }
  return {
    status: _scheduler.GAP_AUTO_FILL_STATUS.ERROR,
    message: 'At least one rule successfully scheduled gap fills, but others failed to schedule'
  };
}
function isCancelled(abortController) {
  return abortController.signal.aborted;
}
async function filterGapsWithOverlappingBackfills(gaps, rulesClientContext, logMessage) {
  const filteredGaps = [];
  const actionsClient = await rulesClientContext.getActionsClient();
  const backfillClient = rulesClientContext.backfillClient;
  const gapsByRuleId = (0, _lodash.groupBy)(gaps, 'ruleId');
  for (const [ruleId, ruleGaps] of Object.entries(gapsByRuleId)) {
    const overlappingBackfills = await backfillClient.findOverlappingBackfills({
      ruleId,
      ranges: ruleGaps.map(gap => ({
        start: gap.range.gte,
        end: gap.range.lte
      })),
      savedObjectsRepository: rulesClientContext.internalSavedObjectsRepository,
      actionsClient
    });
    if (overlappingBackfills.length === 0) {
      filteredGaps.push(...ruleGaps);
    } else {
      ruleGaps.forEach(gap => {
        const isGapOverlappingWithBackfills = overlappingBackfills.some(backfill => {
          if ('error' in backfill) {
            return false;
          }
          if (!backfill.start || !backfill.end) {
            return false;
          }
          return (0, _interval_utils.getOverlap)({
            gte: new Date(backfill.start),
            lte: new Date(backfill.end)
          }, gap.range) !== null;
        });
        if (!isGapOverlappingWithBackfills) {
          filteredGaps.push(gap);
        }
      });
    }
  }
  if (filteredGaps.length < gaps.length) {
    logMessage(`Filtered out ${gaps.length - filteredGaps.length} gaps that have overlapping backfills`);
  }
  return filteredGaps;
}
async function initRun({
  fakeRequest,
  getRulesClientWithRequest,
  eventLogger,
  taskInstance,
  startTime
}) {
  var _taskInstance$params;
  if (!getRulesClientWithRequest || !fakeRequest) {
    throw new Error('Missing request or rules client factory');
  }
  const rulesClient = await getRulesClientWithRequest(fakeRequest);
  if (!rulesClient) {
    throw new Error('Missing rules client');
  }
  const rulesClientContext = rulesClient.getContext();
  const configId = taskInstance === null || taskInstance === void 0 ? void 0 : (_taskInstance$params = taskInstance.params) === null || _taskInstance$params === void 0 ? void 0 : _taskInstance$params.configId;
  if (!configId) {
    throw new Error('Missing configId');
  }
  const soClient = rulesClientContext.unsecuredSavedObjectsClient;
  const schedulerSo = configId ? await soClient.get(_saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE, configId) : null;
  if (!schedulerSo) {
    throw new Error('Missing gap_auto_fill_scheduler saved object');
  }
  const soAttrs = schedulerSo.attributes;
  const config = {
    name: soAttrs.name,
    numRetries: soAttrs.numRetries,
    gapFillRange: soAttrs.gapFillRange,
    schedule: soAttrs.schedule,
    maxBackfills: soAttrs.maxBackfills,
    ruleTypes: soAttrs.ruleTypes
  };
  const logEvent = (0, _gap_auto_fill_scheduler_event_log.createGapAutoFillSchedulerEventLogger)({
    eventLogger,
    context: rulesClientContext,
    taskInstance,
    startTime,
    config: soAttrs
  });
  return {
    rulesClient,
    rulesClientContext,
    config,
    logEvent
  };
}
async function checkBackfillCapacity({
  rulesClient,
  maxBackfills,
  logMessage,
  initiatorId
}) {
  try {
    const findRes = await rulesClient.findBackfill({
      page: 1,
      perPage: 1,
      initiator: _constants.backfillInitiator.SYSTEM,
      initiatorId
    });
    const currentCount = findRes.total;
    const remainingCapacity = Math.max(0, maxBackfills - currentCount);
    const canSchedule = remainingCapacity > 0;
    return {
      canSchedule,
      currentCount,
      maxBackfills,
      remainingCapacity
    };
  } catch (e) {
    logMessage(`Failed to check system backfills count: ${e && e.message}`);
    return {
      canSchedule: false,
      currentCount: 0,
      maxBackfills,
      remainingCapacity: maxBackfills
    };
  }
}