"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRepositionOnScroll = exports.createRepositionOnScroll = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Returns the value of the `repositionOnScroll` from the props, component's defaults context
 * or default to `false`.
 *
 * @param args The arguments for `getRepositionOnScroll`. See {@link RepositionOnScrollArgs}.
 * @returns The value of the `repositionOnScroll`.
 */
var getRepositionOnScroll = exports.getRepositionOnScroll = function getRepositionOnScroll(args) {
  var repositionOnScroll = args.repositionOnScroll,
    componentDefaults = args.componentDefaults;
  if (repositionOnScroll !== undefined) return repositionOnScroll;
  if (componentDefaults && 'repositionOnScroll' in componentDefaults && typeof componentDefaults.repositionOnScroll === 'boolean') {
    var contextValue = componentDefaults.repositionOnScroll;
    return contextValue !== null && contextValue !== void 0 ? contextValue : false;
  }
  return false;
};
/**
 * Creates a manager for handling `repositionOnScroll` logic in overlay components.
 * This utility abstracts the adding, updating, and removing of window scroll event listeners.
 *
 * @param getArgs A function that returns the arguments for `getRepositionOnScroll`. See {@link RepositionOnScrollArgs}.
 * @returns An object with `subscribe`, `update`, and `cleanup` methods to manage the scroll listener.
 */
var createRepositionOnScroll = exports.createRepositionOnScroll = function createRepositionOnScroll(getArgs) {
  var lastResolvedRepositionOnScroll;
  var subscribe = function subscribe() {
    var repositionOnScroll = getRepositionOnScroll(getArgs());
    lastResolvedRepositionOnScroll = repositionOnScroll;
    if (repositionOnScroll) {
      window.addEventListener('scroll', getArgs().repositionFn, true);
    }
  };
  var update = function update() {
    var repositionOnScroll = getRepositionOnScroll(getArgs());
    if (lastResolvedRepositionOnScroll !== repositionOnScroll) {
      if (repositionOnScroll) {
        window.addEventListener('scroll', getArgs().repositionFn, true);
      } else {
        window.removeEventListener('scroll', getArgs().repositionFn, true);
      }
      lastResolvedRepositionOnScroll = repositionOnScroll;
    }
  };
  var cleanup = function cleanup() {
    window.removeEventListener('scroll', getArgs().repositionFn, true);
  };
  return {
    subscribe: subscribe,
    update: update,
    cleanup: cleanup
  };
};