"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.securityTelemetry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _api = require("@opentelemetry/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SecurityTelemetry {
  constructor() {
    (0, _defineProperty2.default)(this, "meter", _api.metrics.getMeter('kibana.security'));
    (0, _defineProperty2.default)(this, "loginDuration", void 0);
    (0, _defineProperty2.default)(this, "userProfileActivationDuration", void 0);
    (0, _defineProperty2.default)(this, "sessionCreationDuration", void 0);
    (0, _defineProperty2.default)(this, "logoutCounter", void 0);
    (0, _defineProperty2.default)(this, "privilegeRegistrationDuration", void 0);
    // Adds more boundaries in 50-500ms range where most operations typically fall
    (0, _defineProperty2.default)(this, "DEFAULT_BUCKET_BOUNDARIES", [50, 75, 100, 150, 200, 250, 300, 400, 500, 650, 800, 1000, 1500, 2000, 3000, 5000, 7500, 10000, 20000]);
    // Provides detailed buckets in the 1.5s-4s range where most operations occur
    (0, _defineProperty2.default)(this, "PRIVILEGE_REGISTRATION_BUCKET_BOUNDARIES", [0, 200, 350, 500, 650, 800, 1000, 1500, 2000, 2500, 3000, 3500, 4000, 4500, 5000, 10000, 20000]);
    // Provides detailed buckets in the under 1.5s range where most operations occur
    (0, _defineProperty2.default)(this, "LOGIN_DURATION_BUCKET_BOUNDARIES", [0, 50, 100, 220, 280, 340, 400, 500, 650, 800, 1000, 1500, 2000, 3000, 5000, 7500, 10000, 20000]);
    (0, _defineProperty2.default)(this, "recordLoginDuration", (duration, attributes) => {
      const transformedAttributes = this.transformAttributes(attributes);
      this.loginDuration.record(duration, transformedAttributes);
    });
    (0, _defineProperty2.default)(this, "recordUserProfileActivationDuration", (duration, attributes) => {
      const transformedAttributes = this.transformAttributes(attributes);
      this.userProfileActivationDuration.record(duration, transformedAttributes);
    });
    (0, _defineProperty2.default)(this, "recordSessionCreationDuration", (duration, attributes) => {
      const transformedAttributes = this.transformAttributes(attributes);
      this.sessionCreationDuration.record(duration, transformedAttributes);
    });
    (0, _defineProperty2.default)(this, "recordLogoutAttempt", attributes => {
      const transformedAttributes = this.transformAttributes(attributes);
      this.logoutCounter.add(1, transformedAttributes);
    });
    (0, _defineProperty2.default)(this, "recordPrivilegeRegistrationDuration", (duration, attributes) => {
      const transformedAttributes = this.transformAttributes(attributes);
      this.privilegeRegistrationDuration.record(duration, transformedAttributes);
    });
    this.loginDuration = this.meter.createHistogram('auth.login.duration', {
      description: 'Duration of login attempts',
      unit: 'ms',
      valueType: _api.ValueType.DOUBLE,
      advice: {
        explicitBucketBoundaries: this.LOGIN_DURATION_BUCKET_BOUNDARIES
      }
    });
    this.userProfileActivationDuration = this.meter.createHistogram('auth.user_profile.activation.duration', {
      description: 'Duration of user profile activation attempts',
      unit: 'ms',
      valueType: _api.ValueType.DOUBLE,
      advice: {
        explicitBucketBoundaries: this.DEFAULT_BUCKET_BOUNDARIES
      }
    });
    this.sessionCreationDuration = this.meter.createHistogram('auth.session.creation.duration', {
      description: 'Duration of session creation attempts',
      unit: 'ms',
      valueType: _api.ValueType.DOUBLE,
      advice: {
        explicitBucketBoundaries: this.DEFAULT_BUCKET_BOUNDARIES
      }
    });
    this.logoutCounter = this.meter.createCounter('auth.logout.attempts', {
      description: 'Number of logout attempts',
      unit: '1',
      valueType: _api.ValueType.INT
    });
    this.privilegeRegistrationDuration = this.meter.createHistogram('auth.privilege.registration.duration', {
      description: 'Duration of privilege registration',
      unit: 'ms',
      valueType: _api.ValueType.DOUBLE,
      advice: {
        explicitBucketBoundaries: this.PRIVILEGE_REGISTRATION_BUCKET_BOUNDARIES
      }
    });
  }
  transformAttributes(attributes) {
    const {
      application,
      providerType,
      outcome,
      deletedPrivileges,
      ...rest
    } = attributes;
    const transformed = {
      ...(application ? {
        application
      } : {}),
      ...(deletedPrivileges ? {
        'deleted.privileges': deletedPrivileges
      } : {}),
      ...(providerType ? {
        'auth.provider.type': providerType
      } : {}),
      ...(outcome ? {
        'auth.outcome': outcome
      } : {}),
      ...rest
    };
    return transformed;
  }
}
const securityTelemetry = exports.securityTelemetry = new SecurityTelemetry();