"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerInstallationRoutes = void 0;
var _i18n = require("@kbn/i18n");
var _common = require("../../common");
var _tasks = require("../tasks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerInstallationRoutes = ({
  router,
  getServices
}) => {
  router.post({
    path: _common.INSTALL_API_PATH,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: false,
    options: {
      access: 'internal',
      timeout: {
        idleSocket: 3 * 60 * 1000
      } // install can take time.
    }
  }, async (ctx, req, res) => {
    const {
      sampleDataManager,
      taskManager,
      logger
    } = getServices();
    const sampleType = _common.DatasetSampleType.elasticsearch;
    const core = await ctx.core;
    const esClient = core.elasticsearch.client.asCurrentUser;
    const soClient = core.savedObjects.client;
    try {
      const currentStatus = await sampleDataManager.getSampleDataStatus({
        sampleType,
        esClient,
        soClient
      });
      if (currentStatus.status === _common.InstallationStatus.Installing) {
        return res.ok({
          body: {
            status: currentStatus.status,
            taskId: currentStatus.taskId || ''
          }
        });
      }
      if (currentStatus.status === _common.InstallationStatus.Installed && currentStatus.indexName && currentStatus.dashboardId) {
        return res.ok({
          body: {
            status: currentStatus.status,
            indexName: currentStatus.indexName,
            dashboardId: currentStatus.dashboardId
          }
        });
      }
      const taskId = await (0, _tasks.scheduleInstallSampleDataTask)({
        taskManager,
        logger,
        sampleType
      });
      return res.ok({
        body: {
          status: _common.InstallationStatus.Installing,
          taskId
        }
      });
    } catch (e) {
      var _e$meta, _e$meta2;
      logger.error('Failed to schedule sample data installation', e);
      return res.customError({
        statusCode: e !== null && e !== void 0 && e.meta && (_e$meta = e.meta) !== null && _e$meta !== void 0 && _e$meta.statusCode ? (_e$meta2 = e.meta) === null || _e$meta2 === void 0 ? void 0 : _e$meta2.statusCode : 500,
        body: {
          message: _i18n.i18n.translate('xpack.sampleDataIngest.server.installSample.errorMessage', {
            defaultMessage: 'Failed to install sample data due to an exception.'
          })
        }
      });
    }
  });
  router.get({
    path: _common.STATUS_API_PATH,
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, async (ctx, req, res) => {
    const {
      sampleDataManager
    } = getServices();
    const core = await ctx.core;
    const esClient = core.elasticsearch.client.asCurrentUser;
    const soClient = core.savedObjects.client;
    try {
      const statusData = await sampleDataManager.getSampleDataStatus({
        sampleType: _common.DatasetSampleType.elasticsearch,
        esClient,
        soClient
      });
      return res.ok({
        body: {
          status: statusData.status,
          indexName: statusData.indexName,
          dashboardId: statusData.dashboardId,
          taskId: statusData.taskId,
          error: statusData.error
        }
      });
    } catch (e) {
      var _e$meta3, _e$meta4;
      return res.customError({
        statusCode: e !== null && e !== void 0 && e.meta && (_e$meta3 = e.meta) !== null && _e$meta3 !== void 0 && _e$meta3.statusCode ? (_e$meta4 = e.meta) === null || _e$meta4 === void 0 ? void 0 : _e$meta4.statusCode : 500,
        body: {
          message: _i18n.i18n.translate('xpack.sampleDataIngest.server.getStatus.errorMessage', {
            defaultMessage: 'Failed to get status due to an exception.'
          })
        }
      });
    }
  });
};
exports.registerInstallationRoutes = registerInstallationRoutes;