"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getBulkAgentDetailsRoute = void 0;
var _zod = require("@kbn/zod");
var _zodHelpers = require("@kbn/zod-helpers");
var _common = require("@kbn/spaces-plugin/common");
var _constants = require("../../../common/constants");
var _common2 = require("../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Request body schema for bulk agent details endpoint.
 * Fetches agent metadata (names, IDs) for multiple agents in a single request.
 */
const GetBulkAgentDetailsRequestBody = _zod.z.object({
  agentIds: _zod.z.array(_zod.z.string()).min(1).max(1000) // Limit to 1000 agents per request (supports up to 10 pages of 100)
});
const getBulkAgentDetailsRoute = (router, osqueryContext) => {
  router.versioned.post({
    access: 'internal',
    path: '/internal/osquery/fleet_wrapper/agents/_bulk',
    security: {
      authz: {
        requiredPrivileges: [`${_common2.PLUGIN_ID}-read`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(GetBulkAgentDetailsRequestBody)
      }
    }
  }, async (context, request, response) => {
    const logger = osqueryContext.logFactory.get('bulkAgentDetails');
    const space = await osqueryContext.service.getActiveSpace(request);

    // Deduplicate agent IDs to prevent redundant queries
    const agentIds = [...new Set(request.body.agentIds)];
    try {
      var _agentService$asInter, _space$id;
      // Use Fleet's bulk getByIds method (with ignoreMissing to handle deleted agents)
      const agentService = osqueryContext.service.getAgentService();
      if (!agentService) {
        return response.ok({
          body: {
            agents: []
          }
        });
      }
      const agents = await ((_agentService$asInter = agentService.asInternalScopedUser((_space$id = space === null || space === void 0 ? void 0 : space.id) !== null && _space$id !== void 0 ? _space$id : _common.DEFAULT_SPACE_ID)) === null || _agentService$asInter === void 0 ? void 0 : _agentService$asInter.getByIds(agentIds, {
        ignoreMissing: true
      }));
      return response.ok({
        body: {
          agents: agents || []
        }
      });
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : String(err);
      logger.error(`Failed to fetch bulk agent details: ${errorMessage}`);
      return response.customError({
        statusCode: 500,
        body: {
          message: 'Failed to fetch agent details'
        }
      });
    }
  });
};
exports.getBulkAgentDetailsRoute = getBulkAgentDetailsRoute;