"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateConversation$ = exports.placeholderConversation = exports.getConversation = exports.createConversation$ = exports.conversationExists = void 0;
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _events = require("./events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Persist a new conversation and emit the corresponding event
 */
const createConversation$ = ({
  agentId,
  conversationClient,
  conversationId,
  title$,
  roundCompletedEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    title: title$,
    roundCompletedEvent: roundCompletedEvents$
  }).pipe((0, _rxjs.switchMap)(({
    title,
    roundCompletedEvent
  }) => {
    return conversationClient.create({
      id: conversationId,
      title,
      agent_id: agentId,
      rounds: [roundCompletedEvent.data.round]
    });
  }), (0, _rxjs.switchMap)(createdConversation => {
    return (0, _rxjs.of)((0, _events.createConversationCreatedEvent)(createdConversation));
  }));
};

/**
 * Update an existing conversation and emit the corresponding event
 */
exports.createConversation$ = createConversation$;
const updateConversation$ = ({
  conversationClient,
  conversation,
  title$,
  roundCompletedEvents$
}) => {
  return (0, _rxjs.forkJoin)({
    title: title$,
    roundCompletedEvent: roundCompletedEvents$
  }).pipe((0, _rxjs.switchMap)(({
    title,
    roundCompletedEvent
  }) => {
    return conversationClient.update({
      id: conversation.id,
      title,
      rounds: [...conversation.rounds, roundCompletedEvent.data.round]
    });
  }), (0, _rxjs.switchMap)(updatedConversation => {
    return (0, _rxjs.of)((0, _events.createConversationUpdatedEvent)(updatedConversation));
  }));
};

/**
 * Check if a conversation exists
 */
exports.updateConversation$ = updateConversation$;
const conversationExists = async ({
  conversationId,
  conversationClient
}) => {
  return conversationClient.exists(conversationId);
};
exports.conversationExists = conversationExists;
/**
 * Get a conversation by ID, or create a placeholder for new conversations
 * Also determines the operation type (CREATE or UPDATE) based on the same logic
 */
const getConversation = async ({
  agentId,
  conversationId,
  autoCreateConversationWithId = false,
  conversationClient
}) => {
  // Case 1: No conversation ID - create new with placeholder
  if (!conversationId) {
    return {
      ...placeholderConversation({
        agentId
      }),
      operation: 'CREATE'
    };
  }

  // Case 2: Conversation ID specified and autoCreate is false - update existing
  if (!autoCreateConversationWithId) {
    return {
      ...(await conversationClient.get(conversationId)),
      operation: 'UPDATE'
    };
  }

  // Case 3: Conversation ID specified and autoCreate is true - check if exists
  const exists = await conversationExists({
    conversationId,
    conversationClient
  });
  if (exists) {
    return {
      ...(await conversationClient.get(conversationId)),
      operation: 'UPDATE'
    };
  } else {
    return {
      ...placeholderConversation({
        conversationId,
        agentId
      }),
      operation: 'CREATE'
    };
  }
};
exports.getConversation = getConversation;
const placeholderConversation = ({
  agentId,
  conversationId
}) => {
  return {
    id: conversationId !== null && conversationId !== void 0 ? conversationId : (0, _uuid.v4)(),
    title: 'New conversation',
    agent_id: agentId,
    rounds: [],
    updated_at: new Date().toISOString(),
    created_at: new Date().toISOString(),
    user: {
      id: 'unknown',
      username: 'unknown'
    }
  };
};
exports.placeholderConversation = placeholderConversation;