"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatResearcherActionHistory = exports.formatAnswerActionHistory = void 0;
var _messages = require("@kbn/onechat-genai-utils/langchain/messages");
var _prompts = require("@kbn/onechat-genai-utils/prompts");
var _agents = require("@kbn/onechat-common/agents");
var _actions = require("../actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatResearcherActionHistory = ({
  actions
}) => {
  const formatted = [];
  for (let i = 0; i < actions.length; i++) {
    const action = actions[i];
    if ((0, _actions.isToolCallAction)(action)) {
      // in case of forceful handover, we have a tool_call action without the corresponding tool result
      // so we want to skip it because we need a [ai, user, ai, user, ...] flow
      if (i === actions.length - 1 || !(0, _actions.isExecuteToolAction)(actions[i + 1])) {
        continue;
      }
      formatted.push((0, _messages.createToolCallMessage)(action.tool_calls));
    }
    if ((0, _actions.isExecuteToolAction)(action)) {
      formatted.push(...action.tool_results.map(result => (0, _messages.createToolResultMessage)({
        content: result.content,
        toolCallId: result.toolCallId
      })));
    }
    if ((0, _actions.isHandoverAction)(action)) {
      // returns a single [AI, user] tuple
      formatted.push(...formatHandoverAction(action));
    }
    if ((0, _actions.isAgentErrorAction)(action)) {
      // returns a single [AI, user] tuple
      formatted.push(...formatErrorAction(action));
    }
  }
  return formatted;
};
exports.formatResearcherActionHistory = formatResearcherActionHistory;
const formatAnswerActionHistory = ({
  actions
}) => {
  const formatted = [];
  for (let i = 0; i < actions.length; i++) {
    const action = actions[i];
    if ((0, _actions.isAgentErrorAction)(action)) {
      // returns a single [AI, user] tuple
      formatted.push(...formatErrorAction(action));
    }
    // [...] we don't need to format AnswerAction because it will terminate the execution
  }
  return formatted;
};
exports.formatAnswerActionHistory = formatAnswerActionHistory;
const formatHandoverAction = ({
  message,
  forceful
}) => {
  if (forceful) {
    return [(0, _messages.createAIMessage)((0, _prompts.cleanPrompt)(`[researcher agent] The research process was interrupted because it exceeded the maximum allowed steps, I cannot perform any more actions.
        Handing over to the answering agent for a final answer based on the information gathered so far.`)), (0, _messages.createUserMessage)((0, _prompts.cleanPrompt)('[dispatcher] Ack. Forwarding to answering agent. Please proceed to respond without invoking any additional tools, using only the information gathered so far.'))];
  } else {
    return [(0, _messages.createAIMessage)((0, _prompts.cleanPrompt)(`[researcher agent] Finished the research step. Handover notes for the answering agent:
        """
        ${message}
        """`)), (0, _messages.createUserMessage)((0, _prompts.cleanPrompt)('[dispatcher] Ack. Forwarding to answering agent. Please proceed to respond without invoking any additional tools, using only the information gathered so far.'))];
  }
};
const formatErrorAction = ({
  error
}) => {
  // tool not found -> we format that as a tool call returning an error.
  if (isExecutionError(error, _agents.AgentExecutionErrorCode.toolNotFound)) {
    const toolCallId = (0, _messages.generateFakeToolCallId)();
    const callArgs = typeof error.meta.toolArgs === 'string' ? {
      args: error.meta.toolArgs
    } : error.meta.toolArgs;
    return [(0, _messages.createToolCallMessage)({
      toolCallId,
      toolName: error.meta.toolName,
      args: callArgs
    }), (0, _messages.createToolResultMessage)({
      toolCallId,
      content: `ERROR: called a tool which was not available - ${error.message}`
    })];
  }

  // tool call validation -> we format that as a tool call returning an error.
  if (isExecutionError(error, _agents.AgentExecutionErrorCode.toolValidationError)) {
    const toolCallId = (0, _messages.generateFakeToolCallId)();
    const callArgs = typeof error.meta.toolArgs === 'string' ? {
      args: error.meta.toolArgs
    } : error.meta.toolArgs;
    return [(0, _messages.createToolCallMessage)({
      toolCallId,
      toolName: error.meta.toolName,
      args: callArgs
    }), (0, _messages.createToolResultMessage)({
      toolCallId,
      content: `ERROR: called a tool which was not available - ${error.meta.validationError} ${error.message}`
    })];
  }

  // empty response -> we format that as an empty AI message and user message asking to try again.
  if (isExecutionError(error, _agents.AgentExecutionErrorCode.emptyResponse)) {
    return [(0, _messages.createAIMessage)(``), (0, _messages.createUserMessage)('Looks like you did not provide any answer. Please try again.')];
  }

  // other error types are not recoverable -> we do not represent them.
  return [];
};
const isExecutionError = (error, code) => {
  return error.meta.errCode === code;
};