"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerToolsRoutes = registerToolsRoutes;
var _configSchema = require("@kbn/config-schema");
var _nodePath = _interopRequireDefault(require("node:path"));
var _onechatCommon = require("@kbn/onechat-common");
var _wrap_handler = require("./wrap_handler");
var _tool_conversion = require("../services/tools/utils/tool_conversion");
var _features = require("../../common/features");
var _constants = require("../../common/constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerToolsRoutes({
  router,
  getInternalServices,
  logger
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // list tools API
  router.versioned.get({
    path: `${_constants.publicApiPath}/tools`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'List tools',
    description: 'List all available tools. Use this endpoint to retrieve complete tool definitions including their schemas and configuration requirements.',
    options: {
      tags: ['tools', 'oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: false,
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/tools_list.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const tools = await registry.list({});
    return response.ok({
      body: {
        results: tools.map(_tool_conversion.toDescriptor)
      }
    });
  }));

  // get tool by ID
  router.versioned.get({
    path: `${_constants.publicApiPath}/tools/{toolId}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'Get a tool by id',
    description: 'Get a specific tool by ID. Use this endpoint to retrieve the complete tool definition including its schema and configuration requirements.',
    options: {
      tags: ['tools', 'oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          toolId: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the tool to retrieve.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/tools_get_by_id.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      toolId
    } = request.params;
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const tool = await registry.get(toolId);
    return response.ok({
      body: await (0, _tool_conversion.toDescriptorWithSchema)(tool)
    });
  }));

  // create tool
  router.versioned.post({
    path: `${_constants.publicApiPath}/tools`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    },
    access: 'public',
    summary: 'Create a tool',
    description: 'Create a new tool. Use this endpoint to define a custom tool with specific functionality and configuration for use by agents.',
    options: {
      tags: ['tools', 'oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'Unique identifier for the tool.'
            }
          }),
          type: _configSchema.schema.oneOf(
          // @ts-expect-error TS2769: No overload matches this call
          _onechatCommon.editableToolTypes.map(type => _configSchema.schema.literal(type)), {
            meta: {
              description: 'The type of tool to create (e.g., esql, index_search).'
            }
          }),
          description: _configSchema.schema.string({
            defaultValue: '',
            meta: {
              description: 'Description of what the tool does.'
            }
          }),
          tags: _configSchema.schema.arrayOf(_configSchema.schema.string({
            meta: {
              description: 'Tag for categorizing the tool.'
            }
          }), {
            defaultValue: [],
            meta: {
              description: 'Optional tags for categorizing and organizing tools.'
            }
          }),
          // actual config validation is done in the tool service
          configuration: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
            meta: {
              description: 'Tool-specific configuration parameters. See examples for details.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/tools_create.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tools: toolService
    } = getInternalServices();
    const createRequest = request.body;
    const registry = await toolService.getRegistry({
      request
    });
    const tool = await registry.create(createRequest);
    return response.ok({
      body: await (0, _tool_conversion.toDescriptorWithSchema)(tool)
    });
  }));

  // update tool
  router.versioned.put({
    path: `${_constants.publicApiPath}/tools/{toolId}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    },
    access: 'public',
    summary: 'Update a tool',
    description: "Update an existing tool. Use this endpoint to modify any aspect of the tool's configuration or metadata.",
    options: {
      tags: ['tools', 'oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          toolId: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the tool to update.'
            }
          })
        }),
        body: _configSchema.schema.object({
          description: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Updated description of what the tool does.'
            }
          })),
          tags: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
            meta: {
              description: 'Updated tag for categorizing the tool.'
            }
          }), {
            meta: {
              description: 'Updated tags for categorizing and organizing tools.'
            }
          })),
          // actual config validation is done in the tool service
          configuration: _configSchema.schema.maybe(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
            meta: {
              description: 'Updated tool-specific configuration parameters. See examples for details.'
            }
          }))
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/tools_update.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tools: toolService
    } = getInternalServices();
    const {
      toolId
    } = request.params;
    const update = request.body;
    const registry = await toolService.getRegistry({
      request
    });
    const tool = await registry.update(toolId, update);
    return response.ok({
      body: await (0, _tool_conversion.toDescriptorWithSchema)(tool)
    });
  }));

  // delete tool
  router.versioned.delete({
    path: `${_constants.publicApiPath}/tools/{toolId}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    },
    access: 'public',
    summary: 'Delete a tool',
    description: 'Delete a tool by ID. This action cannot be undone.',
    options: {
      tags: ['tools', 'oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          toolId: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the tool to delete.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/tools_delete.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      toolId
    } = request.params;
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const success = await registry.delete(toolId);
    return response.ok({
      body: {
        success
      }
    });
  }));
  // execute a tool
  router.versioned.post({
    path: `${_constants.publicApiPath}/tools/_execute`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'Execute a Tool',
    description: 'Execute a tool with parameters. Use this endpoint to run a tool directly with specified inputs and optional external connector integration.',
    options: {
      timeout: {
        idleSocket: _utils.AGENT_SOCKET_TIMEOUT_MS
      },
      tags: ['tools', 'oas-tag:agent builder'],
      availability: {
        stability: 'experimental',
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: _configSchema.schema.object({
          tool_id: _configSchema.schema.string({
            meta: {
              description: 'The ID of the tool to execute.'
            }
          }),
          tool_params: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
            meta: {
              description: 'Parameters to pass to the tool execution. See examples for details'
            }
          }),
          connector_id: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Optional connector ID for tools that require external integrations.'
            }
          }))
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/tools_execute.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      tool_id: id,
      tool_params: toolParams,
      connector_id: defaultConnectorId
    } = request.body;
    const {
      tools: toolService
    } = getInternalServices();
    const registry = await toolService.getRegistry({
      request
    });
    const tool = await registry.get(id);
    const toolSchema = await tool.getSchema();
    const validation = toolSchema.safeParse(toolParams);
    if (validation.error) {
      return response.badRequest({
        body: {
          message: `Invalid parameters: ${validation.error.message}`
        }
      });
    }
    const toolResult = await registry.execute({
      toolId: id,
      toolParams,
      defaultConnectorId
    });
    return response.ok({
      body: {
        results: toolResult.results
      }
    });
  }));
}