"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.esqlFormValidationSchema = void 0;
var _esqlUtils = require("@kbn/esql-utils");
var _esqlValidationAutocomplete = require("@kbn/esql-validation-autocomplete");
var _i18n = require("@kbn/i18n");
var _tools = require("@kbn/onechat-common/tools");
var _zod = require("@kbn/zod");
var _shared_tool_validation = require("./shared_tool_validation");
var _tool_form_types = require("../types/tool_form_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const esqlI18nMessages = {
  // Specific errors will be provided by the ES|QL editor
  esql: {
    requiredError: _i18n.i18n.translate('xpack.onechat.tools.newTool.validation.esql.requiredError', {
      defaultMessage: 'ES|QL query is required.'
    }),
    esqlError: _i18n.i18n.translate('xpack.onechat.tools.newTool.validation.esql.esqlError', {
      defaultMessage: 'Please fix the errors in your ES|QL query.'
    })
  },
  params: {
    nameRequiredError: _i18n.i18n.translate('xpack.onechat.tools.newTool.validation.params.nameRequiredError', {
      defaultMessage: 'Parameter name is required.'
    }),
    nameFormatError: _i18n.i18n.translate('xpack.onechat.tools.newTool.validation.params.nameFormatError', {
      defaultMessage: 'Parameter name must start with a letter or underscore and contain only letters, numbers, and underscores.'
    }),
    descriptionRequiredError: _i18n.i18n.translate('xpack.onechat.tools.newTool.validation.params.descriptionRequiredError', {
      defaultMessage: 'Parameter description is required.'
    }),
    duplicateError: name => _i18n.i18n.translate('xpack.onechat.tools.newTool.validation.params.duplicateError', {
      defaultMessage: 'Duplicate parameter: "{name}".',
      values: {
        name
      }
    })
  }
};
const esqlFormValidationSchema = exports.esqlFormValidationSchema = _zod.z.object({
  // Use shared validation schemas for common fields
  toolId: _shared_tool_validation.sharedValidationSchemas.toolId,
  description: _shared_tool_validation.sharedValidationSchemas.description,
  labels: _shared_tool_validation.sharedValidationSchemas.labels,
  // ESQL specific validation
  esql: _zod.z.string().min(1, {
    message: esqlI18nMessages.esql.requiredError
  }).refine(async esql => {
    const result = await (0, _esqlValidationAutocomplete.validateQuery)(esql);
    return result.errors.length === 0;
  }, {
    message: esqlI18nMessages.esql.esqlError
  }),
  params: _zod.z.array(_zod.z.object({
    name: _zod.z.string().min(1, {
      message: esqlI18nMessages.params.nameRequiredError
    }).regex(/^[a-zA-Z_][a-zA-Z0-9_]*$/, {
      message: esqlI18nMessages.params.nameFormatError
    }),
    description: _zod.z.string().min(1, {
      message: esqlI18nMessages.params.descriptionRequiredError
    }),
    type: _zod.z.custom(data => Object.values(_tools.EsqlToolFieldType).includes(data)),
    source: _zod.z.nativeEnum(_tool_form_types.EsqlParamSource),
    optional: _zod.z.boolean()
  })).superRefine((params, ctx) => {
    params.forEach(({
      name
    }, index) => {
      const otherParamNames = new Set(params.filter((_, i) => i !== index).map(param => param.name));
      if (otherParamNames.has(name)) {
        ctx.addIssue({
          code: _zod.z.ZodIssueCode.custom,
          message: esqlI18nMessages.params.duplicateError(name),
          path: [index, 'name']
        });
      }
    });
  }),
  type: _zod.z.literal(_tools.ToolType.esql)
}).superRefine(({
  esql,
  params
}, ctx) => {
  const inferredParams = (0, _esqlUtils.getESQLQueryVariables)(esql);
  const definedParams = new Set(params.map(param => param.name));
  for (const param of inferredParams) {
    if (!definedParams.has(param)) {
      ctx.addIssue({
        code: _zod.z.ZodIssueCode.custom,
        message: esqlI18nMessages.esql.esqlError,
        path: ['esql']
      });
      return;
    }
  }
});