"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isRegressionAnalysis = exports.isOutlierAnalysis = exports.isClassificationAnalysis = exports.getTrainingPercent = exports.getPredictionFieldName = exports.getPredictedFieldName = exports.getDependentVar = exports.getDefaultPredictionFieldName = exports.getAnalysisType = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Type guard for DFA outlier analysis configurations
 *
 * @param {unknown} arg The config to identify
 * @returns {arg is OutlierAnalysis}
 */
const isOutlierAnalysis = arg => {
  if (typeof arg !== 'object' || arg === null) return false;
  const keys = Object.keys(arg);
  return keys.length === 1 && keys[0] === _constants.ANALYSIS_CONFIG_TYPE.OUTLIER_DETECTION;
};

/**
 * Type guard for DFA regression analysis configurations
 *
 * @param {unknown} arg The config to identify
 * @returns {arg is RegressionAnalysis}
 */
exports.isOutlierAnalysis = isOutlierAnalysis;
const isRegressionAnalysis = arg => {
  if (typeof arg !== 'object' || arg === null) return false;
  const keys = Object.keys(arg);
  return keys.length === 1 && keys[0] === _constants.ANALYSIS_CONFIG_TYPE.REGRESSION;
};

/**
 * Type guard for DFA classification analysis configurations
 *
 * @param {unknown} arg The config to identify
 * @returns {arg is ClassificationAnalysis}
 */
exports.isRegressionAnalysis = isRegressionAnalysis;
const isClassificationAnalysis = arg => {
  if (typeof arg !== 'object' || arg === null) return false;
  const keys = Object.keys(arg);
  return keys.length === 1 && keys[0] === _constants.ANALYSIS_CONFIG_TYPE.CLASSIFICATION;
};

/**
 * Helper function to get the dependent variable of a DFA configuration
 *
 * @param {AnalysisConfig} analysis The analysis configuration
 * @returns {(| RegressionAnalysis['regression']['dependent_variable'] | ClassificationAnalysis['classification']['dependent_variable'])}
 */
exports.isClassificationAnalysis = isClassificationAnalysis;
const getDependentVar = analysis => {
  let depVar = '';
  if (isRegressionAnalysis(analysis)) {
    depVar = analysis.regression.dependent_variable;
  }
  if (isClassificationAnalysis(analysis)) {
    depVar = analysis.classification.dependent_variable;
  }
  return depVar;
};

/**
 * Helper function to get the prediction field name of a DFA configuration
 *
 * @param {AnalysisConfig} analysis The analysis configuration
 * @returns {(| RegressionAnalysis['regression']['prediction_field_name'] | ClassificationAnalysis['classification']['prediction_field_name'])}
 */
exports.getDependentVar = getDependentVar;
const getPredictionFieldName = analysis => {
  // If undefined will be defaulted to dependent_variable when config is created
  let predictionFieldName;
  if (isRegressionAnalysis(analysis) && analysis.regression.prediction_field_name !== undefined) {
    predictionFieldName = analysis.regression.prediction_field_name;
  } else if (isClassificationAnalysis(analysis) && analysis.classification.prediction_field_name !== undefined) {
    predictionFieldName = analysis.classification.prediction_field_name;
  }
  return predictionFieldName;
};

/**
 * Helper function to get the default prediction field name
 *
 * @param {AnalysisConfig} analysis The analysis configuration
 * @returns {string}
 */
exports.getPredictionFieldName = getPredictionFieldName;
const getDefaultPredictionFieldName = analysis => {
  return `${getDependentVar(analysis)}_prediction`;
};

/**
 * Helper function to get the predicted field name
 *
 * @param {string} resultsField
 * @param {AnalysisConfig} analysis The analysis configuration
 * @param {?boolean} [forSort]
 * @returns {string}
 */
exports.getDefaultPredictionFieldName = getDefaultPredictionFieldName;
const getPredictedFieldName = (resultsField, analysis, forSort) => {
  // default is 'ml'
  const predictionFieldName = getPredictionFieldName(analysis);
  const predictedField = `${resultsField}.${predictionFieldName ? predictionFieldName : getDefaultPredictionFieldName(analysis)}`;
  return predictedField;
};

/**
 * Helper function to get the analysis type of a DFA configuration
 *
 * @param {AnalysisConfig} analysis The analysis configuration
 * @returns {(DataFrameAnalysisConfigType | 'unknown')}
 */
exports.getPredictedFieldName = getPredictedFieldName;
const getAnalysisType = analysis => {
  const keys = Object.keys(analysis || {});
  if (keys.length === 1) {
    return keys[0];
  }
  return 'unknown';
};

/**
 * Helper function to get the training percent of a DFA configuration
 *
 * @param {AnalysisConfig} analysis The analysis configuration
 * @returns {(| RegressionAnalysis['regression']['training_percent'] | ClassificationAnalysis['classification']['training_percent'] | undefined)}
 */
exports.getAnalysisType = getAnalysisType;
const getTrainingPercent = analysis => {
  let trainingPercent;
  if (isRegressionAnalysis(analysis)) {
    trainingPercent = analysis.regression.training_percent;
  }
  if (isClassificationAnalysis(analysis)) {
    trainingPercent = analysis.classification.training_percent;
  }
  return trainingPercent;
};
exports.getTrainingPercent = getTrainingPercent;