"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsSearchAPIRoute = void 0;
var _boom = require("@hapi/boom");
var _constants = require("../../../../common/constants");
var _schema = require("./schema");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsSearchAPIRoute = (router, {
  contentManagement,
  builder
}) => {
  const searchRoute = router.get({
    path: _constants.LENS_VIS_API_PATH,
    access: _constants.LENS_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Search Lens visualizations',
    description: 'Get list of Lens visualizations.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  searchRoute.addVersion({
    version: _constants.LENS_API_VERSION,
    validate: {
      request: {
        query: _schema.lensSearchRequestQuerySchema
      },
      response: {
        200: {
          body: () => _schema.lensSearchResponseBodySchema,
          description: 'Ok'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    // TODO fix IContentClient to type this client based on the actual
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_constants.LENS_CONTENT_TYPE);
    const {
      query: q,
      page,
      perPage,
      ...reqOptions
    } = req.query;
    try {
      // Note: these types are to enforce loose param typings of client methods
      const query = {
        text: q,
        cursor: page.toString(),
        limit: perPage
      };
      const options = reqOptions;
      const {
        result: {
          hits,
          pagination
        }
      } = await client.search(query, options);

      // TODO: see if this check is actually needed
      const error = hits.find(item => item.error);
      if (error) {
        throw error;
      }
      return res.ok({
        body: {
          data: hits.map(item => {
            return (0, _utils.getLensResponseItem)(builder, item);
          }),
          meta: {
            page,
            perPage,
            total: pagination.total
          }
        }
      });
    } catch (error) {
      if ((0, _boom.isBoom)(error) && error.output.statusCode === 403) {
        return res.forbidden();
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
  });
};
exports.registerLensVisualizationsSearchAPIRoute = registerLensVisualizationsSearchAPIRoute;