"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAndIntegrateCloudConnector = createAndIntegrateCloudConnector;
var _errors = require("../../errors");
var _cloud_connector = require("../cloud_connector");
var _cloud_connector2 = require("../secrets/cloud_connector");
var _integration_helpers = require("./integration_helpers");
var _lifecycle = require("./lifecycle");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Result of cloud connector integration with a package policy
 */

/**
 * Creates or reuses a cloud connector and integrates it with a package policy for agentless workflows
 * Handles the complete integration flow:
 * - Checks if cloud connectors are enabled
 * - If package policy has cloud_connector_id: reuses existing connector and increments usage count
 * - If no cloud_connector_id: extracts and creates secrets, creates new cloud connector
 * - Updates package policy with secret references and cloud connector ID
 *
 * This function is designed for the agentless policy API but can be reused
 * in other contexts where cloud connectors need to be integrated with package policies.
 *
 * @param params - Integration parameters
 * @returns Updated package policy, cloud connector ID, and whether it was created or reused
 * @throws CloudConnectorCreateError if cloud connector creation or reuse fails
 */
async function createAndIntegrateCloudConnector(params) {
  var _agentPolicy$agentles, _agentPolicy$agentles2, _agentPolicy$agentles3, _agentPolicy$agentles4;
  const {
    packagePolicy,
    agentPolicy,
    policyName,
    soClient,
    esClient,
    logger,
    cloudConnectorName: providedCloudConnectorName
  } = params;

  // Check if cloud connectors are enabled for this agentless policy
  const cloudProvider = (_agentPolicy$agentles = agentPolicy.agentless) === null || _agentPolicy$agentles === void 0 ? void 0 : (_agentPolicy$agentles2 = _agentPolicy$agentles.cloud_connectors) === null || _agentPolicy$agentles2 === void 0 ? void 0 : _agentPolicy$agentles2.target_csp;
  const cloudConnectorsEnabled = (_agentPolicy$agentles3 = agentPolicy.agentless) === null || _agentPolicy$agentles3 === void 0 ? void 0 : (_agentPolicy$agentles4 = _agentPolicy$agentles3.cloud_connectors) === null || _agentPolicy$agentles4 === void 0 ? void 0 : _agentPolicy$agentles4.enabled;
  if (!cloudConnectorsEnabled || !cloudProvider) {
    logger.debug('Cloud connectors not enabled for this policy');
    return {
      packagePolicy,
      wasCreated: false
    };
  }
  logger.debug(`Cloud connectors enabled for agentless policy with provider: ${cloudProvider}`);

  // Set supports_cloud_connector flag before extracting vars
  let updatedPackagePolicy = {
    ...packagePolicy,
    supports_cloud_connector: true
  };

  // Check if user provided an existing cloud connector ID to reuse
  const existingCloudConnectorId = updatedPackagePolicy.cloud_connector_id;
  if (existingCloudConnectorId) {
    logger.debug(`Reusing existing cloud connector: ${existingCloudConnectorId}`);
    try {
      // Validate the cloud connector exists and matches the provider
      const existingConnector = await _cloud_connector.cloudConnectorService.getById(soClient, existingCloudConnectorId);
      if (existingConnector.cloudProvider !== cloudProvider) {
        throw new _errors.CloudConnectorCreateError(`Cloud connector ${existingCloudConnectorId} is for ${existingConnector.cloudProvider} but policy requires ${cloudProvider}`);
      }

      // Increment the usage count
      await (0, _lifecycle.incrementCloudConnectorPackageCount)(soClient, existingCloudConnectorId, logger);
      logger.info(`Successfully reused cloud connector: ${existingCloudConnectorId}`);
      return {
        packagePolicy: updatedPackagePolicy,
        cloudConnectorId: existingCloudConnectorId,
        wasCreated: false
      };
    } catch (error) {
      logger.error(`Error reusing cloud connector ${existingCloudConnectorId}: ${error}`);
      throw new _errors.CloudConnectorCreateError(`Failed to reuse cloud connector ${existingCloudConnectorId}: ${error.message}`);
    }
  }

  // No existing cloud connector ID provided, create a new one
  // Extract cloud connector vars and create secrets for them
  // This decouples cloud connector secret handling from extractAndWriteSecrets
  const cloudConnectorVars = await (0, _cloud_connector2.extractAndCreateCloudConnectorSecrets)(cloudProvider, updatedPackagePolicy, esClient, logger);
  if (!cloudConnectorVars) {
    logger.debug('No cloud connector vars found or incomplete');
    return {
      packagePolicy: updatedPackagePolicy,
      wasCreated: false
    };
  }
  logger.debug('Creating new cloud connector for agentless policy with secret references');

  // Use provided cloud connector name from API request if available,
  // otherwise extract from package policy or generate default
  const cloudConnectorName = providedCloudConnectorName || (0, _integration_helpers.getCloudConnectorNameFromPackagePolicy)(updatedPackagePolicy, cloudProvider, policyName);
  try {
    const cloudConnector = await _cloud_connector.cloudConnectorService.create(soClient, {
      name: cloudConnectorName,
      vars: cloudConnectorVars,
      cloudProvider
    });
    logger.info(`Successfully created cloud connector: ${cloudConnector.id}`);

    // Update package policy inputs with the secret references from cloud connector vars
    // This ensures that when extractAndWriteSecrets is called, it sees these as existing secrets
    updatedPackagePolicy = (0, _integration_helpers.updatePackagePolicyWithCloudConnectorSecrets)(updatedPackagePolicy, cloudConnectorVars, cloudProvider);

    // Set cloud connector ID on package policy
    updatedPackagePolicy = {
      ...updatedPackagePolicy,
      cloud_connector_id: cloudConnector.id
    };
    return {
      packagePolicy: updatedPackagePolicy,
      cloudConnectorId: cloudConnector.id,
      wasCreated: true
    };
  } catch (error) {
    logger.error(`Error creating cloud connector: ${error}`);
    throw new _errors.CloudConnectorCreateError(`${error}`);
  }
}