"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._joinFilters = _joinFilters;
exports.bulkUpdateAgents = bulkUpdateAgents;
exports.closePointInTime = closePointInTime;
exports.deleteAgent = deleteAgent;
exports.fetchAllAgentsByKuery = fetchAllAgentsByKuery;
exports.getAgentByAccessAPIKeyId = getAgentByAccessAPIKeyId;
exports.getAgentById = getAgentById;
exports.getAgentPolicyForAgent = getAgentPolicyForAgent;
exports.getAgentPolicyForAgents = getAgentPolicyForAgents;
exports.getAgentTags = getAgentTags;
exports.getAgentVersionsForAgentPolicyIds = getAgentVersionsForAgentPolicyIds;
exports.getAgents = getAgents;
exports.getAgentsById = getAgentsById;
exports.getAgentsByKuery = getAgentsByKuery;
exports.getAllAgentsByKuery = getAllAgentsByKuery;
exports.getByIds = void 0;
exports.openPointInTime = openPointInTime;
exports.removeSOAttributes = removeSOAttributes;
exports.updateAgent = updateAgent;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/spaces-plugin/common");
var _ = require("..");
var _constants = require("../../../common/constants");
var _common2 = require("../../../common");
var _services = require("../../../common/services");
var _constants2 = require("../../constants");
var _errors = require("../../errors");
var _audit_logging = require("../audit_logging");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _helpers = require("../spaces/helpers");
var _agent_namespaces = require("../spaces/agent_namespaces");
var _query_namespaces_filtering = require("../spaces/query_namespaces_filtering");
var _create_es_search_iterable = require("../utils/create_es_search_iterable");
var _retry = require("../epm/elasticsearch/retry");
var _helpers2 = require("./helpers");
var _build_status_runtime_field = require("./build_status_runtime_field");
var _versions = require("./versions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INACTIVE_AGENT_CONDITION = `status:inactive`;
const ACTIVE_AGENT_CONDITION = `NOT (${INACTIVE_AGENT_CONDITION})`;
const ENROLLED_AGENT_CONDITION = `NOT status:unenrolled`;
const includeUnenrolled = kuery => (kuery === null || kuery === void 0 ? void 0 : kuery.toLowerCase().includes('status:*')) || (kuery === null || kuery === void 0 ? void 0 : kuery.toLowerCase().includes('status:unenrolled'));
function _joinFilters(filters) {
  try {
    return filters.filter(filter => filter !== undefined).reduce((acc, kuery) => {
      if (kuery === undefined) {
        return acc;
      }
      const kueryNode = typeof kuery === 'string' ? (0, _esQuery.fromKueryExpression)(removeSOAttributes(kuery)) : kuery;
      if (!acc) {
        return kueryNode;
      }
      return {
        type: 'function',
        function: 'and',
        arguments: [acc, kueryNode]
      };
    }, undefined);
  } catch (err) {
    throw new _errors.FleetError(`Kuery is malformed: ${err.message}`);
  }
}
function removeSOAttributes(kuery) {
  return kuery.replace(/attributes\./g, '').replace(/fleet-agents\./g, '');
}
async function getAgents(esClient, soClient, options) {
  let agents = [];
  if ('agentIds' in options) {
    agents = (await getAgentsById(esClient, soClient, options.agentIds)).filter(maybeAgent => !('notFound' in maybeAgent));
  } else if ('kuery' in options) {
    var _options$showInactive;
    agents = (await getAllAgentsByKuery(esClient, soClient, {
      kuery: options.kuery,
      showAgentless: options.showAgentless,
      showInactive: (_options$showInactive = options.showInactive) !== null && _options$showInactive !== void 0 ? _options$showInactive : false
    })).agents;
  } else {
    throw new _errors.FleetError('Either options.agentIds or options.kuery are required to get agents');
  }
  return agents;
}
async function openPointInTime(esClient, keepAlive = '10m', index = _constants2.AGENTS_INDEX) {
  const pitRes = await (0, _retry.retryTransientEsErrors)(() => esClient.openPointInTime({
    index,
    keep_alive: keepAlive
  }));
  _audit_logging.auditLoggingService.writeCustomAuditLog({
    message: `User opened point in time query [index=${index}] [keepAlive=${keepAlive}] [pitId=${pitRes.id}]`
  });
  return pitRes.id;
}
async function closePointInTime(esClient, pitId) {
  _audit_logging.auditLoggingService.writeCustomAuditLog({
    message: `User closing point in time query [pitId=${pitId}]`
  });
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.closePointInTime({
      id: pitId
    }));
  } catch (error) {
    _.appContextService.getLogger().warn(`Error closing point in time with id: ${pitId}. Error: ${error.message}`);
  }
}
async function getAgentTags(soClient, esClient, options) {
  const {
    kuery,
    showInactive = false
  } = options;
  const filters = [];
  if (kuery && kuery !== '') {
    filters.push(kuery);
  }
  if (showInactive === false) {
    filters.push(ACTIVE_AGENT_CONDITION);
  }
  if (!includeUnenrolled(kuery)) {
    filters.push(ENROLLED_AGENT_CONDITION);
  }
  const kueryNode = _joinFilters(filters);
  const query = kueryNode ? {
    query: (0, _esQuery.toElasticsearchQuery)(kueryNode)
  } : {};
  const runtimeFields = await (0, _build_status_runtime_field.buildAgentStatusRuntimeField)(soClient);
  try {
    var _result$aggregations, _buckets$map;
    const result = await (0, _retry.retryTransientEsErrors)(() => esClient.search({
      index: _constants2.AGENTS_INDEX,
      size: 0,
      ...query,
      fields: Object.keys(runtimeFields),
      runtime_mappings: runtimeFields,
      aggs: {
        tags: {
          terms: {
            field: 'tags',
            size: _constants.SO_SEARCH_LIMIT
          }
        }
      }
    }));
    const buckets = (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : _result$aggregations.tags.buckets;
    return ((_buckets$map = buckets === null || buckets === void 0 ? void 0 : buckets.map(bucket => bucket.key)) !== null && _buckets$map !== void 0 ? _buckets$map : []).sort((a, b) => a.toLowerCase().localeCompare(b.toLowerCase()));
  } catch (err) {
    if ((0, _errors.isESClientError)(err) && err.meta.statusCode === 404) {
      return [];
    }
    throw err;
  }
}
async function getAgentsByKuery(esClient, soClient, options) {
  var _options$sortField, _options$sortOrder;
  const {
    page = 1,
    perPage = 20,
    sortField = (_options$sortField = options.sortField) !== null && _options$sortField !== void 0 ? _options$sortField : 'enrolled_at',
    sortOrder = (_options$sortOrder = options.sortOrder) !== null && _options$sortOrder !== void 0 ? _options$sortOrder : 'desc',
    kuery,
    showAgentless = true,
    showInactive = false,
    getStatusSummary = false,
    showUpgradeable,
    searchAfter,
    openPit,
    pitId,
    pitKeepAlive = '1m',
    aggregations,
    spaceId
  } = options;
  const filters = await _getSpaceAwarenessFilter(spaceId);
  if (kuery && kuery !== '') {
    filters.push(kuery);
  }

  // Hides agents enrolled in agentless policies by excluding the first 1000 agentless policy IDs
  // from the search. This limitation is to avoid hitting the `max_clause_count` limit.
  // In the future, we should hopefully be able to filter agentless agents using metadata:
  // https://github.com/elastic/elastic-agent/issues/7946
  if (showAgentless === false) {
    const agentlessPolicies = await _.agentPolicyService.list(soClient, {
      perPage: 1000,
      kuery: `${_constants2.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE}.supports_agentless:true`
    });
    if (agentlessPolicies.items.length > 0) {
      filters.push(`NOT policy_id: (${agentlessPolicies.items.map(policy => `"${policy.id}"`).join(' or ')})`);
    }
  }
  if (showInactive === false) {
    filters.push(ACTIVE_AGENT_CONDITION);
  }
  if (!includeUnenrolled(kuery)) {
    filters.push(ENROLLED_AGENT_CONDITION);
  }
  const kueryNode = _joinFilters(filters);
  const runtimeFields = await (0, _build_status_runtime_field.buildAgentStatusRuntimeField)(soClient);
  const sort = (0, _common2.getSortConfig)(sortField, sortOrder);
  const statusSummary = {
    online: 0,
    error: 0,
    inactive: 0,
    offline: 0,
    updating: 0,
    unenrolled: 0,
    degraded: 0,
    enrolling: 0,
    unenrolling: 0,
    orphaned: 0,
    uninstalled: 0
  };
  const pitIdToUse = pitId || (openPit ? await openPointInTime(esClient, pitKeepAlive) : undefined);
  const queryAgents = async queryOptions => {
    const aggs = {
      ...(aggregations || getStatusSummary ? {
        aggs: {
          ...(aggregations ? aggregations : {}),
          ...(getStatusSummary ? {
            status: {
              terms: {
                field: 'status'
              }
            }
          } : {})
        }
      } : {})
    };
    return esClient.search({
      ...('from' in queryOptions ? {
        from: queryOptions.from
      } : {
        search_after: queryOptions.searchAfter
      }),
      size: queryOptions.size,
      track_total_hits: true,
      rest_total_hits_as_int: true,
      runtime_mappings: runtimeFields,
      fields: Object.keys(runtimeFields),
      sort,
      query: kueryNode ? (0, _esQuery.toElasticsearchQuery)(kueryNode) : undefined,
      ...(pitIdToUse ? {
        pit: {
          id: pitIdToUse,
          keep_alive: pitKeepAlive
        }
      } : {
        index: _constants2.AGENTS_INDEX,
        ignore_unavailable: true
      }),
      ...aggs
    });
  };
  let res;
  try {
    res = await (0, _retry.retryTransientEsErrors)(() => queryAgents(searchAfter ? {
      searchAfter,
      size: perPage
    } : {
      from: (page - 1) * perPage,
      size: perPage
    }));
  } catch (err) {
    _.appContextService.getLogger().error(`Error getting agents by kuery: ${JSON.stringify(err)}`);
    throw err;
  }
  let agents = res.hits.hits.map(_helpers2.searchHitToAgent);
  let total = res.hits.total;
  // filtering for a range on the version string will not work,
  // nor does filtering on a flattened field (local_metadata), so filter here
  if (showUpgradeable) {
    const latestAgentVersion = await (0, _versions.getLatestAvailableAgentVersion)();
    // fixing a bug where upgradeable filter was not returning right results https://github.com/elastic/kibana/issues/117329
    // query all agents, then filter upgradeable, and return the requested page and correct total
    // if there are more than SO_SEARCH_LIMIT agents, the logic falls back to same as before
    if (total < _constants.SO_SEARCH_LIMIT) {
      const response = await queryAgents({
        from: 0,
        size: _constants.SO_SEARCH_LIMIT
      });
      agents = response.hits.hits.map(_helpers2.searchHitToAgent).filter(agent => (0, _services.isAgentUpgradeAvailable)(agent, latestAgentVersion));
      total = agents.length;
      const start = (page - 1) * perPage;
      agents = agents.slice(start, start + perPage);
    } else {
      agents = agents.filter(agent => (0, _services.isAgentUpgradeAvailable)(agent, latestAgentVersion));
    }
  }
  if (getStatusSummary) {
    if (showUpgradeable) {
      // when showUpgradeable is selected, calculate the summary status manually from the upgradeable agents above
      // the bucket count doesn't take in account the upgradeable agents
      agents.forEach(agent => {
        if (!(agent !== null && agent !== void 0 && agent.status)) return;
        if (!statusSummary[agent.status]) statusSummary[agent.status] = 0;
        statusSummary[agent.status]++;
      });
    } else {
      var _res$aggregations;
      (_res$aggregations = res.aggregations) === null || _res$aggregations === void 0 ? void 0 : _res$aggregations.status.buckets.forEach(bucket => {
        statusSummary[bucket.key] = bucket.doc_count;
      });
    }
  }
  return {
    agents,
    total,
    ...(searchAfter ? {
      page: 0
    } : {
      page
    }),
    perPage,
    ...(pitIdToUse ? {
      pit: pitIdToUse
    } : {}),
    ...(aggregations ? {
      aggregations: res.aggregations
    } : {}),
    ...(getStatusSummary ? {
      statusSummary
    } : {})
  };
}
async function getAllAgentsByKuery(esClient, soClient, options) {
  const res = await getAgentsByKuery(esClient, soClient, {
    ...options,
    page: 1,
    perPage: _constants.SO_SEARCH_LIMIT
  });
  return {
    agents: res.agents,
    total: res.total
  };
}

/**
 * Fetch all agents by kuery in batches.
 * @param esClient
 * @param soClient
 * @param options
 */
async function fetchAllAgentsByKuery(esClient, soClient, options) {
  const {
    kuery = '',
    perPage = _constants.SO_SEARCH_LIMIT,
    sortField = 'enrolled_at',
    sortOrder = 'desc',
    spaceId
  } = options;
  const filters = await _getSpaceAwarenessFilter(spaceId);
  if (kuery && kuery !== '') {
    filters.push(kuery);
  }
  const kueryNode = _joinFilters(filters);
  const query = kueryNode ? {
    query: (0, _esQuery.toElasticsearchQuery)(kueryNode)
  } : {};
  const runtimeFields = await (0, _build_status_runtime_field.buildAgentStatusRuntimeField)(soClient);
  const sort = (0, _common2.getSortConfig)(sortField, sortOrder);
  try {
    return (0, _create_es_search_iterable.createEsSearchIterable)({
      esClient,
      searchRequest: {
        index: _constants2.AGENTS_INDEX,
        size: perPage,
        rest_total_hits_as_int: true,
        track_total_hits: true,
        runtime_mappings: runtimeFields,
        fields: Object.keys(runtimeFields),
        sort,
        ...query
      },
      resultsMapper: data => {
        return data.hits.hits.map(_helpers2.searchHitToAgent);
      }
    });
  } catch (err) {
    _.appContextService.getLogger().error(`Error fetching all agents by kuery: ${JSON.stringify(err)}`);
    throw err;
  }
}
async function getAgentById(esClient, soClient, agentId) {
  const [agentHit] = await getAgentsById(esClient, soClient, [agentId]);
  if ('notFound' in agentHit) {
    throw new _errors.AgentNotFoundError(`Agent ${agentId} not found`);
  }
  if ((await (0, _agent_namespaces.isAgentInNamespace)(agentHit, (0, _get_current_namespace.getCurrentNamespace)(soClient))) !== true) {
    throw new _errors.AgentNotFoundError(`${agentHit.id} not found in namespace`);
  }
  return agentHit;
}

/**
 * Get list of agents by `id`. service method performs space awareness checks.
 * @param esClient
 * @param soClient
 * @param agentIds
 * @param options
 *
 * @throws AgentNotFoundError
 */
const getByIds = async (esClient, soClient, agentIds, options) => {
  const agentsHits = await getAgentsById(esClient, soClient, agentIds);
  const currentNamespace = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  const response = [];
  const throwNotFoundError = agentId => {
    throw new _errors.AgentNotFoundError(`Agent ${agentId} not found`, {
      agentId
    });
  };
  for (const agentHit of agentsHits) {
    const wasFound = !('notFound' in agentHit);
    if (!wasFound) {
      if (!(options !== null && options !== void 0 && options.ignoreMissing)) {
        throwNotFoundError(agentHit.id);
      }
      continue;
    }
    const isAccessible = await (0, _agent_namespaces.isAgentInNamespace)(agentHit, currentNamespace);
    if (!isAccessible) {
      if (!(options !== null && options !== void 0 && options.ignoreMissing)) {
        throwNotFoundError(agentHit.id);
      }
      continue;
    }
    response.push(agentHit);
  }
  return response;
};
exports.getByIds = getByIds;
async function _filterAgents(esClient, soClient, query, options = {}) {
  const {
    page = 1,
    perPage = 20,
    sortField = 'enrolled_at',
    sortOrder = 'desc'
  } = options;
  const runtimeFields = await (0, _build_status_runtime_field.buildAgentStatusRuntimeField)(soClient);
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  let res;
  try {
    res = await (0, _retry.retryTransientEsErrors)(async () => esClient.search({
      from: (page - 1) * perPage,
      size: perPage,
      track_total_hits: true,
      rest_total_hits_as_int: true,
      runtime_mappings: runtimeFields,
      fields: Object.keys(runtimeFields),
      sort: [{
        [sortField]: {
          order: sortOrder
        }
      }],
      query: await (0, _query_namespaces_filtering.addNamespaceFilteringToQuery)({
        bool: {
          filter: [query]
        }
      }, currentSpaceId),
      index: _constants2.AGENTS_INDEX,
      ignore_unavailable: true
    }));
  } catch (err) {
    _.appContextService.getLogger().error(`Error querying agents: ${JSON.stringify(err)}`);
    throw err;
  }
  const agents = res.hits.hits.map(_helpers2.searchHitToAgent);
  const total = res.hits.total;
  return {
    agents,
    total,
    page,
    perPage
  };
}
async function getAgentsById(esClient, soClient, agentIds) {
  if (!agentIds.length) {
    return [];
  }
  const idsQuery = {
    terms: {
      _id: agentIds
    }
  };
  const {
    agents
  } = await _filterAgents(esClient, soClient, idsQuery, {
    perPage: agentIds.length
  });

  // return agents in the same order as agentIds
  return agentIds.map(agentId => agents.find(agent => agent.id === agentId) || {
    id: agentId,
    notFound: true
  });
}

// given a list of agentPolicyIds, return a map of agent version => count of agents
// this is used to get all fleet server versions
async function getAgentVersionsForAgentPolicyIds(esClient, soClient, agentPolicyIds) {
  const result = [];
  if (!agentPolicyIds.length) {
    return result;
  }
  try {
    const {
      hits: {
        hits
      }
    } = await (0, _retry.retryTransientEsErrors)(() => esClient.search({
      query: {
        bool: {
          filter: [{
            terms: {
              policy_id: agentPolicyIds
            }
          }]
        }
      },
      index: _constants2.AGENTS_INDEX,
      ignore_unavailable: true
    }));
    const groupedHits = (0, _lodash.groupBy)(hits, hit => {
      var _hit$_source;
      return (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.policy_id;
    });
    for (const [policyId, policyHits] of Object.entries(groupedHits)) {
      const versionCounts = {};
      for (const hit of policyHits) {
        var _hit$_source2, _hit$_source2$local_m, _hit$_source2$local_m2, _hit$_source2$local_m3;
        const agentVersion = (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : (_hit$_source2$local_m = _hit$_source2.local_metadata) === null || _hit$_source2$local_m === void 0 ? void 0 : (_hit$_source2$local_m2 = _hit$_source2$local_m.elastic) === null || _hit$_source2$local_m2 === void 0 ? void 0 : (_hit$_source2$local_m3 = _hit$_source2$local_m2.agent) === null || _hit$_source2$local_m3 === void 0 ? void 0 : _hit$_source2$local_m3.version;
        if (!agentVersion) {
          continue;
        }
        versionCounts[agentVersion] = (versionCounts[agentVersion] || 0) + 1;
      }
      result.push({
        policyId,
        versionCounts
      });
    }
  } catch (error) {
    if (error.statusCode !== 404) {
      throw error;
    }
  }
  return result;
}
async function getAgentByAccessAPIKeyId(esClient, soClient, accessAPIKeyId) {
  const query = {
    term: {
      access_api_key_id: accessAPIKeyId
    }
  };
  const {
    agents
  } = await _filterAgents(esClient, soClient, query);
  const agent = agents.length ? agents[0] : null;
  if (!agent) {
    throw new _errors.AgentNotFoundError('Agent not found');
  }
  if (agent.access_api_key_id !== accessAPIKeyId) {
    throw new _errors.FleetError('Agent api key id is not matching');
  }
  if (!agent.active) {
    throw new _errors.FleetUnauthorizedError('Agent inactive');
  }
  return agent;
}
async function updateAgent(esClient, agentId, data) {
  _audit_logging.auditLoggingService.writeCustomAuditLog({
    message: `User updated agent [id=${agentId}]`
  });
  await (0, _retry.retryTransientEsErrors)(() => esClient.update({
    id: agentId,
    index: _constants2.AGENTS_INDEX,
    doc: (0, _helpers2.agentSOAttributesToFleetServerAgentDoc)(data),
    refresh: 'wait_for'
  }));
}
async function bulkUpdateAgents(esClient, updateData, errors) {
  if (updateData.length === 0) {
    return;
  }
  const operations = updateData.flatMap(({
    agentId,
    data
  }) => [{
    update: {
      _id: agentId,
      retry_on_conflict: 5
    }
  }, {
    doc: {
      ...(0, _helpers2.agentSOAttributesToFleetServerAgentDoc)(data)
    }
  }]);
  const res = await (0, _retry.retryTransientEsErrors)(() => esClient.bulk({
    operations,
    index: _constants2.AGENTS_INDEX,
    refresh: 'wait_for'
  }));
  res.items.filter(item => item.update.error).forEach(item => {
    // @ts-expect-error it not assignable to ErrorCause
    errors[item.update._id] = item.update.error;
  });
}
async function deleteAgent(esClient, agentId) {
  try {
    await (0, _retry.retryTransientEsErrors)(() => esClient.update({
      id: agentId,
      index: _constants2.AGENTS_INDEX,
      doc: {
        active: false
      }
    }));
  } catch (err) {
    if ((0, _errors.isESClientError)(err) && err.meta.statusCode === 404) {
      throw new _errors.AgentNotFoundError('Agent not found');
    }
    throw err;
  }
}
async function _getAgentDocById(esClient, agentId) {
  try {
    const res = await (0, _retry.retryTransientEsErrors)(() => esClient.get({
      id: agentId,
      index: _constants2.AGENTS_INDEX
    }));
    if (!res._source) {
      throw new _errors.AgentNotFoundError(`Agent ${agentId} not found`);
    }
    return res._source;
  } catch (err) {
    if ((0, _errors.isESClientError)(err) && err.meta.statusCode === 404) {
      throw new _errors.AgentNotFoundError(`Agent ${agentId} not found`);
    }
    throw err;
  }
}
async function getAgentPolicyForAgent(soClient, esClient, agentId) {
  const agent = await _getAgentDocById(esClient, agentId);
  if (!agent.policy_id) {
    return;
  }
  const agentPolicy = await _.agentPolicyService.get(soClient, agent.policy_id, false);
  if (agentPolicy) {
    return agentPolicy;
  }
}
// Get all the policies for a list of agents
async function getAgentPolicyForAgents(soClient, agents) {
  const policyIds = new Set(agents.map(agent => agent.policy_id));
  const agentPolicies = await _.agentPolicyService.getByIds(soClient, Array.from(policyIds).filter(id => id !== undefined));
  return agentPolicies;
}
async function _getSpaceAwarenessFilter(spaceId) {
  const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
  if (!useSpaceAwareness || !spaceId) {
    return [];
  }
  if (spaceId === _common.DEFAULT_SPACE_ID) {
    return [_agent_namespaces.DEFAULT_NAMESPACES_FILTER];
  } else {
    return [`namespaces:"${spaceId}" or namespaces:"${_constants.ALL_SPACES_ID}"`];
  }
}