"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.children = children;
exports.childrenFoAnyNode = childrenFoAnyNode;
exports.resolveItem = exports.lastItem = exports.firstItem = void 0;
exports.singleItems = singleItems;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Normalizes AST "item" list to only contain *single* items.
 *
 * @param items A list of single or nested items.
 */
function* singleItems(items) {
  for (const item of items) {
    if (Array.isArray(item)) {
      yield* singleItems(item);
    } else {
      yield item;
    }
  }
}

/**
 * Returns the first normalized "single item" from the "item" list.
 *
 * @param items Returns the first "single item" from the "item" list.
 * @returns A "single item", if any.
 */
const firstItem = items => {
  for (const item of singleItems(items)) {
    return item;
  }
};
exports.firstItem = firstItem;
const resolveItem = items => {
  return Array.isArray(items) ? resolveItem(items[0]) : items;
};

/**
 * Returns the last normalized "single item" from the "item" list.
 *
 * @param items Returns the last "single item" from the "item" list.
 * @returns A "single item", if any.
 */
exports.resolveItem = resolveItem;
const lastItem = items => {
  const last = items[items.length - 1];
  if (!last) return undefined;
  if (Array.isArray(last)) return lastItem(last);
  return last;
};
exports.lastItem = lastItem;
function* children(node) {
  switch (node.type) {
    case 'function':
    case 'command':
    case 'header-command':
    case 'order':
    case 'option':
      {
        yield* singleItems(node.args);
        break;
      }
    case 'list':
      {
        yield* singleItems(node.values);
        break;
      }
    case 'map':
      {
        yield* node.entries;
        break;
      }
    case 'map-entry':
      {
        yield node.key;
        yield node.value;
        break;
      }
    case 'inlineCast':
      {
        if (Array.isArray(node.value)) {
          yield* singleItems(node.value);
        } else {
          yield node.value;
        }
        break;
      }
    case 'parens':
      {
        yield node.child;
        break;
      }
  }
}
function* childrenFoAnyNode(node) {
  if ('args' in node && Array.isArray(node.args)) {
    yield* singleItems(node.args);
    return;
  }
  switch (node.type) {
    case 'query':
      {
        if (node.header) {
          yield* node.header;
        }
        yield* node.commands;
        break;
      }
    case 'source':
      {
        if (node.prefix) {
          yield node.prefix;
        }
        if (node.index) {
          yield node.index;
        }
        if (node.selector) {
          yield node.selector;
        }
        break;
      }
    default:
      {
        yield* children(node);
        break;
      }
  }
}