"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SignatureAnalyzer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _expressions = require("../../expressions");
var _helpers = require("../helpers");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** Centralizes signature analysis using getValidSignaturesAndTypesToSuggestNext API. */

class SignatureAnalyzer {
  constructor(signatures, paramDefinitions, firstArgumentType, currentParameterIndex, hasMoreMandatoryArgs) {
    (0, _defineProperty2.default)(this, "signatures", void 0);
    (0, _defineProperty2.default)(this, "paramDefinitions", void 0);
    (0, _defineProperty2.default)(this, "firstArgumentType", void 0);
    (0, _defineProperty2.default)(this, "currentParameterIndex", void 0);
    (0, _defineProperty2.default)(this, "hasMoreMandatoryArgs", void 0);
    this.signatures = signatures;
    this.paramDefinitions = paramDefinitions;
    this.firstArgumentType = firstArgumentType;
    this.currentParameterIndex = currentParameterIndex;
    this.hasMoreMandatoryArgs = hasMoreMandatoryArgs;
  }

  /**
   * Creates SignatureAnalyzer directly from an AST node using getValidSignaturesAndTypesToSuggestNext.
   * This is the preferred factory method as it uses the existing API.
   */
  static fromNode(node, context, fnDefinition) {
    var _validationResult$enr;
    if (!fnDefinition) {
      return null;
    }
    const validationResult = (0, _helpers.getValidSignaturesAndTypesToSuggestNext)(node, context, fnDefinition);
    const firstArgumentType = (_validationResult$enr = validationResult.enrichedArgs[0]) === null || _validationResult$enr === void 0 ? void 0 : _validationResult$enr.dataType;
    return new SignatureAnalyzer(validationResult.validSignatures, validationResult.compatibleParamDefs, firstArgumentType, validationResult.argIndex, validationResult.hasMoreMandatoryArgs);
  }

  /**
   * Creates a SignatureAnalyzer from a FunctionParameterContext.
   * Returns null if context is invalid.
   *
   * Use this method when you already have a FunctionParameterContext constructed.
   * Use fromNode() when you have direct access to the AST node.
   */
  static from(context) {
    var _ref, _context$validSignatu, _context$paramDefinit, _context$currentParam;
    if (!(context !== null && context !== void 0 && context.functionDefinition)) {
      return null;
    }
    const signatures = (_ref = (_context$validSignatu = context.validSignatures) !== null && _context$validSignatu !== void 0 ? _context$validSignatu : context.functionDefinition.signatures) !== null && _ref !== void 0 ? _ref : [];
    const paramDefinitions = (_context$paramDefinit = context.paramDefinitions) !== null && _context$paramDefinit !== void 0 ? _context$paramDefinit : [];
    const firstArgumentType = context.firstArgumentType;
    const currentParameterIndex = (_context$currentParam = context.currentParameterIndex) !== null && _context$currentParam !== void 0 ? _context$currentParam : 0;
    const hasMoreMandatoryArgs = Boolean(context.hasMoreMandatoryArgs);
    return new SignatureAnalyzer(signatures, paramDefinitions, firstArgumentType, currentParameterIndex, hasMoreMandatoryArgs);
  }

  /**
   * Returns true if function requires type homogeneity (COALESCE, CONCAT, etc.)
   *
   * Homogeneous functions require all parameters to be the same type.
   * Examples: COALESCE(text, text, text) or CONCAT(keyword, keyword)
   */
  get isHomogeneous() {
    return this.signatures.every(sig => {
      var _sig$params$;
      const isVariadic = sig.minParams != null;
      if (!isVariadic && sig.params.length < 2) {
        return false;
      }
      const firstParamType = (_sig$params$ = sig.params[0]) === null || _sig$params$ === void 0 ? void 0 : _sig$params$.type;
      if (!firstParamType || firstParamType === 'any') {
        return false;
      }
      return sig.params.every(param => {
        if (param.type === firstParamType) {
          return true;
        }

        // keyword and text are interchangeable in ES|QL
        if ((firstParamType === 'keyword' || firstParamType === 'text') && (param.type === 'keyword' || param.type === 'text')) {
          return true;
        }
        return false;
      });
    });
  }

  /**
   * Returns true if function is variadic (accepts unlimited parameters).
   * Examples: CONCAT, COALESCE, CASE
   */
  get isVariadic() {
    return this.signatures.some(sig => sig.minParams != null);
  }

  /**
   * Returns maximum number of parameters across all signatures.
   */
  get maxParams() {
    if (this.signatures.length === 0) {
      return 0;
    }
    return Math.max(...this.signatures.map(sig => sig.params.length));
  }

  /** Returns true if current parameter index is at or beyond max params.*/
  get isAtMaxParams() {
    if (this.isVariadic) {
      return false;
    }
    return this.currentParameterIndex >= this.maxParams - 1;
  }

  /**
   * Returns true if more parameters can be added.
   * Considers: mandatory args, variadic functions, max params.
   */
  get hasMoreParams() {
    if (this.hasMoreMandatoryArgs) {
      return true;
    }
    if (this.isVariadic) {
      return true;
    }
    return !this.isAtMaxParams && this.maxParams > 0;
  }

  /**
   * Checks if given type is compatible with current parameter position.
   */
  isCurrentTypeCompatible(givenType, isLiteral) {
    if (this.paramDefinitions.length > 0) {
      return this.paramDefinitions.some(def => (0, _expressions.argMatchesParamType)(givenType, def.type, isLiteral, false));
    }
    if (this.isVariadic && this.firstArgumentType) {
      return (0, _expressions.argMatchesParamType)(givenType, this.firstArgumentType, isLiteral, false);
    }
    return false;
  }

  /**
   * Returns compatible parameter definitions for the current position.
   */
  getCompatibleParamDefs() {
    return this.paramDefinitions;
  }

  /**
   * Returns true if there are more mandatory arguments required.
   */
  getHasMoreMandatoryArgs() {
    return this.hasMoreMandatoryArgs;
  }

  /**
   * Returns the first argument type (used for homogeneity checks).
   */
  getFirstArgumentType() {
    return this.firstArgumentType;
  }

  /**
   * Returns the current parameter index.
   */
  getCurrentParameterIndex() {
    return this.currentParameterIndex;
  }

  /**
   * Returns valid signatures for the current function.
   */
  getValidSignatures() {
    return this.signatures;
  }

  // ============================================================================
  // Public API: Type Analysis
  // ============================================================================

  /**
   * Returns true if function accepts arbitrary expressions in parameters.
   *
   * This pattern indicates functions where parameters can contain complex expressions
   * - Variadic with multiple parameters (minParams >= 2)
   * - Unknown return type (depends on arguments)
   * - Mixed parameter types (boolean + any)
   *
   * Example: CASE(condition1, value1, condition2, value2, ..., default)
   */
  get acceptsArbitraryExpressions() {
    return (0, _utils.acceptsArbitraryExpressions)(this.signatures);
  }

  /**
   * Gets accepted types for the current/next parameter.
   *
   * Special handling for:
   * - Functions with arbitrary expressions → returns ['any']
   * - Boolean homogeneity → returns ['any'] (user can build boolean expressions)
   * - Other homogeneous types → returns types matching first parameter
   */
  getAcceptedTypes() {
    // Special case 1: functions accepting arbitrary expressions (CASE, etc.)
    if (this.acceptsArbitraryExpressions) {
      return ['any'];
    }

    // Special case 2: boolean homogeneity allows any type (to build boolean expressions)
    if (this.isHomogeneous && this.firstArgumentType === 'boolean') {
      return ['any'];
    }

    // Special case 3: homogeneous variadic function with first argument type set
    // This handles COALESCE, CONCAT, GREATEST, LEAST beyond their defined params
    // IMPORTANT: Must come BEFORE paramDefinitions check because variadic functions
    // have empty paramDefinitions for parameters beyond those explicitly defined
    if (this.isHomogeneous && this.firstArgumentType && this.firstArgumentType !== 'unknown') {
      const isTextual = this.firstArgumentType === 'text' || this.firstArgumentType === 'keyword';
      return isTextual ? ['text', 'keyword'] : [this.firstArgumentType];
    }

    // (e.g., BUCKET accepts date_period as constantOnly, but TO_DATEPERIOD() returns date_period)
    if (this.paramDefinitions.length > 0) {
      const types = this.paramDefinitions.map(({
        type
      }) => type);
      return this.ensureKeywordAndText(types);
    }
    return ['any'];
  }

  /**
   * Checks if an expression type matches accepted parameter types.
   */
  typeMatches(expressionType, isLiteral) {
    // If no param definitions, check against firstArgumentType for variadic functions
    if (this.paramDefinitions.length === 0) {
      if (this.isVariadic && this.firstArgumentType) {
        return (0, _expressions.argMatchesParamType)(expressionType, this.firstArgumentType, isLiteral, false);
      }
      return false;
    }

    // Check if expression type matches any param definition
    return this.paramDefinitions.some(def => (0, _expressions.argMatchesParamType)(expressionType, def.type, isLiteral, false));
  }

  /** Returns complete parameter state for comma/operator decision engines. */
  getParameterState(expressionType, isLiteral) {
    return {
      typeMatches: this.isCurrentTypeCompatible(expressionType, isLiteral),
      isLiteral,
      hasMoreParams: this.hasMoreParams,
      isVariadic: this.isVariadic
    };
  }
  ensureKeywordAndText(types) {
    const hasKeyword = types.includes('keyword');
    const hasText = types.includes('text');
    if (hasKeyword && !hasText) {
      return [...types, 'text'];
    }
    if (hasText && !hasKeyword) {
      return [...types, 'keyword'];
    }
    return types;
  }
}
exports.SignatureAnalyzer = SignatureAnalyzer;