"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleInOperator = handleInOperator;
exports.handleLikeOperator = handleLikeOperator;
exports.handleNullCheckOperator = handleNullCheckOperator;
var _functions = require("../../../../functions");
var _utils = require("./utils");
var _dispatcher = require("../dispatcher");
var _all_operators = require("../../../../../all_operators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const WHITESPACE_NORMALIZE_REGEX = /\s+/g;
const TRAILING_WHITESPACE_REGEX = /\s+$/;
const NULL_CHECK_CANDIDATES = ['is null', 'is not null'];

/**
 * Handles IS NULL / IS NOT NULL partial operators.
 * Generates suggestions directly without creating synthetic nodes.
 * Supports prefix matching: "IS N" suggests both IS NULL and IS NOT NULL.
 */
async function handleNullCheckOperator({
  textBeforeCursor
}, {
  innerText
}) {
  const text = textBeforeCursor || innerText;
  const queryNormalized = text.toLowerCase().replace(WHITESPACE_NORMALIZE_REGEX, ' ').replace(TRAILING_WHITESPACE_REGEX, ' ');
  const suggestions = [];
  for (const name of NULL_CHECK_CANDIDATES) {
    const def = (0, _functions.getFunctionDefinition)(name);
    if (!def) {
      continue;
    }
    const candidateLower = name.toLowerCase();
    const matches = [...candidateLower].some((_, i) => queryNormalized.endsWith(candidateLower.slice(0, i + 1)));
    if (matches) {
      suggestions.push({
        label: name.toUpperCase(),
        text: name.toUpperCase(),
        kind: 'Operator',
        detail: def.description,
        sortText: 'D'
      });
    }
  }
  return suggestions.length > 0 ? suggestions : null;
}
async function handleLikeOperator(detection, context) {
  return handleInfixOperator(detection, context, _all_operators.patternMatchOperators.map(op => op.name), _utils.createSyntheticLikeOperatorNode);
}
async function handleInOperator(detection, context) {
  return handleInfixOperator(detection, context, _all_operators.inOperators.map(op => op.name), _utils.createSyntheticListOperatorNode);
}

/**
 * Handles infix operators with content (IN, LIKE, RLIKE, etc.).
 * Uses existing AST node if available, otherwise creates synthetic node.
 */
async function handleInfixOperator({
  operatorName,
  textBeforeCursor
}, context, operatorNames, createSyntheticNode) {
  var _expressionRoot$name$, _expressionRoot$name;
  const {
    innerText,
    expressionRoot
  } = context;
  const text = textBeforeCursor || innerText;
  const hasValidAstNode = (expressionRoot === null || expressionRoot === void 0 ? void 0 : expressionRoot.type) === 'function' && operatorNames.includes((_expressionRoot$name$ = (_expressionRoot$name = expressionRoot.name) === null || _expressionRoot$name === void 0 ? void 0 : _expressionRoot$name.toLowerCase()) !== null && _expressionRoot$name$ !== void 0 ? _expressionRoot$name$ : '');
  if (hasValidAstNode) {
    return (0, _dispatcher.dispatchOperators)({
      ...context,
      innerText: text
    });
  }
  const syntheticNode = createSyntheticNode(operatorName, text, expressionRoot);
  return (0, _dispatcher.dispatchOperators)({
    ...context,
    expressionRoot: syntheticNode,
    innerText: text
  });
}