"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EmbeddableStateTransfer = exports.EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _public = require("@kbn/kibana-utils-plugin/public");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY = exports.EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY = 'EMBEDDABLE_STATE_TRANSFER';

/**
 * A wrapper around the session storage which provides strongly typed helper methods
 * for common incoming and outgoing states used by the embeddable infrastructure.
 *
 * @public
 */
class EmbeddableStateTransfer {
  constructor(navigateToApp, currentAppId$, appList, customStorage) {
    (0, _defineProperty2.default)(this, "isTransferInProgress", void 0);
    (0, _defineProperty2.default)(this, "storage", void 0);
    (0, _defineProperty2.default)(this, "appList", void 0);
    /**
     * Fetches an internationalized app title when given an appId.
     * @param appId - The id of the app to fetch the title for
     */
    (0, _defineProperty2.default)(this, "getAppNameFromId", appId => {
      var _this$appList, _this$appList$get;
      return (_this$appList = this.appList) === null || _this$appList === void 0 ? void 0 : (_this$appList$get = _this$appList.get(appId)) === null || _this$appList$get === void 0 ? void 0 : _this$appList$get.title;
    });
    this.navigateToApp = navigateToApp;
    this.storage = customStorage ? customStorage : new _public.Storage(sessionStorage);
    this.isTransferInProgress = false;
    this.appList = appList;
    currentAppId$.subscribe(() => {
      this.isTransferInProgress = false;
    });
  }
  /**
   * Fetches an {@link EmbeddableEditorState | editor state} from the sessionStorage for the provided app id
   *
   * @param appId - The app to fetch incomingEditorState for
   * @param removeAfterFetch - Whether to remove the package state after fetch to prevent duplicates.
   */
  getIncomingEditorState(appId, removeAfterFetch) {
    const states = this.getIncomingPackagesState(_types.isEmbeddableEditorState, appId, _types.EMBEDDABLE_EDITOR_STATE_KEY, {
      keysToRemoveAfterFetch: removeAfterFetch ? [_types.EMBEDDABLE_EDITOR_STATE_KEY] : undefined
    });
    // warn if states are longer than 1
    if (states && states.length > 1) {
      // eslint-disable-next-line no-console
      console.warn(`Multiple incoming editor states found for appId ${appId}:`, states);
    }
    return states === null || states === void 0 ? void 0 : states[0];
  }

  /**
   * Clears the {@link EmbeddableEditorState | editor state} from the sessionStorage for the provided app id
   *
   * @param appId - The app to fetch incomingEditorState for
   * @param removeAfterFetch - Whether to remove the package state after fetch to prevent duplicates.
   */
  clearEditorState(appId) {
    const currentState = this.storage.get(EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY);
    if (currentState) {
      if (appId) {
        var _currentState$EMBEDDA;
        (_currentState$EMBEDDA = currentState[_types.EMBEDDABLE_EDITOR_STATE_KEY]) === null || _currentState$EMBEDDA === void 0 ? true : delete _currentState$EMBEDDA[appId];
      } else {
        delete currentState[_types.EMBEDDABLE_EDITOR_STATE_KEY];
      }
      this.storage.set(EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY, currentState);
    }
  }

  /**
   * Fetches an {@link EmbeddablePackageState | embeddable package} from the sessionStorage for the given AppId
   *
   * @param appId - The app to fetch EmbeddablePackageState for
   * @param removeAfterFetch - Whether to remove the package state after fetch to prevent duplicates.
   */
  getIncomingEmbeddablePackage(appId, removeAfterFetch) {
    return this.getIncomingPackagesState(_types.isEmbeddablePackageState, appId, _types.EMBEDDABLE_PACKAGE_STATE_KEY, {
      keysToRemoveAfterFetch: removeAfterFetch ? [_types.EMBEDDABLE_PACKAGE_STATE_KEY] : undefined
    });
  }

  /**
   * A wrapper around the {@link ApplicationStart.navigateToApp} method which navigates to the specified appId
   * with {@link EmbeddableEditorState | embeddable editor state}
   */
  async navigateToEditor(appId, options) {
    this.isTransferInProgress = true;
    await this.navigateToWithState(appId, _types.EMBEDDABLE_EDITOR_STATE_KEY, {
      ...options,
      state: options !== null && options !== void 0 && options.state ? [options.state] : undefined
    });
  }

  /**
   * A wrapper around the {@link ApplicationStart.navigateToApp} method which navigates to the specified appId
   * with multiple {@link EmbeddablePackageState | embeddable package state}
   */
  async navigateToWithEmbeddablePackages(appId, options) {
    this.isTransferInProgress = true;
    await this.navigateToWithState(appId, _types.EMBEDDABLE_PACKAGE_STATE_KEY, {
      ...options
    });
  }
  removeKeysFromStorage(state, options) {
    var _options$keysToRemove;
    if (options !== null && options !== void 0 && (_options$keysToRemove = options.keysToRemoveAfterFetch) !== null && _options$keysToRemove !== void 0 && _options$keysToRemove.length) {
      const stateReplace = {
        ...state
      };
      options.keysToRemoveAfterFetch.forEach(keyToRemove => {
        delete stateReplace[keyToRemove];
      });
      this.storage.set(EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY, stateReplace);
    }
  }

  /**
   * Retrieves incoming embeddable package states from session storage, handling arrays.
   * Always returns an array format. Filters results using the provided type guard.
   *
   * @param guard - Type guard function to validate state items
   * @param appId - The application ID to fetch state for
   * @param key - The storage key to retrieve state from
   * @param options - Optional configuration including keys to remove after fetch
   * @returns Array of valid package states, or undefined if no valid states found
   */
  getIncomingPackagesState(guard, appId, key, options) {
    var _embeddableState$key;
    const embeddableState = this.storage.get(EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY);
    if (!embeddableState) {
      return undefined;
    }
    const incomingState = (_embeddableState$key = embeddableState[key]) === null || _embeddableState$key === void 0 ? void 0 : _embeddableState$key[appId];
    if (!incomingState) {
      return undefined;
    }
    if (Array.isArray(incomingState)) {
      const validStates = incomingState.filter(item => guard(item));
      if (validStates.length > 0) {
        this.removeKeysFromStorage(embeddableState, options);
        return validStates.map(item => (0, _lodash.cloneDeep)(item));
      }
      return undefined;
    }
    return undefined;
  }
  async navigateToWithState(appId, key, options) {
    var _this$storage$get;
    const existingAppState = ((_this$storage$get = this.storage.get(EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY)) === null || _this$storage$get === void 0 ? void 0 : _this$storage$get[key]) || {};
    const stateObject = {
      ...this.storage.get(EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY),
      [key]: {
        ...existingAppState,
        [appId]: options === null || options === void 0 ? void 0 : options.state
      }
    };
    this.storage.set(EMBEDDABLE_STATE_TRANSFER_STORAGE_KEY, stateObject);
    await this.navigateToApp(appId, {
      path: options === null || options === void 0 ? void 0 : options.path,
      openInNewTab: options === null || options === void 0 ? void 0 : options.openInNewTab,
      skipAppLeave: options === null || options === void 0 ? void 0 : options.skipAppLeave
    });
  }
}
exports.EmbeddableStateTransfer = EmbeddableStateTransfer;