"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateGlobalStateAndReplaceUrl = exports.updateGlobalState = exports.updateAppStateAndReplaceUrl = exports.updateAppState = exports.pushCurrentTabStateToUrl = void 0;
var _constants = require("../../../../../../common/constants");
var _common = require("../../../../../../common");
var _state_comparators = require("../../utils/state_comparators");
var _internal_state = require("../internal_state");
var _selectors = require("../selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const mergeAppState = (currentState, {
  tabId,
  appState
}) => {
  const currentAppState = (0, _selectors.selectTab)(currentState, tabId).appState;
  const mergedAppState = {
    ...currentAppState,
    ...appState
  };
  return {
    mergedAppState,
    hasStateChanges: !(0, _state_comparators.isEqualState)(currentAppState, mergedAppState)
  };
};

/**
 * Partially update the tab app state, merging with existing state and pushing to URL history
 */
const updateAppState = payload => async function updateAppStateThunkFn(dispatch, getState) {
  const {
    mergedAppState,
    hasStateChanges
  } = mergeAppState(getState(), payload);
  if (hasStateChanges) {
    dispatch(_internal_state.internalStateSlice.actions.setAppState({
      tabId: payload.tabId,
      appState: mergedAppState
    }));
  }
};

/**
 * Partially update the tab app state, merging with existing state and replacing URL history
 */
exports.updateAppState = updateAppState;
const updateAppStateAndReplaceUrl = payload => async function updateAppStateAndReplaceUrlThunkFn(dispatch, getState, {
  urlStateStorage
}) {
  const currentState = getState();
  if (currentState.tabs.unsafeCurrentId !== payload.tabId) {
    return dispatch(updateAppState(payload));
  }
  const {
    mergedAppState
  } = mergeAppState(currentState, payload);
  await urlStateStorage.set(_common.APP_STATE_URL_KEY, mergedAppState, {
    replace: true
  });
};
exports.updateAppStateAndReplaceUrl = updateAppStateAndReplaceUrl;
const mergeGlobalState = (currentState, {
  tabId,
  globalState
}) => {
  const currentGlobalState = (0, _selectors.selectTab)(currentState, tabId).globalState;
  const mergedGlobalState = {
    ...currentGlobalState,
    ...globalState
  };
  return {
    mergedGlobalState,
    hasStateChanges: !(0, _state_comparators.isEqualState)(currentGlobalState, mergedGlobalState)
  };
};

/**
 * Partially update the tab global state, merging with existing state and pushing to URL history
 */
const updateGlobalState = payload => async function updateGlobalStateThunkFn(dispatch, getState) {
  const {
    mergedGlobalState,
    hasStateChanges
  } = mergeGlobalState(getState(), payload);
  if (hasStateChanges) {
    dispatch(_internal_state.internalStateSlice.actions.setGlobalState({
      tabId: payload.tabId,
      globalState: mergedGlobalState
    }));
  }
};

/**
 * Partially update the tab global state, merging with existing state and replacing URL history
 */
exports.updateGlobalState = updateGlobalState;
const updateGlobalStateAndReplaceUrl = payload => async function updateGlobalStateAndReplaceUrlThunkFn(dispatch, getState, {
  urlStateStorage
}) {
  const currentState = getState();
  if (currentState.tabs.unsafeCurrentId !== payload.tabId) {
    return dispatch(updateGlobalState(payload));
  }
  const {
    mergedGlobalState
  } = mergeGlobalState(currentState, payload);
  const globalUrlState = {
    time: mergedGlobalState.timeRange,
    refreshInterval: mergedGlobalState.refreshInterval,
    filters: mergedGlobalState.filters
  };
  await urlStateStorage.set(_constants.GLOBAL_STATE_URL_KEY, globalUrlState, {
    replace: true
  });
};

/**
 * Push the current tab app state and global state to the URL, replacing URL history
 */
exports.updateGlobalStateAndReplaceUrl = updateGlobalStateAndReplaceUrl;
const pushCurrentTabStateToUrl = ({
  tabId
}) => async function pushCurrentTabStateToUrlThunkFn(dispatch) {
  await Promise.all([dispatch(updateGlobalStateAndReplaceUrl({
    tabId,
    globalState: {}
  })), dispatch(updateAppStateAndReplaceUrl({
    tabId,
    appState: {}
  }))]);
};
exports.pushCurrentTabStateToUrl = pushCurrentTabStateToUrl;