"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchGraph = void 0;
var _v = require("@kbn/cloud-security-posture-common/types/graph/v1");
var _v2 = require("@kbn/cloud-security-posture-common/schema/graph/v1");
var _helpers = require("@kbn/cloud-security-posture-common/utils/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchGraph = async ({
  esClient,
  logger,
  start,
  end,
  originEventIds,
  showUnknownTarget,
  indexPatterns,
  spaceId,
  esQuery
}) => {
  const originAlertIds = originEventIds.filter(originEventId => originEventId.isAlert);

  // FROM clause currently doesn't support parameters, Therefore, we validate the index patterns to prevent injection attacks.
  // Regex to match invalid characters in index patterns: upper case characters, \, /, ?, ", <, >, |, (space), #, or ,
  indexPatterns.forEach((indexPattern, idx) => {
    if (!_v2.INDEX_PATTERN_REGEX.test(indexPattern)) {
      throw new Error(`Invalid index pattern [${indexPattern}] at index ${idx}. Cannot contain characters \\, /, ?, ", <, >, |, (space character), #, or ,`);
    }
  });
  const isEnrichPolicyExists = await checkEnrichPolicyExists(esClient, logger, spaceId);
  const SECURITY_ALERTS_PARTIAL_IDENTIFIER = '.alerts-security.alerts-';
  const alertsMappingsIncluded = indexPatterns.some(indexPattern => indexPattern.includes(SECURITY_ALERTS_PARTIAL_IDENTIFIER));
  const query = buildEsqlQuery({
    indexPatterns,
    originEventIds,
    originAlertIds,
    isEnrichPolicyExists,
    spaceId,
    alertsMappingsIncluded
  });
  logger.trace(`Executing query [${query}]`);
  const eventIds = originEventIds.map(originEventId => originEventId.id);
  return await esClient.asCurrentUser.helpers.esql({
    columnar: false,
    filter: buildDslFilter(eventIds, showUnknownTarget, start, end, esQuery),
    query,
    // @ts-ignore - types are not up to date
    params: [...originEventIds.map((originEventId, idx) => ({
      [`og_id${idx}`]: originEventId.id
    })), ...originEventIds.filter(originEventId => originEventId.isAlert).map((originEventId, idx) => ({
      [`og_alrt_id${idx}`]: originEventId.id
    }))]
  }).toRecords();
};
exports.fetchGraph = fetchGraph;
const buildDslFilter = (eventIds, showUnknownTarget, start, end, esQuery) => {
  var _esQuery$bool$filter, _esQuery$bool$must, _esQuery$bool$should, _esQuery$bool$must_no;
  return {
    bool: {
      filter: [{
        range: {
          '@timestamp': {
            gte: start,
            lte: end
          }
        }
      }, ...(showUnknownTarget ? [] : [{
        exists: {
          field: 'target.entity.id'
        }
      }]), {
        bool: {
          should: [...(esQuery !== null && esQuery !== void 0 && (_esQuery$bool$filter = esQuery.bool.filter) !== null && _esQuery$bool$filter !== void 0 && _esQuery$bool$filter.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$must = esQuery.bool.must) !== null && _esQuery$bool$must !== void 0 && _esQuery$bool$must.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$should = esQuery.bool.should) !== null && _esQuery$bool$should !== void 0 && _esQuery$bool$should.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$must_no = esQuery.bool.must_not) !== null && _esQuery$bool$must_no !== void 0 && _esQuery$bool$must_no.length ? [esQuery] : []), {
            terms: {
              'event.id': eventIds
            }
          }],
          minimum_should_match: 1
        }
      }]
    }
  };
};
const checkEnrichPolicyExists = async (esClient, logger, spaceId) => {
  try {
    const {
      policies
    } = await esClient.asInternalUser.enrich.getPolicy({
      name: (0, _helpers.getEnrichPolicyId)(spaceId)
    });
    return policies.some(policy => {
      var _policy$config$match;
      return ((_policy$config$match = policy.config.match) === null || _policy$config$match === void 0 ? void 0 : _policy$config$match.name) === (0, _helpers.getEnrichPolicyId)(spaceId);
    });
  } catch (error) {
    logger.error(`Error fetching enrich policy ${error.message}`);
    logger.error(error);
    return false;
  }
};
const buildEsqlQuery = ({
  indexPatterns,
  originEventIds,
  originAlertIds,
  isEnrichPolicyExists,
  spaceId,
  alertsMappingsIncluded
}) => {
  const SECURITY_ALERTS_PARTIAL_IDENTIFIER = '.alerts-security.alerts-';
  const enrichPolicyName = (0, _helpers.getEnrichPolicyId)(spaceId);
  const query = `FROM ${indexPatterns.filter(indexPattern => indexPattern.length > 0).join(',')} METADATA _id, _index
| WHERE event.action IS NOT NULL AND actor.entity.id IS NOT NULL
| MV_EXPAND actor.entity.id
| MV_EXPAND target.entity.id
${isEnrichPolicyExists ? `
| ENRICH ${enrichPolicyName} ON actor.entity.id WITH actorEntityName = entity.name, actorEntityType = entity.type, actorEntitySubType = entity.sub_type, actorHostIp = host.ip
| ENRICH ${enrichPolicyName} ON target.entity.id WITH targetEntityName = entity.name, targetEntityType = entity.type, targetEntitySubType = entity.sub_type, targetHostIp = host.ip

// Construct actor and target entities data
| EVAL actorDocData = CONCAT("{",
    "\\"id\\":\\"", actor.entity.id, "\\"",
    ",\\"type\\":\\"", "${_v2.DOCUMENT_TYPE_ENTITY}", "\\"",
    ",\\"entity\\":", "{",
      "\\"name\\":\\"", COALESCE(actorEntityName, ""), "\\"",
      ",\\"type\\":\\"", COALESCE(actorEntityType, ""), "\\"",
      ",\\"sub_type\\":\\"", COALESCE(actorEntitySubType, ""), "\\"",
      CASE (
        actorHostIp IS NOT NULL,
        CONCAT(",\\"host\\":", "{", "\\"ip\\":\\"", TO_STRING(actorHostIp), "\\"", "}"),
        ""
      ),
    "}",
  "}")
| EVAL targetDocData = CONCAT("{",
    "\\"id\\":\\"", COALESCE(target.entity.id, ""), "\\"",
    ",\\"type\\":\\"", "${_v2.DOCUMENT_TYPE_ENTITY}", "\\"",
    ",\\"entity\\":", "{",
      "\\"name\\":\\"", COALESCE(targetEntityName, ""), "\\"",
      ",\\"type\\":\\"", COALESCE(targetEntityType, ""), "\\"",
      ",\\"sub_type\\":\\"", COALESCE(targetEntitySubType, ""), "\\"",
      CASE (
        targetHostIp IS NOT NULL,
        CONCAT(",\\"host\\":", "{", "\\"ip\\":\\"", TO_STRING(targetHostIp), "\\"", "}"),
        ""
      ),
    "}",
  "}")
` : `
// Fallback to null string with non-enriched actor
| EVAL actorEntityName = TO_STRING(null)
| EVAL actorEntityType = TO_STRING(null)
| EVAL actorEntitySubType = TO_STRING(null)
| EVAL actorHostIp = TO_STRING(null)
| EVAL actorDocData = TO_STRING(null)

// Fallback to null string with non-enriched target
| EVAL targetEntityName = TO_STRING(null)
| EVAL targetEntityType = TO_STRING(null)
| EVAL targetEntitySubType = TO_STRING(null)
| EVAL targetHostIp = TO_STRING(null)
| EVAL targetDocData = TO_STRING(null)
`}
// Map host and source values to enriched contextual data
| EVAL sourceIps = source.ip
| EVAL sourceCountryCodes = source.geo.country_iso_code
// Origin event and alerts allow us to identify the start position of graph traversal
| EVAL isOrigin = ${originEventIds.length > 0 ? `event.id in (${originEventIds.map((_id, idx) => `?og_id${idx}`).join(', ')})` : 'false'}
| EVAL isOriginAlert = isOrigin AND ${originAlertIds.length > 0 ? `event.id in (${originAlertIds.map((_id, idx) => `?og_alrt_id${idx}`).join(', ')})` : 'false'}
| EVAL isAlert = _index LIKE "*${SECURITY_ALERTS_PARTIAL_IDENTIFIER}*"
// Aggregate document's data for popover expansion and metadata enhancements
// We format it as JSON string, the best alternative so far. Tried to use tuple using MV_APPEND
// but it flattens the data and we lose the structure
| EVAL docType = CASE (isAlert, "${_v.DOCUMENT_TYPE_ALERT}", "${_v.DOCUMENT_TYPE_EVENT}")
| EVAL docData = CONCAT("{",
    "\\"id\\":\\"", _id, "\\"",
    CASE (event.id IS NOT NULL AND event.id != "", CONCAT(",\\"event\\":","{","\\"id\\":\\"", event.id, "\\"","}"), ""),
    ",\\"type\\":\\"", docType, "\\"",
    ",\\"index\\":\\"", _index, "\\"",
    ${
  // ESQL complains about missing field's mapping when we don't fetch from alerts index
  alertsMappingsIncluded ? `CASE (isAlert, CONCAT(",\\"alert\\":", "{",
      "\\"ruleName\\":\\"", kibana.alert.rule.name, "\\"",
    "}"), ""),` : ''}
  "}")

| STATS badge = COUNT(*),
  uniqueEventsCount = COUNT_DISTINCT(CASE(isAlert == false, _id, null)),
  uniqueAlertsCount = COUNT_DISTINCT(CASE(isAlert == true, _id, null)),
  isAlert = MV_MAX(VALUES(isAlert)),
  docs = VALUES(docData),
  sourceIps = MV_DEDUPE(VALUES(sourceIps)),
  sourceCountryCodes = MV_DEDUPE(VALUES(sourceCountryCodes)),
  // actor attributes
  actorNodeId = CASE(
    // deterministic group IDs - use raw entity ID for single values, MD5 hash for multiple
    MV_COUNT(VALUES(actor.entity.id)) == 1, TO_STRING(VALUES(actor.entity.id)),
    MD5(MV_CONCAT(MV_SORT(VALUES(actor.entity.id)), ","))
  ),
  actorIdsCount = COUNT_DISTINCT(actor.entity.id),
  actorEntityName = VALUES(actorEntityName),
  actorHostIp = VALUES(actorHostIp),
  actorsDocData = VALUES(actorDocData),
  // target attributes
  targetNodeId = CASE(
    // deterministic group IDs - use raw entity ID for single values, MD5 hash for multiple
    COUNT_DISTINCT(target.entity.id) == 0, null,
    CASE(
      MV_COUNT(VALUES(target.entity.id)) == 1, TO_STRING(VALUES(target.entity.id)),
      MD5(MV_CONCAT(MV_SORT(VALUES(target.entity.id)), ","))
    )
  ),
  targetIdsCount = COUNT_DISTINCT(target.entity.id),
  targetEntityName = VALUES(targetEntityName),
  targetHostIp = VALUES(targetHostIp),
  targetsDocData = VALUES(targetDocData)
    BY action = event.action,
      actorEntityType,
      actorEntitySubType,
      targetEntityType,
      targetEntitySubType,
      isOrigin,
      isOriginAlert
| LIMIT 1000
| SORT action DESC, isOrigin`;
  return query;
};