"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CasePlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _common = require("@kbn/spaces-plugin/common");
var _constants = require("../common/constants");
var _factory = require("./client/factory");
var _features = require("./features");
var _register_routes = require("./routes/api/register_routes");
var _get_external_routes = require("./routes/api/get_external_routes");
var _telemetry = require("./telemetry");
var _get_internal_routes = require("./routes/api/get_internal_routes");
var _persistable_state_registry = require("./attachment_framework/persistable_state_registry");
var _external_reference_registry = require("./attachment_framework/external_reference_registry");
var _services = require("./services");
var _constants2 = require("./common/constants");
var _internal_attachments = require("./internal_attachments");
var _files = require("./files");
var _connectors = require("./connectors");
var _saved_object_types = require("./saved_object_types");
var _incremental_id_task_manager = require("./tasks/incremental_id/incremental_id_task_manager");
var _cases_analytics = require("./cases_analytics");
var _scheduler_task = require("./cases_analytics/tasks/scheduler_task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CasePlugin {
  constructor(initializerContext) {
    (0, _defineProperty2.default)(this, "caseConfig", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "kibanaVersion", void 0);
    (0, _defineProperty2.default)(this, "clientFactory", void 0);
    (0, _defineProperty2.default)(this, "securityPluginSetup", void 0);
    (0, _defineProperty2.default)(this, "lensEmbeddableFactory", void 0);
    (0, _defineProperty2.default)(this, "persistableStateAttachmentTypeRegistry", void 0);
    (0, _defineProperty2.default)(this, "externalReferenceAttachmentTypeRegistry", void 0);
    (0, _defineProperty2.default)(this, "userProfileService", void 0);
    (0, _defineProperty2.default)(this, "incrementalIdTaskManager", void 0);
    (0, _defineProperty2.default)(this, "isServerless", void 0);
    (0, _defineProperty2.default)(this, "createRouteHandlerContext", ({
      core
    }) => {
      return async (context, request, response) => {
        return {
          getCasesClient: async () => {
            const [{
              savedObjects
            }] = await core.getStartServices();
            const coreContext = await context.core;
            return this.clientFactory.create({
              request,
              scopedClusterClient: coreContext.elasticsearch.client.asCurrentUser,
              savedObjectsService: savedObjects
            });
          }
        };
      };
    });
    (0, _defineProperty2.default)(this, "getCasesClientWithRequest", core => async request => {
      const client = core.elasticsearch.client;
      return this.clientFactory.create({
        request,
        scopedClusterClient: client.asScoped(request).asCurrentUser,
        savedObjectsService: core.savedObjects
      });
    });
    this.initializerContext = initializerContext;
    this.caseConfig = initializerContext.config.get();
    this.kibanaVersion = initializerContext.env.packageInfo.version;
    this.logger = this.initializerContext.logger.get();
    this.clientFactory = new _factory.CasesClientFactory(this.logger);
    this.persistableStateAttachmentTypeRegistry = new _persistable_state_registry.PersistableStateAttachmentTypeRegistry();
    this.externalReferenceAttachmentTypeRegistry = new _external_reference_registry.ExternalReferenceAttachmentTypeRegistry();
    this.userProfileService = new _services.UserProfileService(this.logger);
    this.isServerless = initializerContext.env.packageInfo.buildFlavor === 'serverless';
  }
  setup(core, plugins) {
    var _plugins$usageCollect, _plugins$cloud;
    this.logger.debug(`Setting up Case Workflow with core contract [${Object.keys(core)}] and plugins [${Object.keys(plugins)}]`);
    (0, _internal_attachments.registerInternalAttachments)(this.externalReferenceAttachmentTypeRegistry, this.persistableStateAttachmentTypeRegistry);
    (0, _files.registerCaseFileKinds)(this.caseConfig.files, plugins.files, core.security.fips.isEnabled());
    (0, _cases_analytics.registerCasesAnalyticsIndexesTasks)({
      taskManager: plugins.taskManager,
      logger: this.logger,
      core,
      analyticsConfig: this.caseConfig.analytics
    });
    this.securityPluginSetup = plugins.security;
    this.lensEmbeddableFactory = plugins.lens.lensEmbeddableFactory;
    if (this.caseConfig.stack.enabled) {
      // V1 is deprecated, but has to be maintained for the time being
      // https://github.com/elastic/kibana/pull/186800#issue-2369812818
      const casesFeatures = (0, _features.getCasesKibanaFeatures)();
      plugins.features.registerKibanaFeature(casesFeatures.v1);
      plugins.features.registerKibanaFeature(casesFeatures.v2);
      plugins.features.registerKibanaFeature(casesFeatures.v3);
    }
    (0, _saved_object_types.registerSavedObjects)({
      core,
      logger: this.logger,
      persistableStateAttachmentTypeRegistry: this.persistableStateAttachmentTypeRegistry,
      lensEmbeddableFactory: this.lensEmbeddableFactory
    });
    core.http.registerRouteHandlerContext(_constants.APP_ID, this.createRouteHandlerContext({
      core
    }));
    if (plugins.taskManager) {
      if (plugins.usageCollection) {
        (0, _telemetry.createCasesTelemetry)({
          core,
          taskManager: plugins.taskManager,
          usageCollection: plugins.usageCollection,
          logger: this.logger,
          kibanaVersion: this.kibanaVersion
        });
      }
      if (this.caseConfig.incrementalId.enabled) {
        this.incrementalIdTaskManager = new _incremental_id_task_manager.IncrementalIdTaskManager(plugins.taskManager, this.caseConfig.incrementalId, this.logger, plugins.usageCollection);
      }
    }
    const router = core.http.createRouter();
    const telemetryUsageCounter = (_plugins$usageCollect = plugins.usageCollection) === null || _plugins$usageCollect === void 0 ? void 0 : _plugins$usageCollect.createUsageCounter(_constants.APP_ID);
    (0, _register_routes.registerRoutes)({
      router,
      routes: [...(0, _get_external_routes.getExternalRoutes)({
        isServerless: this.isServerless,
        docLinks: core.docLinks
      }), ...(0, _get_internal_routes.getInternalRoutes)(this.userProfileService)],
      logger: this.logger,
      kibanaVersion: this.kibanaVersion,
      telemetryUsageCounter
    });
    plugins.licensing.featureUsage.register(_constants2.LICENSING_CASE_ASSIGNMENT_FEATURE, 'platinum');
    plugins.licensing.featureUsage.register(_constants2.LICENSING_CASE_OBSERVABLES_FEATURE, 'platinum');
    const getCasesClient = async request => {
      const [coreStart] = await core.getStartServices();
      return this.getCasesClientWithRequest(coreStart)(request);
    };
    const getSpaceId = request => {
      var _plugins$spaces$space, _plugins$spaces;
      if (!request) {
        return _common.DEFAULT_SPACE_ID;
      }
      return (_plugins$spaces$space = (_plugins$spaces = plugins.spaces) === null || _plugins$spaces === void 0 ? void 0 : _plugins$spaces.spacesService.getSpaceId(request)) !== null && _plugins$spaces$space !== void 0 ? _plugins$spaces$space : _common.DEFAULT_SPACE_ID;
    };
    const serverlessProjectType = this.isServerless ? (_plugins$cloud = plugins.cloud) === null || _plugins$cloud === void 0 ? void 0 : _plugins$cloud.serverless.projectType : undefined;
    (0, _connectors.registerConnectorTypes)({
      actions: plugins.actions,
      alerting: plugins.alerting,
      core,
      logger: this.logger,
      getCasesClient,
      getSpaceId,
      serverlessProjectType
    });
    return {
      attachmentFramework: {
        registerExternalReference: externalReferenceAttachmentType => {
          this.externalReferenceAttachmentTypeRegistry.register(externalReferenceAttachmentType);
        },
        registerPersistableState: persistableStateAttachmentType => {
          this.persistableStateAttachmentTypeRegistry.register(persistableStateAttachmentType);
        }
      },
      config: this.caseConfig
    };
  }
  start(core, plugins) {
    this.logger.debug(`Starting Case Workflow`);
    if (plugins.taskManager) {
      var _this$caseConfig$anal;
      (0, _telemetry.scheduleCasesTelemetryTask)(plugins.taskManager, this.logger);
      if (this.caseConfig.incrementalId.enabled) {
        var _this$incrementalIdTa;
        void ((_this$incrementalIdTa = this.incrementalIdTaskManager) === null || _this$incrementalIdTa === void 0 ? void 0 : _this$incrementalIdTa.setupIncrementIdTask(plugins.taskManager, core));
      }
      if ((_this$caseConfig$anal = this.caseConfig.analytics.index) !== null && _this$caseConfig$anal !== void 0 && _this$caseConfig$anal.enabled) {
        const internalSavedObjectsRepository = core.savedObjects.createInternalRepository([_constants.CASE_SAVED_OBJECT]);
        const internalSavedObjectsClient = new _server.SavedObjectsClient(internalSavedObjectsRepository);
        (0, _scheduler_task.scheduleCAISchedulerTask)({
          taskManager: plugins.taskManager,
          logger: this.logger
        }).catch(() => {}); // it shouldn't reject, but just in case
        (0, _cases_analytics.createCasesAnalyticsIndexes)({
          esClient: core.elasticsearch.client.asInternalUser,
          logger: this.logger,
          isServerless: this.isServerless,
          taskManager: plugins.taskManager,
          savedObjectsClient: internalSavedObjectsClient
        }).catch(() => {}); // it shouldn't reject, but just in case
      }
    }
    this.userProfileService.initialize({
      spaces: plugins.spaces,
      // securityPluginSetup will be set to a defined value in the setup() function
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      securityPluginSetup: this.securityPluginSetup,
      securityPluginStart: plugins.security,
      licensingPluginStart: plugins.licensing
    });
    this.clientFactory.initialize({
      // securityPluginSetup will be set to a defined value in the setup() function
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      securityPluginSetup: this.securityPluginSetup,
      securityPluginStart: plugins.security,
      securityServiceStart: core.security,
      spacesPluginStart: plugins.spaces,
      featuresPluginStart: plugins.features,
      actionsPluginStart: plugins.actions,
      licensingPluginStart: plugins.licensing,
      /**
       * Lens will be always defined as
       * it is declared as required plugin in kibana.json
       */
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
      lensEmbeddableFactory: this.lensEmbeddableFactory,
      persistableStateAttachmentTypeRegistry: this.persistableStateAttachmentTypeRegistry,
      externalReferenceAttachmentTypeRegistry: this.externalReferenceAttachmentTypeRegistry,
      publicBaseUrl: core.http.basePath.publicBaseUrl,
      notifications: plugins.notifications,
      ruleRegistry: plugins.ruleRegistry,
      filesPluginStart: plugins.files
    });
    return {
      getCasesClientWithRequest: this.getCasesClientWithRequest(core),
      getExternalReferenceAttachmentTypeRegistry: () => this.externalReferenceAttachmentTypeRegistry,
      getPersistableStateAttachmentTypeRegistry: () => this.persistableStateAttachmentTypeRegistry,
      config: this.caseConfig
    };
  }
  stop() {
    this.logger.debug(`Stopping Case Workflow`);
  }
}
exports.CasePlugin = CasePlugin;