"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getBackfillTelemetryPerDay = getBackfillTelemetryPerDay;
var _get_telemetry_from_event_log = require("./get_telemetry_from_event_log");
var _parse_simple_rule_type_bucket = require("./parse_simple_rule_type_bucket");
var _parse_and_log_error = require("./parse_and_log_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getBackfillExecutionCount({
  esClient,
  eventLogIndex,
  logger
}) {
  try {
    var _results$hits$total;
    const query = {
      index: eventLogIndex,
      size: 0,
      track_total_hits: true,
      query: (0, _get_telemetry_from_event_log.getProviderAndActionFilterForTimeRange)('execute-backfill'),
      aggs: {
        by_execution_status: {
          terms: {
            field: 'event.outcome'
          }
        }
      }
    };
    const results = await esClient.search(query);
    const totalBackfillExecutions = typeof results.hits.total === 'number' ? results.hits.total : (_results$hits$total = results.hits.total) === null || _results$hits$total === void 0 ? void 0 : _results$hits$total.value;
    const aggregations = results.aggregations;
    return {
      hasErrors: false,
      countExecutions: totalBackfillExecutions !== null && totalBackfillExecutions !== void 0 ? totalBackfillExecutions : 0,
      countBackfillsByExecutionStatus: (0, _parse_simple_rule_type_bucket.parseSimpleRuleTypeBucket)(aggregations.by_execution_status.buckets)
    };
  } catch (err) {
    const errorMessage = (0, _parse_and_log_error.parseAndLogError)(err, `getBackfillExecutionCount`, logger);
    return {
      hasErrors: true,
      errorMessage,
      countExecutions: 0,
      countBackfillsByExecutionStatus: {}
    };
  }
}
async function getGapInfo({
  esClient,
  eventLogIndex,
  logger
}) {
  try {
    var _results$hits$total2, _aggregations$total_u, _aggregations$total_f;
    const query = {
      index: eventLogIndex,
      track_total_hits: true,
      size: 0,
      query: (0, _get_telemetry_from_event_log.getProviderAndActionFilterForTimeRange)('gap'),
      aggs: {
        total_unfilled_duration_ms: {
          sum: {
            field: 'kibana.alert.rule.gap.unfilled_duration_ms'
          }
        },
        total_filled_duration_ms: {
          sum: {
            field: 'kibana.alert.rule.gap.filled_duration_ms'
          }
        }
      }
    };
    const results = await esClient.search(query);
    const totalGapsReported = typeof results.hits.total === 'number' ? results.hits.total : (_results$hits$total2 = results.hits.total) === null || _results$hits$total2 === void 0 ? void 0 : _results$hits$total2.value;
    const aggregations = results.aggregations;
    return {
      hasErrors: false,
      countGaps: totalGapsReported !== null && totalGapsReported !== void 0 ? totalGapsReported : 0,
      totalUnfilledGapDurationMs: (_aggregations$total_u = aggregations.total_unfilled_duration_ms.value) !== null && _aggregations$total_u !== void 0 ? _aggregations$total_u : 0,
      totalFilledGapDurationMs: (_aggregations$total_f = aggregations.total_filled_duration_ms.value) !== null && _aggregations$total_f !== void 0 ? _aggregations$total_f : 0
    };
  } catch (err) {
    const errorMessage = (0, _parse_and_log_error.parseAndLogError)(err, `getGapInfo`, logger);
    return {
      hasErrors: true,
      errorMessage,
      countGaps: 0,
      totalUnfilledGapDurationMs: 0,
      totalFilledGapDurationMs: 0
    };
  }
}
async function getBackfillTelemetryPerDay(opts) {
  const backfillResults = await getBackfillExecutionCount(opts);
  const gapResults = await getGapInfo(opts);
  const errorMessage = [backfillResults.errorMessage, gapResults.errorMessage].filter(message => !!message).join(',');
  return {
    hasErrors: backfillResults.hasErrors || gapResults.hasErrors,
    ...(errorMessage ? {
      errorMessage
    } : {}),
    countExecutions: backfillResults.countExecutions,
    countBackfillsByExecutionStatus: backfillResults.countBackfillsByExecutionStatus,
    countGaps: gapResults.countGaps,
    totalUnfilledGapDurationMs: gapResults.totalUnfilledGapDurationMs,
    totalFilledGapDurationMs: gapResults.totalFilledGapDurationMs
  };
}