"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.instrumentAsyncMethods = instrumentAsyncMethods;
var _with_span = require("./with_span");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Track instrumented functions to avoid double wrapping (no prototype pollution)
const INSTRUMENTED_FUNCS = new WeakSet();

// Helper to detect async functions
const isAsyncFunction = fn => {
  var _fn$constructor;
  return typeof fn === 'function' && ((_fn$constructor = fn.constructor) === null || _fn$constructor === void 0 ? void 0 : _fn$constructor.name) === 'AsyncFunction';
};
const TRANSPILED_ASYNC_MARKERS = ['__awaiter', '_asyncToGenerator', 'regeneratorRuntime'];

// Detect async functions transpiled to helpers such as __awaiter or _asyncToGenerator
const isTranspiledAsyncFunction = fn => {
  if (typeof fn !== 'function') {
    return false;
  }
  const source = Function.prototype.toString.call(fn);
  return TRANSPILED_ASYNC_MARKERS.some(marker => source.includes(marker));
};
const IGNORE_LIST = ['kibanaServer.request'];

/**
 * Wrap each async method on a class instance or plain object in a withSpan() call using the method name.
 * Mutates the target (and its prototype chain) in-place.
 */
function instrumentAsyncMethods(name, instance, getSpanOptions) {
  if (instance === null || typeof instance !== 'object') {
    return;
  }
  const visited = new Set();
  let current = instance;
  const shouldStopTraversal = target => target === null || typeof target !== 'object' || target === Object.prototype;
  while (current && typeof current === 'object' && !visited.has(current)) {
    visited.add(current);
    const propertyNames = Object.getOwnPropertyNames(current);
    for (const propertyName of propertyNames) {
      if (propertyName === 'constructor') {
        continue;
      }
      const spanName = `${name}.${propertyName}`;
      if (IGNORE_LIST.includes(spanName)) {
        return;
      }
      const descriptor = Object.getOwnPropertyDescriptor(current, propertyName);
      if (!descriptor || descriptor.get || descriptor.set) {
        continue;
      }
      const fn = descriptor.value;
      if (typeof fn !== 'function') {
        continue;
      }
      if (INSTRUMENTED_FUNCS.has(fn)) {
        continue;
      }
      if (!isAsyncFunction(fn) && !isTranspiledAsyncFunction(fn)) {
        continue;
      }
      const originalFn = fn;
      const wrappedFn = function wrapped(...args) {
        let spanOptions = {
          name: spanName
        };
        if (getSpanOptions) {
          spanOptions = getSpanOptions(spanOptions);
        }
        return (0, _with_span.withSpan)(spanOptions, () => originalFn.apply(this, args));
      };
      INSTRUMENTED_FUNCS.add(originalFn);
      INSTRUMENTED_FUNCS.add(wrappedFn);
      Object.defineProperty(current, propertyName, {
        ...descriptor,
        value: wrappedFn
      });
    }
    const nextPrototype = Object.getPrototypeOf(current);
    if (shouldStopTraversal(nextPrototype)) {
      break;
    }
    current = nextPrototype;
  }
}