"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AlertsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _error_with_type = require("../lib/error_with_type");
var _legacy_alerts_client = require("./legacy_alerts_client");
var _resource_installer_utils = require("../alerts_service/resource_installer_utils");
var _lib = require("./lib");
var _alerts_service = require("../alerts_service");
var _alert_conflict_resolver = require("./lib/alert_conflict_resolver");
var _maintenance_windows = require("../task_runner/maintenance_windows");
var _config = require("../config");
var _get_summarized_alerts_query = require("./lib/get_summarized_alerts_query");
var _retry_transient_es_errors = require("../lib/retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AlertsClient {
  constructor(options) {
    var _this$options$ruleTyp, _this$options$ruleTyp2;
    (0, _defineProperty2.default)(this, "legacyAlertsClient", void 0);
    // Query for alerts from the previous execution in order to identify the
    // correct index to use if and when we need to make updates to existing active or
    // recovered alerts
    (0, _defineProperty2.default)(this, "trackedAlerts", void 0);
    (0, _defineProperty2.default)(this, "startedAtString", null);
    (0, _defineProperty2.default)(this, "runTimestampString", void 0);
    (0, _defineProperty2.default)(this, "rule", void 0);
    (0, _defineProperty2.default)(this, "ruleType", void 0);
    (0, _defineProperty2.default)(this, "isServerless", void 0);
    (0, _defineProperty2.default)(this, "indexTemplateAndPattern", void 0);
    (0, _defineProperty2.default)(this, "reportedAlerts", {});
    (0, _defineProperty2.default)(this, "_isUsingDataStreams", void 0);
    (0, _defineProperty2.default)(this, "ruleInfoMessage", void 0);
    (0, _defineProperty2.default)(this, "logTags", void 0);
    this.options = options;
    this.legacyAlertsClient = new _legacy_alerts_client.LegacyAlertsClient({
      alertingEventLogger: this.options.alertingEventLogger,
      logger: this.options.logger,
      maintenanceWindowsService: this.options.maintenanceWindowsService,
      request: this.options.request,
      ruleType: this.options.ruleType,
      spaceId: this.options.spaceId
    });
    this.indexTemplateAndPattern = (0, _resource_installer_utils.getIndexTemplateAndPattern)({
      context: (_this$options$ruleTyp = this.options.ruleType.alerts) === null || _this$options$ruleTyp === void 0 ? void 0 : _this$options$ruleTyp.context,
      namespace: (_this$options$ruleTyp2 = this.options.ruleType.alerts) !== null && _this$options$ruleTyp2 !== void 0 && _this$options$ruleTyp2.isSpaceAware ? this.options.namespace : _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING
    });
    this.trackedAlerts = {
      indices: {},
      active: {},
      recovered: {},
      seqNo: {},
      primaryTerm: {},
      get(uuid) {
        var _this$active$uuid;
        return (_this$active$uuid = this.active[uuid]) !== null && _this$active$uuid !== void 0 ? _this$active$uuid : this.recovered[uuid];
      },
      getById(id) {
        var _Object$values$find;
        return (_Object$values$find = Object.values(this.active).find(alert => (0, _lodash.get)(alert, _ruleDataUtils.ALERT_INSTANCE_ID) === id)) !== null && _Object$values$find !== void 0 ? _Object$values$find : Object.values(this.recovered).find(alert => (0, _lodash.get)(alert, _ruleDataUtils.ALERT_INSTANCE_ID) === id);
      }
    };
    this.rule = (0, _lib.formatRule)({
      rule: this.options.rule,
      ruleType: this.options.ruleType
    });
    this.ruleType = options.ruleType;
    this._isUsingDataStreams = this.options.dataStreamAdapter.isUsingDataStreams();
    this.ruleInfoMessage = `for ${this.ruleType.id}:${this.options.rule.id} '${this.options.rule.name}'`;
    this.logTags = {
      tags: [this.ruleType.id, this.options.rule.id, 'alerts-client']
    };
    this.isServerless = options.isServerless;
  }
  async initializeExecution(opts) {
    this.startedAtString = opts.startedAt ? opts.startedAt.toISOString() : null;
    const {
      runTimestamp
    } = opts;
    if (runTimestamp) {
      this.runTimestampString = runTimestamp.toISOString();
    }
    await this.legacyAlertsClient.initializeExecution(opts);

    // No need to fetch the tracked alerts for the non-lifecycle rules
    if (this.ruleType.autoRecoverAlerts) {
      const maxAlertLimit = this.legacyAlertsClient.getMaxAlertLimit();
      const getTrackedAlerts = async () => {
        // We can use inner_hits to get the alerts for the most recent executions
        // but this may return too many alerts, therefore we make two queries:
        // 1. Get the most recent execution UUIDs
        // 2. Get the alerts for those execution UUIDs
        // We can optimize this in the future once https://github.com/elastic/kibana/issues/235846 is
        // implemented to allow filtering the ongoing recovered alerts. Then we can just query for
        // the alerts of the latest execution by setting the size to 1 and adding inner_hits.
        const executions = await this.search({
          size: opts.flappingSettings.lookBackWindow,
          query: {
            bool: {
              must: [{
                term: {
                  [_ruleDataUtils.ALERT_RULE_UUID]: this.options.rule.id
                }
              }]
            }
          },
          collapse: {
            field: _ruleDataUtils.ALERT_RULE_EXECUTION_UUID
          },
          _source: false,
          sort: [{
            [_ruleDataUtils.TIMESTAMP]: {
              order: 'desc'
            }
          }]
        });
        const executionUuids = (executions.hits || []).map(hit => (0, _lodash.get)(hit.fields, _ruleDataUtils.ALERT_RULE_EXECUTION_UUID)).flat();
        const alerts = await this.search({
          size: (maxAlertLimit || _config.DEFAULT_MAX_ALERTS) * 2,
          seq_no_primary_term: true,
          query: {
            bool: {
              must: [{
                term: {
                  [_ruleDataUtils.ALERT_RULE_UUID]: this.options.rule.id
                }
              }],
              must_not: [{
                term: {
                  [_ruleDataUtils.ALERT_STATUS]: _ruleDataUtils.ALERT_STATUS_UNTRACKED
                }
              }],
              filter: [{
                terms: {
                  [_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]: executionUuids
                }
              }]
            }
          }
        });
        return alerts.hits;
      };
      try {
        const results = await getTrackedAlerts();
        for (const hit of results) {
          const alertHit = hit._source;
          const alertUuid = (0, _lodash.get)(alertHit, _ruleDataUtils.ALERT_UUID);
          if ((0, _lodash.get)(alertHit, _ruleDataUtils.ALERT_STATUS) === _ruleDataUtils.ALERT_STATUS_ACTIVE) {
            this.trackedAlerts.active[alertUuid] = alertHit;
          }
          if ((0, _lodash.get)(alertHit, _ruleDataUtils.ALERT_STATUS) === _ruleDataUtils.ALERT_STATUS_RECOVERED) {
            this.trackedAlerts.recovered[alertUuid] = alertHit;
          }
          this.trackedAlerts.indices[alertUuid] = hit._index;
          this.trackedAlerts.seqNo[alertUuid] = hit._seq_no;
          this.trackedAlerts.primaryTerm[alertUuid] = hit._primary_term;
        }
      } catch (err) {
        this.options.logger.error(`Error searching for tracked alerts by UUID ${this.ruleInfoMessage} - ${err.message}`, this.logTags);
        throw err;
      }
    }
  }
  async search(queryBody) {
    const esClient = await this.options.elasticsearchClientPromise;
    const index = this.isUsingDataStreams() ? this.indexTemplateAndPattern.alias : this.indexTemplateAndPattern.pattern;
    const {
      hits: {
        hits,
        total
      },
      aggregations
    } = await esClient.search({
      index,
      ...queryBody,
      ignore_unavailable: true
    });
    return {
      hits,
      total,
      aggregations
    };
  }
  async msearch(searches) {
    const esClient = await this.options.elasticsearchClientPromise;
    const index = this.isUsingDataStreams() ? this.indexTemplateAndPattern.alias : this.indexTemplateAndPattern.pattern;
    const {
      responses
    } = await esClient.msearch({
      index,
      searches,
      ignore_unavailable: true
    });
    return responses;
  }
  report(alert) {
    var _legacyAlert$getStart;
    const context = alert.context ? alert.context : {};
    const state = !(0, _lodash.isEmpty)(alert.state) ? alert.state : null;

    // Create a legacy alert
    const legacyAlert = this.legacyAlertsClient.factory().create(alert.id).scheduleActions(alert.actionGroup, context);
    if (state) {
      legacyAlert.replaceState(state);
    }

    // Save the alert payload
    if (alert.payload) {
      this.reportedAlerts[alert.id] = alert.payload;
    }
    return {
      uuid: legacyAlert.getUuid(),
      start: (_legacyAlert$getStart = legacyAlert.getStart()) !== null && _legacyAlert$getStart !== void 0 ? _legacyAlert$getStart : this.startedAtString,
      alertDoc: this.trackedAlerts.getById(alert.id)
    };
  }
  setAlertData(alert) {
    const context = alert.context ? alert.context : {};

    // Allow setting context and payload on known alerts only
    // Alerts are known if they have been reported in this execution or are recovered
    const alertToUpdate = this.legacyAlertsClient.getAlert(alert.id);
    if (!alertToUpdate) {
      throw new Error(`Cannot set alert data for alert ${alert.id} because it has not been reported and it is not recovered.`);
    }

    // Set the alert context
    alertToUpdate.setContext(context);

    // Save the alert payload
    if (alert.payload) {
      this.reportedAlerts[alert.id] = alert.payload;
    }
  }
  isTrackedAlert(id) {
    const alert = this.trackedAlerts.getById(id);
    const uuid = alert === null || alert === void 0 ? void 0 : alert[_ruleDataUtils.ALERT_UUID];
    if (uuid) {
      return !!this.trackedAlerts.active[uuid];
    }
    return false;
  }
  hasReachedAlertLimit() {
    return this.legacyAlertsClient.hasReachedAlertLimit();
  }
  getMaxAlertLimit() {
    return this.legacyAlertsClient.getMaxAlertLimit();
  }
  checkLimitUsage() {
    return this.legacyAlertsClient.checkLimitUsage();
  }
  async processAlerts() {
    await this.legacyAlertsClient.processAlerts();
  }
  determineFlappingAlerts() {
    this.legacyAlertsClient.determineFlappingAlerts();
  }
  determineDelayedAlerts(opts) {
    this.legacyAlertsClient.determineDelayedAlerts(opts);
  }
  logAlerts(opts) {
    this.legacyAlertsClient.logAlerts(opts);
  }
  getProcessedAlerts(type) {
    return this.legacyAlertsClient.getProcessedAlerts(type);
  }
  getRawAlertInstancesForState(shouldOptimizeTaskState) {
    return this.legacyAlertsClient.getRawAlertInstancesForState(shouldOptimizeTaskState);
  }
  factory() {
    return this.legacyAlertsClient.factory();
  }
  async getSummarizedAlerts({
    ruleId,
    spaceId,
    excludedAlertInstanceIds,
    alertsFilter,
    start,
    end,
    executionUuid
  }) {
    var _this$ruleType$alerts, _this$ruleType$autoRe;
    if (!ruleId || !spaceId) {
      throw new Error(`Must specify both rule ID and space ID for AAD alert query.`);
    }
    const queryByExecutionUuid = !!executionUuid;
    const queryByTimeRange = !!start && !!end;
    // Either executionUuid or start/end dates must be specified, but not both
    if (!queryByExecutionUuid && !queryByTimeRange || queryByExecutionUuid && queryByTimeRange) {
      throw new Error(`Must specify either execution UUID or time range for AAD alert query.`);
    }
    const maxAlertLimit = this.legacyAlertsClient.getMaxAlertLimit();
    const getQueryParams = {
      executionUuid,
      start,
      end,
      ruleId,
      excludedAlertInstanceIds,
      alertsFilter,
      maxAlertLimit
    };
    const formatAlert = (_this$ruleType$alerts = this.ruleType.alerts) === null || _this$ruleType$alerts === void 0 ? void 0 : _this$ruleType$alerts.formatAlert;
    const isLifecycleAlert = (_this$ruleType$autoRe = this.ruleType.autoRecoverAlerts) !== null && _this$ruleType$autoRe !== void 0 ? _this$ruleType$autoRe : false;
    if (isLifecycleAlert) {
      const queryBodies = (0, _lib.getLifecycleAlertsQueries)(getQueryParams);
      const responses = await Promise.all(queryBodies.map(queryBody => this.search(queryBody)));
      return {
        new: (0, _lib.getHitsWithCount)(responses[0], formatAlert),
        ongoing: (0, _lib.getHitsWithCount)(responses[1], formatAlert),
        recovered: (0, _lib.getHitsWithCount)(responses[2], formatAlert)
      };
    }
    const response = await this.search((0, _lib.getContinualAlertsQuery)(getQueryParams));
    return {
      new: (0, _lib.getHitsWithCount)(response, formatAlert),
      ongoing: {
        count: 0,
        data: []
      },
      recovered: {
        count: 0,
        data: []
      }
    };
  }
  async persistAlerts() {
    var _this$ruleType$alerts2, _this$startedAtString;
    if (!((_this$ruleType$alerts2 = this.ruleType.alerts) !== null && _this$ruleType$alerts2 !== void 0 && _this$ruleType$alerts2.shouldWrite)) {
      var _this$ruleType$alerts3;
      this.options.logger.debug(`Resources registered and installed for ${(_this$ruleType$alerts3 = this.ruleType.alerts) === null || _this$ruleType$alerts3 === void 0 ? void 0 : _this$ruleType$alerts3.context} context but "shouldWrite" is set to false ${this.ruleInfoMessage}.`, this.logTags);
      return;
    }
    const currentTime = (_this$startedAtString = this.startedAtString) !== null && _this$startedAtString !== void 0 ? _this$startedAtString : new Date().toISOString();
    const esClient = await this.options.elasticsearchClientPromise;
    const createAlertsInAllSpaces = (0, _lib.shouldCreateAlertsInAllSpaces)({
      ruleTypeId: this.ruleType.id,
      ruleTypeAlertDef: this.ruleType.alerts,
      logger: this.options.logger
    });
    const {
      rawActiveAlerts,
      rawRecoveredAlerts
    } = this.getRawAlertInstancesForState();
    const activeAlerts = this.legacyAlertsClient.getProcessedAlerts(_ruleDataUtils.ALERT_STATUS_ACTIVE);
    const recoveredAlerts = this.legacyAlertsClient.getProcessedAlerts(_ruleDataUtils.ALERT_STATUS_RECOVERED);

    // TODO - Lifecycle alerts set some other fields based on alert status
    // Example: workflow status - default to 'open' if not set
    // event action: new alert = 'new', active alert: 'active', otherwise 'close'

    const activeAlertsToIndex = [];
    for (const id of (0, _lodash.keys)(rawActiveAlerts)) {
      // See if there's an existing active alert document
      if (activeAlerts[id]) {
        const trackedAlert = this.trackedAlerts.get(activeAlerts[id].getUuid());
        if (!!trackedAlert && (0, _lodash.get)(trackedAlert, _ruleDataUtils.ALERT_STATUS) === _ruleDataUtils.ALERT_STATUS_ACTIVE) {
          const isImproving = (0, _lib.isAlertImproving)(trackedAlert, activeAlerts[id], this.ruleType.actionGroups);
          activeAlertsToIndex.push((0, _lib.buildOngoingAlert)({
            alert: trackedAlert,
            legacyAlert: activeAlerts[id],
            rule: this.rule,
            isImproving,
            runTimestamp: this.runTimestampString,
            timestamp: currentTime,
            payload: this.reportedAlerts[id],
            kibanaVersion: this.options.kibanaVersion,
            dangerouslyCreateAlertsInAllSpaces: createAlertsInAllSpaces
          }));
        } else {
          // skip writing the alert document if the number of consecutive
          // active alerts is less than the rule alertDelay threshold
          if (activeAlerts[id].getActiveCount() < this.options.rule.alertDelay) {
            continue;
          }
          activeAlertsToIndex.push((0, _lib.buildNewAlert)({
            legacyAlert: activeAlerts[id],
            rule: this.rule,
            runTimestamp: this.runTimestampString,
            timestamp: currentTime,
            payload: this.reportedAlerts[id],
            kibanaVersion: this.options.kibanaVersion,
            dangerouslyCreateAlertsInAllSpaces: createAlertsInAllSpaces
          }));
        }
      } else {
        this.options.logger.error(`Error writing alert(${id}) to ${this.indexTemplateAndPattern.alias} - alert(${id}) doesn't exist in active alerts ${this.ruleInfoMessage}.`, this.logTags);
      }
    }
    const recoveredAlertsToIndex = [];
    for (const id of (0, _lodash.keys)(rawRecoveredAlerts)) {
      const trackedAlert = this.trackedAlerts.getById(id);
      // See if there's an existing alert document
      // If there is not, log an error because there should be
      if (trackedAlert) {
        recoveredAlertsToIndex.push(recoveredAlerts[id] ? (0, _lib.buildRecoveredAlert)({
          alert: trackedAlert,
          legacyAlert: recoveredAlerts[id],
          rule: this.rule,
          runTimestamp: this.runTimestampString,
          timestamp: currentTime,
          payload: this.reportedAlerts[id],
          recoveryActionGroup: this.options.ruleType.recoveryActionGroup.id,
          kibanaVersion: this.options.kibanaVersion,
          dangerouslyCreateAlertsInAllSpaces: createAlertsInAllSpaces
        }) : (0, _lib.buildUpdatedRecoveredAlert)({
          alert: trackedAlert,
          legacyRawAlert: rawRecoveredAlerts[id],
          runTimestamp: this.runTimestampString,
          timestamp: currentTime,
          rule: this.rule
        }));
      }
    }
    const alertsToIndex = [...activeAlertsToIndex, ...recoveredAlertsToIndex].filter(alert => {
      const alertUuid = (0, _lodash.get)(alert, _ruleDataUtils.ALERT_UUID);
      const alertIndex = this.trackedAlerts.indices[alertUuid];
      if (!alertIndex) {
        return true;
      } else if (!(0, _alerts_service.isValidAlertIndexName)(alertIndex)) {
        this.options.logger.warn(`Could not update alert ${alertUuid} in ${alertIndex}. Partial and restored alert indices are not supported ${this.ruleInfoMessage}.`, this.logTags);
        return false;
      }
      return true;
    });
    if (alertsToIndex.length > 0) {
      const bulkBody = (0, _lodash.flatMap)(alertsToIndex.map(alert => {
        const alertUuid = (0, _lodash.get)(alert, _ruleDataUtils.ALERT_UUID);
        return [getBulkMeta(alertUuid, this.trackedAlerts.indices[alertUuid], this.trackedAlerts.seqNo[alertUuid], this.trackedAlerts.primaryTerm[alertUuid], this.isUsingDataStreams()), alert];
      }));
      try {
        const response = await esClient.bulk({
          // On serverless we can force a refresh to we don't wait for the longer refresh interval
          // When too many refresh calls are done in a short period of time, they are throttled by stateless Elasticsearch
          refresh: this.isServerless ? true : 'wait_for',
          index: this.indexTemplateAndPattern.alias,
          require_alias: !this.isUsingDataStreams(),
          body: bulkBody
        });

        // If there were individual indexing errors, they will be returned in the success response
        if (response && response.errors) {
          this.throwIfHasClusterBlockException(response);
          await (0, _alert_conflict_resolver.resolveAlertConflicts)({
            logger: this.options.logger,
            esClient,
            bulkRequest: {
              refresh: 'wait_for',
              index: this.indexTemplateAndPattern.alias,
              require_alias: !this.isUsingDataStreams(),
              operations: bulkBody
            },
            bulkResponse: response,
            ruleId: this.options.rule.id,
            ruleName: this.options.rule.name,
            ruleType: this.ruleType.id
          });
        }
      } catch (err) {
        this.options.logger.error(`Error writing ${alertsToIndex.length} alerts to ${this.indexTemplateAndPattern.alias} ${this.ruleInfoMessage} - ${err.message}`, this.logTags);
        throw err;
      }
    }
    function getBulkMeta(uuid, index, seqNo, primaryTerm, isUsingDataStreams) {
      if (index && seqNo != null && primaryTerm != null) {
        return {
          index: {
            _id: uuid,
            _index: index,
            if_seq_no: seqNo,
            if_primary_term: primaryTerm,
            require_alias: false
          }
        };
      }
      return {
        create: {
          _id: uuid,
          ...(isUsingDataStreams ? {} : {
            require_alias: true
          })
        }
      };
    }
  }
  async updatePersistedAlerts({
    alertsToUpdateWithMaintenanceWindows,
    alertsToUpdateWithLastScheduledActions
  }) {
    const idsToUpdate = new Set([...Object.keys(alertsToUpdateWithMaintenanceWindows), ...Object.keys(alertsToUpdateWithLastScheduledActions)]);
    if (idsToUpdate.size === 0) {
      return;
    }
    try {
      const esClient = await this.options.elasticsearchClientPromise;
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => {
        return esClient.updateByQuery({
          query: {
            terms: {
              _id: [...idsToUpdate]
            }
          },
          conflicts: 'proceed',
          index: this.indexTemplateAndPattern.alias,
          script: {
            source: `
                if (params.toScheduledAction.containsKey(ctx._source['${_ruleDataUtils.ALERT_UUID}'])) {
                  ctx._source['${_ruleDataUtils.ALERT_SCHEDULED_ACTION_GROUP}'] = params.toScheduledAction[ctx._source['${_ruleDataUtils.ALERT_UUID}']].group;
                  ctx._source['${_ruleDataUtils.ALERT_SCHEDULED_ACTION_DATE}'] = params.toScheduledAction[ctx._source['${_ruleDataUtils.ALERT_UUID}']].date;
                  if (params.toScheduledAction[ctx._source['${_ruleDataUtils.ALERT_UUID}']].containsKey('throttling')) {
                    ctx._source['${_ruleDataUtils.ALERT_SCHEDULED_ACTION_THROTTLING}'] = params.toScheduledAction[ctx._source['${_ruleDataUtils.ALERT_UUID}']].throttling;
                  }
                }
                if (params.toMaintenanceWindows.containsKey(ctx._source['${_ruleDataUtils.ALERT_UUID}'])) {
                  ctx._source['${_ruleDataUtils.ALERT_MAINTENANCE_WINDOW_IDS}'] = params.toMaintenanceWindows[ctx._source['${_ruleDataUtils.ALERT_UUID}']];
                }
              `,
            lang: 'painless',
            params: {
              toScheduledAction: alertsToUpdateWithLastScheduledActions,
              toMaintenanceWindows: alertsToUpdateWithMaintenanceWindows
            }
          }
        });
      }, {
        logger: this.options.logger
      });
    } catch (err) {
      this.options.logger.error(`Error updating alerts. (last scheduled actions or maintenance windows) ${this.ruleInfoMessage}: ${err}`, this.logTags);
      throw err;
    }
  }
  async getMaintenanceWindowScopedQueryAlerts({
    ruleId,
    spaceId,
    executionUuid,
    maintenanceWindows
  }) {
    var _this$ruleType$autoRe2;
    if (!ruleId || !spaceId || !executionUuid) {
      throw new Error(`Must specify rule ID, space ID, and executionUuid for scoped query AAD alert query.`);
    }
    const isLifecycleAlert = (_this$ruleType$autoRe2 = this.ruleType.autoRecoverAlerts) !== null && _this$ruleType$autoRe2 !== void 0 ? _this$ruleType$autoRe2 : false;
    const maxAlertLimit = this.legacyAlertsClient.getMaxAlertLimit();
    const searches = (0, _lib.getMaintenanceWindowAlertsQuery)({
      executionUuid,
      ruleId,
      maintenanceWindows,
      action: isLifecycleAlert ? 'open' : undefined,
      maxAlertLimit
    });
    const responses = await this.msearch(searches);
    const alertsByMaintenanceWindowIds = {};
    responses.forEach(response => {
      if ('error' in response) {
        this.options.logger.error(`Error fetching scoped query alerts for maintenance windows ${this.ruleInfoMessage}: ${response.error.reason}`, this.logTags);
        return;
      }
      response.hits.hits.forEach(({
        fields
      }) => {
        if (!fields) {
          return;
        }
        const mwIdField = fields[_get_summarized_alerts_query.RUNTIME_MAINTENANCE_WINDOW_ID_FIELD];
        if (!alertsByMaintenanceWindowIds[mwIdField]) {
          alertsByMaintenanceWindowIds[mwIdField] = [];
        }
        alertsByMaintenanceWindowIds[mwIdField].push((0, _lodash.get)(fields, _ruleDataUtils.ALERT_UUID)[0]);
      });
    });
    return alertsByMaintenanceWindowIds;
  }
  async getAlertsToUpdateWithMaintenanceWindows() {
    try {
      // check if there are any alerts
      const newAlerts = Object.values(this.legacyAlertsClient.getProcessedAlerts('new'));
      const activeAlerts = Object.values(this.legacyAlertsClient.getProcessedAlerts('active'));
      const recoveredAlerts = Object.values(this.legacyAlertsClient.getProcessedAlerts('recovered'));

      // return if there are no alerts written
      if (!newAlerts.length && !activeAlerts.length && !recoveredAlerts.length || !this.options.maintenanceWindowsService) {
        return {};
      }
      const {
        maintenanceWindows
      } = await this.options.maintenanceWindowsService.getMaintenanceWindows({
        eventLogger: this.options.alertingEventLogger,
        request: this.options.request,
        ruleTypeCategory: this.ruleType.category,
        spaceId: this.options.spaceId
      });
      const maintenanceWindowsWithScopedQuery = (0, _maintenance_windows.filterMaintenanceWindows)({
        maintenanceWindows: maintenanceWindows !== null && maintenanceWindows !== void 0 ? maintenanceWindows : [],
        withScopedQuery: true
      });
      const maintenanceWindowsWithoutScopedQueryIds = (0, _maintenance_windows.filterMaintenanceWindowsIds)({
        maintenanceWindows: maintenanceWindows !== null && maintenanceWindows !== void 0 ? maintenanceWindows : [],
        withScopedQuery: false
      });
      if (maintenanceWindowsWithScopedQuery.length === 0) {
        return {};
      }

      // Run aggs to get all scoped query alert IDs, returns a record<maintenanceWindowId, alertIds>,
      // indicating the maintenance window has matches a number of alerts with the scoped query.
      const alertsByMaintenanceWindowIds = await this.getMaintenanceWindowScopedQueryAlerts({
        ruleId: this.options.rule.id,
        spaceId: this.options.rule.spaceId,
        executionUuid: this.options.rule.executionId,
        maintenanceWindows: maintenanceWindowsWithScopedQuery
      });
      const alertsAffectedByScopedQuery = [];
      const appliedMaintenanceWindowIds = [];
      for (const [scopedQueryMaintenanceWindowId, alertIds] of Object.entries(alertsByMaintenanceWindowIds)) {
        // Go through matched alerts, find the in memory object
        alertIds.forEach(alertId => {
          const newAlert = newAlerts.find(alert => alert.matchesUuid(alertId));
          if (!newAlert) {
            return;
          }
          const newMaintenanceWindowIds = [
          // Keep existing Ids
          ...newAlert.getMaintenanceWindowIds(),
          // Add the ids that don't have scoped queries
          ...maintenanceWindowsWithoutScopedQueryIds,
          // Add the scoped query id
          scopedQueryMaintenanceWindowId];

          // Update in memory alert with new maintenance window IDs
          newAlert.setMaintenanceWindowIds([...new Set(newMaintenanceWindowIds)]);
          alertsAffectedByScopedQuery.push(alertId);
          appliedMaintenanceWindowIds.push(...newMaintenanceWindowIds);
        });
      }
      const uniqueAlertsId = [...new Set(alertsAffectedByScopedQuery)];
      const uniqueMaintenanceWindowIds = [...new Set(appliedMaintenanceWindowIds)];
      if (uniqueMaintenanceWindowIds && uniqueMaintenanceWindowIds.length > 0) {
        this.options.alertingEventLogger.setMaintenanceWindowIds(uniqueMaintenanceWindowIds);
      }
      const alertsAffectedByMaintenanceWindows = {};
      uniqueAlertsId.forEach(id => {
        const newAlert = newAlerts.find(alert => alert.matchesUuid(id));
        if (newAlert) {
          alertsAffectedByMaintenanceWindows[id] = newAlert.getMaintenanceWindowIds();
        }
      });
      return alertsAffectedByMaintenanceWindows;
    } catch (err) {
      this.options.logger.error(`Error getting alerts affected by maintenance windows: ${err.message}`, this.logTags);
      return {};
    }
  }
  getAlertsToUpdateWithLastScheduledActions() {
    const {
      rawActiveAlerts
    } = this.getRawAlertInstancesForState(true);
    const result = {};
    try {
      for (const key in rawActiveAlerts) {
        if (key) {
          const {
            meta
          } = rawActiveAlerts[key];
          const uuid = meta === null || meta === void 0 ? void 0 : meta.uuid;
          const last = meta === null || meta === void 0 ? void 0 : meta.lastScheduledActions;
          if (!uuid || !last) continue;
          const {
            group,
            date,
            actions
          } = last;
          result[uuid] = actions ? {
            group,
            date,
            throttling: actions
          } : {
            group,
            date
          };
        }
      }
    } catch (err) {
      this.options.logger.error(`Error getting alerts to update with last scheduled actions: ${err.message}`, this.logTags);
    }
    return result;
  }
  client() {
    return {
      report: alert => this.report(alert),
      isTrackedAlert: id => this.isTrackedAlert(id),
      setAlertData: alert => this.setAlertData(alert),
      getAlertLimitValue: () => this.factory().alertLimit.getValue(),
      setAlertLimitReached: reached => this.factory().alertLimit.setLimitReached(reached),
      getRecoveredAlerts: () => {
        var _getRecoveredAlerts;
        const {
          getRecoveredAlerts
        } = this.factory().done();
        const recoveredLegacyAlerts = (_getRecoveredAlerts = getRecoveredAlerts()) !== null && _getRecoveredAlerts !== void 0 ? _getRecoveredAlerts : [];
        return recoveredLegacyAlerts.map(alert => ({
          alert,
          hit: this.trackedAlerts.get(alert.getUuid())
        }));
      }
    };
  }
  isUsingDataStreams() {
    return this._isUsingDataStreams;
  }
  throwIfHasClusterBlockException(response) {
    response.items.forEach(item => {
      const op = item.create || item.index || item.update || item.delete;
      if (op !== null && op !== void 0 && op.error && op.error.type === _error_with_type.CLUSTER_BLOCK_EXCEPTION) {
        throw new _error_with_type.ErrorWithType({
          message: op.error.reason || 'Unknown reason',
          type: _error_with_type.CLUSTER_BLOCK_EXCEPTION,
          stack: op.error.stack_trace
        });
      }
    });
  }
}
exports.AlertsClient = AlertsClient;