"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchMiniHistogramsForSignificantItems = void 0;
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _get_category_query = require("@kbn/aiops-log-pattern-analysis/get_category_query");
var _constants = require("../constants");
var _get_histogram_query = require("./get_histogram_query");
var _mini_histogram_utils = require("./mini_histogram_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchMiniHistogramsForSignificantItems = async (esClient, params, significantItems, overallTimeSeries, logger,
// The default value of 1 means no sampling will be used
randomSamplerProbability = 1, emitError, abortSignal) => {
  const histogramQuery = (0, _get_histogram_query.getHistogramQuery)(params);
  const histogramAggs = significantItems.reduce((aggs, significantItem, index) => {
    let filter;
    if ((0, _mlAggUtils.isSignificantItem)(significantItem) && significantItem.type === 'keyword') {
      filter = {
        term: {
          [significantItem.fieldName]: significantItem.fieldValue
        }
      };
    } else if ((0, _mlAggUtils.isSignificantItem)(significantItem) && significantItem.type === 'log_pattern') {
      filter = (0, _get_category_query.getCategoryQuery)(significantItem.fieldName, [{
        key: `${significantItem.key}`,
        count: significantItem.doc_count,
        examples: [],
        regex: ''
      }]);
    } else {
      throw new Error('Invalid significant item type.');
    }
    aggs[`${_mini_histogram_utils.MINI_HISTOGRAM_AGG_PREFIX}${index}`] = {
      filter,
      aggs: (0, _mini_histogram_utils.getMiniHistogramAgg)(params)
    };
    return aggs;
  }, {});
  const {
    wrap,
    unwrap
  } = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: randomSamplerProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  const resp = await esClient.search({
    index: params.index,
    query: histogramQuery,
    aggs: wrap(histogramAggs),
    size: 0
  }, {
    signal: abortSignal,
    maxRetries: 0
  });
  if (resp.aggregations === undefined) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(resp)) {
      if (logger) {
        logger.error(`Failed to fetch the histogram data chunk, got: \n${JSON.stringify(resp, null, 2)}`);
      }
      if (emitError) {
        emitError(`Failed to fetch the histogram data chunk.`);
      }
    }
    return [];
  }
  const unwrappedResp = unwrap(resp.aggregations);
  return significantItems.map((significantItem, index) => ({
    fieldName: significantItem.fieldName,
    fieldValue: significantItem.fieldValue,
    histogram: (0, _mini_histogram_utils.getMiniHistogramDataFromAggResponse)(overallTimeSeries, unwrappedResp, index)
  }));
};
exports.fetchMiniHistogramsForSignificantItems = fetchMiniHistogramsForSignificantItems;