"use strict";
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsNested = void 0;
exports.EcsNested = {
    agent: {
        description: 'The agent fields contain the data about the software entity, if any, that collects, detects, or observes events on a host, or takes measurements on a host.\nExamples include Beats. Agents may also run on observers. ECS agent.* fields shall be populated with details of the agent running on the host or observer where the event happened or the measurement was taken.',
        fields: {
            'agent.build.original': {
                dashed_name: 'agent-build-original',
                description: 'Extended build information for the agent.\nThis field is intended to contain any build information that a data source may provide, no specific formatting is required.',
                example: 'metricbeat version 7.6.0 (amd64), libbeat 7.6.0 [6a23e8f8f30f5001ba344e4e54d8d9cb82cb107c built 2020-02-05 23:10:10 +0000 UTC]',
                flat_name: 'agent.build.original',
                ignore_above: 1024,
                level: 'core',
                name: 'build.original',
                normalize: [],
                short: 'Extended build information for the agent.',
                type: 'keyword',
            },
            'agent.ephemeral_id': {
                dashed_name: 'agent-ephemeral-id',
                description: 'Ephemeral identifier of this agent (if one exists).\nThis id normally changes across restarts, but `agent.id` does not.',
                example: '8a4f500f',
                flat_name: 'agent.ephemeral_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'ephemeral_id',
                normalize: [],
                short: 'Ephemeral identifier of this agent.',
                type: 'keyword',
            },
            'agent.id': {
                dashed_name: 'agent-id',
                description: 'Unique identifier of this agent (if one exists).\nExample: For Beats this would be beat.id.',
                example: '8a4f500d',
                flat_name: 'agent.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                short: 'Unique identifier of this agent.',
                type: 'keyword',
            },
            'agent.name': {
                dashed_name: 'agent-name',
                description: 'Custom name of the agent.\nThis is a name that can be given to an agent. This can be helpful if for example two Filebeat instances are running on the same host but a human readable separation is needed on which Filebeat instance data is coming from.',
                example: 'foo',
                flat_name: 'agent.name',
                ignore_above: 1024,
                level: 'core',
                name: 'name',
                normalize: [],
                short: 'Custom name of the agent.',
                type: 'keyword',
            },
            'agent.type': {
                dashed_name: 'agent-type',
                description: 'Type of the agent.\nThe agent type always stays the same and should be given by the agent used. In case of Filebeat the agent would always be Filebeat also if two Filebeat instances are run on the same machine.',
                example: 'filebeat',
                flat_name: 'agent.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: [],
                short: 'Type of the agent.',
                type: 'keyword',
            },
            'agent.version': {
                dashed_name: 'agent-version',
                description: 'Version of the agent.',
                example: '6.0.0-rc2',
                flat_name: 'agent.version',
                ignore_above: 1024,
                level: 'core',
                name: 'version',
                normalize: [],
                short: 'Version of the agent.',
                type: 'keyword',
            },
        },
        footnote: 'Examples: In the case of Beats for logs, the agent.name is filebeat. For APM, it is the agent running in the app/service. The agent information does not change if data is sent through queuing systems like Kafka, Redis, or processing systems such as Logstash or APM Server.',
        group: 2,
        name: 'agent',
        prefix: 'agent.',
        short: 'Fields about the monitoring agent.',
        title: 'Agent',
        type: 'group',
    },
    as: {
        description: 'An autonomous system (AS) is a collection of connected Internet Protocol (IP) routing prefixes under the control of one or more network operators on behalf of a single administrative entity or domain that presents a common, clearly defined routing policy to the internet.',
        fields: {
            'as.number': {
                dashed_name: 'as-number',
                description: 'Unique number allocated to the autonomous system. The autonomous system number (ASN) uniquely identifies each network on the Internet.',
                example: 15169,
                flat_name: 'as.number',
                level: 'extended',
                name: 'number',
                normalize: [],
                short: 'Unique number allocated to the autonomous system.',
                type: 'long',
            },
            'as.organization.name': {
                dashed_name: 'as-organization-name',
                description: 'Organization name.',
                example: 'Google LLC',
                flat_name: 'as.organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'as.organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'organization.name',
                normalize: [],
                short: 'Organization name.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'as',
        prefix: 'as.',
        reusable: {
            expected: [
                { as: 'as', at: 'client', full: 'client.as' },
                { as: 'as', at: 'destination', full: 'destination.as' },
                { as: 'as', at: 'server', full: 'server.as' },
                { as: 'as', at: 'source', full: 'source.as' },
                { as: 'as', at: 'threat.indicator', full: 'threat.indicator.as' },
                {
                    as: 'as',
                    at: 'threat.enrichments.indicator',
                    full: 'threat.enrichments.indicator.as',
                },
            ],
            top_level: false,
        },
        short: 'Fields describing an Autonomous System (Internet routing prefix).',
        title: 'Autonomous System',
        type: 'group',
    },
    base: {
        description: 'The `base` field set contains all fields which are at the root of the events. These fields are common across all types of events.',
        fields: {
            '@timestamp': {
                dashed_name: 'timestamp',
                description: 'Date/time when the event originated.\nThis is the date/time extracted from the event, typically representing when the event was generated by the source.\nIf the event source has no original timestamp, this value is typically populated by the first time the event was received by the pipeline.\nRequired field for all events.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: '@timestamp',
                level: 'core',
                name: '@timestamp',
                normalize: [],
                required: true,
                short: 'Date/time when the event originated.',
                type: 'date',
            },
            labels: {
                dashed_name: 'labels',
                description: 'Custom key/value pairs.\nCan be used to add meta information to events. Should not contain nested objects. All values are stored as keyword.\nExample: `docker` and `k8s` labels.',
                example: '{"application": "foo-bar", "env": "production"}',
                flat_name: 'labels',
                level: 'core',
                name: 'labels',
                normalize: [],
                object_type: 'keyword',
                short: 'Custom key/value pairs.',
                type: 'object',
            },
            message: {
                dashed_name: 'message',
                description: 'For log events the message field contains the log message, optimized for viewing in a log viewer.\nFor structured logs without an original message field, other fields can be concatenated to form a human-readable summary of the event.\nIf multiple messages exist, they can be combined into one message.',
                example: 'Hello World',
                flat_name: 'message',
                level: 'core',
                name: 'message',
                normalize: [],
                short: 'Log message optimized for viewing in a log viewer.',
                type: 'match_only_text',
            },
            tags: {
                dashed_name: 'tags',
                description: 'List of keywords used to tag each event.',
                example: '["production", "env2"]',
                flat_name: 'tags',
                ignore_above: 1024,
                level: 'core',
                name: 'tags',
                normalize: ['array'],
                short: 'List of keywords used to tag each event.',
                type: 'keyword',
            },
        },
        group: 1,
        name: 'base',
        prefix: '',
        root: true,
        short: 'All fields defined directly at the root of the events.',
        title: 'Base',
        type: 'group',
    },
    client: {
        description: 'A client is defined as the initiator of a network connection for events regarding sessions, connections, or bidirectional flow records.\nFor TCP events, the client is the initiator of the TCP connection that sends the SYN packet(s). For other protocols, the client is generally the initiator or requestor in the network transaction. Some systems use the term "originator" to refer the client in TCP connections. The client fields describe details about the system acting as the client in the network event. Client fields are usually populated in conjunction with server fields. Client fields are generally not populated for packet-level events.\nClient / server representations can add semantic context to an exchange, which is helpful to visualize the data in certain situations. If your context falls in that category, you should still ensure that source and destination are filled appropriately.',
        fields: {
            'client.address': {
                dashed_name: 'client-address',
                description: 'Some event client addresses are defined ambiguously. The event will sometimes list an IP, a domain or a unix socket.  You should always store the raw address in the `.address` field.\nThen it should be duplicated to `.ip` or `.domain`, depending on which one it is.',
                flat_name: 'client.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'address',
                normalize: [],
                short: 'Client network address.',
                type: 'keyword',
            },
            'client.as.number': {
                dashed_name: 'client-as-number',
                description: 'Unique number allocated to the autonomous system. The autonomous system number (ASN) uniquely identifies each network on the Internet.',
                example: 15169,
                flat_name: 'client.as.number',
                level: 'extended',
                name: 'number',
                normalize: [],
                original_fieldset: 'as',
                short: 'Unique number allocated to the autonomous system.',
                type: 'long',
            },
            'client.as.organization.name': {
                dashed_name: 'client-as-organization-name',
                description: 'Organization name.',
                example: 'Google LLC',
                flat_name: 'client.as.organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'client.as.organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'organization.name',
                normalize: [],
                original_fieldset: 'as',
                short: 'Organization name.',
                type: 'keyword',
            },
            'client.bytes': {
                dashed_name: 'client-bytes',
                description: 'Bytes sent from the client to the server.',
                example: 184,
                flat_name: 'client.bytes',
                format: 'bytes',
                level: 'core',
                name: 'bytes',
                normalize: [],
                short: 'Bytes sent from the client to the server.',
                type: 'long',
            },
            'client.domain': {
                dashed_name: 'client-domain',
                description: 'The domain name of the client system.\nThis value may be a host name, a fully qualified domain name, or another host naming format. The value may derive from the original event or be added from enrichment.',
                example: 'foo.example.com',
                flat_name: 'client.domain',
                ignore_above: 1024,
                level: 'core',
                name: 'domain',
                normalize: [],
                short: 'The domain name of the client.',
                type: 'keyword',
            },
            'client.geo.city_name': {
                dashed_name: 'client-geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'client.geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'City name.',
                type: 'keyword',
            },
            'client.geo.continent_code': {
                dashed_name: 'client-geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'client.geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Continent code.',
                type: 'keyword',
            },
            'client.geo.continent_name': {
                dashed_name: 'client-geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'client.geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'client.geo.country_iso_code': {
                dashed_name: 'client-geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'client.geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'client.geo.country_name': {
                dashed_name: 'client-geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'client.geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country name.',
                type: 'keyword',
            },
            'client.geo.location': {
                dashed_name: 'client-geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'client.geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'client.geo.name': {
                dashed_name: 'client-geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'client.geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'client.geo.postal_code': {
                dashed_name: 'client-geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'client.geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Postal code.',
                type: 'keyword',
            },
            'client.geo.region_iso_code': {
                dashed_name: 'client-geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'client.geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'client.geo.region_name': {
                dashed_name: 'client-geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'client.geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region name.',
                type: 'keyword',
            },
            'client.geo.timezone': {
                dashed_name: 'client-geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'client.geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Time zone.',
                type: 'keyword',
            },
            'client.ip': {
                dashed_name: 'client-ip',
                description: 'IP address of the client (IPv4 or IPv6).',
                flat_name: 'client.ip',
                level: 'core',
                name: 'ip',
                normalize: [],
                short: 'IP address of the client.',
                type: 'ip',
            },
            'client.mac': {
                dashed_name: 'client-mac',
                description: 'MAC address of the client.\nThe notation format from RFC 7042 is suggested: Each octet (that is, 8-bit byte) is represented by two [uppercase] hexadecimal digits giving the value of the octet as an unsigned integer. Successive octets are separated by a hyphen.',
                example: '00-00-5E-00-53-23',
                flat_name: 'client.mac',
                ignore_above: 1024,
                level: 'core',
                name: 'mac',
                normalize: [],
                pattern: '^[A-F0-9]{2}(-[A-F0-9]{2}){5,}$',
                short: 'MAC address of the client.',
                type: 'keyword',
            },
            'client.nat.ip': {
                dashed_name: 'client-nat-ip',
                description: 'Translated IP of source based NAT sessions (e.g. internal client to internet).\nTypically connections traversing load balancers, firewalls, or routers.',
                flat_name: 'client.nat.ip',
                level: 'extended',
                name: 'nat.ip',
                normalize: [],
                short: 'Client NAT ip address',
                type: 'ip',
            },
            'client.nat.port': {
                dashed_name: 'client-nat-port',
                description: 'Translated port of source based NAT sessions (e.g. internal client to internet).\nTypically connections traversing load balancers, firewalls, or routers.',
                flat_name: 'client.nat.port',
                format: 'string',
                level: 'extended',
                name: 'nat.port',
                normalize: [],
                short: 'Client NAT port',
                type: 'long',
            },
            'client.packets': {
                dashed_name: 'client-packets',
                description: 'Packets sent from the client to the server.',
                example: 12,
                flat_name: 'client.packets',
                level: 'core',
                name: 'packets',
                normalize: [],
                short: 'Packets sent from the client to the server.',
                type: 'long',
            },
            'client.port': {
                dashed_name: 'client-port',
                description: 'Port of the client.',
                flat_name: 'client.port',
                format: 'string',
                level: 'core',
                name: 'port',
                normalize: [],
                short: 'Port of the client.',
                type: 'long',
            },
            'client.registered_domain': {
                dashed_name: 'client-registered-domain',
                description: 'The highest registered client domain, stripped of the subdomain.\nFor example, the registered domain for "foo.example.com" is "example.com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last two labels will not work well for TLDs such as "co.uk".',
                example: 'example.com',
                flat_name: 'client.registered_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'registered_domain',
                normalize: [],
                short: 'The highest registered client domain, stripped of the subdomain.',
                type: 'keyword',
            },
            'client.subdomain': {
                dashed_name: 'client-subdomain',
                description: 'The subdomain portion of a fully qualified domain name includes all of the names except the host name under the registered_domain.  In a partially qualified domain, or if the the qualification level of the full name cannot be determined, subdomain contains all of the names below the registered domain.\nFor example the subdomain portion of "www.east.mydomain.co.uk" is "east". If the domain has multiple levels of subdomain, such as "sub2.sub1.example.com", the subdomain field should contain "sub2.sub1", with no trailing period.',
                example: 'east',
                flat_name: 'client.subdomain',
                ignore_above: 1024,
                level: 'extended',
                name: 'subdomain',
                normalize: [],
                short: 'The subdomain of the domain.',
                type: 'keyword',
            },
            'client.top_level_domain': {
                dashed_name: 'client-top-level-domain',
                description: 'The effective top level domain (eTLD), also known as the domain suffix, is the last part of the domain name. For example, the top level domain for example.com is "com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last label will not work well for effective TLDs such as "co.uk".',
                example: 'co.uk',
                flat_name: 'client.top_level_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'top_level_domain',
                normalize: [],
                short: 'The effective top level domain (com, org, net, co.uk).',
                type: 'keyword',
            },
            'client.user.domain': {
                dashed_name: 'client-user-domain',
                description: 'Name of the directory the user is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'client.user.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'user',
                short: 'Name of the directory the user is a member of.',
                type: 'keyword',
            },
            'client.user.email': {
                dashed_name: 'client-user-email',
                description: 'User email address.',
                flat_name: 'client.user.email',
                ignore_above: 1024,
                level: 'extended',
                name: 'email',
                normalize: [],
                original_fieldset: 'user',
                short: 'User email address.',
                type: 'keyword',
            },
            'client.user.full_name': {
                dashed_name: 'client-user-full-name',
                description: "User's full name, if available.",
                example: 'Albert Einstein',
                flat_name: 'client.user.full_name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'client.user.full_name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'full_name',
                normalize: [],
                original_fieldset: 'user',
                short: "User's full name, if available.",
                type: 'keyword',
            },
            'client.user.group.domain': {
                dashed_name: 'client-user-group-domain',
                description: 'Name of the directory the group is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'client.user.group.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the directory the group is a member of.',
                type: 'keyword',
            },
            'client.user.group.id': {
                dashed_name: 'client-user-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'client.user.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'client.user.group.name': {
                dashed_name: 'client-user-group-name',
                description: 'Name of the group.',
                flat_name: 'client.user.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'client.user.hash': {
                dashed_name: 'client-user-hash',
                description: 'Unique user hash to correlate information for a user in anonymized form.\nUseful if `user.id` or `user.name` contain confidential information and cannot be used.',
                flat_name: 'client.user.hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'hash',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique user hash to correlate information for a user in anonymized form.',
                type: 'keyword',
            },
            'client.user.id': {
                dashed_name: 'client-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'client.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'client.user.name': {
                dashed_name: 'client-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'client.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'client.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'client.user.roles': {
                dashed_name: 'client-user-roles',
                description: 'Array of user roles at the time of the event.',
                example: '["kibana_admin", "reporting_user"]',
                flat_name: 'client.user.roles',
                ignore_above: 1024,
                level: 'extended',
                name: 'roles',
                normalize: ['array'],
                original_fieldset: 'user',
                short: 'Array of user roles at the time of the event.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'client',
        nestings: ['client.as', 'client.geo', 'client.user'],
        prefix: 'client.',
        reused_here: [
            {
                full: 'client.as',
                schema_name: 'as',
                short: 'Fields describing an Autonomous System (Internet routing prefix).',
            },
            {
                full: 'client.geo',
                schema_name: 'geo',
                short: 'Fields describing a location.',
            },
            {
                full: 'client.user',
                schema_name: 'user',
                short: 'Fields to describe the user relevant to the event.',
            },
        ],
        short: 'Fields about the client side of a network connection, used with server.',
        title: 'Client',
        type: 'group',
    },
    cloud: {
        description: 'Fields related to the cloud or infrastructure the events are coming from.',
        fields: {
            'cloud.account.id': {
                dashed_name: 'cloud-account-id',
                description: 'The cloud account or organization id used to identify different entities in a multi-tenant environment.\nExamples: AWS account id, Google Cloud ORG Id, or other unique identifier.',
                example: 666777888999,
                flat_name: 'cloud.account.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'account.id',
                normalize: [],
                short: 'The cloud account or organization id.',
                type: 'keyword',
            },
            'cloud.account.name': {
                dashed_name: 'cloud-account-name',
                description: 'The cloud account name or alias used to identify different entities in a multi-tenant environment.\nExamples: AWS account name, Google Cloud ORG display name.',
                example: 'elastic-dev',
                flat_name: 'cloud.account.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'account.name',
                normalize: [],
                short: 'The cloud account name.',
                type: 'keyword',
            },
            'cloud.availability_zone': {
                dashed_name: 'cloud-availability-zone',
                description: 'Availability zone in which this host, resource, or service is located.',
                example: 'us-east-1c',
                flat_name: 'cloud.availability_zone',
                ignore_above: 1024,
                level: 'extended',
                name: 'availability_zone',
                normalize: [],
                short: 'Availability zone in which this host, resource, or service is located.',
                type: 'keyword',
            },
            'cloud.instance.id': {
                dashed_name: 'cloud-instance-id',
                description: 'Instance ID of the host machine.',
                example: 'i-1234567890abcdef0',
                flat_name: 'cloud.instance.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'instance.id',
                normalize: [],
                short: 'Instance ID of the host machine.',
                type: 'keyword',
            },
            'cloud.instance.name': {
                dashed_name: 'cloud-instance-name',
                description: 'Instance name of the host machine.',
                flat_name: 'cloud.instance.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'instance.name',
                normalize: [],
                short: 'Instance name of the host machine.',
                type: 'keyword',
            },
            'cloud.machine.type': {
                dashed_name: 'cloud-machine-type',
                description: 'Machine type of the host machine.',
                example: 't2.medium',
                flat_name: 'cloud.machine.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'machine.type',
                normalize: [],
                short: 'Machine type of the host machine.',
                type: 'keyword',
            },
            'cloud.origin.account.id': {
                dashed_name: 'cloud-origin-account-id',
                description: 'The cloud account or organization id used to identify different entities in a multi-tenant environment.\nExamples: AWS account id, Google Cloud ORG Id, or other unique identifier.',
                example: 666777888999,
                flat_name: 'cloud.origin.account.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'account.id',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud account or organization id.',
                type: 'keyword',
            },
            'cloud.origin.account.name': {
                dashed_name: 'cloud-origin-account-name',
                description: 'The cloud account name or alias used to identify different entities in a multi-tenant environment.\nExamples: AWS account name, Google Cloud ORG display name.',
                example: 'elastic-dev',
                flat_name: 'cloud.origin.account.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'account.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud account name.',
                type: 'keyword',
            },
            'cloud.origin.availability_zone': {
                dashed_name: 'cloud-origin-availability-zone',
                description: 'Availability zone in which this host, resource, or service is located.',
                example: 'us-east-1c',
                flat_name: 'cloud.origin.availability_zone',
                ignore_above: 1024,
                level: 'extended',
                name: 'availability_zone',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Availability zone in which this host, resource, or service is located.',
                type: 'keyword',
            },
            'cloud.origin.instance.id': {
                dashed_name: 'cloud-origin-instance-id',
                description: 'Instance ID of the host machine.',
                example: 'i-1234567890abcdef0',
                flat_name: 'cloud.origin.instance.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'instance.id',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Instance ID of the host machine.',
                type: 'keyword',
            },
            'cloud.origin.instance.name': {
                dashed_name: 'cloud-origin-instance-name',
                description: 'Instance name of the host machine.',
                flat_name: 'cloud.origin.instance.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'instance.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Instance name of the host machine.',
                type: 'keyword',
            },
            'cloud.origin.machine.type': {
                dashed_name: 'cloud-origin-machine-type',
                description: 'Machine type of the host machine.',
                example: 't2.medium',
                flat_name: 'cloud.origin.machine.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'machine.type',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Machine type of the host machine.',
                type: 'keyword',
            },
            'cloud.origin.project.id': {
                dashed_name: 'cloud-origin-project-id',
                description: 'The cloud project identifier.\nExamples: Google Cloud Project id, Azure Project id.',
                example: 'my-project',
                flat_name: 'cloud.origin.project.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'project.id',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud project id.',
                type: 'keyword',
            },
            'cloud.origin.project.name': {
                dashed_name: 'cloud-origin-project-name',
                description: 'The cloud project name.\nExamples: Google Cloud Project name, Azure Project name.',
                example: 'my project',
                flat_name: 'cloud.origin.project.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'project.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud project name.',
                type: 'keyword',
            },
            'cloud.origin.provider': {
                dashed_name: 'cloud-origin-provider',
                description: 'Name of the cloud provider. Example values are aws, azure, gcp, or digitalocean.',
                example: 'aws',
                flat_name: 'cloud.origin.provider',
                ignore_above: 1024,
                level: 'extended',
                name: 'provider',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Name of the cloud provider.',
                type: 'keyword',
            },
            'cloud.origin.region': {
                dashed_name: 'cloud-origin-region',
                description: 'Region in which this host, resource, or service is located.',
                example: 'us-east-1',
                flat_name: 'cloud.origin.region',
                ignore_above: 1024,
                level: 'extended',
                name: 'region',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Region in which this host, resource, or service is located.',
                type: 'keyword',
            },
            'cloud.origin.service.name': {
                dashed_name: 'cloud-origin-service-name',
                description: 'The cloud service name is intended to distinguish services running on different platforms within a provider, eg AWS EC2 vs Lambda, GCP GCE vs App Engine, Azure VM vs App Server.\nExamples: app engine, app service, cloud run, fargate, lambda.',
                example: 'lambda',
                flat_name: 'cloud.origin.service.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'service.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud service name.',
                type: 'keyword',
            },
            'cloud.project.id': {
                dashed_name: 'cloud-project-id',
                description: 'The cloud project identifier.\nExamples: Google Cloud Project id, Azure Project id.',
                example: 'my-project',
                flat_name: 'cloud.project.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'project.id',
                normalize: [],
                short: 'The cloud project id.',
                type: 'keyword',
            },
            'cloud.project.name': {
                dashed_name: 'cloud-project-name',
                description: 'The cloud project name.\nExamples: Google Cloud Project name, Azure Project name.',
                example: 'my project',
                flat_name: 'cloud.project.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'project.name',
                normalize: [],
                short: 'The cloud project name.',
                type: 'keyword',
            },
            'cloud.provider': {
                dashed_name: 'cloud-provider',
                description: 'Name of the cloud provider. Example values are aws, azure, gcp, or digitalocean.',
                example: 'aws',
                flat_name: 'cloud.provider',
                ignore_above: 1024,
                level: 'extended',
                name: 'provider',
                normalize: [],
                short: 'Name of the cloud provider.',
                type: 'keyword',
            },
            'cloud.region': {
                dashed_name: 'cloud-region',
                description: 'Region in which this host, resource, or service is located.',
                example: 'us-east-1',
                flat_name: 'cloud.region',
                ignore_above: 1024,
                level: 'extended',
                name: 'region',
                normalize: [],
                short: 'Region in which this host, resource, or service is located.',
                type: 'keyword',
            },
            'cloud.service.name': {
                dashed_name: 'cloud-service-name',
                description: 'The cloud service name is intended to distinguish services running on different platforms within a provider, eg AWS EC2 vs Lambda, GCP GCE vs App Engine, Azure VM vs App Server.\nExamples: app engine, app service, cloud run, fargate, lambda.',
                example: 'lambda',
                flat_name: 'cloud.service.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'service.name',
                normalize: [],
                short: 'The cloud service name.',
                type: 'keyword',
            },
            'cloud.target.account.id': {
                dashed_name: 'cloud-target-account-id',
                description: 'The cloud account or organization id used to identify different entities in a multi-tenant environment.\nExamples: AWS account id, Google Cloud ORG Id, or other unique identifier.',
                example: 666777888999,
                flat_name: 'cloud.target.account.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'account.id',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud account or organization id.',
                type: 'keyword',
            },
            'cloud.target.account.name': {
                dashed_name: 'cloud-target-account-name',
                description: 'The cloud account name or alias used to identify different entities in a multi-tenant environment.\nExamples: AWS account name, Google Cloud ORG display name.',
                example: 'elastic-dev',
                flat_name: 'cloud.target.account.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'account.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud account name.',
                type: 'keyword',
            },
            'cloud.target.availability_zone': {
                dashed_name: 'cloud-target-availability-zone',
                description: 'Availability zone in which this host, resource, or service is located.',
                example: 'us-east-1c',
                flat_name: 'cloud.target.availability_zone',
                ignore_above: 1024,
                level: 'extended',
                name: 'availability_zone',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Availability zone in which this host, resource, or service is located.',
                type: 'keyword',
            },
            'cloud.target.instance.id': {
                dashed_name: 'cloud-target-instance-id',
                description: 'Instance ID of the host machine.',
                example: 'i-1234567890abcdef0',
                flat_name: 'cloud.target.instance.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'instance.id',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Instance ID of the host machine.',
                type: 'keyword',
            },
            'cloud.target.instance.name': {
                dashed_name: 'cloud-target-instance-name',
                description: 'Instance name of the host machine.',
                flat_name: 'cloud.target.instance.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'instance.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Instance name of the host machine.',
                type: 'keyword',
            },
            'cloud.target.machine.type': {
                dashed_name: 'cloud-target-machine-type',
                description: 'Machine type of the host machine.',
                example: 't2.medium',
                flat_name: 'cloud.target.machine.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'machine.type',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Machine type of the host machine.',
                type: 'keyword',
            },
            'cloud.target.project.id': {
                dashed_name: 'cloud-target-project-id',
                description: 'The cloud project identifier.\nExamples: Google Cloud Project id, Azure Project id.',
                example: 'my-project',
                flat_name: 'cloud.target.project.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'project.id',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud project id.',
                type: 'keyword',
            },
            'cloud.target.project.name': {
                dashed_name: 'cloud-target-project-name',
                description: 'The cloud project name.\nExamples: Google Cloud Project name, Azure Project name.',
                example: 'my project',
                flat_name: 'cloud.target.project.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'project.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud project name.',
                type: 'keyword',
            },
            'cloud.target.provider': {
                dashed_name: 'cloud-target-provider',
                description: 'Name of the cloud provider. Example values are aws, azure, gcp, or digitalocean.',
                example: 'aws',
                flat_name: 'cloud.target.provider',
                ignore_above: 1024,
                level: 'extended',
                name: 'provider',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Name of the cloud provider.',
                type: 'keyword',
            },
            'cloud.target.region': {
                dashed_name: 'cloud-target-region',
                description: 'Region in which this host, resource, or service is located.',
                example: 'us-east-1',
                flat_name: 'cloud.target.region',
                ignore_above: 1024,
                level: 'extended',
                name: 'region',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'Region in which this host, resource, or service is located.',
                type: 'keyword',
            },
            'cloud.target.service.name': {
                dashed_name: 'cloud-target-service-name',
                description: 'The cloud service name is intended to distinguish services running on different platforms within a provider, eg AWS EC2 vs Lambda, GCP GCE vs App Engine, Azure VM vs App Server.\nExamples: app engine, app service, cloud run, fargate, lambda.',
                example: 'lambda',
                flat_name: 'cloud.target.service.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'service.name',
                normalize: [],
                original_fieldset: 'cloud',
                short: 'The cloud service name.',
                type: 'keyword',
            },
        },
        footnote: "Examples: If Metricbeat is running on an EC2 host and fetches data from its host, the cloud info contains the data about this machine. If Metricbeat runs on a remote machine outside the cloud and fetches data from a service running in the cloud, the field contains cloud data from the machine the service is running on.\nThe cloud fields may be self-nested under cloud.origin.* and cloud.target.* to describe origin or target service's cloud information in the context of incoming or outgoing requests, respectively. However, the fieldsets cloud.origin.* and cloud.target.* must not be confused with the root cloud fieldset that is used to describe the cloud context of the actual service under observation. The fieldset cloud.origin.* may only be used in the context of incoming requests or events to provide the originating service's cloud information. The fieldset cloud.target.* may only be used in the context of outgoing requests or events to describe the target service's cloud information.",
        group: 2,
        name: 'cloud',
        nestings: ['cloud.origin', 'cloud.target'],
        prefix: 'cloud.',
        reusable: {
            expected: [
                {
                    as: 'origin',
                    at: 'cloud',
                    beta: 'Reusing the `cloud` fields in this location is currently considered beta.',
                    full: 'cloud.origin',
                    short_override: 'Provides the cloud information of the origin entity in case of an incoming request or event.',
                },
                {
                    as: 'target',
                    at: 'cloud',
                    beta: 'Reusing the `cloud` fields in this location is currently considered beta.',
                    full: 'cloud.target',
                    short_override: 'Provides the cloud information of the target entity in case of an outgoing request or event.',
                },
            ],
            top_level: true,
        },
        reused_here: [
            {
                beta: 'Reusing the `cloud` fields in this location is currently considered beta.',
                full: 'cloud.origin',
                schema_name: 'cloud',
                short: 'Provides the cloud information of the origin entity in case of an incoming request or event.',
            },
            {
                beta: 'Reusing the `cloud` fields in this location is currently considered beta.',
                full: 'cloud.target',
                schema_name: 'cloud',
                short: 'Provides the cloud information of the target entity in case of an outgoing request or event.',
            },
        ],
        short: 'Fields about the cloud resource.',
        title: 'Cloud',
        type: 'group',
    },
    code_signature: {
        description: 'These fields contain information about binary code signatures.',
        fields: {
            'code_signature.digest_algorithm': {
                dashed_name: 'code-signature-digest-algorithm',
                description: 'The hashing algorithm used to sign the process.\nThis value can distinguish signatures when a file is signed multiple times by the same signer but with a different digest algorithm.',
                example: 'sha256',
                flat_name: 'code_signature.digest_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'digest_algorithm',
                normalize: [],
                short: 'Hashing algorithm used to sign the process.',
                type: 'keyword',
            },
            'code_signature.exists': {
                dashed_name: 'code-signature-exists',
                description: 'Boolean to capture if a signature is present.',
                example: 'true',
                flat_name: 'code_signature.exists',
                level: 'core',
                name: 'exists',
                normalize: [],
                short: 'Boolean to capture if a signature is present.',
                type: 'boolean',
            },
            'code_signature.signing_id': {
                dashed_name: 'code-signature-signing-id',
                description: 'The identifier used to sign the process.\nThis is used to identify the application manufactured by a software vendor. The field is relevant to Apple *OS only.',
                example: 'com.apple.xpc.proxy',
                flat_name: 'code_signature.signing_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'signing_id',
                normalize: [],
                short: 'The identifier used to sign the process.',
                type: 'keyword',
            },
            'code_signature.status': {
                dashed_name: 'code-signature-status',
                description: 'Additional information about the certificate status.\nThis is useful for logging cryptographic errors with the certificate validity or trust status. Leave unpopulated if the validity or trust of the certificate was unchecked.',
                example: 'ERROR_UNTRUSTED_ROOT',
                flat_name: 'code_signature.status',
                ignore_above: 1024,
                level: 'extended',
                name: 'status',
                normalize: [],
                short: 'Additional information about the certificate status.',
                type: 'keyword',
            },
            'code_signature.subject_name': {
                dashed_name: 'code-signature-subject-name',
                description: 'Subject name of the code signer',
                example: 'Microsoft Corporation',
                flat_name: 'code_signature.subject_name',
                ignore_above: 1024,
                level: 'core',
                name: 'subject_name',
                normalize: [],
                short: 'Subject name of the code signer',
                type: 'keyword',
            },
            'code_signature.team_id': {
                dashed_name: 'code-signature-team-id',
                description: 'The team identifier used to sign the process.\nThis is used to identify the team or vendor of a software product. The field is relevant to Apple *OS only.',
                example: 'EQHXZ8M8AV',
                flat_name: 'code_signature.team_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'team_id',
                normalize: [],
                short: 'The team identifier used to sign the process.',
                type: 'keyword',
            },
            'code_signature.timestamp': {
                dashed_name: 'code-signature-timestamp',
                description: 'Date and time when the code signature was generated and signed.',
                example: '2021-01-01T12:10:30Z',
                flat_name: 'code_signature.timestamp',
                level: 'extended',
                name: 'timestamp',
                normalize: [],
                short: 'When the signature was generated and signed.',
                type: 'date',
            },
            'code_signature.trusted': {
                dashed_name: 'code-signature-trusted',
                description: 'Stores the trust status of the certificate chain.\nValidating the trust of the certificate chain may be complicated, and this field should only be populated by tools that actively check the status.',
                example: 'true',
                flat_name: 'code_signature.trusted',
                level: 'extended',
                name: 'trusted',
                normalize: [],
                short: 'Stores the trust status of the certificate chain.',
                type: 'boolean',
            },
            'code_signature.valid': {
                dashed_name: 'code-signature-valid',
                description: 'Boolean to capture if the digital signature is verified against the binary content.\nLeave unpopulated if a certificate was unchecked.',
                example: 'true',
                flat_name: 'code_signature.valid',
                level: 'extended',
                name: 'valid',
                normalize: [],
                short: 'Boolean to capture if the digital signature is verified against the binary content.',
                type: 'boolean',
            },
        },
        group: 2,
        name: 'code_signature',
        prefix: 'code_signature.',
        reusable: {
            expected: [
                { as: 'code_signature', at: 'file', full: 'file.code_signature' },
                { as: 'code_signature', at: 'process', full: 'process.code_signature' },
                { as: 'code_signature', at: 'dll', full: 'dll.code_signature' },
            ],
            top_level: false,
        },
        short: 'These fields contain information about binary code signatures.',
        title: 'Code Signature',
        type: 'group',
    },
    container: {
        description: 'Container fields are used for meta information about the specific container that is the source of information.\nThese fields help correlate data based containers from any runtime.',
        fields: {
            'container.cpu.usage': {
                dashed_name: 'container-cpu-usage',
                description: 'Percent CPU used which is normalized by the number of CPU cores and it ranges from 0 to 1. Scaling factor: 1000.',
                flat_name: 'container.cpu.usage',
                level: 'extended',
                name: 'cpu.usage',
                normalize: [],
                scaling_factor: 1000,
                short: 'Percent CPU used, between 0 and 1.',
                type: 'scaled_float',
            },
            'container.disk.read.bytes': {
                dashed_name: 'container-disk-read-bytes',
                description: 'The total number of bytes (gauge) read successfully (aggregated from all disks) since the last metric collection.',
                flat_name: 'container.disk.read.bytes',
                level: 'extended',
                name: 'disk.read.bytes',
                normalize: [],
                short: 'The number of bytes read by all disks.',
                type: 'long',
            },
            'container.disk.write.bytes': {
                dashed_name: 'container-disk-write-bytes',
                description: 'The total number of bytes (gauge) written successfully (aggregated from all disks) since the last metric collection.',
                flat_name: 'container.disk.write.bytes',
                level: 'extended',
                name: 'disk.write.bytes',
                normalize: [],
                short: 'The number of bytes written on all disks.',
                type: 'long',
            },
            'container.id': {
                dashed_name: 'container-id',
                description: 'Unique container id.',
                flat_name: 'container.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                short: 'Unique container id.',
                type: 'keyword',
            },
            'container.image.hash.all': {
                dashed_name: 'container-image-hash-all',
                description: 'An array of digests of the image the container was built on. Each digest consists of the hash algorithm and value in this format: `algorithm:value`. Algorithm names should align with the field names in the ECS hash field set.',
                example: '[sha256:f8fefc80e3273dc756f288a63945820d6476ad64883892c771b5e2ece6bf1b26]',
                flat_name: 'container.image.hash.all',
                ignore_above: 1024,
                level: 'extended',
                name: 'image.hash.all',
                normalize: ['array'],
                short: 'An array of digests of the image the container was built on.',
                type: 'keyword',
            },
            'container.image.name': {
                dashed_name: 'container-image-name',
                description: 'Name of the image the container was built on.',
                flat_name: 'container.image.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'image.name',
                normalize: [],
                short: 'Name of the image the container was built on.',
                type: 'keyword',
            },
            'container.image.tag': {
                dashed_name: 'container-image-tag',
                description: 'Container image tags.',
                flat_name: 'container.image.tag',
                ignore_above: 1024,
                level: 'extended',
                name: 'image.tag',
                normalize: ['array'],
                short: 'Container image tags.',
                type: 'keyword',
            },
            'container.labels': {
                dashed_name: 'container-labels',
                description: 'Image labels.',
                flat_name: 'container.labels',
                level: 'extended',
                name: 'labels',
                normalize: [],
                object_type: 'keyword',
                short: 'Image labels.',
                type: 'object',
            },
            'container.memory.usage': {
                dashed_name: 'container-memory-usage',
                description: 'Memory usage percentage and it ranges from 0 to 1. Scaling factor: 1000.',
                flat_name: 'container.memory.usage',
                level: 'extended',
                name: 'memory.usage',
                normalize: [],
                scaling_factor: 1000,
                short: 'Percent memory used, between 0 and 1.',
                type: 'scaled_float',
            },
            'container.name': {
                dashed_name: 'container-name',
                description: 'Container name.',
                flat_name: 'container.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Container name.',
                type: 'keyword',
            },
            'container.network.egress.bytes': {
                dashed_name: 'container-network-egress-bytes',
                description: 'The number of bytes (gauge) sent out on all network interfaces by the container since the last metric collection.',
                flat_name: 'container.network.egress.bytes',
                level: 'extended',
                name: 'network.egress.bytes',
                normalize: [],
                short: 'The number of bytes sent on all network interfaces.',
                type: 'long',
            },
            'container.network.ingress.bytes': {
                dashed_name: 'container-network-ingress-bytes',
                description: 'The number of bytes received (gauge) on all network interfaces by the container since the last metric collection.',
                flat_name: 'container.network.ingress.bytes',
                level: 'extended',
                name: 'network.ingress.bytes',
                normalize: [],
                short: 'The number of bytes received on all network interfaces.',
                type: 'long',
            },
            'container.runtime': {
                dashed_name: 'container-runtime',
                description: 'Runtime managing this container.',
                example: 'docker',
                flat_name: 'container.runtime',
                ignore_above: 1024,
                level: 'extended',
                name: 'runtime',
                normalize: [],
                short: 'Runtime managing this container.',
                type: 'keyword',
            },
            'container.security_context.privileged': {
                dashed_name: 'container-security-context-privileged',
                description: 'Indicates whether the container is running in privileged mode.',
                flat_name: 'container.security_context.privileged',
                level: 'extended',
                name: 'security_context.privileged',
                normalize: [],
                short: 'Indicates whether the container is running in privileged mode.',
                type: 'boolean',
            },
        },
        group: 2,
        name: 'container',
        prefix: 'container.',
        short: 'Fields describing the container that generated this event.',
        title: 'Container',
        type: 'group',
    },
    data_stream: {
        beta: 'These fields are in beta and are subject to change.',
        description: 'The data_stream fields take part in defining the new data stream naming scheme.\nIn the new data stream naming scheme the value of the data stream fields combine to the name of the actual data stream in the following manner: `{data_stream.type}-{data_stream.dataset}-{data_stream.namespace}`. This means the fields can only contain characters that are valid as part of names of data streams. More details about this can be found in this https://www.elastic.co/blog/an-introduction-to-the-elastic-data-stream-naming-scheme[blog post].\nAn Elasticsearch data stream consists of one or more backing indices, and a data stream name forms part of the backing indices names. Due to this convention, data streams must also follow index naming restrictions. For example, data stream names cannot include `\\`, `/`, `*`, `?`, `"`, `<`, `>`, `|`, ` ` (space character), `,`, or `#`. Please see the Elasticsearch reference for additional https://www.elastic.co/guide/en/elasticsearch/reference/current/indices-create-index.html#indices-create-api-path-params[restrictions].',
        fields: {
            'data_stream.dataset': {
                dashed_name: 'data-stream-dataset',
                description: 'The field can contain anything that makes sense to signify the source of the data.\nExamples include `nginx.access`, `prometheus`, `endpoint` etc. For data streams that otherwise fit, but that do not have dataset set we use the value "generic" for the dataset value. `event.dataset` should have the same value as `data_stream.dataset`.\nBeyond the Elasticsearch data stream naming criteria noted above, the `dataset` value has additional restrictions:\n  * Must not contain `-`\n  * No longer than 100 characters',
                example: 'nginx.access',
                flat_name: 'data_stream.dataset',
                level: 'extended',
                name: 'dataset',
                normalize: [],
                short: 'The field can contain anything that makes sense to signify the source of the data.',
                type: 'constant_keyword',
            },
            'data_stream.namespace': {
                dashed_name: 'data-stream-namespace',
                description: 'A user defined namespace. Namespaces are useful to allow grouping of data.\nMany users already organize their indices this way, and the data stream naming scheme now provides this best practice as a default. Many users will populate this field with `default`. If no value is used, it falls back to `default`.\nBeyond the Elasticsearch index naming criteria noted above, `namespace` value has the additional restrictions:\n  * Must not contain `-`\n  * No longer than 100 characters',
                example: 'production',
                flat_name: 'data_stream.namespace',
                level: 'extended',
                name: 'namespace',
                normalize: [],
                short: 'A user defined namespace. Namespaces are useful to allow grouping of data.',
                type: 'constant_keyword',
            },
            'data_stream.type': {
                dashed_name: 'data-stream-type',
                description: 'An overarching type for the data stream.\nCurrently allowed values are "logs" and "metrics". We expect to also add "traces" and "synthetics" in the near future.',
                example: 'logs',
                flat_name: 'data_stream.type',
                level: 'extended',
                name: 'type',
                normalize: [],
                short: 'An overarching type for the data stream.',
                type: 'constant_keyword',
            },
        },
        group: 2,
        name: 'data_stream',
        prefix: 'data_stream.',
        short: 'The data_stream fields take part in defining the new data stream naming scheme.',
        title: 'Data Stream',
        type: 'group',
    },
    destination: {
        description: 'Destination fields capture details about the receiver of a network exchange/packet. These fields are populated from a network event, packet, or other event containing details of a network transaction.\nDestination fields are usually populated in conjunction with source fields. The source and destination fields are considered the baseline and should always be filled if an event contains source and destination details from a network transaction. If the event also contains identification of the client and server roles, then the client and server fields should also be populated.',
        fields: {
            'destination.address': {
                dashed_name: 'destination-address',
                description: 'Some event destination addresses are defined ambiguously. The event will sometimes list an IP, a domain or a unix socket.  You should always store the raw address in the `.address` field.\nThen it should be duplicated to `.ip` or `.domain`, depending on which one it is.',
                flat_name: 'destination.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'address',
                normalize: [],
                short: 'Destination network address.',
                type: 'keyword',
            },
            'destination.as.number': {
                dashed_name: 'destination-as-number',
                description: 'Unique number allocated to the autonomous system. The autonomous system number (ASN) uniquely identifies each network on the Internet.',
                example: 15169,
                flat_name: 'destination.as.number',
                level: 'extended',
                name: 'number',
                normalize: [],
                original_fieldset: 'as',
                short: 'Unique number allocated to the autonomous system.',
                type: 'long',
            },
            'destination.as.organization.name': {
                dashed_name: 'destination-as-organization-name',
                description: 'Organization name.',
                example: 'Google LLC',
                flat_name: 'destination.as.organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'destination.as.organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'organization.name',
                normalize: [],
                original_fieldset: 'as',
                short: 'Organization name.',
                type: 'keyword',
            },
            'destination.bytes': {
                dashed_name: 'destination-bytes',
                description: 'Bytes sent from the destination to the source.',
                example: 184,
                flat_name: 'destination.bytes',
                format: 'bytes',
                level: 'core',
                name: 'bytes',
                normalize: [],
                short: 'Bytes sent from the destination to the source.',
                type: 'long',
            },
            'destination.domain': {
                dashed_name: 'destination-domain',
                description: 'The domain name of the destination system.\nThis value may be a host name, a fully qualified domain name, or another host naming format. The value may derive from the original event or be added from enrichment.',
                example: 'foo.example.com',
                flat_name: 'destination.domain',
                ignore_above: 1024,
                level: 'core',
                name: 'domain',
                normalize: [],
                short: 'The domain name of the destination.',
                type: 'keyword',
            },
            'destination.geo.city_name': {
                dashed_name: 'destination-geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'destination.geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'City name.',
                type: 'keyword',
            },
            'destination.geo.continent_code': {
                dashed_name: 'destination-geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'destination.geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Continent code.',
                type: 'keyword',
            },
            'destination.geo.continent_name': {
                dashed_name: 'destination-geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'destination.geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'destination.geo.country_iso_code': {
                dashed_name: 'destination-geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'destination.geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'destination.geo.country_name': {
                dashed_name: 'destination-geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'destination.geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country name.',
                type: 'keyword',
            },
            'destination.geo.location': {
                dashed_name: 'destination-geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'destination.geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'destination.geo.name': {
                dashed_name: 'destination-geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'destination.geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'destination.geo.postal_code': {
                dashed_name: 'destination-geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'destination.geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Postal code.',
                type: 'keyword',
            },
            'destination.geo.region_iso_code': {
                dashed_name: 'destination-geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'destination.geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'destination.geo.region_name': {
                dashed_name: 'destination-geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'destination.geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region name.',
                type: 'keyword',
            },
            'destination.geo.timezone': {
                dashed_name: 'destination-geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'destination.geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Time zone.',
                type: 'keyword',
            },
            'destination.ip': {
                dashed_name: 'destination-ip',
                description: 'IP address of the destination (IPv4 or IPv6).',
                flat_name: 'destination.ip',
                level: 'core',
                name: 'ip',
                normalize: [],
                short: 'IP address of the destination.',
                type: 'ip',
            },
            'destination.mac': {
                dashed_name: 'destination-mac',
                description: 'MAC address of the destination.\nThe notation format from RFC 7042 is suggested: Each octet (that is, 8-bit byte) is represented by two [uppercase] hexadecimal digits giving the value of the octet as an unsigned integer. Successive octets are separated by a hyphen.',
                example: '00-00-5E-00-53-23',
                flat_name: 'destination.mac',
                ignore_above: 1024,
                level: 'core',
                name: 'mac',
                normalize: [],
                pattern: '^[A-F0-9]{2}(-[A-F0-9]{2}){5,}$',
                short: 'MAC address of the destination.',
                type: 'keyword',
            },
            'destination.nat.ip': {
                dashed_name: 'destination-nat-ip',
                description: 'Translated ip of destination based NAT sessions (e.g. internet to private DMZ)\nTypically used with load balancers, firewalls, or routers.',
                flat_name: 'destination.nat.ip',
                level: 'extended',
                name: 'nat.ip',
                normalize: [],
                short: 'Destination NAT ip',
                type: 'ip',
            },
            'destination.nat.port': {
                dashed_name: 'destination-nat-port',
                description: 'Port the source session is translated to by NAT Device.\nTypically used with load balancers, firewalls, or routers.',
                flat_name: 'destination.nat.port',
                format: 'string',
                level: 'extended',
                name: 'nat.port',
                normalize: [],
                short: 'Destination NAT Port',
                type: 'long',
            },
            'destination.packets': {
                dashed_name: 'destination-packets',
                description: 'Packets sent from the destination to the source.',
                example: 12,
                flat_name: 'destination.packets',
                level: 'core',
                name: 'packets',
                normalize: [],
                short: 'Packets sent from the destination to the source.',
                type: 'long',
            },
            'destination.port': {
                dashed_name: 'destination-port',
                description: 'Port of the destination.',
                flat_name: 'destination.port',
                format: 'string',
                level: 'core',
                name: 'port',
                normalize: [],
                short: 'Port of the destination.',
                type: 'long',
            },
            'destination.registered_domain': {
                dashed_name: 'destination-registered-domain',
                description: 'The highest registered destination domain, stripped of the subdomain.\nFor example, the registered domain for "foo.example.com" is "example.com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last two labels will not work well for TLDs such as "co.uk".',
                example: 'example.com',
                flat_name: 'destination.registered_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'registered_domain',
                normalize: [],
                short: 'The highest registered destination domain, stripped of the subdomain.',
                type: 'keyword',
            },
            'destination.subdomain': {
                dashed_name: 'destination-subdomain',
                description: 'The subdomain portion of a fully qualified domain name includes all of the names except the host name under the registered_domain.  In a partially qualified domain, or if the the qualification level of the full name cannot be determined, subdomain contains all of the names below the registered domain.\nFor example the subdomain portion of "www.east.mydomain.co.uk" is "east". If the domain has multiple levels of subdomain, such as "sub2.sub1.example.com", the subdomain field should contain "sub2.sub1", with no trailing period.',
                example: 'east',
                flat_name: 'destination.subdomain',
                ignore_above: 1024,
                level: 'extended',
                name: 'subdomain',
                normalize: [],
                short: 'The subdomain of the domain.',
                type: 'keyword',
            },
            'destination.top_level_domain': {
                dashed_name: 'destination-top-level-domain',
                description: 'The effective top level domain (eTLD), also known as the domain suffix, is the last part of the domain name. For example, the top level domain for example.com is "com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last label will not work well for effective TLDs such as "co.uk".',
                example: 'co.uk',
                flat_name: 'destination.top_level_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'top_level_domain',
                normalize: [],
                short: 'The effective top level domain (com, org, net, co.uk).',
                type: 'keyword',
            },
            'destination.user.domain': {
                dashed_name: 'destination-user-domain',
                description: 'Name of the directory the user is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'destination.user.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'user',
                short: 'Name of the directory the user is a member of.',
                type: 'keyword',
            },
            'destination.user.email': {
                dashed_name: 'destination-user-email',
                description: 'User email address.',
                flat_name: 'destination.user.email',
                ignore_above: 1024,
                level: 'extended',
                name: 'email',
                normalize: [],
                original_fieldset: 'user',
                short: 'User email address.',
                type: 'keyword',
            },
            'destination.user.full_name': {
                dashed_name: 'destination-user-full-name',
                description: "User's full name, if available.",
                example: 'Albert Einstein',
                flat_name: 'destination.user.full_name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'destination.user.full_name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'full_name',
                normalize: [],
                original_fieldset: 'user',
                short: "User's full name, if available.",
                type: 'keyword',
            },
            'destination.user.group.domain': {
                dashed_name: 'destination-user-group-domain',
                description: 'Name of the directory the group is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'destination.user.group.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the directory the group is a member of.',
                type: 'keyword',
            },
            'destination.user.group.id': {
                dashed_name: 'destination-user-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'destination.user.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'destination.user.group.name': {
                dashed_name: 'destination-user-group-name',
                description: 'Name of the group.',
                flat_name: 'destination.user.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'destination.user.hash': {
                dashed_name: 'destination-user-hash',
                description: 'Unique user hash to correlate information for a user in anonymized form.\nUseful if `user.id` or `user.name` contain confidential information and cannot be used.',
                flat_name: 'destination.user.hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'hash',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique user hash to correlate information for a user in anonymized form.',
                type: 'keyword',
            },
            'destination.user.id': {
                dashed_name: 'destination-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'destination.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'destination.user.name': {
                dashed_name: 'destination-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'destination.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'destination.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'destination.user.roles': {
                dashed_name: 'destination-user-roles',
                description: 'Array of user roles at the time of the event.',
                example: '["kibana_admin", "reporting_user"]',
                flat_name: 'destination.user.roles',
                ignore_above: 1024,
                level: 'extended',
                name: 'roles',
                normalize: ['array'],
                original_fieldset: 'user',
                short: 'Array of user roles at the time of the event.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'destination',
        nestings: ['destination.as', 'destination.geo', 'destination.user'],
        prefix: 'destination.',
        reused_here: [
            {
                full: 'destination.as',
                schema_name: 'as',
                short: 'Fields describing an Autonomous System (Internet routing prefix).',
            },
            {
                full: 'destination.geo',
                schema_name: 'geo',
                short: 'Fields describing a location.',
            },
            {
                full: 'destination.user',
                schema_name: 'user',
                short: 'Fields to describe the user relevant to the event.',
            },
        ],
        short: 'Fields about the destination side of a network connection, used with source.',
        title: 'Destination',
        type: 'group',
    },
    device: {
        beta: 'These fields are in beta and are subject to change.',
        description: 'Fields that describe a device instance and its characteristics.  Data collected for applications and processes running on a (mobile) device can be enriched with these fields to describe the identity, type and other characteristics of the device.\nThis field group definition is based on the Device namespace of the OpenTelemetry Semantic Conventions (https://opentelemetry.io/docs/reference/specification/resource/semantic_conventions/device/).',
        fields: {
            'device.id': {
                dashed_name: 'device-id',
                description: 'The unique identifier of a device. The identifier must not change across application sessions but stay fixed for an instance of a (mobile) device. \nOn iOS, this value must be equal to the vendor identifier (https://developer.apple.com/documentation/uikit/uidevice/1620059-identifierforvendor). On Android, this value must be equal to the Firebase Installation ID or a globally unique UUID which is persisted across sessions in your application.\nFor GDPR and data protection law reasons this identifier should not carry information that would allow to identify a user.',
                example: '00000000-54b3-e7c7-0000-000046bffd97',
                flat_name: 'device.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                short: 'The unique identifier of a device.',
                type: 'keyword',
            },
            'device.manufacturer': {
                dashed_name: 'device-manufacturer',
                description: 'The vendor name of the device manufacturer.',
                example: 'Samsung',
                flat_name: 'device.manufacturer',
                ignore_above: 1024,
                level: 'extended',
                name: 'manufacturer',
                normalize: [],
                short: 'The vendor name of the device manufacturer.',
                type: 'keyword',
            },
            'device.model.identifier': {
                dashed_name: 'device-model-identifier',
                description: 'The machine readable identifier of the device model.',
                example: 'SM-G920F',
                flat_name: 'device.model.identifier',
                ignore_above: 1024,
                level: 'extended',
                name: 'model.identifier',
                normalize: [],
                short: 'The machine readable identifier of the device model.',
                type: 'keyword',
            },
            'device.model.name': {
                dashed_name: 'device-model-name',
                description: 'The human readable marketing name of the device model.',
                example: 'Samsung Galaxy S6',
                flat_name: 'device.model.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'model.name',
                normalize: [],
                short: 'The human readable marketing name of the device model.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'device',
        prefix: 'device.',
        short: 'Fields characterizing a (mobile) device a process or application is running on.',
        title: 'Device',
        type: 'group',
    },
    dll: {
        description: 'These fields contain information about code libraries dynamically loaded into processes.\n\nMany operating systems refer to "shared code libraries" with different names, but this field set refers to all of the following:\n* Dynamic-link library (`.dll`) commonly used on Windows\n* Shared Object (`.so`) commonly used on Unix-like operating systems\n* Dynamic library (`.dylib`) commonly used on macOS',
        fields: {
            'dll.code_signature.digest_algorithm': {
                dashed_name: 'dll-code-signature-digest-algorithm',
                description: 'The hashing algorithm used to sign the process.\nThis value can distinguish signatures when a file is signed multiple times by the same signer but with a different digest algorithm.',
                example: 'sha256',
                flat_name: 'dll.code_signature.digest_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'digest_algorithm',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Hashing algorithm used to sign the process.',
                type: 'keyword',
            },
            'dll.code_signature.exists': {
                dashed_name: 'dll-code-signature-exists',
                description: 'Boolean to capture if a signature is present.',
                example: 'true',
                flat_name: 'dll.code_signature.exists',
                level: 'core',
                name: 'exists',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if a signature is present.',
                type: 'boolean',
            },
            'dll.code_signature.signing_id': {
                dashed_name: 'dll-code-signature-signing-id',
                description: 'The identifier used to sign the process.\nThis is used to identify the application manufactured by a software vendor. The field is relevant to Apple *OS only.',
                example: 'com.apple.xpc.proxy',
                flat_name: 'dll.code_signature.signing_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'signing_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The identifier used to sign the process.',
                type: 'keyword',
            },
            'dll.code_signature.status': {
                dashed_name: 'dll-code-signature-status',
                description: 'Additional information about the certificate status.\nThis is useful for logging cryptographic errors with the certificate validity or trust status. Leave unpopulated if the validity or trust of the certificate was unchecked.',
                example: 'ERROR_UNTRUSTED_ROOT',
                flat_name: 'dll.code_signature.status',
                ignore_above: 1024,
                level: 'extended',
                name: 'status',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Additional information about the certificate status.',
                type: 'keyword',
            },
            'dll.code_signature.subject_name': {
                dashed_name: 'dll-code-signature-subject-name',
                description: 'Subject name of the code signer',
                example: 'Microsoft Corporation',
                flat_name: 'dll.code_signature.subject_name',
                ignore_above: 1024,
                level: 'core',
                name: 'subject_name',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Subject name of the code signer',
                type: 'keyword',
            },
            'dll.code_signature.team_id': {
                dashed_name: 'dll-code-signature-team-id',
                description: 'The team identifier used to sign the process.\nThis is used to identify the team or vendor of a software product. The field is relevant to Apple *OS only.',
                example: 'EQHXZ8M8AV',
                flat_name: 'dll.code_signature.team_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'team_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The team identifier used to sign the process.',
                type: 'keyword',
            },
            'dll.code_signature.timestamp': {
                dashed_name: 'dll-code-signature-timestamp',
                description: 'Date and time when the code signature was generated and signed.',
                example: '2021-01-01T12:10:30Z',
                flat_name: 'dll.code_signature.timestamp',
                level: 'extended',
                name: 'timestamp',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'When the signature was generated and signed.',
                type: 'date',
            },
            'dll.code_signature.trusted': {
                dashed_name: 'dll-code-signature-trusted',
                description: 'Stores the trust status of the certificate chain.\nValidating the trust of the certificate chain may be complicated, and this field should only be populated by tools that actively check the status.',
                example: 'true',
                flat_name: 'dll.code_signature.trusted',
                level: 'extended',
                name: 'trusted',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Stores the trust status of the certificate chain.',
                type: 'boolean',
            },
            'dll.code_signature.valid': {
                dashed_name: 'dll-code-signature-valid',
                description: 'Boolean to capture if the digital signature is verified against the binary content.\nLeave unpopulated if a certificate was unchecked.',
                example: 'true',
                flat_name: 'dll.code_signature.valid',
                level: 'extended',
                name: 'valid',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if the digital signature is verified against the binary content.',
                type: 'boolean',
            },
            'dll.hash.md5': {
                dashed_name: 'dll-hash-md5',
                description: 'MD5 hash.',
                flat_name: 'dll.hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                original_fieldset: 'hash',
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'dll.hash.sha1': {
                dashed_name: 'dll-hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'dll.hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'dll.hash.sha256': {
                dashed_name: 'dll-hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'dll.hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'dll.hash.sha384': {
                dashed_name: 'dll-hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'dll.hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'dll.hash.sha512': {
                dashed_name: 'dll-hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'dll.hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'dll.hash.ssdeep': {
                dashed_name: 'dll-hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'dll.hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'dll.hash.tlsh': {
                dashed_name: 'dll-hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'dll.hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                original_fieldset: 'hash',
                short: 'TLSH hash.',
                type: 'keyword',
            },
            'dll.name': {
                dashed_name: 'dll-name',
                description: 'Name of the library.\nThis generally maps to the name of the file on disk.',
                example: 'kernel32.dll',
                flat_name: 'dll.name',
                ignore_above: 1024,
                level: 'core',
                name: 'name',
                normalize: [],
                short: 'Name of the library.',
                type: 'keyword',
            },
            'dll.path': {
                dashed_name: 'dll-path',
                description: 'Full file path of the library.',
                example: 'C:\\Windows\\System32\\kernel32.dll',
                flat_name: 'dll.path',
                ignore_above: 1024,
                level: 'extended',
                name: 'path',
                normalize: [],
                short: 'Full file path of the library.',
                type: 'keyword',
            },
            'dll.pe.architecture': {
                dashed_name: 'dll-pe-architecture',
                description: 'CPU architecture target for the file.',
                example: 'x64',
                flat_name: 'dll.pe.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'pe',
                short: 'CPU architecture target for the file.',
                type: 'keyword',
            },
            'dll.pe.company': {
                dashed_name: 'dll-pe-company',
                description: 'Internal company name of the file, provided at compile-time.',
                example: 'Microsoft Corporation',
                flat_name: 'dll.pe.company',
                ignore_above: 1024,
                level: 'extended',
                name: 'company',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal company name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'dll.pe.description': {
                dashed_name: 'dll-pe-description',
                description: 'Internal description of the file, provided at compile-time.',
                example: 'Paint',
                flat_name: 'dll.pe.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal description of the file, provided at compile-time.',
                type: 'keyword',
            },
            'dll.pe.file_version': {
                dashed_name: 'dll-pe-file-version',
                description: 'Internal version of the file, provided at compile-time.',
                example: '6.3.9600.17415',
                flat_name: 'dll.pe.file_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'file_version',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Process name.',
                type: 'keyword',
            },
            'dll.pe.go_import_hash': {
                dashed_name: 'dll-pe-go-import-hash',
                description: 'A hash of the Go language imports in a PE file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'dll.pe.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the Go language imports in a PE file.',
                type: 'keyword',
            },
            'dll.pe.go_imports': {
                dashed_name: 'dll-pe-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'dll.pe.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'pe',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'dll.pe.go_imports_names_entropy': {
                dashed_name: 'dll-pe-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'dll.pe.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'dll.pe.go_imports_names_var_entropy': {
                dashed_name: 'dll-pe-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'dll.pe.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'dll.pe.go_stripped': {
                dashed_name: 'dll-pe-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'dll.pe.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'dll.pe.imphash': {
                dashed_name: 'dll-pe-imphash',
                description: 'A hash of the imports in a PE file. An imphash -- or import hash -- can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nLearn more at https://www.fireeye.com/blog/threat-research/2014/01/tracking-malware-import-hashing.html.',
                example: '0c6803c4e922103c4dca5963aad36ddf',
                flat_name: 'dll.pe.imphash',
                ignore_above: 1024,
                level: 'extended',
                name: 'imphash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'dll.pe.import_hash': {
                dashed_name: 'dll-pe-import-hash',
                description: 'A hash of the imports in a PE file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'dll.pe.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'dll.pe.imports': {
                dashed_name: 'dll-pe-imports',
                description: 'List of imported element names and types.',
                flat_name: 'dll.pe.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'dll.pe.imports_names_entropy': {
                dashed_name: 'dll-pe-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'dll.pe.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'dll.pe.imports_names_var_entropy': {
                dashed_name: 'dll-pe-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'dll.pe.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'dll.pe.original_file_name': {
                dashed_name: 'dll-pe-original-file-name',
                description: 'Internal name of the file, provided at compile-time.',
                example: 'MSPAINT.EXE',
                flat_name: 'dll.pe.original_file_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'original_file_name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'dll.pe.pehash': {
                dashed_name: 'dll-pe-pehash',
                description: 'A hash of the PE header and data from one or more PE sections. An pehash can be used to cluster files by transforming structural information about a file into a hash value.\nLearn more at https://www.usenix.org/legacy/events/leet09/tech/full_papers/wicherski/wicherski_html/index.html.',
                example: '73ff189b63cd6be375a7ff25179a38d347651975',
                flat_name: 'dll.pe.pehash',
                ignore_above: 1024,
                level: 'extended',
                name: 'pehash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the PE header and data from one or more PE sections.',
                type: 'keyword',
            },
            'dll.pe.product': {
                dashed_name: 'dll-pe-product',
                description: 'Internal product name of the file, provided at compile-time.',
                example: 'Microsoft® Windows® Operating System',
                flat_name: 'dll.pe.product',
                ignore_above: 1024,
                level: 'extended',
                name: 'product',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal product name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'dll.pe.sections': {
                dashed_name: 'dll-pe-sections',
                description: 'An array containing an object for each section of the PE file.\nThe keys that should be present in these objects are defined by sub-fields underneath `pe.sections.*`.',
                flat_name: 'dll.pe.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'Section information of the PE file.',
                type: 'nested',
            },
            'dll.pe.sections.entropy': {
                dashed_name: 'dll-pe-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'dll.pe.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'dll.pe.sections.name': {
                dashed_name: 'dll-pe-sections-name',
                description: 'PE Section List name.',
                flat_name: 'dll.pe.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List name.',
                type: 'keyword',
            },
            'dll.pe.sections.physical_size': {
                dashed_name: 'dll-pe-sections-physical-size',
                description: 'PE Section List physical size.',
                flat_name: 'dll.pe.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List physical size.',
                type: 'long',
            },
            'dll.pe.sections.var_entropy': {
                dashed_name: 'dll-pe-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'dll.pe.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'dll.pe.sections.virtual_size': {
                dashed_name: 'dll-pe-sections-virtual-size',
                description: 'PE Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'dll.pe.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
        },
        group: 2,
        name: 'dll',
        nestings: ['dll.code_signature', 'dll.hash', 'dll.pe'],
        prefix: 'dll.',
        reused_here: [
            {
                full: 'dll.hash',
                schema_name: 'hash',
                short: 'Hashes, usually file hashes.',
            },
            {
                full: 'dll.pe',
                schema_name: 'pe',
                short: 'These fields contain Windows Portable Executable (PE) metadata.',
            },
            {
                full: 'dll.code_signature',
                schema_name: 'code_signature',
                short: 'These fields contain information about binary code signatures.',
            },
        ],
        short: 'These fields contain information about code libraries dynamically loaded into processes.',
        title: 'DLL',
        type: 'group',
    },
    dns: {
        description: 'Fields describing DNS queries and answers.\nDNS events should either represent a single DNS query prior to getting answers (`dns.type:query`) or they should represent a full exchange and contain the query details as well as all of the answers that were provided for this query (`dns.type:answer`).',
        fields: {
            'dns.answers': {
                dashed_name: 'dns-answers',
                description: 'An array containing an object for each answer section returned by the server.\nThe main keys that should be present in these objects are defined by ECS. Records that have more information may contain more keys than what ECS defines.\nNot all DNS data sources give all details about DNS answers. At minimum, answer objects must contain the `data` key. If more information is available, map as much of it to ECS as possible, and add any additional fields to the answer objects as custom fields.',
                flat_name: 'dns.answers',
                level: 'extended',
                name: 'answers',
                normalize: ['array'],
                short: 'Array of DNS answers.',
                type: 'object',
            },
            'dns.answers.class': {
                dashed_name: 'dns-answers-class',
                description: 'The class of DNS data contained in this resource record.',
                example: 'IN',
                flat_name: 'dns.answers.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'answers.class',
                normalize: [],
                short: 'The class of DNS data contained in this resource record.',
                type: 'keyword',
            },
            'dns.answers.data': {
                dashed_name: 'dns-answers-data',
                description: 'The data describing the resource.\nThe meaning of this data depends on the type and class of the resource record.',
                example: '10.10.10.10',
                flat_name: 'dns.answers.data',
                ignore_above: 1024,
                level: 'extended',
                name: 'answers.data',
                normalize: [],
                short: 'The data describing the resource.',
                type: 'keyword',
            },
            'dns.answers.name': {
                dashed_name: 'dns-answers-name',
                description: "The domain name to which this resource record pertains.\nIf a chain of CNAME is being resolved, each answer's `name` should be the one that corresponds with the answer's `data`. It should not simply be the original `question.name` repeated.",
                example: 'www.example.com',
                flat_name: 'dns.answers.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'answers.name',
                normalize: [],
                short: 'The domain name to which this resource record pertains.',
                type: 'keyword',
            },
            'dns.answers.ttl': {
                dashed_name: 'dns-answers-ttl',
                description: 'The time interval in seconds that this resource record may be cached before it should be discarded. Zero values mean that the data should not be cached.',
                example: 180,
                flat_name: 'dns.answers.ttl',
                level: 'extended',
                name: 'answers.ttl',
                normalize: [],
                short: 'The time interval in seconds that this resource record may be cached before it should be discarded.',
                type: 'long',
            },
            'dns.answers.type': {
                dashed_name: 'dns-answers-type',
                description: 'The type of data contained in this resource record.',
                example: 'CNAME',
                flat_name: 'dns.answers.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'answers.type',
                normalize: [],
                short: 'The type of data contained in this resource record.',
                type: 'keyword',
            },
            'dns.header_flags': {
                dashed_name: 'dns-header-flags',
                description: 'Array of 2 letter DNS header flags.',
                example: '["RD", "RA"]',
                expected_values: ['AA', 'TC', 'RD', 'RA', 'AD', 'CD', 'DO'],
                flat_name: 'dns.header_flags',
                ignore_above: 1024,
                level: 'extended',
                name: 'header_flags',
                normalize: ['array'],
                short: 'Array of DNS header flags.',
                type: 'keyword',
            },
            'dns.id': {
                dashed_name: 'dns-id',
                description: 'The DNS packet identifier assigned by the program that generated the query. The identifier is copied to the response.',
                example: 62111,
                flat_name: 'dns.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                short: 'The DNS packet identifier assigned by the program that generated the query. The identifier is copied to the response.',
                type: 'keyword',
            },
            'dns.op_code': {
                dashed_name: 'dns-op-code',
                description: 'The DNS operation code that specifies the kind of query in the message. This value is set by the originator of a query and copied into the response.',
                example: 'QUERY',
                flat_name: 'dns.op_code',
                ignore_above: 1024,
                level: 'extended',
                name: 'op_code',
                normalize: [],
                short: 'The DNS operation code that specifies the kind of query in the message.',
                type: 'keyword',
            },
            'dns.question.class': {
                dashed_name: 'dns-question-class',
                description: 'The class of records being queried.',
                example: 'IN',
                flat_name: 'dns.question.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'question.class',
                normalize: [],
                short: 'The class of records being queried.',
                type: 'keyword',
            },
            'dns.question.name': {
                dashed_name: 'dns-question-name',
                description: 'The name being queried.\nIf the name field contains non-printable characters (below 32 or above 126), those characters should be represented as escaped base 10 integers (\\DDD). Back slashes and quotes should be escaped. Tabs, carriage returns, and line feeds should be converted to \\t, \\r, and \\n respectively.',
                example: 'www.example.com',
                flat_name: 'dns.question.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'question.name',
                normalize: [],
                short: 'The name being queried.',
                type: 'keyword',
            },
            'dns.question.registered_domain': {
                dashed_name: 'dns-question-registered-domain',
                description: 'The highest registered domain, stripped of the subdomain.\nFor example, the registered domain for "foo.example.com" is "example.com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last two labels will not work well for TLDs such as "co.uk".',
                example: 'example.com',
                flat_name: 'dns.question.registered_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'question.registered_domain',
                normalize: [],
                short: 'The highest registered domain, stripped of the subdomain.',
                type: 'keyword',
            },
            'dns.question.subdomain': {
                dashed_name: 'dns-question-subdomain',
                description: 'The subdomain is all of the labels under the registered_domain.\nIf the domain has multiple levels of subdomain, such as "sub2.sub1.example.com", the subdomain field should contain "sub2.sub1", with no trailing period.',
                example: 'www',
                flat_name: 'dns.question.subdomain',
                ignore_above: 1024,
                level: 'extended',
                name: 'question.subdomain',
                normalize: [],
                short: 'The subdomain of the domain.',
                type: 'keyword',
            },
            'dns.question.top_level_domain': {
                dashed_name: 'dns-question-top-level-domain',
                description: 'The effective top level domain (eTLD), also known as the domain suffix, is the last part of the domain name. For example, the top level domain for example.com is "com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last label will not work well for effective TLDs such as "co.uk".',
                example: 'co.uk',
                flat_name: 'dns.question.top_level_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'question.top_level_domain',
                normalize: [],
                short: 'The effective top level domain (com, org, net, co.uk).',
                type: 'keyword',
            },
            'dns.question.type': {
                dashed_name: 'dns-question-type',
                description: 'The type of record being queried.',
                example: 'AAAA',
                flat_name: 'dns.question.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'question.type',
                normalize: [],
                short: 'The type of record being queried.',
                type: 'keyword',
            },
            'dns.resolved_ip': {
                dashed_name: 'dns-resolved-ip',
                description: 'Array containing all IPs seen in `answers.data`.\nThe `answers` array can be difficult to use, because of the variety of data formats it can contain. Extracting all IP addresses seen in there to `dns.resolved_ip` makes it possible to index them as IP addresses, and makes them easier to visualize and query for.',
                example: '["10.10.10.10", "10.10.10.11"]',
                flat_name: 'dns.resolved_ip',
                level: 'extended',
                name: 'resolved_ip',
                normalize: ['array'],
                short: 'Array containing all IPs seen in answers.data',
                type: 'ip',
            },
            'dns.response_code': {
                dashed_name: 'dns-response-code',
                description: 'The DNS response code.',
                example: 'NOERROR',
                flat_name: 'dns.response_code',
                ignore_above: 1024,
                level: 'extended',
                name: 'response_code',
                normalize: [],
                short: 'The DNS response code.',
                type: 'keyword',
            },
            'dns.type': {
                dashed_name: 'dns-type',
                description: 'The type of DNS event captured, query or answer.\nIf your source of DNS events only gives you DNS queries, you should only create dns events of type `dns.type:query`.\nIf your source of DNS events gives you answers as well, you should create one event per query (optionally as soon as the query is seen). And a second event containing all query details as well as an array of answers.',
                example: 'answer',
                flat_name: 'dns.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                short: 'The type of DNS event captured, query or answer.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'dns',
        prefix: 'dns.',
        short: 'Fields describing DNS queries and answers.',
        title: 'DNS',
        type: 'group',
    },
    ecs: {
        description: 'Meta-information specific to ECS.',
        fields: {
            'ecs.version': {
                dashed_name: 'ecs-version',
                description: 'ECS version this event conforms to. `ecs.version` is a required field and must exist in all events.\nWhen querying across multiple indices -- which may conform to slightly different ECS versions -- this field lets integrations adjust to the schema version of the events.',
                example: '1.0.0',
                flat_name: 'ecs.version',
                ignore_above: 1024,
                level: 'core',
                name: 'version',
                normalize: [],
                required: true,
                short: 'ECS version this event conforms to.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'ecs',
        prefix: 'ecs.',
        short: 'Meta-information specific to ECS.',
        title: 'ECS',
        type: 'group',
    },
    elf: {
        beta: 'These fields are in beta and are subject to change.',
        description: 'These fields contain Linux Executable Linkable Format (ELF) metadata.',
        fields: {
            'elf.architecture': {
                dashed_name: 'elf-architecture',
                description: 'Machine architecture of the ELF file.',
                example: 'x86-64',
                flat_name: 'elf.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                short: 'Machine architecture of the ELF file.',
                type: 'keyword',
            },
            'elf.byte_order': {
                dashed_name: 'elf-byte-order',
                description: 'Byte sequence of ELF file.',
                example: 'Little Endian',
                flat_name: 'elf.byte_order',
                ignore_above: 1024,
                level: 'extended',
                name: 'byte_order',
                normalize: [],
                short: 'Byte sequence of ELF file.',
                type: 'keyword',
            },
            'elf.cpu_type': {
                dashed_name: 'elf-cpu-type',
                description: 'CPU type of the ELF file.',
                example: 'Intel',
                flat_name: 'elf.cpu_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'cpu_type',
                normalize: [],
                short: 'CPU type of the ELF file.',
                type: 'keyword',
            },
            'elf.creation_date': {
                dashed_name: 'elf-creation-date',
                description: "Extracted when possible from the file's metadata. Indicates when it was built or compiled. It can also be faked by malware creators.",
                flat_name: 'elf.creation_date',
                level: 'extended',
                name: 'creation_date',
                normalize: [],
                short: 'Build or compile date.',
                type: 'date',
            },
            'elf.exports': {
                dashed_name: 'elf-exports',
                description: 'List of exported element names and types.',
                flat_name: 'elf.exports',
                level: 'extended',
                name: 'exports',
                normalize: ['array'],
                short: 'List of exported element names and types.',
                type: 'flattened',
            },
            'elf.go_import_hash': {
                dashed_name: 'elf-go-import-hash',
                description: 'A hash of the Go language imports in an ELF file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'elf.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                short: 'A hash of the Go language imports in an ELF file.',
                type: 'keyword',
            },
            'elf.go_imports': {
                dashed_name: 'elf-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'elf.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'elf.go_imports_names_entropy': {
                dashed_name: 'elf-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'elf.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'elf.go_imports_names_var_entropy': {
                dashed_name: 'elf-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'elf.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'elf.go_stripped': {
                dashed_name: 'elf-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'elf.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'elf.header.abi_version': {
                dashed_name: 'elf-header-abi-version',
                description: 'Version of the ELF Application Binary Interface (ABI).',
                flat_name: 'elf.header.abi_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.abi_version',
                normalize: [],
                short: 'Version of the ELF Application Binary Interface (ABI).',
                type: 'keyword',
            },
            'elf.header.class': {
                dashed_name: 'elf-header-class',
                description: 'Header class of the ELF file.',
                flat_name: 'elf.header.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.class',
                normalize: [],
                short: 'Header class of the ELF file.',
                type: 'keyword',
            },
            'elf.header.data': {
                dashed_name: 'elf-header-data',
                description: 'Data table of the ELF header.',
                flat_name: 'elf.header.data',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.data',
                normalize: [],
                short: 'Data table of the ELF header.',
                type: 'keyword',
            },
            'elf.header.entrypoint': {
                dashed_name: 'elf-header-entrypoint',
                description: 'Header entrypoint of the ELF file.',
                flat_name: 'elf.header.entrypoint',
                format: 'string',
                level: 'extended',
                name: 'header.entrypoint',
                normalize: [],
                short: 'Header entrypoint of the ELF file.',
                type: 'long',
            },
            'elf.header.object_version': {
                dashed_name: 'elf-header-object-version',
                description: '"0x1" for original ELF files.',
                flat_name: 'elf.header.object_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.object_version',
                normalize: [],
                short: '"0x1" for original ELF files.',
                type: 'keyword',
            },
            'elf.header.os_abi': {
                dashed_name: 'elf-header-os-abi',
                description: 'Application Binary Interface (ABI) of the Linux OS.',
                flat_name: 'elf.header.os_abi',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.os_abi',
                normalize: [],
                short: 'Application Binary Interface (ABI) of the Linux OS.',
                type: 'keyword',
            },
            'elf.header.type': {
                dashed_name: 'elf-header-type',
                description: 'Header type of the ELF file.',
                flat_name: 'elf.header.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.type',
                normalize: [],
                short: 'Header type of the ELF file.',
                type: 'keyword',
            },
            'elf.header.version': {
                dashed_name: 'elf-header-version',
                description: 'Version of the ELF header.',
                flat_name: 'elf.header.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.version',
                normalize: [],
                short: 'Version of the ELF header.',
                type: 'keyword',
            },
            'elf.import_hash': {
                dashed_name: 'elf-import-hash',
                description: 'A hash of the imports in an ELF file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is an ELF implementation of the Windows PE imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'elf.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                short: 'A hash of the imports in an ELF file.',
                type: 'keyword',
            },
            'elf.imports': {
                dashed_name: 'elf-imports',
                description: 'List of imported element names and types.',
                flat_name: 'elf.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'elf.imports_names_entropy': {
                dashed_name: 'elf-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'elf.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'elf.imports_names_var_entropy': {
                dashed_name: 'elf-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'elf.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'elf.sections': {
                dashed_name: 'elf-sections',
                description: 'An array containing an object for each section of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.sections.*`.',
                flat_name: 'elf.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                short: 'Section information of the ELF file.',
                type: 'nested',
            },
            'elf.sections.chi2': {
                dashed_name: 'elf-sections-chi2',
                description: 'Chi-square probability distribution of the section.',
                flat_name: 'elf.sections.chi2',
                format: 'number',
                level: 'extended',
                name: 'sections.chi2',
                normalize: [],
                short: 'Chi-square probability distribution of the section.',
                type: 'long',
            },
            'elf.sections.entropy': {
                dashed_name: 'elf-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'elf.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'elf.sections.flags': {
                dashed_name: 'elf-sections-flags',
                description: 'ELF Section List flags.',
                flat_name: 'elf.sections.flags',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.flags',
                normalize: [],
                short: 'ELF Section List flags.',
                type: 'keyword',
            },
            'elf.sections.name': {
                dashed_name: 'elf-sections-name',
                description: 'ELF Section List name.',
                flat_name: 'elf.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                short: 'ELF Section List name.',
                type: 'keyword',
            },
            'elf.sections.physical_offset': {
                dashed_name: 'elf-sections-physical-offset',
                description: 'ELF Section List offset.',
                flat_name: 'elf.sections.physical_offset',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.physical_offset',
                normalize: [],
                short: 'ELF Section List offset.',
                type: 'keyword',
            },
            'elf.sections.physical_size': {
                dashed_name: 'elf-sections-physical-size',
                description: 'ELF Section List physical size.',
                flat_name: 'elf.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                short: 'ELF Section List physical size.',
                type: 'long',
            },
            'elf.sections.type': {
                dashed_name: 'elf-sections-type',
                description: 'ELF Section List type.',
                flat_name: 'elf.sections.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.type',
                normalize: [],
                short: 'ELF Section List type.',
                type: 'keyword',
            },
            'elf.sections.var_entropy': {
                dashed_name: 'elf-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'elf.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'elf.sections.virtual_address': {
                dashed_name: 'elf-sections-virtual-address',
                description: 'ELF Section List virtual address.',
                flat_name: 'elf.sections.virtual_address',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_address',
                normalize: [],
                short: 'ELF Section List virtual address.',
                type: 'long',
            },
            'elf.sections.virtual_size': {
                dashed_name: 'elf-sections-virtual-size',
                description: 'ELF Section List virtual size.',
                flat_name: 'elf.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                short: 'ELF Section List virtual size.',
                type: 'long',
            },
            'elf.segments': {
                dashed_name: 'elf-segments',
                description: 'An array containing an object for each segment of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.segments.*`.',
                flat_name: 'elf.segments',
                level: 'extended',
                name: 'segments',
                normalize: ['array'],
                short: 'ELF object segment list.',
                type: 'nested',
            },
            'elf.segments.sections': {
                dashed_name: 'elf-segments-sections',
                description: 'ELF object segment sections.',
                flat_name: 'elf.segments.sections',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.sections',
                normalize: [],
                short: 'ELF object segment sections.',
                type: 'keyword',
            },
            'elf.segments.type': {
                dashed_name: 'elf-segments-type',
                description: 'ELF object segment type.',
                flat_name: 'elf.segments.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.type',
                normalize: [],
                short: 'ELF object segment type.',
                type: 'keyword',
            },
            'elf.shared_libraries': {
                dashed_name: 'elf-shared-libraries',
                description: 'List of shared libraries used by this ELF object.',
                flat_name: 'elf.shared_libraries',
                ignore_above: 1024,
                level: 'extended',
                name: 'shared_libraries',
                normalize: ['array'],
                short: 'List of shared libraries used by this ELF object.',
                type: 'keyword',
            },
            'elf.telfhash': {
                dashed_name: 'elf-telfhash',
                description: 'telfhash symbol hash for ELF file.',
                flat_name: 'elf.telfhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'telfhash',
                normalize: [],
                short: 'telfhash hash for ELF file.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'elf',
        prefix: 'elf.',
        reusable: {
            expected: [
                {
                    as: 'elf',
                    at: 'file',
                    beta: 'This field reuse is beta and subject to change.',
                    full: 'file.elf',
                },
                {
                    as: 'elf',
                    at: 'process',
                    beta: 'This field reuse is beta and subject to change.',
                    full: 'process.elf',
                },
            ],
            top_level: false,
        },
        short: 'These fields contain Linux Executable Linkable Format (ELF) metadata.',
        title: 'ELF Header',
        type: 'group',
    },
    email: {
        description: 'Event details relating to an email transaction.\nThis field set focuses on the email message header, body, and attachments. Network protocols that send and receive email messages such as SMTP are outside the scope of the `email.*` fields.',
        fields: {
            'email.attachments': {
                dashed_name: 'email-attachments',
                description: 'A list of objects describing the attachment files sent along with an email message.',
                flat_name: 'email.attachments',
                level: 'extended',
                name: 'attachments',
                normalize: ['array'],
                short: 'List of objects describing the attachments.',
                type: 'nested',
            },
            'email.attachments.file.extension': {
                dashed_name: 'email-attachments-file-extension',
                description: 'Attachment file extension, excluding the leading dot.',
                example: 'txt',
                flat_name: 'email.attachments.file.extension',
                ignore_above: 1024,
                level: 'extended',
                name: 'attachments.file.extension',
                normalize: [],
                short: 'Attachment file extension.',
                type: 'keyword',
            },
            'email.attachments.file.hash.md5': {
                dashed_name: 'email-attachments-file-hash-md5',
                description: 'MD5 hash.',
                flat_name: 'email.attachments.file.hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                original_fieldset: 'hash',
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'email.attachments.file.hash.sha1': {
                dashed_name: 'email-attachments-file-hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'email.attachments.file.hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'email.attachments.file.hash.sha256': {
                dashed_name: 'email-attachments-file-hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'email.attachments.file.hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'email.attachments.file.hash.sha384': {
                dashed_name: 'email-attachments-file-hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'email.attachments.file.hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'email.attachments.file.hash.sha512': {
                dashed_name: 'email-attachments-file-hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'email.attachments.file.hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'email.attachments.file.hash.ssdeep': {
                dashed_name: 'email-attachments-file-hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'email.attachments.file.hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'email.attachments.file.hash.tlsh': {
                dashed_name: 'email-attachments-file-hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'email.attachments.file.hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                original_fieldset: 'hash',
                short: 'TLSH hash.',
                type: 'keyword',
            },
            'email.attachments.file.mime_type': {
                dashed_name: 'email-attachments-file-mime-type',
                description: 'The MIME media type of the attachment.\nThis value will typically be extracted from the `Content-Type` MIME header field.',
                example: 'text/plain',
                flat_name: 'email.attachments.file.mime_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'attachments.file.mime_type',
                normalize: [],
                short: 'MIME type of the attachment file.',
                type: 'keyword',
            },
            'email.attachments.file.name': {
                dashed_name: 'email-attachments-file-name',
                description: 'Name of the attachment file including the file extension.',
                example: 'attachment.txt',
                flat_name: 'email.attachments.file.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'attachments.file.name',
                normalize: [],
                short: 'Name of the attachment file.',
                type: 'keyword',
            },
            'email.attachments.file.size': {
                dashed_name: 'email-attachments-file-size',
                description: 'Attachment file size in bytes.',
                example: 64329,
                flat_name: 'email.attachments.file.size',
                level: 'extended',
                name: 'attachments.file.size',
                normalize: [],
                short: 'Attachment file size.',
                type: 'long',
            },
            'email.bcc.address': {
                dashed_name: 'email-bcc-address',
                description: 'The email address of BCC recipient',
                example: 'bcc.user1@example.com',
                flat_name: 'email.bcc.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'bcc.address',
                normalize: ['array'],
                short: 'Email address of BCC recipient',
                type: 'keyword',
            },
            'email.cc.address': {
                dashed_name: 'email-cc-address',
                description: 'The email address of CC recipient',
                example: 'cc.user1@example.com',
                flat_name: 'email.cc.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'cc.address',
                normalize: ['array'],
                short: 'Email address of CC recipient',
                type: 'keyword',
            },
            'email.content_type': {
                dashed_name: 'email-content-type',
                description: 'Information about how the message is to be displayed.\nTypically a MIME type.',
                example: 'text/plain',
                flat_name: 'email.content_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'content_type',
                normalize: [],
                short: 'MIME type of the email message.',
                type: 'keyword',
            },
            'email.delivery_timestamp': {
                dashed_name: 'email-delivery-timestamp',
                description: 'The date and time when the email message was received by the service or client.',
                example: '2020-11-10T22:12:34.8196921Z',
                flat_name: 'email.delivery_timestamp',
                level: 'extended',
                name: 'delivery_timestamp',
                normalize: [],
                short: 'Date and time when message was delivered.',
                type: 'date',
            },
            'email.direction': {
                dashed_name: 'email-direction',
                description: 'The direction of the message based on the sending and receiving domains.',
                example: 'inbound',
                flat_name: 'email.direction',
                ignore_above: 1024,
                level: 'extended',
                name: 'direction',
                normalize: [],
                short: 'Direction of the message.',
                type: 'keyword',
            },
            'email.from.address': {
                dashed_name: 'email-from-address',
                description: 'The email address of the sender, typically from the RFC 5322 `From:` header field.',
                example: 'sender@example.com',
                flat_name: 'email.from.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'from.address',
                normalize: ['array'],
                short: "The sender's email address.",
                type: 'keyword',
            },
            'email.local_id': {
                dashed_name: 'email-local-id',
                description: 'Unique identifier given to the email by the source that created the event.\nIdentifier is not persistent across hops.',
                example: 'c26dbea0-80d5-463b-b93c-4e8b708219ce',
                flat_name: 'email.local_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'local_id',
                normalize: [],
                short: 'Unique identifier given by the source.',
                type: 'keyword',
            },
            'email.message_id': {
                dashed_name: 'email-message-id',
                description: 'Identifier from the RFC 5322 `Message-ID:` email header that refers to a particular email message.',
                example: '81ce15$8r2j59@mail01.example.com',
                flat_name: 'email.message_id',
                level: 'extended',
                name: 'message_id',
                normalize: [],
                short: 'Value from the Message-ID header.',
                type: 'wildcard',
            },
            'email.origination_timestamp': {
                dashed_name: 'email-origination-timestamp',
                description: 'The date and time the email message was composed. Many email clients will fill in this value automatically when the message is sent by a user.',
                example: '2020-11-10T22:12:34.8196921Z',
                flat_name: 'email.origination_timestamp',
                level: 'extended',
                name: 'origination_timestamp',
                normalize: [],
                short: 'Date and time the email was composed.',
                type: 'date',
            },
            'email.reply_to.address': {
                dashed_name: 'email-reply-to-address',
                description: 'The address that replies should be delivered to based on the value in the RFC 5322 `Reply-To:` header.',
                example: 'reply.here@example.com',
                flat_name: 'email.reply_to.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'reply_to.address',
                normalize: ['array'],
                short: 'Address replies should be delivered to.',
                type: 'keyword',
            },
            'email.sender.address': {
                dashed_name: 'email-sender-address',
                description: 'Per RFC 5322, specifies the address responsible for the actual transmission of the message.',
                flat_name: 'email.sender.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'sender.address',
                normalize: [],
                short: 'Address of the message sender.',
                type: 'keyword',
            },
            'email.subject': {
                dashed_name: 'email-subject',
                description: 'A brief summary of the topic of the message.',
                example: 'Please see this important message.',
                flat_name: 'email.subject',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'email.subject.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'subject',
                normalize: [],
                short: 'The subject of the email message.',
                type: 'keyword',
            },
            'email.to.address': {
                dashed_name: 'email-to-address',
                description: 'The email address of recipient',
                example: 'user1@example.com',
                flat_name: 'email.to.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'to.address',
                normalize: ['array'],
                short: 'Email address of recipient',
                type: 'keyword',
            },
            'email.x_mailer': {
                dashed_name: 'email-x-mailer',
                description: 'The name of the application that was used to draft and send the original email message.',
                example: 'Spambot v2.5',
                flat_name: 'email.x_mailer',
                ignore_above: 1024,
                level: 'extended',
                name: 'x_mailer',
                normalize: [],
                short: 'Application that drafted email.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'email',
        nestings: ['email.attachments.file.hash'],
        prefix: 'email.',
        reused_here: [
            {
                full: 'email.attachments.file.hash',
                schema_name: 'hash',
                short: 'Hashes, usually file hashes.',
            },
        ],
        short: 'Describes an email transaction.',
        title: 'Email',
        type: 'group',
    },
    error: {
        description: 'These fields can represent errors of any kind.\nUse them for errors that happen while fetching events or in cases where the event itself contains an error.',
        fields: {
            'error.code': {
                dashed_name: 'error-code',
                description: 'Error code describing the error.',
                flat_name: 'error.code',
                ignore_above: 1024,
                level: 'core',
                name: 'code',
                normalize: [],
                short: 'Error code describing the error.',
                type: 'keyword',
            },
            'error.id': {
                dashed_name: 'error-id',
                description: 'Unique identifier for the error.',
                flat_name: 'error.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                short: 'Unique identifier for the error.',
                type: 'keyword',
            },
            'error.message': {
                dashed_name: 'error-message',
                description: 'Error message.',
                flat_name: 'error.message',
                level: 'core',
                name: 'message',
                normalize: [],
                short: 'Error message.',
                type: 'match_only_text',
            },
            'error.stack_trace': {
                dashed_name: 'error-stack-trace',
                description: 'The stack trace of this error in plain text.',
                flat_name: 'error.stack_trace',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'error.stack_trace.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'stack_trace',
                normalize: [],
                short: 'The stack trace of this error in plain text.',
                type: 'wildcard',
            },
            'error.type': {
                dashed_name: 'error-type',
                description: 'The type of the error, for example the class name of the exception.',
                example: 'java.lang.NullPointerException',
                flat_name: 'error.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                short: 'The type of the error, for example the class name of the exception.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'error',
        prefix: 'error.',
        short: 'Fields about errors of any kind.',
        title: 'Error',
        type: 'group',
    },
    event: {
        description: 'The event fields are used for context information about the log or metric event itself.\nA log is defined as an event containing details of something that happened. Log events must include the time at which the thing happened. Examples of log events include a process starting on a host, a network packet being sent from a source to a destination, or a network connection between a client and a server being initiated or closed. A metric is defined as an event containing one or more numerical measurements and the time at which the measurement was taken. Examples of metric events include memory pressure measured on a host and device temperature. See the `event.kind` definition in this section for additional details about metric and state events.',
        fields: {
            'event.action': {
                dashed_name: 'event-action',
                description: 'The action captured by the event.\nThis describes the information in the event. It is more specific than `event.category`. Examples are `group-add`, `process-started`, `file-created`. The value is normally defined by the implementer.',
                example: 'user-password-change',
                flat_name: 'event.action',
                ignore_above: 1024,
                level: 'core',
                name: 'action',
                normalize: [],
                short: 'The action captured by the event.',
                type: 'keyword',
            },
            'event.agent_id_status': {
                dashed_name: 'event-agent-id-status',
                description: "Agents are normally responsible for populating the `agent.id` field value. If the system receiving events is capable of validating the value based on authentication information for the client then this field can be used to reflect the outcome of that validation.\nFor example if the agent's connection is authenticated with mTLS and the client cert contains the ID of the agent to which the cert was issued then the `agent.id` value in events can be checked against the certificate. If the values match then `event.agent_id_status: verified` is added to the event, otherwise one of the other allowed values should be used.\nIf no validation is performed then the field should be omitted.\nThe allowed values are:\n`verified` - The `agent.id` field value matches expected value obtained from auth metadata.\n`mismatch` - The `agent.id` field value does not match the expected value obtained from auth metadata.\n`missing` - There was no `agent.id` field in the event to validate.\n`auth_metadata_missing` - There was no auth metadata or it was missing information about the agent ID.",
                example: 'verified',
                flat_name: 'event.agent_id_status',
                ignore_above: 1024,
                level: 'extended',
                name: 'agent_id_status',
                normalize: [],
                short: "Validation status of the event's agent.id field.",
                type: 'keyword',
            },
            'event.category': {
                allowed_values: [
                    {
                        description: 'Events in this category annotate API calls that occured on a system. Typical sources for those events could be from the Operating System level through the native libraries (for example Windows Win32, Linux libc, etc.), or managed sources of events (such as ETW, syslog), but can also include network protocols (such as SOAP, RPC, Websocket, REST, etc.)',
                        expected_event_types: [
                            'access',
                            'admin',
                            'allowed',
                            'change',
                            'creation',
                            'deletion',
                            'denied',
                            'end',
                            'info',
                            'start',
                            'user',
                        ],
                        name: 'api',
                    },
                    {
                        description: 'Events in this category are related to the challenge and response process in which credentials are supplied and verified to allow the creation of a session. Common sources for these logs are Windows event logs and ssh logs. Visualize and analyze events in this category to look for failed logins, and other authentication-related activity.',
                        expected_event_types: ['start', 'end', 'info'],
                        name: 'authentication',
                    },
                    {
                        description: 'Events in the configuration category have to deal with creating, modifying, or deleting the settings or parameters of an application, process, or system.\nExample sources include security policy change logs, configuration auditing logging, and system integrity monitoring.',
                        expected_event_types: [
                            'access',
                            'change',
                            'creation',
                            'deletion',
                            'info',
                        ],
                        name: 'configuration',
                    },
                    {
                        description: 'The database category denotes events and metrics relating to a data storage and retrieval system. Note that use of this category is not limited to relational database systems. Examples include event logs from MS SQL, MySQL, Elasticsearch, MongoDB, etc. Use this category to visualize and analyze database activity such as accesses and changes.',
                        expected_event_types: ['access', 'change', 'info', 'error'],
                        name: 'database',
                    },
                    {
                        description: 'Events in the driver category have to do with operating system device drivers and similar software entities such as Windows drivers, kernel extensions, kernel modules, etc.\nUse events and metrics in this category to visualize and analyze driver-related activity and status on hosts.',
                        expected_event_types: ['change', 'end', 'info', 'start'],
                        name: 'driver',
                    },
                    {
                        description: 'This category is used for events relating to email messages, email attachments, and email network or protocol activity.\nEmails events can be produced by email security gateways, mail transfer agents, email cloud service providers, or mail server monitoring applications.',
                        expected_event_types: ['info'],
                        name: 'email',
                    },
                    {
                        description: 'Relating to a set of information that has been created on, or has existed on a filesystem. Use this category of events to visualize and analyze the creation, access, and deletions of files. Events in this category can come from both host-based and network-based sources. An example source of a network-based detection of a file transfer would be the Zeek file.log.',
                        expected_event_types: [
                            'access',
                            'change',
                            'creation',
                            'deletion',
                            'info',
                        ],
                        name: 'file',
                    },
                    {
                        description: 'Use this category to visualize and analyze information such as host inventory or host lifecycle events.\nMost of the events in this category can usually be observed from the outside, such as from a hypervisor or a control plane\'s point of view. Some can also be seen from within, such as "start" or "end".\nNote that this category is for information about hosts themselves; it is not meant to capture activity "happening on a host".',
                        expected_event_types: ['access', 'change', 'end', 'info', 'start'],
                        name: 'host',
                    },
                    {
                        description: 'Identity and access management (IAM) events relating to users, groups, and administration. Use this category to visualize and analyze IAM-related logs and data from active directory, LDAP, Okta, Duo, and other IAM systems.',
                        expected_event_types: [
                            'admin',
                            'change',
                            'creation',
                            'deletion',
                            'group',
                            'info',
                            'user',
                        ],
                        name: 'iam',
                    },
                    {
                        description: 'Relating to intrusion detections from IDS/IPS systems and functions, both network and host-based. Use this category to visualize and analyze intrusion detection alerts from systems such as Snort, Suricata, and Palo Alto threat detections.',
                        expected_event_types: ['allowed', 'denied', 'info'],
                        name: 'intrusion_detection',
                    },
                    {
                        description: 'Events in this category refer to the loading of a library, such as (dll / so / dynlib), into a process. Use this category to visualize and analyze library loading related activity on hosts.  Keep in mind that driver related activity will be captured under the "driver" category above.',
                        expected_event_types: ['start'],
                        name: 'library',
                    },
                    {
                        description: 'Malware detection events and alerts. Use this category to visualize and analyze malware detections from EDR/EPP systems such as Elastic Endpoint Security, Symantec Endpoint Protection, Crowdstrike, and network IDS/IPS systems such as Suricata, or other sources of malware-related events such as Palo Alto Networks threat logs and Wildfire logs.',
                        expected_event_types: ['info'],
                        name: 'malware',
                    },
                    {
                        description: 'Relating to all network activity, including network connection lifecycle, network traffic, and essentially any event that includes an IP address. Many events containing decoded network protocol transactions fit into this category. Use events in this category to visualize or analyze counts of network ports, protocols, addresses, geolocation information, etc.',
                        expected_event_types: [
                            'access',
                            'allowed',
                            'connection',
                            'denied',
                            'end',
                            'info',
                            'protocol',
                            'start',
                        ],
                        name: 'network',
                    },
                    {
                        description: 'Relating to software packages installed on hosts. Use this category to visualize and analyze inventory of software installed on various hosts, or to determine host vulnerability in the absence of vulnerability scan data.',
                        expected_event_types: [
                            'access',
                            'change',
                            'deletion',
                            'info',
                            'installation',
                            'start',
                        ],
                        name: 'package',
                    },
                    {
                        description: 'Use this category of events to visualize and analyze process-specific information such as lifecycle events or process ancestry.',
                        expected_event_types: ['access', 'change', 'end', 'info', 'start'],
                        name: 'process',
                    },
                    {
                        description: 'Having to do with settings and assets stored in the Windows registry. Use this category to visualize and analyze activity such as registry access and modifications.',
                        expected_event_types: ['access', 'change', 'creation', 'deletion'],
                        name: 'registry',
                    },
                    {
                        description: 'The session category is applied to events and metrics regarding logical persistent connections to hosts and services. Use this category to visualize and analyze interactive or automated persistent connections between assets. Data for this category may come from Windows Event logs, SSH logs, or stateless sessions such as HTTP cookie-based sessions, etc.',
                        expected_event_types: ['start', 'end', 'info'],
                        name: 'session',
                    },
                    {
                        description: "Use this category to visualize and analyze events describing threat actors' targets, motives, or behaviors.",
                        expected_event_types: ['indicator'],
                        name: 'threat',
                    },
                    {
                        description: 'Relating to vulnerability scan results. Use this category to analyze vulnerabilities detected by Tenable, Qualys, internal scanners, and other vulnerability management sources.',
                        expected_event_types: ['info'],
                        name: 'vulnerability',
                    },
                    {
                        description: 'Relating to web server access. Use this category to create a dashboard of web server/proxy activity from apache, IIS, nginx web servers, etc. Note: events from network observers such as Zeek http log may also be included in this category.',
                        expected_event_types: ['access', 'error', 'info'],
                        name: 'web',
                    },
                ],
                dashed_name: 'event-category',
                description: 'This is one of four ECS Categorization Fields, and indicates the second level in the ECS category hierarchy.\n`event.category` represents the "big buckets" of ECS categories. For example, filtering on `event.category:process` yields all events relating to process activity. This field is closely related to `event.type`, which is used as a subcategory.\nThis field is an array. This will allow proper categorization of some events that fall in multiple categories.',
                example: 'authentication',
                flat_name: 'event.category',
                ignore_above: 1024,
                level: 'core',
                name: 'category',
                normalize: ['array'],
                short: 'Event category. The second categorization field in the hierarchy.',
                type: 'keyword',
            },
            'event.code': {
                dashed_name: 'event-code',
                description: 'Identification code for this event, if one exists.\nSome event sources use event codes to identify messages unambiguously, regardless of message language or wording adjustments over time. An example of this is the Windows Event ID.',
                example: 4648,
                flat_name: 'event.code',
                ignore_above: 1024,
                level: 'extended',
                name: 'code',
                normalize: [],
                short: 'Identification code for this event.',
                type: 'keyword',
            },
            'event.created': {
                dashed_name: 'event-created',
                description: "`event.created` contains the date/time when the event was first read by an agent, or by your pipeline.\nThis field is distinct from `@timestamp` in that `@timestamp` typically contain the time extracted from the original event.\nIn most situations, these two timestamps will be slightly different. The difference can be used to calculate the delay between your source generating an event, and the time when your agent first processed it. This can be used to monitor your agent's or pipeline's ability to keep up with your event source.\nIn case the two timestamps are identical, `@timestamp` should be used.",
                example: '2016-05-23T08:05:34.857Z',
                flat_name: 'event.created',
                level: 'core',
                name: 'created',
                normalize: [],
                short: 'Time when the event was first read by an agent or by your pipeline.',
                type: 'date',
            },
            'event.dataset': {
                dashed_name: 'event-dataset',
                description: "Name of the dataset.\nIf an event source publishes more than one type of log or events (e.g. access log, error log), the dataset is used to specify which one the event comes from.\nIt's recommended but not required to start the dataset name with the module name, followed by a dot, then the dataset name.",
                example: 'apache.access',
                flat_name: 'event.dataset',
                ignore_above: 1024,
                level: 'core',
                name: 'dataset',
                normalize: [],
                short: 'Name of the dataset.',
                type: 'keyword',
            },
            'event.duration': {
                dashed_name: 'event-duration',
                description: 'Duration of the event in nanoseconds.\nIf `event.start` and `event.end` are known this value should be the difference between the end and start time.',
                flat_name: 'event.duration',
                format: 'duration',
                input_format: 'nanoseconds',
                level: 'core',
                name: 'duration',
                normalize: [],
                output_format: 'asMilliseconds',
                output_precision: 1,
                short: 'Duration of the event in nanoseconds.',
                type: 'long',
            },
            'event.end': {
                dashed_name: 'event-end',
                description: '`event.end` contains the date when the event ended or when the activity was last observed.',
                flat_name: 'event.end',
                level: 'extended',
                name: 'end',
                normalize: [],
                short: '`event.end` contains the date when the event ended or when the activity was last observed.',
                type: 'date',
            },
            'event.hash': {
                dashed_name: 'event-hash',
                description: 'Hash (perhaps logstash fingerprint) of raw field to be able to demonstrate log integrity.',
                example: '123456789012345678901234567890ABCD',
                flat_name: 'event.hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'hash',
                normalize: [],
                short: 'Hash (perhaps logstash fingerprint) of raw field to be able to demonstrate log integrity.',
                type: 'keyword',
            },
            'event.id': {
                dashed_name: 'event-id',
                description: 'Unique ID to describe the event.',
                example: '8a4f500d',
                flat_name: 'event.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                short: 'Unique ID to describe the event.',
                type: 'keyword',
            },
            'event.ingested': {
                dashed_name: 'event-ingested',
                description: "Timestamp when an event arrived in the central data store.\nThis is different from `@timestamp`, which is when the event originally occurred.  It's also different from `event.created`, which is meant to capture the first time an agent saw the event.\nIn normal conditions, assuming no tampering, the timestamps should chronologically look like this: `@timestamp` < `event.created` < `event.ingested`.",
                example: '2016-05-23T08:05:35.101Z',
                flat_name: 'event.ingested',
                level: 'core',
                name: 'ingested',
                normalize: [],
                short: 'Timestamp when an event arrived in the central data store.',
                type: 'date',
            },
            'event.kind': {
                allowed_values: [
                    {
                        description: 'This value indicates an event such as an alert or notable event, triggered by a detection rule executing externally to the Elastic Stack.\n`event.kind:alert` is often populated for events coming from firewalls, intrusion detection systems, endpoint detection and response systems, and so on.\nThis value is not used by Elastic solutions for alert documents that are created by rules executing within the Kibana alerting framework.',
                        name: 'alert',
                    },
                    {
                        beta: 'This event categorization value is beta and subject to change.',
                        description: 'This value indicates events whose primary purpose is to store an inventory of assets/entities and their attributes. Assets/entities are objects (such as users and hosts) that are expected to be subjects of detailed analysis within the system.\nExamples include lists of user identities or accounts ingested from directory services such as Active Directory (AD), inventory of hosts pulled from configuration management databases (CMDB), and lists of cloud storage buckets pulled from cloud provider APIs.\nThis value is used by Elastic Security for asset management solutions. `event.kind: asset` is not used for normal system events or logs that are coming from an asset/entity, nor is it used for system events or logs coming from a directory or CMDB system.',
                        name: 'asset',
                    },
                    {
                        description: 'The `enrichment` value indicates an event collected to provide additional context, often to other events.\nAn example is collecting indicators of compromise (IOCs) from a threat intelligence provider with the intent to use those values to enrich other events. The IOC events from the intelligence provider should be categorized as `event.kind:enrichment`.',
                        name: 'enrichment',
                    },
                    {
                        description: 'This value is the most general and most common value for this field. It is used to represent events that indicate that something happened.',
                        name: 'event',
                    },
                    {
                        description: 'This value is used to indicate that this event describes a numeric measurement taken at given point in time.\nExamples include CPU utilization, memory usage, or device temperature.\nMetric events are often collected on a predictable frequency, such as once every few seconds, or once a minute, but can also be used to describe ad-hoc numeric metric queries.',
                        name: 'metric',
                    },
                    {
                        description: "The state value is similar to metric, indicating that this event describes a measurement taken at given point in time, except that the measurement does not result in a numeric value, but rather one of a fixed set of categorical values that represent conditions or states.\nExamples include periodic events reporting Elasticsearch cluster state (green/yellow/red), the state of a TCP connection (open, closed, fin_wait, etc.), the state of a host with respect to a software vulnerability (vulnerable, not vulnerable), and the state of a system regarding compliance with a regulatory standard (compliant, not compliant).\nNote that an event that describes a change of state would not use `event.kind:state`, but instead would use 'event.kind:event' since a state change fits the more general event definition of something that happened.\nState events are often collected on a predictable frequency, such as once every few seconds, once a minute, once an hour, or once a day, but can also be used to describe ad-hoc state queries.",
                        name: 'state',
                    },
                    {
                        description: 'This value indicates that an error occurred during the ingestion of this event, and that event data may be missing, inconsistent, or incorrect. `event.kind:pipeline_error` is often associated with parsing errors.',
                        name: 'pipeline_error',
                    },
                    {
                        description: 'This value is used by Elastic solutions (e.g., Security, Observability) for alert documents that are created by rules executing within the Kibana alerting framework.\nUsage of this value is reserved, and data ingestion pipelines must not populate `event.kind` with the value "signal".',
                        name: 'signal',
                    },
                ],
                dashed_name: 'event-kind',
                description: 'This is one of four ECS Categorization Fields, and indicates the highest level in the ECS category hierarchy.\n`event.kind` gives high-level information about what type of information the event contains, without being specific to the contents of the event. For example, values of this field distinguish alert events from metric events.\nThe value of this field can be used to inform how these kinds of events should be handled. They may warrant different retention, different access control, it may also help understand whether the data is coming in at a regular interval or not.',
                example: 'alert',
                flat_name: 'event.kind',
                ignore_above: 1024,
                level: 'core',
                name: 'kind',
                normalize: [],
                short: 'The kind of the event. The highest categorization field in the hierarchy.',
                type: 'keyword',
            },
            'event.module': {
                dashed_name: 'event-module',
                description: 'Name of the module this data is coming from.\nIf your monitoring agent supports the concept of modules or plugins to process events of a given source (e.g. Apache logs), `event.module` should contain the name of this module.',
                example: 'apache',
                flat_name: 'event.module',
                ignore_above: 1024,
                level: 'core',
                name: 'module',
                normalize: [],
                short: 'Name of the module this data is coming from.',
                type: 'keyword',
            },
            'event.original': {
                dashed_name: 'event-original',
                description: 'Raw text message of entire event. Used to demonstrate log integrity or where the full log message (before splitting it up in multiple parts) may be required, e.g. for reindex.\nThis field is not indexed and doc_values are disabled. It cannot be searched, but it can be retrieved from `_source`. If users wish to override this and index this field, please see `Field data types` in the `Elasticsearch Reference`.',
                doc_values: false,
                example: 'Sep 19 08:26:10 host CEF:0&#124;Security&#124; threatmanager&#124;1.0&#124;100&#124; worm successfully stopped&#124;10&#124;src=10.0.0.1 dst=2.1.2.2spt=1232',
                flat_name: 'event.original',
                index: false,
                level: 'core',
                name: 'original',
                normalize: [],
                short: 'Raw text message of entire event.',
                type: 'keyword',
            },
            'event.outcome': {
                allowed_values: [
                    {
                        description: 'Indicates that this event describes a failed result. A common example is `event.category:file AND event.type:access AND event.outcome:failure` to indicate that a file access was attempted, but was not successful.',
                        name: 'failure',
                    },
                    {
                        description: 'Indicates that this event describes a successful result. A common example is `event.category:file AND event.type:create AND event.outcome:success` to indicate that a file was successfully created.',
                        name: 'success',
                    },
                    {
                        description: "Indicates that this event describes only an attempt for which the result is unknown from the perspective of the event producer. For example, if the event contains information only about the request side of a transaction that results in a response, populating `event.outcome:unknown` in the request event is appropriate. The unknown value should not be used when an outcome doesn't make logical sense for the event. In such cases `event.outcome` should not be populated.",
                        name: 'unknown',
                    },
                ],
                dashed_name: 'event-outcome',
                description: 'This is one of four ECS Categorization Fields, and indicates the lowest level in the ECS category hierarchy.\n`event.outcome` simply denotes whether the event represents a success or a failure from the perspective of the entity that produced the event.\nNote that when a single transaction is described in multiple events, each event may populate different values of `event.outcome`, according to their perspective.\nAlso note that in the case of a compound event (a single event that contains multiple logical events), this field should be populated with the value that best captures the overall success or failure from the perspective of the event producer.\nFurther note that not all events will have an associated outcome. For example, this field is generally not populated for metric events, events with `event.type:info`, or any events for which an outcome does not make logical sense.',
                example: 'success',
                flat_name: 'event.outcome',
                ignore_above: 1024,
                level: 'core',
                name: 'outcome',
                normalize: [],
                short: 'The outcome of the event. The lowest level categorization field in the hierarchy.',
                type: 'keyword',
            },
            'event.provider': {
                dashed_name: 'event-provider',
                description: 'Source of the event.\nEvent transports such as Syslog or the Windows Event Log typically mention the source of an event. It can be the name of the software that generated the event (e.g. Sysmon, httpd), or of a subsystem of the operating system (kernel, Microsoft-Windows-Security-Auditing).',
                example: 'kernel',
                flat_name: 'event.provider',
                ignore_above: 1024,
                level: 'extended',
                name: 'provider',
                normalize: [],
                short: 'Source of the event.',
                type: 'keyword',
            },
            'event.reason': {
                dashed_name: 'event-reason',
                description: 'Reason why this event happened, according to the source.\nThis describes the why of a particular action or outcome captured in the event. Where `event.action` captures the action from the event, `event.reason` describes why that action was taken. For example, a web proxy with an `event.action` which denied the request may also populate `event.reason` with the reason why (e.g. `blocked site`).',
                example: 'Terminated an unexpected process',
                flat_name: 'event.reason',
                ignore_above: 1024,
                level: 'extended',
                name: 'reason',
                normalize: [],
                short: 'Reason why this event happened, according to the source',
                type: 'keyword',
            },
            'event.reference': {
                dashed_name: 'event-reference',
                description: 'Reference URL linking to additional information about this event.\nThis URL links to a static definition of this event. Alert events, indicated by `event.kind:alert`, are a common use case for this field.',
                example: 'https://system.example.com/event/#0001234',
                flat_name: 'event.reference',
                ignore_above: 1024,
                level: 'extended',
                name: 'reference',
                normalize: [],
                short: 'Event reference URL',
                type: 'keyword',
            },
            'event.risk_score': {
                dashed_name: 'event-risk-score',
                description: "Risk score or priority of the event (e.g. security solutions). Use your system's original value here.",
                flat_name: 'event.risk_score',
                level: 'core',
                name: 'risk_score',
                normalize: [],
                short: "Risk score or priority of the event (e.g. security solutions). Use your system's original value here.",
                type: 'float',
            },
            'event.risk_score_norm': {
                dashed_name: 'event-risk-score-norm',
                description: 'Normalized risk score or priority of the event, on a scale of 0 to 100.\nThis is mainly useful if you use more than one system that assigns risk scores, and you want to see a normalized value across all systems.',
                flat_name: 'event.risk_score_norm',
                level: 'extended',
                name: 'risk_score_norm',
                normalize: [],
                short: 'Normalized risk score or priority of the event (0-100).',
                type: 'float',
            },
            'event.sequence': {
                dashed_name: 'event-sequence',
                description: 'Sequence number of the event.\nThe sequence number is a value published by some event sources, to make the exact ordering of events unambiguous, regardless of the timestamp precision.',
                flat_name: 'event.sequence',
                format: 'string',
                level: 'extended',
                name: 'sequence',
                normalize: [],
                short: 'Sequence number of the event.',
                type: 'long',
            },
            'event.severity': {
                dashed_name: 'event-severity',
                description: "The numeric severity of the event according to your event source.\nWhat the different severity values mean can be different between sources and use cases. It's up to the implementer to make sure severities are consistent across events from the same source.\nThe Syslog severity belongs in `log.syslog.severity.code`. `event.severity` is meant to represent the severity according to the event source (e.g. firewall, IDS). If the event source does not publish its own severity, you may optionally copy the `log.syslog.severity.code` to `event.severity`.",
                example: 7,
                flat_name: 'event.severity',
                format: 'string',
                level: 'core',
                name: 'severity',
                normalize: [],
                short: 'Numeric severity of the event.',
                type: 'long',
            },
            'event.start': {
                dashed_name: 'event-start',
                description: '`event.start` contains the date when the event started or when the activity was first observed.',
                flat_name: 'event.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                short: '`event.start` contains the date when the event started or when the activity was first observed.',
                type: 'date',
            },
            'event.timezone': {
                dashed_name: 'event-timezone',
                description: 'This field should be populated when the event\'s timestamp does not include timezone information already (e.g. default Syslog timestamps). It\'s optional otherwise.\nAcceptable timezone formats are: a canonical ID (e.g. "Europe/Amsterdam"), abbreviated (e.g. "EST") or an HH:mm differential (e.g. "-05:00").',
                flat_name: 'event.timezone',
                ignore_above: 1024,
                level: 'extended',
                name: 'timezone',
                normalize: [],
                short: 'Event time zone.',
                type: 'keyword',
            },
            'event.type': {
                allowed_values: [
                    {
                        description: 'The access event type is used for the subset of events within a category that indicate that something was accessed. Common examples include `event.category:database AND event.type:access`, or `event.category:file AND event.type:access`. Note for file access, both directory listings and file opens should be included in this subcategory. You can further distinguish access operations using the ECS `event.action` field.',
                        name: 'access',
                    },
                    {
                        description: 'The admin event type is used for the subset of events within a category that are related to admin objects. For example, administrative changes within an IAM framework that do not specifically affect a user or group (e.g., adding new applications to a federation solution or connecting discrete forests in Active Directory) would fall into this subcategory. Common example: `event.category:iam AND event.type:change AND event.type:admin`. You can further distinguish admin operations using the ECS `event.action` field.',
                        name: 'admin',
                    },
                    {
                        description: 'The allowed event type is used for the subset of events within a category that indicate that something was allowed. Common examples include `event.category:network AND event.type:connection AND event.type:allowed` (to indicate a network firewall event for which the firewall disposition was to allow the connection to complete) and `event.category:intrusion_detection AND event.type:allowed` (to indicate a network intrusion prevention system event for which the IPS disposition was to allow the connection to complete). You can further distinguish allowed operations using the ECS `event.action` field, populating with values of your choosing, such as "allow", "detect", or "pass".',
                        name: 'allowed',
                    },
                    {
                        description: 'The change event type is used for the subset of events within a category that indicate that something has changed. If semantics best describe an event as modified, then include them in this subcategory. Common examples include `event.category:process AND event.type:change`, and `event.category:file AND event.type:change`. You can further distinguish change operations using the ECS `event.action` field.',
                        name: 'change',
                    },
                    {
                        description: 'Used primarily with `event.category:network` this value is used for the subset of network traffic that includes sufficient information for the event to be included in flow or connection analysis. Events in this subcategory will contain at least source and destination IP addresses, source and destination TCP/UDP ports, and will usually contain counts of bytes and/or packets transferred. Events in this subcategory may contain unidirectional or bidirectional information, including summary information. Use this subcategory to visualize and analyze network connections. Flow analysis, including Netflow, IPFIX, and other flow-related events fit in this subcategory. Note that firewall events from many Next-Generation Firewall (NGFW) devices will also fit into this subcategory.  A common filter for flow/connection information would be `event.category:network AND event.type:connection AND event.type:end` (to view or analyze all completed network connections, ignoring mid-flow reports). You can further distinguish connection events using the ECS `event.action` field, populating with values of your choosing, such as "timeout", or "reset".',
                        name: 'connection',
                    },
                    {
                        description: 'The "creation" event type is used for the subset of events within a category that indicate that something was created. A common example is `event.category:file AND event.type:creation`.',
                        name: 'creation',
                    },
                    {
                        description: 'The deletion event type is used for the subset of events within a category that indicate that something was deleted. A common example is `event.category:file AND event.type:deletion` to indicate that a file has been deleted.',
                        name: 'deletion',
                    },
                    {
                        description: 'The denied event type is used for the subset of events within a category that indicate that something was denied. Common examples include `event.category:network AND event.type:denied` (to indicate a network firewall event for which the firewall disposition was to deny the connection) and `event.category:intrusion_detection AND event.type:denied` (to indicate a network intrusion prevention system event for which the IPS disposition was to deny the connection to complete). You can further distinguish denied operations using the ECS `event.action` field, populating with values of your choosing, such as "blocked", "dropped", or "quarantined".',
                        name: 'denied',
                    },
                    {
                        description: 'The end event type is used for the subset of events within a category that indicate something has ended. A common example is `event.category:process AND event.type:end`.',
                        name: 'end',
                    },
                    {
                        description: 'The error event type is used for the subset of events within a category that indicate or describe an error. A common example is `event.category:database AND event.type:error`. Note that pipeline errors that occur during the event ingestion process should not use this `event.type` value. Instead, they should use `event.kind:pipeline_error`.',
                        name: 'error',
                    },
                    {
                        description: 'The group event type is used for the subset of events within a category that are related to group objects. Common example: `event.category:iam AND event.type:creation AND event.type:group`. You can further distinguish group operations using the ECS `event.action` field.',
                        name: 'group',
                    },
                    {
                        description: 'The indicator event type is used for the subset of events within a category that contain details about indicators of compromise (IOCs).\nA common example is `event.category:threat AND event.type:indicator`.',
                        name: 'indicator',
                    },
                    {
                        description: 'The info event type is used for the subset of events within a category that indicate that they are purely informational, and don\'t report a state change, or any type of action. For example, an initial run of a file integrity monitoring system (FIM), where an agent reports all files under management, would fall into the "info" subcategory. Similarly, an event containing a dump of all currently running processes (as opposed to reporting that a process started/ended) would fall into the "info" subcategory. An additional common examples is `event.category:intrusion_detection AND event.type:info`.',
                        name: 'info',
                    },
                    {
                        description: 'The installation event type is used for the subset of events within a category that indicate that something was installed. A common example is `event.category:package` AND `event.type:installation`.',
                        name: 'installation',
                    },
                    {
                        description: 'The protocol event type is used for the subset of events within a category that indicate that they contain protocol details or analysis, beyond simply identifying the protocol. Generally, network events that contain specific protocol details will fall into this subcategory. A common example is `event.category:network AND event.type:protocol AND event.type:connection AND event.type:end` (to indicate that the event is a network connection event sent at the end of a connection that also includes a protocol detail breakdown). Note that events that only indicate the name or id of the protocol should not use the protocol value. Further note that when the protocol subcategory is used, the identified protocol is populated in the ECS `network.protocol` field.',
                        name: 'protocol',
                    },
                    {
                        description: 'The start event type is used for the subset of events within a category that indicate something has started. A common example is `event.category:process AND event.type:start`.',
                        name: 'start',
                    },
                    {
                        description: 'The user event type is used for the subset of events within a category that are related to user objects. Common example: `event.category:iam AND event.type:deletion AND event.type:user`. You can further distinguish user operations using the ECS `event.action` field.',
                        name: 'user',
                    },
                ],
                dashed_name: 'event-type',
                description: 'This is one of four ECS Categorization Fields, and indicates the third level in the ECS category hierarchy.\n`event.type` represents a categorization "sub-bucket" that, when used along with the `event.category` field values, enables filtering events down to a level appropriate for single visualization.\nThis field is an array. This will allow proper categorization of some events that fall in multiple event types.',
                flat_name: 'event.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: ['array'],
                short: 'Event type. The third categorization field in the hierarchy.',
                type: 'keyword',
            },
            'event.url': {
                dashed_name: 'event-url',
                description: 'URL linking to an external system to continue investigation of this event.\nThis URL links to another system where in-depth investigation of the specific occurrence of this event can take place. Alert events, indicated by `event.kind:alert`, are a common use case for this field.',
                example: 'https://mysystem.example.com/alert/5271dedb-f5b0-4218-87f0-4ac4870a38fe',
                flat_name: 'event.url',
                ignore_above: 1024,
                level: 'extended',
                name: 'url',
                normalize: [],
                short: 'Event investigation URL',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'event',
        prefix: 'event.',
        short: 'Fields breaking down the event details.',
        title: 'Event',
        type: 'group',
    },
    faas: {
        beta: 'These fields are in beta and are subject to change.',
        description: 'The user fields describe information about the function as a service (FaaS) that is relevant to the event.',
        fields: {
            'faas.coldstart': {
                dashed_name: 'faas-coldstart',
                description: 'Boolean value indicating a cold start of a function.',
                flat_name: 'faas.coldstart',
                level: 'extended',
                name: 'coldstart',
                normalize: [],
                short: 'Boolean value indicating a cold start of a function.',
                type: 'boolean',
            },
            'faas.execution': {
                dashed_name: 'faas-execution',
                description: 'The execution ID of the current function execution.',
                example: 'af9d5aa4-a685-4c5f-a22b-444f80b3cc28',
                flat_name: 'faas.execution',
                ignore_above: 1024,
                level: 'extended',
                name: 'execution',
                normalize: [],
                short: 'The execution ID of the current function execution.',
                type: 'keyword',
            },
            'faas.id': {
                dashed_name: 'faas-id',
                description: "The unique identifier of a serverless function.\nFor AWS Lambda it's the function ARN (Amazon Resource Name) without a version or alias suffix.",
                example: 'arn:aws:lambda:us-west-2:123456789012:function:my-function',
                flat_name: 'faas.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                short: 'The unique identifier of a serverless function.',
                type: 'keyword',
            },
            'faas.name': {
                dashed_name: 'faas-name',
                description: 'The name of a serverless function.',
                example: 'my-function',
                flat_name: 'faas.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'The name of a serverless function.',
                type: 'keyword',
            },
            'faas.trigger.request_id': {
                dashed_name: 'faas-trigger-request-id',
                description: 'The ID of the trigger request , message, event, etc.',
                example: 123456789,
                flat_name: 'faas.trigger.request_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'trigger.request_id',
                normalize: [],
                short: 'The ID of the trigger request , message, event, etc.',
                type: 'keyword',
            },
            'faas.trigger.type': {
                dashed_name: 'faas-trigger-type',
                description: 'The trigger for the function execution.',
                example: 'http',
                expected_values: ['http', 'pubsub', 'datasource', 'timer', 'other'],
                flat_name: 'faas.trigger.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'trigger.type',
                normalize: [],
                short: 'The trigger for the function execution.',
                type: 'keyword',
            },
            'faas.version': {
                dashed_name: 'faas-version',
                description: 'The version of a serverless function.',
                example: '123',
                flat_name: 'faas.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'version',
                normalize: [],
                short: 'The version of a serverless function.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'faas',
        prefix: 'faas.',
        short: 'Fields describing functions as a service.',
        title: 'FaaS',
        type: 'group',
    },
    file: {
        description: 'A file is defined as a set of information that has been created on, or has existed on a filesystem.\nFile objects can be associated with host events, network events, and/or file events (e.g., those produced by File Integrity Monitoring [FIM] products or services). File fields provide details about the affected file associated with the event or metric.',
        fields: {
            'file.accessed': {
                dashed_name: 'file-accessed',
                description: 'Last time the file was accessed.\nNote that not all filesystems keep track of access time.',
                flat_name: 'file.accessed',
                level: 'extended',
                name: 'accessed',
                normalize: [],
                short: 'Last time the file was accessed.',
                type: 'date',
            },
            'file.attributes': {
                dashed_name: 'file-attributes',
                description: "Array of file attributes.\nAttributes names will vary by platform. Here's a non-exhaustive list of values that are expected in this field: archive, compressed, directory, encrypted, execute, hidden, read, readonly, system, write.",
                example: '["readonly", "system"]',
                flat_name: 'file.attributes',
                ignore_above: 1024,
                level: 'extended',
                name: 'attributes',
                normalize: ['array'],
                short: 'Array of file attributes.',
                type: 'keyword',
            },
            'file.code_signature.digest_algorithm': {
                dashed_name: 'file-code-signature-digest-algorithm',
                description: 'The hashing algorithm used to sign the process.\nThis value can distinguish signatures when a file is signed multiple times by the same signer but with a different digest algorithm.',
                example: 'sha256',
                flat_name: 'file.code_signature.digest_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'digest_algorithm',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Hashing algorithm used to sign the process.',
                type: 'keyword',
            },
            'file.code_signature.exists': {
                dashed_name: 'file-code-signature-exists',
                description: 'Boolean to capture if a signature is present.',
                example: 'true',
                flat_name: 'file.code_signature.exists',
                level: 'core',
                name: 'exists',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if a signature is present.',
                type: 'boolean',
            },
            'file.code_signature.signing_id': {
                dashed_name: 'file-code-signature-signing-id',
                description: 'The identifier used to sign the process.\nThis is used to identify the application manufactured by a software vendor. The field is relevant to Apple *OS only.',
                example: 'com.apple.xpc.proxy',
                flat_name: 'file.code_signature.signing_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'signing_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The identifier used to sign the process.',
                type: 'keyword',
            },
            'file.code_signature.status': {
                dashed_name: 'file-code-signature-status',
                description: 'Additional information about the certificate status.\nThis is useful for logging cryptographic errors with the certificate validity or trust status. Leave unpopulated if the validity or trust of the certificate was unchecked.',
                example: 'ERROR_UNTRUSTED_ROOT',
                flat_name: 'file.code_signature.status',
                ignore_above: 1024,
                level: 'extended',
                name: 'status',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Additional information about the certificate status.',
                type: 'keyword',
            },
            'file.code_signature.subject_name': {
                dashed_name: 'file-code-signature-subject-name',
                description: 'Subject name of the code signer',
                example: 'Microsoft Corporation',
                flat_name: 'file.code_signature.subject_name',
                ignore_above: 1024,
                level: 'core',
                name: 'subject_name',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Subject name of the code signer',
                type: 'keyword',
            },
            'file.code_signature.team_id': {
                dashed_name: 'file-code-signature-team-id',
                description: 'The team identifier used to sign the process.\nThis is used to identify the team or vendor of a software product. The field is relevant to Apple *OS only.',
                example: 'EQHXZ8M8AV',
                flat_name: 'file.code_signature.team_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'team_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The team identifier used to sign the process.',
                type: 'keyword',
            },
            'file.code_signature.timestamp': {
                dashed_name: 'file-code-signature-timestamp',
                description: 'Date and time when the code signature was generated and signed.',
                example: '2021-01-01T12:10:30Z',
                flat_name: 'file.code_signature.timestamp',
                level: 'extended',
                name: 'timestamp',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'When the signature was generated and signed.',
                type: 'date',
            },
            'file.code_signature.trusted': {
                dashed_name: 'file-code-signature-trusted',
                description: 'Stores the trust status of the certificate chain.\nValidating the trust of the certificate chain may be complicated, and this field should only be populated by tools that actively check the status.',
                example: 'true',
                flat_name: 'file.code_signature.trusted',
                level: 'extended',
                name: 'trusted',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Stores the trust status of the certificate chain.',
                type: 'boolean',
            },
            'file.code_signature.valid': {
                dashed_name: 'file-code-signature-valid',
                description: 'Boolean to capture if the digital signature is verified against the binary content.\nLeave unpopulated if a certificate was unchecked.',
                example: 'true',
                flat_name: 'file.code_signature.valid',
                level: 'extended',
                name: 'valid',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if the digital signature is verified against the binary content.',
                type: 'boolean',
            },
            'file.created': {
                dashed_name: 'file-created',
                description: 'File creation time.\nNote that not all filesystems store the creation time.',
                flat_name: 'file.created',
                level: 'extended',
                name: 'created',
                normalize: [],
                short: 'File creation time.',
                type: 'date',
            },
            'file.ctime': {
                dashed_name: 'file-ctime',
                description: 'Last time the file attributes or metadata changed.\nNote that changes to the file content will update `mtime`. This implies `ctime` will be adjusted at the same time, since `mtime` is an attribute of the file.',
                flat_name: 'file.ctime',
                level: 'extended',
                name: 'ctime',
                normalize: [],
                short: 'Last time the file attributes or metadata changed.',
                type: 'date',
            },
            'file.device': {
                dashed_name: 'file-device',
                description: 'Device that is the source of the file.',
                example: 'sda',
                flat_name: 'file.device',
                ignore_above: 1024,
                level: 'extended',
                name: 'device',
                normalize: [],
                short: 'Device that is the source of the file.',
                type: 'keyword',
            },
            'file.directory': {
                dashed_name: 'file-directory',
                description: 'Directory where the file is located. It should include the drive letter, when appropriate.',
                example: '/home/alice',
                flat_name: 'file.directory',
                ignore_above: 1024,
                level: 'extended',
                name: 'directory',
                normalize: [],
                short: 'Directory where the file is located.',
                type: 'keyword',
            },
            'file.drive_letter': {
                dashed_name: 'file-drive-letter',
                description: 'Drive letter where the file is located. This field is only relevant on Windows.\nThe value should be uppercase, and not include the colon.',
                example: 'C',
                flat_name: 'file.drive_letter',
                ignore_above: 1,
                level: 'extended',
                name: 'drive_letter',
                normalize: [],
                short: 'Drive letter where the file is located.',
                type: 'keyword',
            },
            'file.elf.architecture': {
                dashed_name: 'file-elf-architecture',
                description: 'Machine architecture of the ELF file.',
                example: 'x86-64',
                flat_name: 'file.elf.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Machine architecture of the ELF file.',
                type: 'keyword',
            },
            'file.elf.byte_order': {
                dashed_name: 'file-elf-byte-order',
                description: 'Byte sequence of ELF file.',
                example: 'Little Endian',
                flat_name: 'file.elf.byte_order',
                ignore_above: 1024,
                level: 'extended',
                name: 'byte_order',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Byte sequence of ELF file.',
                type: 'keyword',
            },
            'file.elf.cpu_type': {
                dashed_name: 'file-elf-cpu-type',
                description: 'CPU type of the ELF file.',
                example: 'Intel',
                flat_name: 'file.elf.cpu_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'cpu_type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'CPU type of the ELF file.',
                type: 'keyword',
            },
            'file.elf.creation_date': {
                dashed_name: 'file-elf-creation-date',
                description: "Extracted when possible from the file's metadata. Indicates when it was built or compiled. It can also be faked by malware creators.",
                flat_name: 'file.elf.creation_date',
                level: 'extended',
                name: 'creation_date',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Build or compile date.',
                type: 'date',
            },
            'file.elf.exports': {
                dashed_name: 'file-elf-exports',
                description: 'List of exported element names and types.',
                flat_name: 'file.elf.exports',
                level: 'extended',
                name: 'exports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of exported element names and types.',
                type: 'flattened',
            },
            'file.elf.go_import_hash': {
                dashed_name: 'file-elf-go-import-hash',
                description: 'A hash of the Go language imports in an ELF file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'file.elf.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the Go language imports in an ELF file.',
                type: 'keyword',
            },
            'file.elf.go_imports': {
                dashed_name: 'file-elf-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'file.elf.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'elf',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'file.elf.go_imports_names_entropy': {
                dashed_name: 'file-elf-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'file.elf.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'file.elf.go_imports_names_var_entropy': {
                dashed_name: 'file-elf-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'file.elf.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'file.elf.go_stripped': {
                dashed_name: 'file-elf-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'file.elf.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'file.elf.header.abi_version': {
                dashed_name: 'file-elf-header-abi-version',
                description: 'Version of the ELF Application Binary Interface (ABI).',
                flat_name: 'file.elf.header.abi_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.abi_version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF Application Binary Interface (ABI).',
                type: 'keyword',
            },
            'file.elf.header.class': {
                dashed_name: 'file-elf-header-class',
                description: 'Header class of the ELF file.',
                flat_name: 'file.elf.header.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.class',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header class of the ELF file.',
                type: 'keyword',
            },
            'file.elf.header.data': {
                dashed_name: 'file-elf-header-data',
                description: 'Data table of the ELF header.',
                flat_name: 'file.elf.header.data',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.data',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Data table of the ELF header.',
                type: 'keyword',
            },
            'file.elf.header.entrypoint': {
                dashed_name: 'file-elf-header-entrypoint',
                description: 'Header entrypoint of the ELF file.',
                flat_name: 'file.elf.header.entrypoint',
                format: 'string',
                level: 'extended',
                name: 'header.entrypoint',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header entrypoint of the ELF file.',
                type: 'long',
            },
            'file.elf.header.object_version': {
                dashed_name: 'file-elf-header-object-version',
                description: '"0x1" for original ELF files.',
                flat_name: 'file.elf.header.object_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.object_version',
                normalize: [],
                original_fieldset: 'elf',
                short: '"0x1" for original ELF files.',
                type: 'keyword',
            },
            'file.elf.header.os_abi': {
                dashed_name: 'file-elf-header-os-abi',
                description: 'Application Binary Interface (ABI) of the Linux OS.',
                flat_name: 'file.elf.header.os_abi',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.os_abi',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Application Binary Interface (ABI) of the Linux OS.',
                type: 'keyword',
            },
            'file.elf.header.type': {
                dashed_name: 'file-elf-header-type',
                description: 'Header type of the ELF file.',
                flat_name: 'file.elf.header.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header type of the ELF file.',
                type: 'keyword',
            },
            'file.elf.header.version': {
                dashed_name: 'file-elf-header-version',
                description: 'Version of the ELF header.',
                flat_name: 'file.elf.header.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF header.',
                type: 'keyword',
            },
            'file.elf.import_hash': {
                dashed_name: 'file-elf-import-hash',
                description: 'A hash of the imports in an ELF file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is an ELF implementation of the Windows PE imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'file.elf.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the imports in an ELF file.',
                type: 'keyword',
            },
            'file.elf.imports': {
                dashed_name: 'file-elf-imports',
                description: 'List of imported element names and types.',
                flat_name: 'file.elf.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'file.elf.imports_names_entropy': {
                dashed_name: 'file-elf-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'file.elf.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'file.elf.imports_names_var_entropy': {
                dashed_name: 'file-elf-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'file.elf.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'file.elf.sections': {
                dashed_name: 'file-elf-sections',
                description: 'An array containing an object for each section of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.sections.*`.',
                flat_name: 'file.elf.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'Section information of the ELF file.',
                type: 'nested',
            },
            'file.elf.sections.chi2': {
                dashed_name: 'file-elf-sections-chi2',
                description: 'Chi-square probability distribution of the section.',
                flat_name: 'file.elf.sections.chi2',
                format: 'number',
                level: 'extended',
                name: 'sections.chi2',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Chi-square probability distribution of the section.',
                type: 'long',
            },
            'file.elf.sections.entropy': {
                dashed_name: 'file-elf-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'file.elf.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'file.elf.sections.flags': {
                dashed_name: 'file-elf-sections-flags',
                description: 'ELF Section List flags.',
                flat_name: 'file.elf.sections.flags',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.flags',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List flags.',
                type: 'keyword',
            },
            'file.elf.sections.name': {
                dashed_name: 'file-elf-sections-name',
                description: 'ELF Section List name.',
                flat_name: 'file.elf.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List name.',
                type: 'keyword',
            },
            'file.elf.sections.physical_offset': {
                dashed_name: 'file-elf-sections-physical-offset',
                description: 'ELF Section List offset.',
                flat_name: 'file.elf.sections.physical_offset',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.physical_offset',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List offset.',
                type: 'keyword',
            },
            'file.elf.sections.physical_size': {
                dashed_name: 'file-elf-sections-physical-size',
                description: 'ELF Section List physical size.',
                flat_name: 'file.elf.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List physical size.',
                type: 'long',
            },
            'file.elf.sections.type': {
                dashed_name: 'file-elf-sections-type',
                description: 'ELF Section List type.',
                flat_name: 'file.elf.sections.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List type.',
                type: 'keyword',
            },
            'file.elf.sections.var_entropy': {
                dashed_name: 'file-elf-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'file.elf.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'file.elf.sections.virtual_address': {
                dashed_name: 'file-elf-sections-virtual-address',
                description: 'ELF Section List virtual address.',
                flat_name: 'file.elf.sections.virtual_address',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_address',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual address.',
                type: 'long',
            },
            'file.elf.sections.virtual_size': {
                dashed_name: 'file-elf-sections-virtual-size',
                description: 'ELF Section List virtual size.',
                flat_name: 'file.elf.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual size.',
                type: 'long',
            },
            'file.elf.segments': {
                dashed_name: 'file-elf-segments',
                description: 'An array containing an object for each segment of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.segments.*`.',
                flat_name: 'file.elf.segments',
                level: 'extended',
                name: 'segments',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'ELF object segment list.',
                type: 'nested',
            },
            'file.elf.segments.sections': {
                dashed_name: 'file-elf-segments-sections',
                description: 'ELF object segment sections.',
                flat_name: 'file.elf.segments.sections',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.sections',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment sections.',
                type: 'keyword',
            },
            'file.elf.segments.type': {
                dashed_name: 'file-elf-segments-type',
                description: 'ELF object segment type.',
                flat_name: 'file.elf.segments.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment type.',
                type: 'keyword',
            },
            'file.elf.shared_libraries': {
                dashed_name: 'file-elf-shared-libraries',
                description: 'List of shared libraries used by this ELF object.',
                flat_name: 'file.elf.shared_libraries',
                ignore_above: 1024,
                level: 'extended',
                name: 'shared_libraries',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of shared libraries used by this ELF object.',
                type: 'keyword',
            },
            'file.elf.telfhash': {
                dashed_name: 'file-elf-telfhash',
                description: 'telfhash symbol hash for ELF file.',
                flat_name: 'file.elf.telfhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'telfhash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'telfhash hash for ELF file.',
                type: 'keyword',
            },
            'file.extension': {
                dashed_name: 'file-extension',
                description: 'File extension, excluding the leading dot.\nNote that when the file name has multiple extensions (example.tar.gz), only the last one should be captured ("gz", not "tar.gz").',
                example: 'png',
                flat_name: 'file.extension',
                ignore_above: 1024,
                level: 'extended',
                name: 'extension',
                normalize: [],
                short: 'File extension, excluding the leading dot.',
                type: 'keyword',
            },
            'file.fork_name': {
                dashed_name: 'file-fork-name',
                description: 'A fork is additional data associated with a filesystem object.\nOn Linux, a resource fork is used to store additional data with a filesystem object. A file always has at least one fork for the data portion, and additional forks may exist.\nOn NTFS, this is analogous to an Alternate Data Stream (ADS), and the default data stream for a file is just called $DATA. Zone.Identifier is commonly used by Windows to track contents downloaded from the Internet. An ADS is typically of the form: `C:\\path\\to\\filename.extension:some_fork_name`, and `some_fork_name` is the value that should populate `fork_name`. `filename.extension` should populate `file.name`, and `extension` should populate `file.extension`. The full path, `file.path`, will include the fork name.',
                example: 'Zone.Identifer',
                flat_name: 'file.fork_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'fork_name',
                normalize: [],
                short: 'A fork is additional data associated with a filesystem object.',
                type: 'keyword',
            },
            'file.gid': {
                dashed_name: 'file-gid',
                description: 'Primary group ID (GID) of the file.',
                example: '1001',
                flat_name: 'file.gid',
                ignore_above: 1024,
                level: 'extended',
                name: 'gid',
                normalize: [],
                short: 'Primary group ID (GID) of the file.',
                type: 'keyword',
            },
            'file.group': {
                dashed_name: 'file-group',
                description: 'Primary group name of the file.',
                example: 'alice',
                flat_name: 'file.group',
                ignore_above: 1024,
                level: 'extended',
                name: 'group',
                normalize: [],
                short: 'Primary group name of the file.',
                type: 'keyword',
            },
            'file.hash.md5': {
                dashed_name: 'file-hash-md5',
                description: 'MD5 hash.',
                flat_name: 'file.hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                original_fieldset: 'hash',
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'file.hash.sha1': {
                dashed_name: 'file-hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'file.hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'file.hash.sha256': {
                dashed_name: 'file-hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'file.hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'file.hash.sha384': {
                dashed_name: 'file-hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'file.hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'file.hash.sha512': {
                dashed_name: 'file-hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'file.hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'file.hash.ssdeep': {
                dashed_name: 'file-hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'file.hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'file.hash.tlsh': {
                dashed_name: 'file-hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'file.hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                original_fieldset: 'hash',
                short: 'TLSH hash.',
                type: 'keyword',
            },
            'file.inode': {
                dashed_name: 'file-inode',
                description: 'Inode representing the file in the filesystem.',
                example: '256383',
                flat_name: 'file.inode',
                ignore_above: 1024,
                level: 'extended',
                name: 'inode',
                normalize: [],
                short: 'Inode representing the file in the filesystem.',
                type: 'keyword',
            },
            'file.macho.go_import_hash': {
                dashed_name: 'file-macho-go-import-hash',
                description: 'A hash of the Go language imports in a Mach-O file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'file.macho.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the Go language imports in a Mach-O file.',
                type: 'keyword',
            },
            'file.macho.go_imports': {
                dashed_name: 'file-macho-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'file.macho.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'macho',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'file.macho.go_imports_names_entropy': {
                dashed_name: 'file-macho-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'file.macho.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'file.macho.go_imports_names_var_entropy': {
                dashed_name: 'file-macho-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'file.macho.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'file.macho.go_stripped': {
                dashed_name: 'file-macho-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'file.macho.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'file.macho.import_hash': {
                dashed_name: 'file-macho-import-hash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for symhash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'file.macho.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
            'file.macho.imports': {
                dashed_name: 'file-macho-imports',
                description: 'List of imported element names and types.',
                flat_name: 'file.macho.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'macho',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'file.macho.imports_names_entropy': {
                dashed_name: 'file-macho-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'file.macho.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'file.macho.imports_names_var_entropy': {
                dashed_name: 'file-macho-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'file.macho.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'file.macho.sections': {
                dashed_name: 'file-macho-sections',
                description: 'An array containing an object for each section of the Mach-O file.\nThe keys that should be present in these objects are defined by sub-fields underneath `macho.sections.*`.',
                flat_name: 'file.macho.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'macho',
                short: 'Section information of the Mach-O file.',
                type: 'nested',
            },
            'file.macho.sections.entropy': {
                dashed_name: 'file-macho-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'file.macho.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'file.macho.sections.name': {
                dashed_name: 'file-macho-sections-name',
                description: 'Mach-O Section List name.',
                flat_name: 'file.macho.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List name.',
                type: 'keyword',
            },
            'file.macho.sections.physical_size': {
                dashed_name: 'file-macho-sections-physical-size',
                description: 'Mach-O Section List physical size.',
                flat_name: 'file.macho.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List physical size.',
                type: 'long',
            },
            'file.macho.sections.var_entropy': {
                dashed_name: 'file-macho-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'file.macho.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'file.macho.sections.virtual_size': {
                dashed_name: 'file-macho-sections-virtual-size',
                description: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'file.macho.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'file.macho.symhash': {
                dashed_name: 'file-macho-symhash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a Mach-O implementation of the Windows PE imphash',
                example: 'd3ccf195b62a9279c3c19af1080497ec',
                flat_name: 'file.macho.symhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'symhash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
            'file.mime_type': {
                dashed_name: 'file-mime-type',
                description: 'MIME type should identify the format of the file or stream of bytes using https://www.iana.org/assignments/media-types/media-types.xhtml[IANA official types], where possible. When more than one type is applicable, the most specific type should be used.',
                flat_name: 'file.mime_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'mime_type',
                normalize: [],
                short: 'Media type of file, document, or arrangement of bytes.',
                type: 'keyword',
            },
            'file.mode': {
                dashed_name: 'file-mode',
                description: 'Mode of the file in octal representation.',
                example: '0640',
                flat_name: 'file.mode',
                ignore_above: 1024,
                level: 'extended',
                name: 'mode',
                normalize: [],
                short: 'Mode of the file in octal representation.',
                type: 'keyword',
            },
            'file.mtime': {
                dashed_name: 'file-mtime',
                description: 'Last time the file content was modified.',
                flat_name: 'file.mtime',
                level: 'extended',
                name: 'mtime',
                normalize: [],
                short: 'Last time the file content was modified.',
                type: 'date',
            },
            'file.name': {
                dashed_name: 'file-name',
                description: 'Name of the file including the extension, without the directory.',
                example: 'example.png',
                flat_name: 'file.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Name of the file including the extension, without the directory.',
                type: 'keyword',
            },
            'file.owner': {
                dashed_name: 'file-owner',
                description: "File owner's username.",
                example: 'alice',
                flat_name: 'file.owner',
                ignore_above: 1024,
                level: 'extended',
                name: 'owner',
                normalize: [],
                short: "File owner's username.",
                type: 'keyword',
            },
            'file.path': {
                dashed_name: 'file-path',
                description: 'Full path to the file, including the file name. It should include the drive letter, when appropriate.',
                example: '/home/alice/example.png',
                flat_name: 'file.path',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'file.path.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'path',
                normalize: [],
                short: 'Full path to the file, including the file name.',
                type: 'keyword',
            },
            'file.pe.architecture': {
                dashed_name: 'file-pe-architecture',
                description: 'CPU architecture target for the file.',
                example: 'x64',
                flat_name: 'file.pe.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'pe',
                short: 'CPU architecture target for the file.',
                type: 'keyword',
            },
            'file.pe.company': {
                dashed_name: 'file-pe-company',
                description: 'Internal company name of the file, provided at compile-time.',
                example: 'Microsoft Corporation',
                flat_name: 'file.pe.company',
                ignore_above: 1024,
                level: 'extended',
                name: 'company',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal company name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'file.pe.description': {
                dashed_name: 'file-pe-description',
                description: 'Internal description of the file, provided at compile-time.',
                example: 'Paint',
                flat_name: 'file.pe.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal description of the file, provided at compile-time.',
                type: 'keyword',
            },
            'file.pe.file_version': {
                dashed_name: 'file-pe-file-version',
                description: 'Internal version of the file, provided at compile-time.',
                example: '6.3.9600.17415',
                flat_name: 'file.pe.file_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'file_version',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Process name.',
                type: 'keyword',
            },
            'file.pe.go_import_hash': {
                dashed_name: 'file-pe-go-import-hash',
                description: 'A hash of the Go language imports in a PE file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'file.pe.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the Go language imports in a PE file.',
                type: 'keyword',
            },
            'file.pe.go_imports': {
                dashed_name: 'file-pe-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'file.pe.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'pe',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'file.pe.go_imports_names_entropy': {
                dashed_name: 'file-pe-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'file.pe.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'file.pe.go_imports_names_var_entropy': {
                dashed_name: 'file-pe-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'file.pe.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'file.pe.go_stripped': {
                dashed_name: 'file-pe-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'file.pe.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'file.pe.imphash': {
                dashed_name: 'file-pe-imphash',
                description: 'A hash of the imports in a PE file. An imphash -- or import hash -- can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nLearn more at https://www.fireeye.com/blog/threat-research/2014/01/tracking-malware-import-hashing.html.',
                example: '0c6803c4e922103c4dca5963aad36ddf',
                flat_name: 'file.pe.imphash',
                ignore_above: 1024,
                level: 'extended',
                name: 'imphash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'file.pe.import_hash': {
                dashed_name: 'file-pe-import-hash',
                description: 'A hash of the imports in a PE file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'file.pe.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'file.pe.imports': {
                dashed_name: 'file-pe-imports',
                description: 'List of imported element names and types.',
                flat_name: 'file.pe.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'file.pe.imports_names_entropy': {
                dashed_name: 'file-pe-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'file.pe.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'file.pe.imports_names_var_entropy': {
                dashed_name: 'file-pe-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'file.pe.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'file.pe.original_file_name': {
                dashed_name: 'file-pe-original-file-name',
                description: 'Internal name of the file, provided at compile-time.',
                example: 'MSPAINT.EXE',
                flat_name: 'file.pe.original_file_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'original_file_name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'file.pe.pehash': {
                dashed_name: 'file-pe-pehash',
                description: 'A hash of the PE header and data from one or more PE sections. An pehash can be used to cluster files by transforming structural information about a file into a hash value.\nLearn more at https://www.usenix.org/legacy/events/leet09/tech/full_papers/wicherski/wicherski_html/index.html.',
                example: '73ff189b63cd6be375a7ff25179a38d347651975',
                flat_name: 'file.pe.pehash',
                ignore_above: 1024,
                level: 'extended',
                name: 'pehash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the PE header and data from one or more PE sections.',
                type: 'keyword',
            },
            'file.pe.product': {
                dashed_name: 'file-pe-product',
                description: 'Internal product name of the file, provided at compile-time.',
                example: 'Microsoft® Windows® Operating System',
                flat_name: 'file.pe.product',
                ignore_above: 1024,
                level: 'extended',
                name: 'product',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal product name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'file.pe.sections': {
                dashed_name: 'file-pe-sections',
                description: 'An array containing an object for each section of the PE file.\nThe keys that should be present in these objects are defined by sub-fields underneath `pe.sections.*`.',
                flat_name: 'file.pe.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'Section information of the PE file.',
                type: 'nested',
            },
            'file.pe.sections.entropy': {
                dashed_name: 'file-pe-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'file.pe.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'file.pe.sections.name': {
                dashed_name: 'file-pe-sections-name',
                description: 'PE Section List name.',
                flat_name: 'file.pe.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List name.',
                type: 'keyword',
            },
            'file.pe.sections.physical_size': {
                dashed_name: 'file-pe-sections-physical-size',
                description: 'PE Section List physical size.',
                flat_name: 'file.pe.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List physical size.',
                type: 'long',
            },
            'file.pe.sections.var_entropy': {
                dashed_name: 'file-pe-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'file.pe.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'file.pe.sections.virtual_size': {
                dashed_name: 'file-pe-sections-virtual-size',
                description: 'PE Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'file.pe.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'file.size': {
                dashed_name: 'file-size',
                description: 'File size in bytes.\nOnly relevant when `file.type` is "file".',
                example: 16384,
                flat_name: 'file.size',
                level: 'extended',
                name: 'size',
                normalize: [],
                short: 'File size in bytes.',
                type: 'long',
            },
            'file.target_path': {
                dashed_name: 'file-target-path',
                description: 'Target path for symlinks.',
                flat_name: 'file.target_path',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'file.target_path.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'target_path',
                normalize: [],
                short: 'Target path for symlinks.',
                type: 'keyword',
            },
            'file.type': {
                dashed_name: 'file-type',
                description: 'File type (file, dir, or symlink).',
                example: 'file',
                flat_name: 'file.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                short: 'File type (file, dir, or symlink).',
                type: 'keyword',
            },
            'file.uid': {
                dashed_name: 'file-uid',
                description: 'The user ID (UID) or security identifier (SID) of the file owner.',
                example: '1001',
                flat_name: 'file.uid',
                ignore_above: 1024,
                level: 'extended',
                name: 'uid',
                normalize: [],
                short: 'The user ID (UID) or security identifier (SID) of the file owner.',
                type: 'keyword',
            },
            'file.x509.alternative_names': {
                dashed_name: 'file-x509-alternative-names',
                description: 'List of subject alternative names (SAN). Name types vary by certificate authority and certificate type but commonly contain IP addresses, DNS names (and wildcards), and email addresses.',
                example: '*.elastic.co',
                flat_name: 'file.x509.alternative_names',
                ignore_above: 1024,
                level: 'extended',
                name: 'alternative_names',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of subject alternative names (SAN).',
                type: 'keyword',
            },
            'file.x509.issuer.common_name': {
                dashed_name: 'file-x509-issuer-common-name',
                description: 'List of common name (CN) of issuing certificate authority.',
                example: 'Example SHA2 High Assurance Server CA',
                flat_name: 'file.x509.issuer.common_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.common_name',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of common name (CN) of issuing certificate authority.',
                type: 'keyword',
            },
            'file.x509.issuer.country': {
                dashed_name: 'file-x509-issuer-country',
                description: 'List of country \\(C) codes',
                example: 'US',
                flat_name: 'file.x509.issuer.country',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.country',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of country \\(C) codes',
                type: 'keyword',
            },
            'file.x509.issuer.distinguished_name': {
                dashed_name: 'file-x509-issuer-distinguished-name',
                description: 'Distinguished name (DN) of issuing certificate authority.',
                example: 'C=US, O=Example Inc, OU=www.example.com, CN=Example SHA2 High Assurance Server CA',
                flat_name: 'file.x509.issuer.distinguished_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.distinguished_name',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Distinguished name (DN) of issuing certificate authority.',
                type: 'keyword',
            },
            'file.x509.issuer.locality': {
                dashed_name: 'file-x509-issuer-locality',
                description: 'List of locality names (L)',
                example: 'Mountain View',
                flat_name: 'file.x509.issuer.locality',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.locality',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of locality names (L)',
                type: 'keyword',
            },
            'file.x509.issuer.organization': {
                dashed_name: 'file-x509-issuer-organization',
                description: 'List of organizations (O) of issuing certificate authority.',
                example: 'Example Inc',
                flat_name: 'file.x509.issuer.organization',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.organization',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizations (O) of issuing certificate authority.',
                type: 'keyword',
            },
            'file.x509.issuer.organizational_unit': {
                dashed_name: 'file-x509-issuer-organizational-unit',
                description: 'List of organizational units (OU) of issuing certificate authority.',
                example: 'www.example.com',
                flat_name: 'file.x509.issuer.organizational_unit',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.organizational_unit',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizational units (OU) of issuing certificate authority.',
                type: 'keyword',
            },
            'file.x509.issuer.state_or_province': {
                dashed_name: 'file-x509-issuer-state-or-province',
                description: 'List of state or province names (ST, S, or P)',
                example: 'California',
                flat_name: 'file.x509.issuer.state_or_province',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.state_or_province',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of state or province names (ST, S, or P)',
                type: 'keyword',
            },
            'file.x509.not_after': {
                dashed_name: 'file-x509-not-after',
                description: 'Time at which the certificate is no longer considered valid.',
                example: '2020-07-16T03:15:39Z',
                flat_name: 'file.x509.not_after',
                level: 'extended',
                name: 'not_after',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Time at which the certificate is no longer considered valid.',
                type: 'date',
            },
            'file.x509.not_before': {
                dashed_name: 'file-x509-not-before',
                description: 'Time at which the certificate is first considered valid.',
                example: '2019-08-16T01:40:25Z',
                flat_name: 'file.x509.not_before',
                level: 'extended',
                name: 'not_before',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Time at which the certificate is first considered valid.',
                type: 'date',
            },
            'file.x509.public_key_algorithm': {
                dashed_name: 'file-x509-public-key-algorithm',
                description: 'Algorithm used to generate the public key.',
                example: 'RSA',
                flat_name: 'file.x509.public_key_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'public_key_algorithm',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Algorithm used to generate the public key.',
                type: 'keyword',
            },
            'file.x509.public_key_curve': {
                dashed_name: 'file-x509-public-key-curve',
                description: 'The curve used by the elliptic curve public key algorithm. This is algorithm specific.',
                example: 'nistp521',
                flat_name: 'file.x509.public_key_curve',
                ignore_above: 1024,
                level: 'extended',
                name: 'public_key_curve',
                normalize: [],
                original_fieldset: 'x509',
                short: 'The curve used by the elliptic curve public key algorithm. This is algorithm specific.',
                type: 'keyword',
            },
            'file.x509.public_key_exponent': {
                dashed_name: 'file-x509-public-key-exponent',
                description: 'Exponent used to derive the public key. This is algorithm specific.',
                doc_values: false,
                example: 65537,
                flat_name: 'file.x509.public_key_exponent',
                index: false,
                level: 'extended',
                name: 'public_key_exponent',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Exponent used to derive the public key. This is algorithm specific.',
                type: 'long',
            },
            'file.x509.public_key_size': {
                dashed_name: 'file-x509-public-key-size',
                description: 'The size of the public key space in bits.',
                example: 2048,
                flat_name: 'file.x509.public_key_size',
                level: 'extended',
                name: 'public_key_size',
                normalize: [],
                original_fieldset: 'x509',
                short: 'The size of the public key space in bits.',
                type: 'long',
            },
            'file.x509.serial_number': {
                dashed_name: 'file-x509-serial-number',
                description: 'Unique serial number issued by the certificate authority. For consistency, if this value is alphanumeric, it should be formatted without colons and uppercase characters.',
                example: '55FBB9C7DEBF09809D12CCAA',
                flat_name: 'file.x509.serial_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'serial_number',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Unique serial number issued by the certificate authority.',
                type: 'keyword',
            },
            'file.x509.signature_algorithm': {
                dashed_name: 'file-x509-signature-algorithm',
                description: 'Identifier for certificate signature algorithm. We recommend using names found in Go Lang Crypto library. See https://github.com/golang/go/blob/go1.14/src/crypto/x509/x509.go#L337-L353.',
                example: 'SHA256-RSA',
                flat_name: 'file.x509.signature_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'signature_algorithm',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Identifier for certificate signature algorithm.',
                type: 'keyword',
            },
            'file.x509.subject.common_name': {
                dashed_name: 'file-x509-subject-common-name',
                description: 'List of common names (CN) of subject.',
                example: 'shared.global.example.net',
                flat_name: 'file.x509.subject.common_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.common_name',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of common names (CN) of subject.',
                type: 'keyword',
            },
            'file.x509.subject.country': {
                dashed_name: 'file-x509-subject-country',
                description: 'List of country \\(C) code',
                example: 'US',
                flat_name: 'file.x509.subject.country',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.country',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of country \\(C) code',
                type: 'keyword',
            },
            'file.x509.subject.distinguished_name': {
                dashed_name: 'file-x509-subject-distinguished-name',
                description: 'Distinguished name (DN) of the certificate subject entity.',
                example: 'C=US, ST=California, L=San Francisco, O=Example, Inc., CN=shared.global.example.net',
                flat_name: 'file.x509.subject.distinguished_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.distinguished_name',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Distinguished name (DN) of the certificate subject entity.',
                type: 'keyword',
            },
            'file.x509.subject.locality': {
                dashed_name: 'file-x509-subject-locality',
                description: 'List of locality names (L)',
                example: 'San Francisco',
                flat_name: 'file.x509.subject.locality',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.locality',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of locality names (L)',
                type: 'keyword',
            },
            'file.x509.subject.organization': {
                dashed_name: 'file-x509-subject-organization',
                description: 'List of organizations (O) of subject.',
                example: 'Example, Inc.',
                flat_name: 'file.x509.subject.organization',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.organization',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizations (O) of subject.',
                type: 'keyword',
            },
            'file.x509.subject.organizational_unit': {
                dashed_name: 'file-x509-subject-organizational-unit',
                description: 'List of organizational units (OU) of subject.',
                flat_name: 'file.x509.subject.organizational_unit',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.organizational_unit',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizational units (OU) of subject.',
                type: 'keyword',
            },
            'file.x509.subject.state_or_province': {
                dashed_name: 'file-x509-subject-state-or-province',
                description: 'List of state or province names (ST, S, or P)',
                example: 'California',
                flat_name: 'file.x509.subject.state_or_province',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.state_or_province',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of state or province names (ST, S, or P)',
                type: 'keyword',
            },
            'file.x509.version_number': {
                dashed_name: 'file-x509-version-number',
                description: 'Version of x509 format.',
                example: 3,
                flat_name: 'file.x509.version_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'version_number',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Version of x509 format.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'file',
        nestings: [
            'file.code_signature',
            'file.elf',
            'file.hash',
            'file.macho',
            'file.pe',
            'file.x509',
        ],
        prefix: 'file.',
        reusable: {
            expected: [
                { as: 'file', at: 'threat.indicator', full: 'threat.indicator.file' },
                {
                    as: 'file',
                    at: 'threat.enrichments.indicator',
                    full: 'threat.enrichments.indicator.file',
                },
            ],
            top_level: true,
        },
        reused_here: [
            {
                full: 'file.hash',
                schema_name: 'hash',
                short: 'Hashes, usually file hashes.',
            },
            {
                full: 'file.pe',
                schema_name: 'pe',
                short: 'These fields contain Windows Portable Executable (PE) metadata.',
            },
            {
                full: 'file.x509',
                schema_name: 'x509',
                short: 'These fields contain x509 certificate metadata.',
            },
            {
                full: 'file.code_signature',
                schema_name: 'code_signature',
                short: 'These fields contain information about binary code signatures.',
            },
            {
                beta: 'This field reuse is beta and subject to change.',
                full: 'file.elf',
                schema_name: 'elf',
                short: 'These fields contain Linux Executable Linkable Format (ELF) metadata.',
            },
            {
                beta: 'This field reuse is beta and subject to change.',
                full: 'file.macho',
                schema_name: 'macho',
                short: 'These fields contain Mac OS Mach Object file format (Mach-O) metadata.',
            },
        ],
        short: 'Fields describing files.',
        title: 'File',
        type: 'group',
    },
    geo: {
        description: 'Geo fields can carry data about a specific location related to an event.\nThis geolocation information can be derived from techniques such as Geo IP, or be user-supplied.',
        fields: {
            'geo.city_name': {
                dashed_name: 'geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                short: 'City name.',
                type: 'keyword',
            },
            'geo.continent_code': {
                dashed_name: 'geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                short: 'Continent code.',
                type: 'keyword',
            },
            'geo.continent_name': {
                dashed_name: 'geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'geo.country_iso_code': {
                dashed_name: 'geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'geo.country_name': {
                dashed_name: 'geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                short: 'Country name.',
                type: 'keyword',
            },
            'geo.location': {
                dashed_name: 'geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'geo.name': {
                dashed_name: 'geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'geo.postal_code': {
                dashed_name: 'geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                short: 'Postal code.',
                type: 'keyword',
            },
            'geo.region_iso_code': {
                dashed_name: 'geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'geo.region_name': {
                dashed_name: 'geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                short: 'Region name.',
                type: 'keyword',
            },
            'geo.timezone': {
                dashed_name: 'geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                short: 'Time zone.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'geo',
        prefix: 'geo.',
        reusable: {
            expected: [
                { as: 'geo', at: 'client', full: 'client.geo' },
                { as: 'geo', at: 'destination', full: 'destination.geo' },
                { as: 'geo', at: 'observer', full: 'observer.geo' },
                { as: 'geo', at: 'host', full: 'host.geo' },
                { as: 'geo', at: 'server', full: 'server.geo' },
                { as: 'geo', at: 'source', full: 'source.geo' },
                { as: 'geo', at: 'threat.indicator', full: 'threat.indicator.geo' },
                {
                    as: 'geo',
                    at: 'threat.enrichments.indicator',
                    full: 'threat.enrichments.indicator.geo',
                },
            ],
            top_level: false,
        },
        short: 'Fields describing a location.',
        title: 'Geo',
        type: 'group',
    },
    group: {
        description: 'The group fields are meant to represent groups that are relevant to the event.',
        fields: {
            'group.domain': {
                dashed_name: 'group-domain',
                description: 'Name of the directory the group is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'group.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                short: 'Name of the directory the group is a member of.',
                type: 'keyword',
            },
            'group.id': {
                dashed_name: 'group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'group.name': {
                dashed_name: 'group-name',
                description: 'Name of the group.',
                flat_name: 'group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Name of the group.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'group',
        prefix: 'group.',
        reusable: {
            expected: [
                { as: 'group', at: 'user', full: 'user.group' },
                {
                    as: 'group',
                    at: 'process',
                    full: 'process.group',
                    short_override: 'The effective group (egid).',
                },
                {
                    as: 'real_group',
                    at: 'process',
                    full: 'process.real_group',
                    short_override: 'The real group (rgid).',
                },
                {
                    as: 'saved_group',
                    at: 'process',
                    full: 'process.saved_group',
                    short_override: 'The saved group (sgid).',
                },
                {
                    as: 'supplemental_groups',
                    at: 'process',
                    full: 'process.supplemental_groups',
                    normalize: ['array'],
                    short_override: 'An array of supplemental groups.',
                },
                {
                    as: 'attested_groups',
                    at: 'process',
                    beta: 'Reusing the `group` fields in this location is currently considered beta.',
                    full: 'process.attested_groups',
                    normalize: ['array'],
                    short_override: 'The externally attested groups based on an external source such as the Kube API.',
                },
            ],
            top_level: true,
        },
        short: "User's group relevant to the event.",
        title: 'Group',
        type: 'group',
    },
    hash: {
        description: 'The hash fields represent different bitwise hash algorithms and their values.\nField names for common hashes (e.g. MD5, SHA1) are predefined. Add fields for other hashes by lowercasing the hash algorithm name and using underscore separators as appropriate (snake case, e.g. sha3_512).\nNote that this fieldset is used for common hashes that may be computed over a range of generic bytes. Entity-specific hashes such as ja3 or imphash are placed in the fieldsets to which they relate (tls and pe, respectively).',
        fields: {
            'hash.md5': {
                dashed_name: 'hash-md5',
                description: 'MD5 hash.',
                flat_name: 'hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'hash.sha1': {
                dashed_name: 'hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'hash.sha256': {
                dashed_name: 'hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'hash.sha384': {
                dashed_name: 'hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'hash.sha512': {
                dashed_name: 'hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'hash.ssdeep': {
                dashed_name: 'hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'hash.tlsh': {
                dashed_name: 'hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                short: 'TLSH hash.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'hash',
        prefix: 'hash.',
        reusable: {
            expected: [
                { as: 'hash', at: 'file', full: 'file.hash' },
                { as: 'hash', at: 'process', full: 'process.hash' },
                { as: 'hash', at: 'dll', full: 'dll.hash' },
                {
                    as: 'hash',
                    at: 'email.attachments.file',
                    full: 'email.attachments.file.hash',
                },
            ],
            top_level: false,
        },
        short: 'Hashes, usually file hashes.',
        title: 'Hash',
        type: 'group',
    },
    host: {
        description: 'A host is defined as a general computing instance.\nECS host.* fields should be populated with details about the host on which the event happened, or from which the measurement was taken. Host types include hardware, virtual machines, Docker containers, and Kubernetes nodes.',
        fields: {
            'host.architecture': {
                dashed_name: 'host-architecture',
                description: 'Operating system architecture.',
                example: 'x86_64',
                flat_name: 'host.architecture',
                ignore_above: 1024,
                level: 'core',
                name: 'architecture',
                normalize: [],
                short: 'Operating system architecture.',
                type: 'keyword',
            },
            'host.boot.id': {
                beta: 'This field is beta and subject to change.',
                dashed_name: 'host-boot-id',
                description: 'Linux boot uuid taken from /proc/sys/kernel/random/boot_id. Note the boot_id value from /proc may or may not be the same in containers as on the host. Some container runtimes will bind mount a new boot_id value onto the proc file in each container.',
                example: '88a1f0ed-5ae5-41ee-af6b-41921c311872',
                flat_name: 'host.boot.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'boot.id',
                normalize: [],
                short: 'Linux boot uuid taken from /proc/sys/kernel/random/boot_id',
                type: 'keyword',
            },
            'host.cpu.usage': {
                dashed_name: 'host-cpu-usage',
                description: 'Percent CPU used which is normalized by the number of CPU cores and it ranges from 0 to 1.\nScaling factor: 1000.\nFor example: For a two core host, this value should be the average of the two cores, between 0 and 1.',
                flat_name: 'host.cpu.usage',
                level: 'extended',
                name: 'cpu.usage',
                normalize: [],
                scaling_factor: 1000,
                short: 'Percent CPU used, between 0 and 1.',
                type: 'scaled_float',
            },
            'host.disk.read.bytes': {
                dashed_name: 'host-disk-read-bytes',
                description: 'The total number of bytes (gauge) read successfully (aggregated from all disks) since the last metric collection.',
                flat_name: 'host.disk.read.bytes',
                level: 'extended',
                name: 'disk.read.bytes',
                normalize: [],
                short: 'The number of bytes read by all disks.',
                type: 'long',
            },
            'host.disk.write.bytes': {
                dashed_name: 'host-disk-write-bytes',
                description: 'The total number of bytes (gauge) written successfully (aggregated from all disks) since the last metric collection.',
                flat_name: 'host.disk.write.bytes',
                level: 'extended',
                name: 'disk.write.bytes',
                normalize: [],
                short: 'The number of bytes written on all disks.',
                type: 'long',
            },
            'host.domain': {
                dashed_name: 'host-domain',
                description: "Name of the domain of which the host is a member.\nFor example, on Windows this could be the host's Active Directory domain or NetBIOS domain name. For Linux this could be the domain of the host's LDAP provider.",
                example: 'CONTOSO',
                flat_name: 'host.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                short: 'Name of the directory the group is a member of.',
                type: 'keyword',
            },
            'host.geo.city_name': {
                dashed_name: 'host-geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'host.geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'City name.',
                type: 'keyword',
            },
            'host.geo.continent_code': {
                dashed_name: 'host-geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'host.geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Continent code.',
                type: 'keyword',
            },
            'host.geo.continent_name': {
                dashed_name: 'host-geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'host.geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'host.geo.country_iso_code': {
                dashed_name: 'host-geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'host.geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'host.geo.country_name': {
                dashed_name: 'host-geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'host.geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country name.',
                type: 'keyword',
            },
            'host.geo.location': {
                dashed_name: 'host-geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'host.geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'host.geo.name': {
                dashed_name: 'host-geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'host.geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'host.geo.postal_code': {
                dashed_name: 'host-geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'host.geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Postal code.',
                type: 'keyword',
            },
            'host.geo.region_iso_code': {
                dashed_name: 'host-geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'host.geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'host.geo.region_name': {
                dashed_name: 'host-geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'host.geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region name.',
                type: 'keyword',
            },
            'host.geo.timezone': {
                dashed_name: 'host-geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'host.geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Time zone.',
                type: 'keyword',
            },
            'host.hostname': {
                dashed_name: 'host-hostname',
                description: 'Hostname of the host.\nIt normally contains what the `hostname` command returns on the host machine.',
                flat_name: 'host.hostname',
                ignore_above: 1024,
                level: 'core',
                name: 'hostname',
                normalize: [],
                short: 'Hostname of the host.',
                type: 'keyword',
            },
            'host.id': {
                dashed_name: 'host-id',
                description: 'Unique host id.\nAs hostname is not always unique, use values that are meaningful in your environment.\nExample: The current usage of `beat.name`.',
                flat_name: 'host.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                short: 'Unique host id.',
                type: 'keyword',
            },
            'host.ip': {
                dashed_name: 'host-ip',
                description: 'Host ip addresses.',
                flat_name: 'host.ip',
                level: 'core',
                name: 'ip',
                normalize: ['array'],
                short: 'Host ip addresses.',
                type: 'ip',
            },
            'host.mac': {
                dashed_name: 'host-mac',
                description: 'Host MAC addresses.\nThe notation format from RFC 7042 is suggested: Each octet (that is, 8-bit byte) is represented by two [uppercase] hexadecimal digits giving the value of the octet as an unsigned integer. Successive octets are separated by a hyphen.',
                example: '["00-00-5E-00-53-23", "00-00-5E-00-53-24"]',
                flat_name: 'host.mac',
                ignore_above: 1024,
                level: 'core',
                name: 'mac',
                normalize: ['array'],
                pattern: '^[A-F0-9]{2}(-[A-F0-9]{2}){5,}$',
                short: 'Host MAC addresses.',
                type: 'keyword',
            },
            'host.name': {
                dashed_name: 'host-name',
                description: 'Name of the host.\nIt can contain what hostname returns on Unix systems, the fully qualified domain name (FQDN), or a name specified by the user. The recommended value is the lowercase FQDN of the host.',
                flat_name: 'host.name',
                ignore_above: 1024,
                level: 'core',
                name: 'name',
                normalize: [],
                short: 'Name of the host.',
                type: 'keyword',
            },
            'host.network.egress.bytes': {
                dashed_name: 'host-network-egress-bytes',
                description: 'The number of bytes (gauge) sent out on all network interfaces by the host since the last metric collection.',
                flat_name: 'host.network.egress.bytes',
                level: 'extended',
                name: 'network.egress.bytes',
                normalize: [],
                short: 'The number of bytes sent on all network interfaces.',
                type: 'long',
            },
            'host.network.egress.packets': {
                dashed_name: 'host-network-egress-packets',
                description: 'The number of packets (gauge) sent out on all network interfaces by the host since the last metric collection.',
                flat_name: 'host.network.egress.packets',
                level: 'extended',
                name: 'network.egress.packets',
                normalize: [],
                short: 'The number of packets sent on all network interfaces.',
                type: 'long',
            },
            'host.network.ingress.bytes': {
                dashed_name: 'host-network-ingress-bytes',
                description: 'The number of bytes received (gauge) on all network interfaces by the host since the last metric collection.',
                flat_name: 'host.network.ingress.bytes',
                level: 'extended',
                name: 'network.ingress.bytes',
                normalize: [],
                short: 'The number of bytes received on all network interfaces.',
                type: 'long',
            },
            'host.network.ingress.packets': {
                dashed_name: 'host-network-ingress-packets',
                description: 'The number of packets (gauge) received on all network interfaces by the host since the last metric collection.',
                flat_name: 'host.network.ingress.packets',
                level: 'extended',
                name: 'network.ingress.packets',
                normalize: [],
                short: 'The number of packets received on all network interfaces.',
                type: 'long',
            },
            'host.os.family': {
                dashed_name: 'host-os-family',
                description: 'OS family (such as redhat, debian, freebsd, windows).',
                example: 'debian',
                flat_name: 'host.os.family',
                ignore_above: 1024,
                level: 'extended',
                name: 'family',
                normalize: [],
                original_fieldset: 'os',
                short: 'OS family (such as redhat, debian, freebsd, windows).',
                type: 'keyword',
            },
            'host.os.full': {
                dashed_name: 'host-os-full',
                description: 'Operating system name, including the version or code name.',
                example: 'Mac OS Mojave',
                flat_name: 'host.os.full',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'host.os.full.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'full',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system name, including the version or code name.',
                type: 'keyword',
            },
            'host.os.kernel': {
                dashed_name: 'host-os-kernel',
                description: 'Operating system kernel version as a raw string.',
                example: '4.4.0-112-generic',
                flat_name: 'host.os.kernel',
                ignore_above: 1024,
                level: 'extended',
                name: 'kernel',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system kernel version as a raw string.',
                type: 'keyword',
            },
            'host.os.name': {
                dashed_name: 'host-os-name',
                description: 'Operating system name, without the version.',
                example: 'Mac OS X',
                flat_name: 'host.os.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'host.os.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system name, without the version.',
                type: 'keyword',
            },
            'host.os.platform': {
                dashed_name: 'host-os-platform',
                description: 'Operating system platform (such centos, ubuntu, windows).',
                example: 'darwin',
                flat_name: 'host.os.platform',
                ignore_above: 1024,
                level: 'extended',
                name: 'platform',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system platform (such centos, ubuntu, windows).',
                type: 'keyword',
            },
            'host.os.type': {
                dashed_name: 'host-os-type',
                description: "Use the `os.type` field to categorize the operating system into one of the broad commercial families.\nIf the OS you're dealing with is not listed as an expected value, the field should not be populated. Please let us know by opening an issue with ECS, to propose its addition.",
                example: 'macos',
                expected_values: [
                    'linux',
                    'macos',
                    'unix',
                    'windows',
                    'ios',
                    'android',
                ],
                flat_name: 'host.os.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                original_fieldset: 'os',
                short: 'Which commercial OS family (one of: linux, macos, unix, windows, ios or android).',
                type: 'keyword',
            },
            'host.os.version': {
                dashed_name: 'host-os-version',
                description: 'Operating system version as a raw string.',
                example: '10.14.1',
                flat_name: 'host.os.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'version',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system version as a raw string.',
                type: 'keyword',
            },
            'host.pid_ns_ino': {
                beta: 'This field is beta and subject to change.',
                dashed_name: 'host-pid-ns-ino',
                description: 'This is the inode number of the namespace in the namespace file system (nsfs). Unsigned int inum in include/linux/ns_common.h.',
                example: 256383,
                flat_name: 'host.pid_ns_ino',
                ignore_above: 1024,
                level: 'extended',
                name: 'pid_ns_ino',
                normalize: [],
                short: 'Pid namespace inode',
                type: 'keyword',
            },
            'host.risk.calculated_level': {
                dashed_name: 'host-risk-calculated-level',
                description: 'A risk classification level calculated by an internal system as part of entity analytics and entity risk scoring.',
                example: 'High',
                flat_name: 'host.risk.calculated_level',
                ignore_above: 1024,
                level: 'extended',
                name: 'calculated_level',
                normalize: [],
                original_fieldset: 'risk',
                short: 'A risk classification level calculated by an internal system as part of entity analytics and entity risk scoring.',
                type: 'keyword',
            },
            'host.risk.calculated_score': {
                dashed_name: 'host-risk-calculated-score',
                description: 'A risk classification score calculated by an internal system as part of entity analytics and entity risk scoring.',
                example: 880.73,
                flat_name: 'host.risk.calculated_score',
                level: 'extended',
                name: 'calculated_score',
                normalize: [],
                original_fieldset: 'risk',
                short: 'A risk classification score calculated by an internal system as part of entity analytics and entity risk scoring.',
                type: 'float',
            },
            'host.risk.calculated_score_norm': {
                dashed_name: 'host-risk-calculated-score-norm',
                description: 'A risk classification score calculated by an internal system as part of entity analytics and entity risk scoring, and normalized to a range of 0 to 100.',
                example: 88.73,
                flat_name: 'host.risk.calculated_score_norm',
                level: 'extended',
                name: 'calculated_score_norm',
                normalize: [],
                original_fieldset: 'risk',
                short: 'A normalized risk score calculated by an internal system.',
                type: 'float',
            },
            'host.risk.static_level': {
                dashed_name: 'host-risk-static-level',
                description: 'A risk classification level obtained from outside the system, such as from some external Threat Intelligence Platform.',
                example: 'High',
                flat_name: 'host.risk.static_level',
                ignore_above: 1024,
                level: 'extended',
                name: 'static_level',
                normalize: [],
                original_fieldset: 'risk',
                short: 'A risk classification level obtained from outside the system, such as from some external Threat Intelligence Platform.',
                type: 'keyword',
            },
            'host.risk.static_score': {
                dashed_name: 'host-risk-static-score',
                description: 'A risk classification score obtained from outside the system, such as from some external Threat Intelligence Platform.',
                example: 830,
                flat_name: 'host.risk.static_score',
                level: 'extended',
                name: 'static_score',
                normalize: [],
                original_fieldset: 'risk',
                short: 'A risk classification score obtained from outside the system, such as from some external Threat Intelligence Platform.',
                type: 'float',
            },
            'host.risk.static_score_norm': {
                dashed_name: 'host-risk-static-score-norm',
                description: 'A risk classification score obtained from outside the system, such as from some external Threat Intelligence Platform, and normalized to a range of 0 to 100.',
                example: 83,
                flat_name: 'host.risk.static_score_norm',
                level: 'extended',
                name: 'static_score_norm',
                normalize: [],
                original_fieldset: 'risk',
                short: 'A normalized risk score calculated by an external system.',
                type: 'float',
            },
            'host.type': {
                dashed_name: 'host-type',
                description: 'Type of host.\nFor Cloud providers this can be the machine type like `t2.medium`. If vm, this could be the container, for example, or other information meaningful in your environment.',
                flat_name: 'host.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: [],
                short: 'Type of host.',
                type: 'keyword',
            },
            'host.uptime': {
                dashed_name: 'host-uptime',
                description: 'Seconds the host has been up.',
                example: 1325,
                flat_name: 'host.uptime',
                level: 'extended',
                name: 'uptime',
                normalize: [],
                short: 'Seconds the host has been up.',
                type: 'long',
            },
        },
        group: 2,
        name: 'host',
        nestings: ['host.geo', 'host.os', 'host.risk'],
        prefix: 'host.',
        reused_here: [
            {
                full: 'host.geo',
                schema_name: 'geo',
                short: 'Fields describing a location.',
            },
            {
                full: 'host.os',
                schema_name: 'os',
                short: 'OS fields contain information about the operating system.',
            },
            {
                full: 'host.risk',
                schema_name: 'risk',
                short: 'Fields for describing risk score and level.',
            },
        ],
        short: 'Fields describing the relevant computing instance.',
        title: 'Host',
        type: 'group',
    },
    http: {
        description: 'Fields related to HTTP activity. Use the `url` field set to store the url of the request.',
        fields: {
            'http.request.body.bytes': {
                dashed_name: 'http-request-body-bytes',
                description: 'Size in bytes of the request body.',
                example: 887,
                flat_name: 'http.request.body.bytes',
                format: 'bytes',
                level: 'extended',
                name: 'request.body.bytes',
                normalize: [],
                short: 'Size in bytes of the request body.',
                type: 'long',
            },
            'http.request.body.content': {
                dashed_name: 'http-request-body-content',
                description: 'The full HTTP request body.',
                example: 'Hello world',
                flat_name: 'http.request.body.content',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'http.request.body.content.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'request.body.content',
                normalize: [],
                short: 'The full HTTP request body.',
                type: 'wildcard',
            },
            'http.request.bytes': {
                dashed_name: 'http-request-bytes',
                description: 'Total size in bytes of the request (body and headers).',
                example: 1437,
                flat_name: 'http.request.bytes',
                format: 'bytes',
                level: 'extended',
                name: 'request.bytes',
                normalize: [],
                short: 'Total size in bytes of the request (body and headers).',
                type: 'long',
            },
            'http.request.id': {
                dashed_name: 'http-request-id',
                description: 'A unique identifier for each HTTP request to correlate logs between clients and servers in transactions.\nThe id may be contained in a non-standard HTTP header, such as `X-Request-ID` or `X-Correlation-ID`.',
                example: '123e4567-e89b-12d3-a456-426614174000',
                flat_name: 'http.request.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'request.id',
                normalize: [],
                short: 'HTTP request ID.',
                type: 'keyword',
            },
            'http.request.method': {
                dashed_name: 'http-request-method',
                description: 'HTTP request method.\nThe value should retain its casing from the original event. For example, `GET`, `get`, and `GeT` are all considered valid values for this field.',
                example: 'POST',
                flat_name: 'http.request.method',
                ignore_above: 1024,
                level: 'extended',
                name: 'request.method',
                normalize: [],
                short: 'HTTP request method.',
                type: 'keyword',
            },
            'http.request.mime_type': {
                dashed_name: 'http-request-mime-type',
                description: "Mime type of the body of the request.\nThis value must only be populated based on the content of the request body, not on the `Content-Type` header. Comparing the mime type of a request with the request's Content-Type header can be helpful in detecting threats or misconfigured clients.",
                example: 'image/gif',
                flat_name: 'http.request.mime_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'request.mime_type',
                normalize: [],
                short: 'Mime type of the body of the request.',
                type: 'keyword',
            },
            'http.request.referrer': {
                dashed_name: 'http-request-referrer',
                description: 'Referrer for this HTTP request.',
                example: 'https://blog.example.com/',
                flat_name: 'http.request.referrer',
                ignore_above: 1024,
                level: 'extended',
                name: 'request.referrer',
                normalize: [],
                short: 'Referrer for this HTTP request.',
                type: 'keyword',
            },
            'http.response.body.bytes': {
                dashed_name: 'http-response-body-bytes',
                description: 'Size in bytes of the response body.',
                example: 887,
                flat_name: 'http.response.body.bytes',
                format: 'bytes',
                level: 'extended',
                name: 'response.body.bytes',
                normalize: [],
                short: 'Size in bytes of the response body.',
                type: 'long',
            },
            'http.response.body.content': {
                dashed_name: 'http-response-body-content',
                description: 'The full HTTP response body.',
                example: 'Hello world',
                flat_name: 'http.response.body.content',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'http.response.body.content.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'response.body.content',
                normalize: [],
                short: 'The full HTTP response body.',
                type: 'wildcard',
            },
            'http.response.bytes': {
                dashed_name: 'http-response-bytes',
                description: 'Total size in bytes of the response (body and headers).',
                example: 1437,
                flat_name: 'http.response.bytes',
                format: 'bytes',
                level: 'extended',
                name: 'response.bytes',
                normalize: [],
                short: 'Total size in bytes of the response (body and headers).',
                type: 'long',
            },
            'http.response.mime_type': {
                dashed_name: 'http-response-mime-type',
                description: "Mime type of the body of the response.\nThis value must only be populated based on the content of the response body, not on the `Content-Type` header. Comparing the mime type of a response with the response's Content-Type header can be helpful in detecting misconfigured servers.",
                example: 'image/gif',
                flat_name: 'http.response.mime_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'response.mime_type',
                normalize: [],
                short: 'Mime type of the body of the response.',
                type: 'keyword',
            },
            'http.response.status_code': {
                dashed_name: 'http-response-status-code',
                description: 'HTTP response status code.',
                example: 404,
                flat_name: 'http.response.status_code',
                format: 'string',
                level: 'extended',
                name: 'response.status_code',
                normalize: [],
                short: 'HTTP response status code.',
                type: 'long',
            },
            'http.version': {
                dashed_name: 'http-version',
                description: 'HTTP version.',
                example: 1.1,
                flat_name: 'http.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'version',
                normalize: [],
                short: 'HTTP version.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'http',
        prefix: 'http.',
        short: 'Fields describing an HTTP request.',
        title: 'HTTP',
        type: 'group',
    },
    interface: {
        description: 'The interface fields are used to record ingress and egress interface information when reported by an observer (e.g. firewall, router, load balancer) in the context of the observer handling a network connection.  In the case of a single observer interface (e.g. network sensor on a span port) only the observer.ingress information should be populated.',
        fields: {
            'interface.alias': {
                dashed_name: 'interface-alias',
                description: 'Interface alias as reported by the system, typically used in firewall implementations for e.g. inside, outside, or dmz logical interface naming.',
                example: 'outside',
                flat_name: 'interface.alias',
                ignore_above: 1024,
                level: 'extended',
                name: 'alias',
                normalize: [],
                short: 'Interface alias',
                type: 'keyword',
            },
            'interface.id': {
                dashed_name: 'interface-id',
                description: 'Interface ID as reported by an observer (typically SNMP interface ID).',
                example: 10,
                flat_name: 'interface.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                short: 'Interface ID',
                type: 'keyword',
            },
            'interface.name': {
                dashed_name: 'interface-name',
                description: 'Interface name as reported by the system.',
                example: 'eth0',
                flat_name: 'interface.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Interface name',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'interface',
        prefix: 'interface.',
        reusable: {
            expected: [
                {
                    as: 'interface',
                    at: 'observer.ingress',
                    full: 'observer.ingress.interface',
                },
                {
                    as: 'interface',
                    at: 'observer.egress',
                    full: 'observer.egress.interface',
                },
            ],
            top_level: false,
        },
        short: 'Fields to describe observer interface information.',
        title: 'Interface',
        type: 'group',
    },
    log: {
        description: "Details about the event's logging mechanism or logging transport.\nThe log.* fields are typically populated with details about the logging mechanism used to create and/or transport the event. For example, syslog details belong under `log.syslog.*`.\nThe details specific to your event source are typically not logged under `log.*`, but rather in `event.*` or in other ECS fields.",
        fields: {
            'log.file.path': {
                dashed_name: 'log-file-path',
                description: "Full path to the log file this event came from, including the file name. It should include the drive letter, when appropriate.\nIf the event wasn't read from a log file, do not populate this field.",
                example: '/var/log/fun-times.log',
                flat_name: 'log.file.path',
                ignore_above: 1024,
                level: 'extended',
                name: 'file.path',
                normalize: [],
                short: 'Full path to the log file this event came from.',
                type: 'keyword',
            },
            'log.level': {
                dashed_name: 'log-level',
                description: "Original log level of the log event.\nIf the source of the event provides a log level or textual severity, this is the one that goes in `log.level`. If your source doesn't specify one, you may put your event transport's severity here (e.g. Syslog severity).\nSome examples are `warn`, `err`, `i`, `informational`.",
                example: 'error',
                flat_name: 'log.level',
                ignore_above: 1024,
                level: 'core',
                name: 'level',
                normalize: [],
                short: 'Log level of the log event.',
                type: 'keyword',
            },
            'log.logger': {
                dashed_name: 'log-logger',
                description: 'The name of the logger inside an application. This is usually the name of the class which initialized the logger, or can be a custom name.',
                example: 'org.elasticsearch.bootstrap.Bootstrap',
                flat_name: 'log.logger',
                ignore_above: 1024,
                level: 'core',
                name: 'logger',
                normalize: [],
                short: 'Name of the logger.',
                type: 'keyword',
            },
            'log.origin.file.line': {
                dashed_name: 'log-origin-file-line',
                description: 'The line number of the file containing the source code which originated the log event.',
                example: 42,
                flat_name: 'log.origin.file.line',
                level: 'extended',
                name: 'origin.file.line',
                normalize: [],
                short: 'The line number of the file which originated the log event.',
                type: 'long',
            },
            'log.origin.file.name': {
                dashed_name: 'log-origin-file-name',
                description: 'The name of the file containing the source code which originated the log event.\nNote that this field is not meant to capture the log file. The correct field to capture the log file is `log.file.path`.',
                example: 'Bootstrap.java',
                flat_name: 'log.origin.file.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'origin.file.name',
                normalize: [],
                short: 'The code file which originated the log event.',
                type: 'keyword',
            },
            'log.origin.function': {
                dashed_name: 'log-origin-function',
                description: 'The name of the function or method which originated the log event.',
                example: 'init',
                flat_name: 'log.origin.function',
                ignore_above: 1024,
                level: 'extended',
                name: 'origin.function',
                normalize: [],
                short: 'The function which originated the log event.',
                type: 'keyword',
            },
            'log.syslog': {
                dashed_name: 'log-syslog',
                description: 'The Syslog metadata of the event, if the event was transmitted via Syslog. Please see RFCs 5424 or 3164.',
                flat_name: 'log.syslog',
                level: 'extended',
                name: 'syslog',
                normalize: [],
                short: 'Syslog metadata',
                type: 'object',
            },
            'log.syslog.appname': {
                dashed_name: 'log-syslog-appname',
                description: 'The device or application that originated the Syslog message, if available.',
                example: 'sshd',
                flat_name: 'log.syslog.appname',
                ignore_above: 1024,
                level: 'extended',
                name: 'syslog.appname',
                normalize: [],
                short: 'The device or application that originated the Syslog message.',
                type: 'keyword',
            },
            'log.syslog.facility.code': {
                dashed_name: 'log-syslog-facility-code',
                description: 'The Syslog numeric facility of the log event, if available.\nAccording to RFCs 5424 and 3164, this value should be an integer between 0 and 23.',
                example: 23,
                flat_name: 'log.syslog.facility.code',
                format: 'string',
                level: 'extended',
                name: 'syslog.facility.code',
                normalize: [],
                short: 'Syslog numeric facility of the event.',
                type: 'long',
            },
            'log.syslog.facility.name': {
                dashed_name: 'log-syslog-facility-name',
                description: 'The Syslog text-based facility of the log event, if available.',
                example: 'local7',
                flat_name: 'log.syslog.facility.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'syslog.facility.name',
                normalize: [],
                short: 'Syslog text-based facility of the event.',
                type: 'keyword',
            },
            'log.syslog.hostname': {
                dashed_name: 'log-syslog-hostname',
                description: 'The hostname, FQDN, or IP of the machine that originally sent the Syslog message. This is sourced from the hostname field of the syslog header. Depending on the environment, this value may be different from the host that handled the event, especially if the host handling the events is acting as a collector.',
                example: 'example-host',
                flat_name: 'log.syslog.hostname',
                ignore_above: 1024,
                level: 'extended',
                name: 'syslog.hostname',
                normalize: [],
                short: 'The host that originated the Syslog message.',
                type: 'keyword',
            },
            'log.syslog.msgid': {
                dashed_name: 'log-syslog-msgid',
                description: 'An identifier for the type of Syslog message, if available. Only applicable for RFC 5424 messages.',
                example: 'ID47',
                flat_name: 'log.syslog.msgid',
                ignore_above: 1024,
                level: 'extended',
                name: 'syslog.msgid',
                normalize: [],
                short: 'An identifier for the type of Syslog message.',
                type: 'keyword',
            },
            'log.syslog.priority': {
                dashed_name: 'log-syslog-priority',
                description: 'Syslog numeric priority of the event, if available.\nAccording to RFCs 5424 and 3164, the priority is 8 * facility + severity. This number is therefore expected to contain a value between 0 and 191.',
                example: 135,
                flat_name: 'log.syslog.priority',
                format: 'string',
                level: 'extended',
                name: 'syslog.priority',
                normalize: [],
                short: 'Syslog priority of the event.',
                type: 'long',
            },
            'log.syslog.procid': {
                dashed_name: 'log-syslog-procid',
                description: 'The process name or ID that originated the Syslog message, if available.',
                example: 12345,
                flat_name: 'log.syslog.procid',
                ignore_above: 1024,
                level: 'extended',
                name: 'syslog.procid',
                normalize: [],
                short: 'The process name or ID that originated the Syslog message.',
                type: 'keyword',
            },
            'log.syslog.severity.code': {
                dashed_name: 'log-syslog-severity-code',
                description: "The Syslog numeric severity of the log event, if available.\nIf the event source publishing via Syslog provides a different numeric severity value (e.g. firewall, IDS), your source's numeric severity should go to `event.severity`. If the event source does not specify a distinct severity, you can optionally copy the Syslog severity to `event.severity`.",
                example: 3,
                flat_name: 'log.syslog.severity.code',
                level: 'extended',
                name: 'syslog.severity.code',
                normalize: [],
                short: 'Syslog numeric severity of the event.',
                type: 'long',
            },
            'log.syslog.severity.name': {
                dashed_name: 'log-syslog-severity-name',
                description: "The Syslog numeric severity of the log event, if available.\nIf the event source publishing via Syslog provides a different severity value (e.g. firewall, IDS), your source's text severity should go to `log.level`. If the event source does not specify a distinct severity, you can optionally copy the Syslog severity to `log.level`.",
                example: 'Error',
                flat_name: 'log.syslog.severity.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'syslog.severity.name',
                normalize: [],
                short: 'Syslog text-based severity of the event.',
                type: 'keyword',
            },
            'log.syslog.structured_data': {
                dashed_name: 'log-syslog-structured-data',
                description: 'Structured data expressed in RFC 5424 messages, if available. These are key-value pairs formed from the structured data portion of the syslog message, as defined in RFC 5424 Section 6.3.',
                flat_name: 'log.syslog.structured_data',
                level: 'extended',
                name: 'syslog.structured_data',
                normalize: [],
                short: 'Structured data expressed in RFC 5424 messages.',
                type: 'flattened',
            },
            'log.syslog.version': {
                dashed_name: 'log-syslog-version',
                description: 'The version of the Syslog protocol specification. Only applicable for RFC 5424 messages.',
                example: 1,
                flat_name: 'log.syslog.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'syslog.version',
                normalize: [],
                short: 'Syslog protocol version.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'log',
        prefix: 'log.',
        short: "Details about the event's logging mechanism.",
        title: 'Log',
        type: 'group',
    },
    macho: {
        beta: 'These fields are in beta and are subject to change.',
        description: 'These fields contain Mac OS Mach Object file format (Mach-O) metadata.',
        fields: {
            'macho.go_import_hash': {
                dashed_name: 'macho-go-import-hash',
                description: 'A hash of the Go language imports in a Mach-O file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'macho.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                short: 'A hash of the Go language imports in a Mach-O file.',
                type: 'keyword',
            },
            'macho.go_imports': {
                dashed_name: 'macho-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'macho.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'macho.go_imports_names_entropy': {
                dashed_name: 'macho-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'macho.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'macho.go_imports_names_var_entropy': {
                dashed_name: 'macho-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'macho.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'macho.go_stripped': {
                dashed_name: 'macho-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'macho.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'macho.import_hash': {
                dashed_name: 'macho-import-hash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for symhash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'macho.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
            'macho.imports': {
                dashed_name: 'macho-imports',
                description: 'List of imported element names and types.',
                flat_name: 'macho.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'macho.imports_names_entropy': {
                dashed_name: 'macho-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'macho.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'macho.imports_names_var_entropy': {
                dashed_name: 'macho-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'macho.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'macho.sections': {
                dashed_name: 'macho-sections',
                description: 'An array containing an object for each section of the Mach-O file.\nThe keys that should be present in these objects are defined by sub-fields underneath `macho.sections.*`.',
                flat_name: 'macho.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                short: 'Section information of the Mach-O file.',
                type: 'nested',
            },
            'macho.sections.entropy': {
                dashed_name: 'macho-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'macho.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'macho.sections.name': {
                dashed_name: 'macho-sections-name',
                description: 'Mach-O Section List name.',
                flat_name: 'macho.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                short: 'Mach-O Section List name.',
                type: 'keyword',
            },
            'macho.sections.physical_size': {
                dashed_name: 'macho-sections-physical-size',
                description: 'Mach-O Section List physical size.',
                flat_name: 'macho.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                short: 'Mach-O Section List physical size.',
                type: 'long',
            },
            'macho.sections.var_entropy': {
                dashed_name: 'macho-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'macho.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'macho.sections.virtual_size': {
                dashed_name: 'macho-sections-virtual-size',
                description: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'macho.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                short: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'macho.symhash': {
                dashed_name: 'macho-symhash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a Mach-O implementation of the Windows PE imphash',
                example: 'd3ccf195b62a9279c3c19af1080497ec',
                flat_name: 'macho.symhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'symhash',
                normalize: [],
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'macho',
        prefix: 'macho.',
        reusable: {
            expected: [
                {
                    as: 'macho',
                    at: 'file',
                    beta: 'This field reuse is beta and subject to change.',
                    full: 'file.macho',
                },
                {
                    as: 'macho',
                    at: 'process',
                    beta: 'This field reuse is beta and subject to change.',
                    full: 'process.macho',
                },
            ],
            top_level: false,
        },
        short: 'These fields contain Mac OS Mach Object file format (Mach-O) metadata.',
        title: 'Mach-O Header',
        type: 'group',
    },
    network: {
        description: 'The network is defined as the communication path over which a host or network event happens.\nThe network.* fields should be populated with details about the network activity associated with an event.',
        fields: {
            'network.application': {
                dashed_name: 'network-application',
                description: "When a specific application or service is identified from network connection details (source/dest IPs, ports, certificates, or wire format), this field captures the application's or service's name.\nFor example, the original event identifies the network connection being from a specific web service in a `https` network connection, like `facebook` or `twitter`.\nThe field value must be normalized to lowercase for querying.",
                example: 'aim',
                flat_name: 'network.application',
                ignore_above: 1024,
                level: 'extended',
                name: 'application',
                normalize: [],
                short: 'Application level protocol name.',
                type: 'keyword',
            },
            'network.bytes': {
                dashed_name: 'network-bytes',
                description: 'Total bytes transferred in both directions.\nIf `source.bytes` and `destination.bytes` are known, `network.bytes` is their sum.',
                example: 368,
                flat_name: 'network.bytes',
                format: 'bytes',
                level: 'core',
                name: 'bytes',
                normalize: [],
                short: 'Total bytes transferred in both directions.',
                type: 'long',
            },
            'network.community_id': {
                dashed_name: 'network-community-id',
                description: 'A hash of source and destination IPs and ports, as well as the protocol used in a communication. This is a tool-agnostic standard to identify flows.\nLearn more at https://github.com/corelight/community-id-spec.',
                example: '1:hO+sN4H+MG5MY/8hIrXPqc4ZQz0=',
                flat_name: 'network.community_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'community_id',
                normalize: [],
                short: 'A hash of source and destination IPs and ports.',
                type: 'keyword',
            },
            'network.direction': {
                dashed_name: 'network-direction',
                description: 'Direction of the network traffic.\nWhen mapping events from a host-based monitoring context, populate this field from the host\'s point of view, using the values "ingress" or "egress".\nWhen mapping events from a network or perimeter-based monitoring context, populate this field from the point of view of the network perimeter, using the values "inbound", "outbound", "internal" or "external".\nNote that "internal" is not crossing perimeter boundaries, and is meant to describe communication between two hosts within the perimeter. Note also that "external" is meant to describe traffic between two hosts that are external to the perimeter. This could for example be useful for ISPs or VPN service providers.',
                example: 'inbound',
                expected_values: [
                    'ingress',
                    'egress',
                    'inbound',
                    'outbound',
                    'internal',
                    'external',
                    'unknown',
                ],
                flat_name: 'network.direction',
                ignore_above: 1024,
                level: 'core',
                name: 'direction',
                normalize: [],
                short: 'Direction of the network traffic.',
                type: 'keyword',
            },
            'network.forwarded_ip': {
                dashed_name: 'network-forwarded-ip',
                description: 'Host IP address when the source IP address is the proxy.',
                example: '192.1.1.2',
                flat_name: 'network.forwarded_ip',
                level: 'core',
                name: 'forwarded_ip',
                normalize: [],
                short: 'Host IP address when the source IP address is the proxy.',
                type: 'ip',
            },
            'network.iana_number': {
                dashed_name: 'network-iana-number',
                description: 'IANA Protocol Number (https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml). Standardized list of protocols. This aligns well with NetFlow and sFlow related logs which use the IANA Protocol Number.',
                example: 6,
                flat_name: 'network.iana_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'iana_number',
                normalize: [],
                short: 'IANA Protocol Number.',
                type: 'keyword',
            },
            'network.inner': {
                dashed_name: 'network-inner',
                description: 'Network.inner fields are added in addition to network.vlan fields to describe the innermost VLAN when q-in-q VLAN tagging is present. Allowed fields include vlan.id and vlan.name. Inner vlan fields are typically used when sending traffic with multiple 802.1q encapsulations to a network sensor (e.g. Zeek, Wireshark.)',
                flat_name: 'network.inner',
                level: 'extended',
                name: 'inner',
                normalize: [],
                short: 'Inner VLAN tag information',
                type: 'object',
            },
            'network.inner.vlan.id': {
                dashed_name: 'network-inner-vlan-id',
                description: 'VLAN ID as reported by the observer.',
                example: 10,
                flat_name: 'network.inner.vlan.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'VLAN ID as reported by the observer.',
                type: 'keyword',
            },
            'network.inner.vlan.name': {
                dashed_name: 'network-inner-vlan-name',
                description: 'Optional VLAN name as reported by the observer.',
                example: 'outside',
                flat_name: 'network.inner.vlan.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'Optional VLAN name as reported by the observer.',
                type: 'keyword',
            },
            'network.name': {
                dashed_name: 'network-name',
                description: 'Name given by operators to sections of their network.',
                example: 'Guest Wifi',
                flat_name: 'network.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Name given by operators to sections of their network.',
                type: 'keyword',
            },
            'network.packets': {
                dashed_name: 'network-packets',
                description: 'Total packets transferred in both directions.\nIf `source.packets` and `destination.packets` are known, `network.packets` is their sum.',
                example: 24,
                flat_name: 'network.packets',
                level: 'core',
                name: 'packets',
                normalize: [],
                short: 'Total packets transferred in both directions.',
                type: 'long',
            },
            'network.protocol': {
                dashed_name: 'network-protocol',
                description: 'In the OSI Model this would be the Application Layer protocol. For example, `http`, `dns`, or `ssh`.\nThe field value must be normalized to lowercase for querying.',
                example: 'http',
                flat_name: 'network.protocol',
                ignore_above: 1024,
                level: 'core',
                name: 'protocol',
                normalize: [],
                short: 'Application protocol name.',
                type: 'keyword',
            },
            'network.transport': {
                dashed_name: 'network-transport',
                description: 'Same as network.iana_number, but instead using the Keyword name of the transport layer (udp, tcp, ipv6-icmp, etc.)\nThe field value must be normalized to lowercase for querying.',
                example: 'tcp',
                flat_name: 'network.transport',
                ignore_above: 1024,
                level: 'core',
                name: 'transport',
                normalize: [],
                short: 'Protocol Name corresponding to the field `iana_number`.',
                type: 'keyword',
            },
            'network.type': {
                dashed_name: 'network-type',
                description: 'In the OSI Model this would be the Network Layer. ipv4, ipv6, ipsec, pim, etc\nThe field value must be normalized to lowercase for querying.',
                example: 'ipv4',
                flat_name: 'network.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: [],
                short: 'In the OSI Model this would be the Network Layer. ipv4, ipv6, ipsec, pim, etc',
                type: 'keyword',
            },
            'network.vlan.id': {
                dashed_name: 'network-vlan-id',
                description: 'VLAN ID as reported by the observer.',
                example: 10,
                flat_name: 'network.vlan.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'VLAN ID as reported by the observer.',
                type: 'keyword',
            },
            'network.vlan.name': {
                dashed_name: 'network-vlan-name',
                description: 'Optional VLAN name as reported by the observer.',
                example: 'outside',
                flat_name: 'network.vlan.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'Optional VLAN name as reported by the observer.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'network',
        nestings: ['network.inner.vlan', 'network.vlan'],
        prefix: 'network.',
        reused_here: [
            {
                full: 'network.vlan',
                schema_name: 'vlan',
                short: 'Fields to describe observed VLAN information.',
            },
            {
                full: 'network.inner.vlan',
                schema_name: 'vlan',
                short: 'Fields to describe observed VLAN information.',
            },
        ],
        short: 'Fields describing the communication path over which the event happened.',
        title: 'Network',
        type: 'group',
    },
    observer: {
        description: 'An observer is defined as a special network, security, or application device used to detect, observe, or create network, security, or application-related events and metrics.\nThis could be a custom hardware appliance or a server that has been configured to run special network, security, or application software. Examples include firewalls, web proxies, intrusion detection/prevention systems, network monitoring sensors, web application firewalls, data loss prevention systems, and APM servers. The observer.* fields shall be populated with details of the system, if any, that detects, observes and/or creates a network, security, or application event or metric. Message queues and ETL components used in processing events or metrics are not considered observers in ECS.',
        fields: {
            'observer.egress': {
                dashed_name: 'observer-egress',
                description: 'Observer.egress holds information like interface number and name, vlan, and zone information to classify egress traffic.  Single armed monitoring such as a network sensor on a span port should only use observer.ingress to categorize traffic.',
                flat_name: 'observer.egress',
                level: 'extended',
                name: 'egress',
                normalize: [],
                short: 'Object field for egress information',
                type: 'object',
            },
            'observer.egress.interface.alias': {
                dashed_name: 'observer-egress-interface-alias',
                description: 'Interface alias as reported by the system, typically used in firewall implementations for e.g. inside, outside, or dmz logical interface naming.',
                example: 'outside',
                flat_name: 'observer.egress.interface.alias',
                ignore_above: 1024,
                level: 'extended',
                name: 'alias',
                normalize: [],
                original_fieldset: 'interface',
                short: 'Interface alias',
                type: 'keyword',
            },
            'observer.egress.interface.id': {
                dashed_name: 'observer-egress-interface-id',
                description: 'Interface ID as reported by an observer (typically SNMP interface ID).',
                example: 10,
                flat_name: 'observer.egress.interface.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'interface',
                short: 'Interface ID',
                type: 'keyword',
            },
            'observer.egress.interface.name': {
                dashed_name: 'observer-egress-interface-name',
                description: 'Interface name as reported by the system.',
                example: 'eth0',
                flat_name: 'observer.egress.interface.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'interface',
                short: 'Interface name',
                type: 'keyword',
            },
            'observer.egress.vlan.id': {
                dashed_name: 'observer-egress-vlan-id',
                description: 'VLAN ID as reported by the observer.',
                example: 10,
                flat_name: 'observer.egress.vlan.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'VLAN ID as reported by the observer.',
                type: 'keyword',
            },
            'observer.egress.vlan.name': {
                dashed_name: 'observer-egress-vlan-name',
                description: 'Optional VLAN name as reported by the observer.',
                example: 'outside',
                flat_name: 'observer.egress.vlan.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'Optional VLAN name as reported by the observer.',
                type: 'keyword',
            },
            'observer.egress.zone': {
                dashed_name: 'observer-egress-zone',
                description: 'Network zone of outbound traffic as reported by the observer to categorize the destination area of egress traffic, e.g. Internal, External, DMZ, HR, Legal, etc.',
                example: 'Public_Internet',
                flat_name: 'observer.egress.zone',
                ignore_above: 1024,
                level: 'extended',
                name: 'egress.zone',
                normalize: [],
                short: 'Observer Egress zone',
                type: 'keyword',
            },
            'observer.geo.city_name': {
                dashed_name: 'observer-geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'observer.geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'City name.',
                type: 'keyword',
            },
            'observer.geo.continent_code': {
                dashed_name: 'observer-geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'observer.geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Continent code.',
                type: 'keyword',
            },
            'observer.geo.continent_name': {
                dashed_name: 'observer-geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'observer.geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'observer.geo.country_iso_code': {
                dashed_name: 'observer-geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'observer.geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'observer.geo.country_name': {
                dashed_name: 'observer-geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'observer.geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country name.',
                type: 'keyword',
            },
            'observer.geo.location': {
                dashed_name: 'observer-geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'observer.geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'observer.geo.name': {
                dashed_name: 'observer-geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'observer.geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'observer.geo.postal_code': {
                dashed_name: 'observer-geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'observer.geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Postal code.',
                type: 'keyword',
            },
            'observer.geo.region_iso_code': {
                dashed_name: 'observer-geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'observer.geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'observer.geo.region_name': {
                dashed_name: 'observer-geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'observer.geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region name.',
                type: 'keyword',
            },
            'observer.geo.timezone': {
                dashed_name: 'observer-geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'observer.geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Time zone.',
                type: 'keyword',
            },
            'observer.hostname': {
                dashed_name: 'observer-hostname',
                description: 'Hostname of the observer.',
                flat_name: 'observer.hostname',
                ignore_above: 1024,
                level: 'core',
                name: 'hostname',
                normalize: [],
                short: 'Hostname of the observer.',
                type: 'keyword',
            },
            'observer.ingress': {
                dashed_name: 'observer-ingress',
                description: 'Observer.ingress holds information like interface number and name, vlan, and zone information to classify ingress traffic.  Single armed monitoring such as a network sensor on a span port should only use observer.ingress to categorize traffic.',
                flat_name: 'observer.ingress',
                level: 'extended',
                name: 'ingress',
                normalize: [],
                short: 'Object field for ingress information',
                type: 'object',
            },
            'observer.ingress.interface.alias': {
                dashed_name: 'observer-ingress-interface-alias',
                description: 'Interface alias as reported by the system, typically used in firewall implementations for e.g. inside, outside, or dmz logical interface naming.',
                example: 'outside',
                flat_name: 'observer.ingress.interface.alias',
                ignore_above: 1024,
                level: 'extended',
                name: 'alias',
                normalize: [],
                original_fieldset: 'interface',
                short: 'Interface alias',
                type: 'keyword',
            },
            'observer.ingress.interface.id': {
                dashed_name: 'observer-ingress-interface-id',
                description: 'Interface ID as reported by an observer (typically SNMP interface ID).',
                example: 10,
                flat_name: 'observer.ingress.interface.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'interface',
                short: 'Interface ID',
                type: 'keyword',
            },
            'observer.ingress.interface.name': {
                dashed_name: 'observer-ingress-interface-name',
                description: 'Interface name as reported by the system.',
                example: 'eth0',
                flat_name: 'observer.ingress.interface.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'interface',
                short: 'Interface name',
                type: 'keyword',
            },
            'observer.ingress.vlan.id': {
                dashed_name: 'observer-ingress-vlan-id',
                description: 'VLAN ID as reported by the observer.',
                example: 10,
                flat_name: 'observer.ingress.vlan.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'VLAN ID as reported by the observer.',
                type: 'keyword',
            },
            'observer.ingress.vlan.name': {
                dashed_name: 'observer-ingress-vlan-name',
                description: 'Optional VLAN name as reported by the observer.',
                example: 'outside',
                flat_name: 'observer.ingress.vlan.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'vlan',
                short: 'Optional VLAN name as reported by the observer.',
                type: 'keyword',
            },
            'observer.ingress.zone': {
                dashed_name: 'observer-ingress-zone',
                description: 'Network zone of incoming traffic as reported by the observer to categorize the source area of ingress traffic. e.g. internal, External, DMZ, HR, Legal, etc.',
                example: 'DMZ',
                flat_name: 'observer.ingress.zone',
                ignore_above: 1024,
                level: 'extended',
                name: 'ingress.zone',
                normalize: [],
                short: 'Observer ingress zone',
                type: 'keyword',
            },
            'observer.ip': {
                dashed_name: 'observer-ip',
                description: 'IP addresses of the observer.',
                flat_name: 'observer.ip',
                level: 'core',
                name: 'ip',
                normalize: ['array'],
                short: 'IP addresses of the observer.',
                type: 'ip',
            },
            'observer.mac': {
                dashed_name: 'observer-mac',
                description: 'MAC addresses of the observer.\nThe notation format from RFC 7042 is suggested: Each octet (that is, 8-bit byte) is represented by two [uppercase] hexadecimal digits giving the value of the octet as an unsigned integer. Successive octets are separated by a hyphen.',
                example: '["00-00-5E-00-53-23", "00-00-5E-00-53-24"]',
                flat_name: 'observer.mac',
                ignore_above: 1024,
                level: 'core',
                name: 'mac',
                normalize: ['array'],
                pattern: '^[A-F0-9]{2}(-[A-F0-9]{2}){5,}$',
                short: 'MAC addresses of the observer.',
                type: 'keyword',
            },
            'observer.name': {
                dashed_name: 'observer-name',
                description: 'Custom name of the observer.\nThis is a name that can be given to an observer. This can be helpful for example if multiple firewalls of the same model are used in an organization.\nIf no custom name is needed, the field can be left empty.',
                example: '1_proxySG',
                flat_name: 'observer.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Custom name of the observer.',
                type: 'keyword',
            },
            'observer.os.family': {
                dashed_name: 'observer-os-family',
                description: 'OS family (such as redhat, debian, freebsd, windows).',
                example: 'debian',
                flat_name: 'observer.os.family',
                ignore_above: 1024,
                level: 'extended',
                name: 'family',
                normalize: [],
                original_fieldset: 'os',
                short: 'OS family (such as redhat, debian, freebsd, windows).',
                type: 'keyword',
            },
            'observer.os.full': {
                dashed_name: 'observer-os-full',
                description: 'Operating system name, including the version or code name.',
                example: 'Mac OS Mojave',
                flat_name: 'observer.os.full',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'observer.os.full.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'full',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system name, including the version or code name.',
                type: 'keyword',
            },
            'observer.os.kernel': {
                dashed_name: 'observer-os-kernel',
                description: 'Operating system kernel version as a raw string.',
                example: '4.4.0-112-generic',
                flat_name: 'observer.os.kernel',
                ignore_above: 1024,
                level: 'extended',
                name: 'kernel',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system kernel version as a raw string.',
                type: 'keyword',
            },
            'observer.os.name': {
                dashed_name: 'observer-os-name',
                description: 'Operating system name, without the version.',
                example: 'Mac OS X',
                flat_name: 'observer.os.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'observer.os.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system name, without the version.',
                type: 'keyword',
            },
            'observer.os.platform': {
                dashed_name: 'observer-os-platform',
                description: 'Operating system platform (such centos, ubuntu, windows).',
                example: 'darwin',
                flat_name: 'observer.os.platform',
                ignore_above: 1024,
                level: 'extended',
                name: 'platform',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system platform (such centos, ubuntu, windows).',
                type: 'keyword',
            },
            'observer.os.type': {
                dashed_name: 'observer-os-type',
                description: "Use the `os.type` field to categorize the operating system into one of the broad commercial families.\nIf the OS you're dealing with is not listed as an expected value, the field should not be populated. Please let us know by opening an issue with ECS, to propose its addition.",
                example: 'macos',
                expected_values: [
                    'linux',
                    'macos',
                    'unix',
                    'windows',
                    'ios',
                    'android',
                ],
                flat_name: 'observer.os.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                original_fieldset: 'os',
                short: 'Which commercial OS family (one of: linux, macos, unix, windows, ios or android).',
                type: 'keyword',
            },
            'observer.os.version': {
                dashed_name: 'observer-os-version',
                description: 'Operating system version as a raw string.',
                example: '10.14.1',
                flat_name: 'observer.os.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'version',
                normalize: [],
                original_fieldset: 'os',
                short: 'Operating system version as a raw string.',
                type: 'keyword',
            },
            'observer.product': {
                dashed_name: 'observer-product',
                description: 'The product name of the observer.',
                example: 's200',
                flat_name: 'observer.product',
                ignore_above: 1024,
                level: 'extended',
                name: 'product',
                normalize: [],
                short: 'The product name of the observer.',
                type: 'keyword',
            },
            'observer.serial_number': {
                dashed_name: 'observer-serial-number',
                description: 'Observer serial number.',
                flat_name: 'observer.serial_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'serial_number',
                normalize: [],
                short: 'Observer serial number.',
                type: 'keyword',
            },
            'observer.type': {
                dashed_name: 'observer-type',
                description: 'The type of the observer the data is coming from.\nThere is no predefined list of observer types. Some examples are `forwarder`, `firewall`, `ids`, `ips`, `proxy`, `poller`, `sensor`, `APM server`.',
                example: 'firewall',
                flat_name: 'observer.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: [],
                short: 'The type of the observer the data is coming from.',
                type: 'keyword',
            },
            'observer.vendor': {
                dashed_name: 'observer-vendor',
                description: 'Vendor name of the observer.',
                example: 'Symantec',
                flat_name: 'observer.vendor',
                ignore_above: 1024,
                level: 'core',
                name: 'vendor',
                normalize: [],
                short: 'Vendor name of the observer.',
                type: 'keyword',
            },
            'observer.version': {
                dashed_name: 'observer-version',
                description: 'Observer version.',
                flat_name: 'observer.version',
                ignore_above: 1024,
                level: 'core',
                name: 'version',
                normalize: [],
                short: 'Observer version.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'observer',
        nestings: [
            'observer.egress.interface',
            'observer.egress.vlan',
            'observer.geo',
            'observer.ingress.interface',
            'observer.ingress.vlan',
            'observer.os',
        ],
        prefix: 'observer.',
        reused_here: [
            {
                full: 'observer.geo',
                schema_name: 'geo',
                short: 'Fields describing a location.',
            },
            {
                full: 'observer.ingress.interface',
                schema_name: 'interface',
                short: 'Fields to describe observer interface information.',
            },
            {
                full: 'observer.egress.interface',
                schema_name: 'interface',
                short: 'Fields to describe observer interface information.',
            },
            {
                full: 'observer.os',
                schema_name: 'os',
                short: 'OS fields contain information about the operating system.',
            },
            {
                full: 'observer.ingress.vlan',
                schema_name: 'vlan',
                short: 'Fields to describe observed VLAN information.',
            },
            {
                full: 'observer.egress.vlan',
                schema_name: 'vlan',
                short: 'Fields to describe observed VLAN information.',
            },
        ],
        short: 'Fields describing an entity observing the event from outside the host.',
        title: 'Observer',
        type: 'group',
    },
    orchestrator: {
        description: 'Fields that describe the resources which container orchestrators manage or act upon.',
        fields: {
            'orchestrator.api_version': {
                dashed_name: 'orchestrator-api-version',
                description: 'API version being used to carry out the action',
                example: 'v1beta1',
                flat_name: 'orchestrator.api_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'api_version',
                normalize: [],
                short: 'API version being used to carry out the action',
                type: 'keyword',
            },
            'orchestrator.cluster.id': {
                dashed_name: 'orchestrator-cluster-id',
                description: 'Unique ID of the cluster.',
                flat_name: 'orchestrator.cluster.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'cluster.id',
                normalize: [],
                short: 'Unique ID of the cluster.',
                type: 'keyword',
            },
            'orchestrator.cluster.name': {
                dashed_name: 'orchestrator-cluster-name',
                description: 'Name of the cluster.',
                flat_name: 'orchestrator.cluster.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'cluster.name',
                normalize: [],
                short: 'Name of the cluster.',
                type: 'keyword',
            },
            'orchestrator.cluster.url': {
                dashed_name: 'orchestrator-cluster-url',
                description: 'URL of the API used to manage the cluster.',
                flat_name: 'orchestrator.cluster.url',
                ignore_above: 1024,
                level: 'extended',
                name: 'cluster.url',
                normalize: [],
                short: 'URL of the API used to manage the cluster.',
                type: 'keyword',
            },
            'orchestrator.cluster.version': {
                dashed_name: 'orchestrator-cluster-version',
                description: 'The version of the cluster.',
                flat_name: 'orchestrator.cluster.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'cluster.version',
                normalize: [],
                short: 'The version of the cluster.',
                type: 'keyword',
            },
            'orchestrator.namespace': {
                dashed_name: 'orchestrator-namespace',
                description: 'Namespace in which the action is taking place.',
                example: 'kube-system',
                flat_name: 'orchestrator.namespace',
                ignore_above: 1024,
                level: 'extended',
                name: 'namespace',
                normalize: [],
                short: 'Namespace in which the action is taking place.',
                type: 'keyword',
            },
            'orchestrator.organization': {
                dashed_name: 'orchestrator-organization',
                description: 'Organization affected by the event (for multi-tenant orchestrator setups).',
                example: 'elastic',
                flat_name: 'orchestrator.organization',
                ignore_above: 1024,
                level: 'extended',
                name: 'organization',
                normalize: [],
                short: 'Organization affected by the event (for multi-tenant orchestrator setups).',
                type: 'keyword',
            },
            'orchestrator.resource.annotation': {
                dashed_name: 'orchestrator-resource-annotation',
                description: 'The list of annotations added to the resource.',
                example: "['key1:value1', 'key2:value2', 'key3:value3']",
                flat_name: 'orchestrator.resource.annotation',
                ignore_above: 1024,
                level: 'extended',
                name: 'resource.annotation',
                normalize: ['array'],
                short: 'The list of annotations added to the resource.',
                type: 'keyword',
            },
            'orchestrator.resource.id': {
                dashed_name: 'orchestrator-resource-id',
                description: 'Unique ID of the resource being acted upon.',
                flat_name: 'orchestrator.resource.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'resource.id',
                normalize: [],
                short: 'Unique ID of the resource being acted upon.',
                type: 'keyword',
            },
            'orchestrator.resource.ip': {
                dashed_name: 'orchestrator-resource-ip',
                description: 'IP address assigned to the resource associated with the event being observed. In the case of a Kubernetes Pod, this array would contain only one element: the IP of the Pod (as opposed to the Node on which the Pod is running).',
                flat_name: 'orchestrator.resource.ip',
                level: 'extended',
                name: 'resource.ip',
                normalize: ['array'],
                short: 'IP address assigned to the resource associated with the event being observed.',
                type: 'ip',
            },
            'orchestrator.resource.label': {
                dashed_name: 'orchestrator-resource-label',
                description: 'The list of labels added to the resource.',
                example: "['key1:value1', 'key2:value2', 'key3:value3']",
                flat_name: 'orchestrator.resource.label',
                ignore_above: 1024,
                level: 'extended',
                name: 'resource.label',
                normalize: ['array'],
                short: 'The list of labels added to the resource.',
                type: 'keyword',
            },
            'orchestrator.resource.name': {
                dashed_name: 'orchestrator-resource-name',
                description: 'Name of the resource being acted upon.',
                example: 'test-pod-cdcws',
                flat_name: 'orchestrator.resource.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'resource.name',
                normalize: [],
                short: 'Name of the resource being acted upon.',
                type: 'keyword',
            },
            'orchestrator.resource.parent.type': {
                dashed_name: 'orchestrator-resource-parent-type',
                description: 'Type or kind of the parent resource associated with the event being observed. In Kubernetes, this will be the name of a built-in workload resource (e.g., Deployment, StatefulSet, DaemonSet).',
                example: 'DaemonSet',
                flat_name: 'orchestrator.resource.parent.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'resource.parent.type',
                normalize: [],
                short: 'Type or kind of the parent resource associated with the event being observed.',
                type: 'keyword',
            },
            'orchestrator.resource.type': {
                dashed_name: 'orchestrator-resource-type',
                description: 'Type of resource being acted upon.',
                example: 'service',
                flat_name: 'orchestrator.resource.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'resource.type',
                normalize: [],
                short: 'Type of resource being acted upon.',
                type: 'keyword',
            },
            'orchestrator.type': {
                dashed_name: 'orchestrator-type',
                description: 'Orchestrator cluster type (e.g. kubernetes, nomad or cloudfoundry).',
                example: 'kubernetes',
                flat_name: 'orchestrator.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                short: 'Orchestrator cluster type (e.g. kubernetes, nomad or cloudfoundry).',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'orchestrator',
        prefix: 'orchestrator.',
        short: 'Fields relevant to container orchestrators.',
        title: 'Orchestrator',
        type: 'group',
    },
    organization: {
        description: 'The organization fields enrich data with information about the company or entity the data is associated with.\nThese fields help you arrange or filter data stored in an index by one or multiple organizations.',
        fields: {
            'organization.id': {
                dashed_name: 'organization-id',
                description: 'Unique identifier for the organization.',
                flat_name: 'organization.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                short: 'Unique identifier for the organization.',
                type: 'keyword',
            },
            'organization.name': {
                dashed_name: 'organization-name',
                description: 'Organization name.',
                flat_name: 'organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                short: 'Organization name.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'organization',
        prefix: 'organization.',
        short: 'Fields describing the organization or company the event is associated with.',
        title: 'Organization',
        type: 'group',
    },
    os: {
        description: 'The OS fields contain information about the operating system.',
        fields: {
            'os.family': {
                dashed_name: 'os-family',
                description: 'OS family (such as redhat, debian, freebsd, windows).',
                example: 'debian',
                flat_name: 'os.family',
                ignore_above: 1024,
                level: 'extended',
                name: 'family',
                normalize: [],
                short: 'OS family (such as redhat, debian, freebsd, windows).',
                type: 'keyword',
            },
            'os.full': {
                dashed_name: 'os-full',
                description: 'Operating system name, including the version or code name.',
                example: 'Mac OS Mojave',
                flat_name: 'os.full',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    { flat_name: 'os.full.text', name: 'text', type: 'match_only_text' },
                ],
                name: 'full',
                normalize: [],
                short: 'Operating system name, including the version or code name.',
                type: 'keyword',
            },
            'os.kernel': {
                dashed_name: 'os-kernel',
                description: 'Operating system kernel version as a raw string.',
                example: '4.4.0-112-generic',
                flat_name: 'os.kernel',
                ignore_above: 1024,
                level: 'extended',
                name: 'kernel',
                normalize: [],
                short: 'Operating system kernel version as a raw string.',
                type: 'keyword',
            },
            'os.name': {
                dashed_name: 'os-name',
                description: 'Operating system name, without the version.',
                example: 'Mac OS X',
                flat_name: 'os.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    { flat_name: 'os.name.text', name: 'text', type: 'match_only_text' },
                ],
                name: 'name',
                normalize: [],
                short: 'Operating system name, without the version.',
                type: 'keyword',
            },
            'os.platform': {
                dashed_name: 'os-platform',
                description: 'Operating system platform (such centos, ubuntu, windows).',
                example: 'darwin',
                flat_name: 'os.platform',
                ignore_above: 1024,
                level: 'extended',
                name: 'platform',
                normalize: [],
                short: 'Operating system platform (such centos, ubuntu, windows).',
                type: 'keyword',
            },
            'os.type': {
                dashed_name: 'os-type',
                description: "Use the `os.type` field to categorize the operating system into one of the broad commercial families.\nIf the OS you're dealing with is not listed as an expected value, the field should not be populated. Please let us know by opening an issue with ECS, to propose its addition.",
                example: 'macos',
                expected_values: [
                    'linux',
                    'macos',
                    'unix',
                    'windows',
                    'ios',
                    'android',
                ],
                flat_name: 'os.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                short: 'Which commercial OS family (one of: linux, macos, unix, windows, ios or android).',
                type: 'keyword',
            },
            'os.version': {
                dashed_name: 'os-version',
                description: 'Operating system version as a raw string.',
                example: '10.14.1',
                flat_name: 'os.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'version',
                normalize: [],
                short: 'Operating system version as a raw string.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'os',
        prefix: 'os.',
        reusable: {
            expected: [
                { as: 'os', at: 'observer', full: 'observer.os' },
                { as: 'os', at: 'host', full: 'host.os' },
                { as: 'os', at: 'user_agent', full: 'user_agent.os' },
            ],
            top_level: false,
        },
        short: 'OS fields contain information about the operating system.',
        title: 'Operating System',
        type: 'group',
    },
    package: {
        description: 'These fields contain information about an installed software package. It contains general information about a package, such as name, version or size. It also contains installation details, such as time or location.',
        fields: {
            'package.architecture': {
                dashed_name: 'package-architecture',
                description: 'Package architecture.',
                example: 'x86_64',
                flat_name: 'package.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                short: 'Package architecture.',
                type: 'keyword',
            },
            'package.build_version': {
                dashed_name: 'package-build-version',
                description: 'Additional information about the build version of the installed package.\nFor example use the commit SHA of a non-released package.',
                example: '36f4f7e89dd61b0988b12ee000b98966867710cd',
                flat_name: 'package.build_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'build_version',
                normalize: [],
                short: 'Build version information',
                type: 'keyword',
            },
            'package.checksum': {
                dashed_name: 'package-checksum',
                description: 'Checksum of the installed package for verification.',
                example: '68b329da9893e34099c7d8ad5cb9c940',
                flat_name: 'package.checksum',
                ignore_above: 1024,
                level: 'extended',
                name: 'checksum',
                normalize: [],
                short: 'Checksum of the installed package for verification.',
                type: 'keyword',
            },
            'package.description': {
                dashed_name: 'package-description',
                description: 'Description of the package.',
                example: 'Open source programming language to build simple/reliable/efficient software.',
                flat_name: 'package.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                short: 'Description of the package.',
                type: 'keyword',
            },
            'package.install_scope': {
                dashed_name: 'package-install-scope',
                description: 'Indicating how the package was installed, e.g. user-local, global.',
                example: 'global',
                flat_name: 'package.install_scope',
                ignore_above: 1024,
                level: 'extended',
                name: 'install_scope',
                normalize: [],
                short: 'Indicating how the package was installed, e.g. user-local, global.',
                type: 'keyword',
            },
            'package.installed': {
                dashed_name: 'package-installed',
                description: 'Time when package was installed.',
                flat_name: 'package.installed',
                level: 'extended',
                name: 'installed',
                normalize: [],
                short: 'Time when package was installed.',
                type: 'date',
            },
            'package.license': {
                dashed_name: 'package-license',
                description: 'License under which the package was released.\nUse a short name, e.g. the license identifier from SPDX License List where possible (https://spdx.org/licenses/).',
                example: 'Apache License 2.0',
                flat_name: 'package.license',
                ignore_above: 1024,
                level: 'extended',
                name: 'license',
                normalize: [],
                short: 'Package license',
                type: 'keyword',
            },
            'package.name': {
                dashed_name: 'package-name',
                description: 'Package name',
                example: 'go',
                flat_name: 'package.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Package name',
                type: 'keyword',
            },
            'package.path': {
                dashed_name: 'package-path',
                description: 'Path where the package is installed.',
                example: '/usr/local/Cellar/go/1.12.9/',
                flat_name: 'package.path',
                ignore_above: 1024,
                level: 'extended',
                name: 'path',
                normalize: [],
                short: 'Path where the package is installed.',
                type: 'keyword',
            },
            'package.reference': {
                dashed_name: 'package-reference',
                description: 'Home page or reference URL of the software in this package, if available.',
                example: 'https://golang.org',
                flat_name: 'package.reference',
                ignore_above: 1024,
                level: 'extended',
                name: 'reference',
                normalize: [],
                short: 'Package home page or reference URL',
                type: 'keyword',
            },
            'package.size': {
                dashed_name: 'package-size',
                description: 'Package size in bytes.',
                example: 62231,
                flat_name: 'package.size',
                format: 'string',
                level: 'extended',
                name: 'size',
                normalize: [],
                short: 'Package size in bytes.',
                type: 'long',
            },
            'package.type': {
                dashed_name: 'package-type',
                description: 'Type of package.\nThis should contain the package file type, rather than the package manager name. Examples: rpm, dpkg, brew, npm, gem, nupkg, jar.',
                example: 'rpm',
                flat_name: 'package.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                short: 'Package type',
                type: 'keyword',
            },
            'package.version': {
                dashed_name: 'package-version',
                description: 'Package version',
                example: '1.12.9',
                flat_name: 'package.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'version',
                normalize: [],
                short: 'Package version',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'package',
        prefix: 'package.',
        short: 'These fields contain information about an installed software package.',
        title: 'Package',
        type: 'group',
    },
    pe: {
        description: 'These fields contain Windows Portable Executable (PE) metadata.',
        fields: {
            'pe.architecture': {
                dashed_name: 'pe-architecture',
                description: 'CPU architecture target for the file.',
                example: 'x64',
                flat_name: 'pe.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                short: 'CPU architecture target for the file.',
                type: 'keyword',
            },
            'pe.company': {
                dashed_name: 'pe-company',
                description: 'Internal company name of the file, provided at compile-time.',
                example: 'Microsoft Corporation',
                flat_name: 'pe.company',
                ignore_above: 1024,
                level: 'extended',
                name: 'company',
                normalize: [],
                short: 'Internal company name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'pe.description': {
                dashed_name: 'pe-description',
                description: 'Internal description of the file, provided at compile-time.',
                example: 'Paint',
                flat_name: 'pe.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                short: 'Internal description of the file, provided at compile-time.',
                type: 'keyword',
            },
            'pe.file_version': {
                dashed_name: 'pe-file-version',
                description: 'Internal version of the file, provided at compile-time.',
                example: '6.3.9600.17415',
                flat_name: 'pe.file_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'file_version',
                normalize: [],
                short: 'Process name.',
                type: 'keyword',
            },
            'pe.go_import_hash': {
                dashed_name: 'pe-go-import-hash',
                description: 'A hash of the Go language imports in a PE file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'pe.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                short: 'A hash of the Go language imports in a PE file.',
                type: 'keyword',
            },
            'pe.go_imports': {
                dashed_name: 'pe-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'pe.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'pe.go_imports_names_entropy': {
                dashed_name: 'pe-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'pe.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'pe.go_imports_names_var_entropy': {
                dashed_name: 'pe-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'pe.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'pe.go_stripped': {
                dashed_name: 'pe-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'pe.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'pe.imphash': {
                dashed_name: 'pe-imphash',
                description: 'A hash of the imports in a PE file. An imphash -- or import hash -- can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nLearn more at https://www.fireeye.com/blog/threat-research/2014/01/tracking-malware-import-hashing.html.',
                example: '0c6803c4e922103c4dca5963aad36ddf',
                flat_name: 'pe.imphash',
                ignore_above: 1024,
                level: 'extended',
                name: 'imphash',
                normalize: [],
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'pe.import_hash': {
                dashed_name: 'pe-import-hash',
                description: 'A hash of the imports in a PE file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'pe.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'pe.imports': {
                dashed_name: 'pe-imports',
                description: 'List of imported element names and types.',
                flat_name: 'pe.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'pe.imports_names_entropy': {
                dashed_name: 'pe-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'pe.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'pe.imports_names_var_entropy': {
                dashed_name: 'pe-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'pe.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'pe.original_file_name': {
                dashed_name: 'pe-original-file-name',
                description: 'Internal name of the file, provided at compile-time.',
                example: 'MSPAINT.EXE',
                flat_name: 'pe.original_file_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'original_file_name',
                normalize: [],
                short: 'Internal name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'pe.pehash': {
                dashed_name: 'pe-pehash',
                description: 'A hash of the PE header and data from one or more PE sections. An pehash can be used to cluster files by transforming structural information about a file into a hash value.\nLearn more at https://www.usenix.org/legacy/events/leet09/tech/full_papers/wicherski/wicherski_html/index.html.',
                example: '73ff189b63cd6be375a7ff25179a38d347651975',
                flat_name: 'pe.pehash',
                ignore_above: 1024,
                level: 'extended',
                name: 'pehash',
                normalize: [],
                short: 'A hash of the PE header and data from one or more PE sections.',
                type: 'keyword',
            },
            'pe.product': {
                dashed_name: 'pe-product',
                description: 'Internal product name of the file, provided at compile-time.',
                example: 'Microsoft® Windows® Operating System',
                flat_name: 'pe.product',
                ignore_above: 1024,
                level: 'extended',
                name: 'product',
                normalize: [],
                short: 'Internal product name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'pe.sections': {
                dashed_name: 'pe-sections',
                description: 'An array containing an object for each section of the PE file.\nThe keys that should be present in these objects are defined by sub-fields underneath `pe.sections.*`.',
                flat_name: 'pe.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                short: 'Section information of the PE file.',
                type: 'nested',
            },
            'pe.sections.entropy': {
                dashed_name: 'pe-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'pe.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'pe.sections.name': {
                dashed_name: 'pe-sections-name',
                description: 'PE Section List name.',
                flat_name: 'pe.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                short: 'PE Section List name.',
                type: 'keyword',
            },
            'pe.sections.physical_size': {
                dashed_name: 'pe-sections-physical-size',
                description: 'PE Section List physical size.',
                flat_name: 'pe.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                short: 'PE Section List physical size.',
                type: 'long',
            },
            'pe.sections.var_entropy': {
                dashed_name: 'pe-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'pe.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'pe.sections.virtual_size': {
                dashed_name: 'pe-sections-virtual-size',
                description: 'PE Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'pe.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                short: 'PE Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
        },
        group: 2,
        name: 'pe',
        prefix: 'pe.',
        reusable: {
            expected: [
                { as: 'pe', at: 'file', full: 'file.pe' },
                { as: 'pe', at: 'dll', full: 'dll.pe' },
                { as: 'pe', at: 'process', full: 'process.pe' },
            ],
            top_level: false,
        },
        short: 'These fields contain Windows Portable Executable (PE) metadata.',
        title: 'PE Header',
        type: 'group',
    },
    process: {
        description: 'These fields contain information about a process.\nThese fields can help you correlate metrics information with a process id/name from a log message.  The `process.pid` often stays in the metric itself and is copied to the global field for correlation.',
        fields: {
            'process.args': {
                dashed_name: 'process-args',
                description: 'Array of process arguments, starting with the absolute path to the executable.\nMay be filtered to protect sensitive information.',
                example: '["/usr/bin/ssh", "-l", "user", "10.0.0.16"]',
                flat_name: 'process.args',
                ignore_above: 1024,
                level: 'extended',
                name: 'args',
                normalize: ['array'],
                short: 'Array of process arguments.',
                type: 'keyword',
            },
            'process.args_count': {
                dashed_name: 'process-args-count',
                description: 'Length of the process.args array.\nThis field can be useful for querying or performing bucket analysis on how many arguments were provided to start a process. More arguments may be an indication of suspicious activity.',
                example: 4,
                flat_name: 'process.args_count',
                level: 'extended',
                name: 'args_count',
                normalize: [],
                short: 'Length of the process.args array.',
                type: 'long',
            },
            'process.code_signature.digest_algorithm': {
                dashed_name: 'process-code-signature-digest-algorithm',
                description: 'The hashing algorithm used to sign the process.\nThis value can distinguish signatures when a file is signed multiple times by the same signer but with a different digest algorithm.',
                example: 'sha256',
                flat_name: 'process.code_signature.digest_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'digest_algorithm',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Hashing algorithm used to sign the process.',
                type: 'keyword',
            },
            'process.code_signature.exists': {
                dashed_name: 'process-code-signature-exists',
                description: 'Boolean to capture if a signature is present.',
                example: 'true',
                flat_name: 'process.code_signature.exists',
                level: 'core',
                name: 'exists',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if a signature is present.',
                type: 'boolean',
            },
            'process.code_signature.signing_id': {
                dashed_name: 'process-code-signature-signing-id',
                description: 'The identifier used to sign the process.\nThis is used to identify the application manufactured by a software vendor. The field is relevant to Apple *OS only.',
                example: 'com.apple.xpc.proxy',
                flat_name: 'process.code_signature.signing_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'signing_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The identifier used to sign the process.',
                type: 'keyword',
            },
            'process.code_signature.status': {
                dashed_name: 'process-code-signature-status',
                description: 'Additional information about the certificate status.\nThis is useful for logging cryptographic errors with the certificate validity or trust status. Leave unpopulated if the validity or trust of the certificate was unchecked.',
                example: 'ERROR_UNTRUSTED_ROOT',
                flat_name: 'process.code_signature.status',
                ignore_above: 1024,
                level: 'extended',
                name: 'status',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Additional information about the certificate status.',
                type: 'keyword',
            },
            'process.code_signature.subject_name': {
                dashed_name: 'process-code-signature-subject-name',
                description: 'Subject name of the code signer',
                example: 'Microsoft Corporation',
                flat_name: 'process.code_signature.subject_name',
                ignore_above: 1024,
                level: 'core',
                name: 'subject_name',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Subject name of the code signer',
                type: 'keyword',
            },
            'process.code_signature.team_id': {
                dashed_name: 'process-code-signature-team-id',
                description: 'The team identifier used to sign the process.\nThis is used to identify the team or vendor of a software product. The field is relevant to Apple *OS only.',
                example: 'EQHXZ8M8AV',
                flat_name: 'process.code_signature.team_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'team_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The team identifier used to sign the process.',
                type: 'keyword',
            },
            'process.code_signature.timestamp': {
                dashed_name: 'process-code-signature-timestamp',
                description: 'Date and time when the code signature was generated and signed.',
                example: '2021-01-01T12:10:30Z',
                flat_name: 'process.code_signature.timestamp',
                level: 'extended',
                name: 'timestamp',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'When the signature was generated and signed.',
                type: 'date',
            },
            'process.code_signature.trusted': {
                dashed_name: 'process-code-signature-trusted',
                description: 'Stores the trust status of the certificate chain.\nValidating the trust of the certificate chain may be complicated, and this field should only be populated by tools that actively check the status.',
                example: 'true',
                flat_name: 'process.code_signature.trusted',
                level: 'extended',
                name: 'trusted',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Stores the trust status of the certificate chain.',
                type: 'boolean',
            },
            'process.code_signature.valid': {
                dashed_name: 'process-code-signature-valid',
                description: 'Boolean to capture if the digital signature is verified against the binary content.\nLeave unpopulated if a certificate was unchecked.',
                example: 'true',
                flat_name: 'process.code_signature.valid',
                level: 'extended',
                name: 'valid',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if the digital signature is verified against the binary content.',
                type: 'boolean',
            },
            'process.command_line': {
                dashed_name: 'process-command-line',
                description: 'Full command line that started the process, including the absolute path to the executable, and all arguments.\nSome arguments may be filtered to protect sensitive information.',
                example: '/usr/bin/ssh -l user 10.0.0.16',
                flat_name: 'process.command_line',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.command_line.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'command_line',
                normalize: [],
                short: 'Full command line that started the process.',
                type: 'wildcard',
            },
            'process.elf.architecture': {
                dashed_name: 'process-elf-architecture',
                description: 'Machine architecture of the ELF file.',
                example: 'x86-64',
                flat_name: 'process.elf.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Machine architecture of the ELF file.',
                type: 'keyword',
            },
            'process.elf.byte_order': {
                dashed_name: 'process-elf-byte-order',
                description: 'Byte sequence of ELF file.',
                example: 'Little Endian',
                flat_name: 'process.elf.byte_order',
                ignore_above: 1024,
                level: 'extended',
                name: 'byte_order',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Byte sequence of ELF file.',
                type: 'keyword',
            },
            'process.elf.cpu_type': {
                dashed_name: 'process-elf-cpu-type',
                description: 'CPU type of the ELF file.',
                example: 'Intel',
                flat_name: 'process.elf.cpu_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'cpu_type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'CPU type of the ELF file.',
                type: 'keyword',
            },
            'process.elf.creation_date': {
                dashed_name: 'process-elf-creation-date',
                description: "Extracted when possible from the file's metadata. Indicates when it was built or compiled. It can also be faked by malware creators.",
                flat_name: 'process.elf.creation_date',
                level: 'extended',
                name: 'creation_date',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Build or compile date.',
                type: 'date',
            },
            'process.elf.exports': {
                dashed_name: 'process-elf-exports',
                description: 'List of exported element names and types.',
                flat_name: 'process.elf.exports',
                level: 'extended',
                name: 'exports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of exported element names and types.',
                type: 'flattened',
            },
            'process.elf.go_import_hash': {
                dashed_name: 'process-elf-go-import-hash',
                description: 'A hash of the Go language imports in an ELF file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'process.elf.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the Go language imports in an ELF file.',
                type: 'keyword',
            },
            'process.elf.go_imports': {
                dashed_name: 'process-elf-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'process.elf.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'elf',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'process.elf.go_imports_names_entropy': {
                dashed_name: 'process-elf-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.elf.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.elf.go_imports_names_var_entropy': {
                dashed_name: 'process-elf-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.elf.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.elf.go_stripped': {
                dashed_name: 'process-elf-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'process.elf.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'process.elf.header.abi_version': {
                dashed_name: 'process-elf-header-abi-version',
                description: 'Version of the ELF Application Binary Interface (ABI).',
                flat_name: 'process.elf.header.abi_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.abi_version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF Application Binary Interface (ABI).',
                type: 'keyword',
            },
            'process.elf.header.class': {
                dashed_name: 'process-elf-header-class',
                description: 'Header class of the ELF file.',
                flat_name: 'process.elf.header.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.class',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header class of the ELF file.',
                type: 'keyword',
            },
            'process.elf.header.data': {
                dashed_name: 'process-elf-header-data',
                description: 'Data table of the ELF header.',
                flat_name: 'process.elf.header.data',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.data',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Data table of the ELF header.',
                type: 'keyword',
            },
            'process.elf.header.entrypoint': {
                dashed_name: 'process-elf-header-entrypoint',
                description: 'Header entrypoint of the ELF file.',
                flat_name: 'process.elf.header.entrypoint',
                format: 'string',
                level: 'extended',
                name: 'header.entrypoint',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header entrypoint of the ELF file.',
                type: 'long',
            },
            'process.elf.header.object_version': {
                dashed_name: 'process-elf-header-object-version',
                description: '"0x1" for original ELF files.',
                flat_name: 'process.elf.header.object_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.object_version',
                normalize: [],
                original_fieldset: 'elf',
                short: '"0x1" for original ELF files.',
                type: 'keyword',
            },
            'process.elf.header.os_abi': {
                dashed_name: 'process-elf-header-os-abi',
                description: 'Application Binary Interface (ABI) of the Linux OS.',
                flat_name: 'process.elf.header.os_abi',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.os_abi',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Application Binary Interface (ABI) of the Linux OS.',
                type: 'keyword',
            },
            'process.elf.header.type': {
                dashed_name: 'process-elf-header-type',
                description: 'Header type of the ELF file.',
                flat_name: 'process.elf.header.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header type of the ELF file.',
                type: 'keyword',
            },
            'process.elf.header.version': {
                dashed_name: 'process-elf-header-version',
                description: 'Version of the ELF header.',
                flat_name: 'process.elf.header.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF header.',
                type: 'keyword',
            },
            'process.elf.import_hash': {
                dashed_name: 'process-elf-import-hash',
                description: 'A hash of the imports in an ELF file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is an ELF implementation of the Windows PE imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'process.elf.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the imports in an ELF file.',
                type: 'keyword',
            },
            'process.elf.imports': {
                dashed_name: 'process-elf-imports',
                description: 'List of imported element names and types.',
                flat_name: 'process.elf.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'process.elf.imports_names_entropy': {
                dashed_name: 'process-elf-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.elf.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.elf.imports_names_var_entropy': {
                dashed_name: 'process-elf-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.elf.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.elf.sections': {
                dashed_name: 'process-elf-sections',
                description: 'An array containing an object for each section of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.sections.*`.',
                flat_name: 'process.elf.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'Section information of the ELF file.',
                type: 'nested',
            },
            'process.elf.sections.chi2': {
                dashed_name: 'process-elf-sections-chi2',
                description: 'Chi-square probability distribution of the section.',
                flat_name: 'process.elf.sections.chi2',
                format: 'number',
                level: 'extended',
                name: 'sections.chi2',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Chi-square probability distribution of the section.',
                type: 'long',
            },
            'process.elf.sections.entropy': {
                dashed_name: 'process-elf-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'process.elf.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.elf.sections.flags': {
                dashed_name: 'process-elf-sections-flags',
                description: 'ELF Section List flags.',
                flat_name: 'process.elf.sections.flags',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.flags',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List flags.',
                type: 'keyword',
            },
            'process.elf.sections.name': {
                dashed_name: 'process-elf-sections-name',
                description: 'ELF Section List name.',
                flat_name: 'process.elf.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List name.',
                type: 'keyword',
            },
            'process.elf.sections.physical_offset': {
                dashed_name: 'process-elf-sections-physical-offset',
                description: 'ELF Section List offset.',
                flat_name: 'process.elf.sections.physical_offset',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.physical_offset',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List offset.',
                type: 'keyword',
            },
            'process.elf.sections.physical_size': {
                dashed_name: 'process-elf-sections-physical-size',
                description: 'ELF Section List physical size.',
                flat_name: 'process.elf.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List physical size.',
                type: 'long',
            },
            'process.elf.sections.type': {
                dashed_name: 'process-elf-sections-type',
                description: 'ELF Section List type.',
                flat_name: 'process.elf.sections.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List type.',
                type: 'keyword',
            },
            'process.elf.sections.var_entropy': {
                dashed_name: 'process-elf-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'process.elf.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.elf.sections.virtual_address': {
                dashed_name: 'process-elf-sections-virtual-address',
                description: 'ELF Section List virtual address.',
                flat_name: 'process.elf.sections.virtual_address',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_address',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual address.',
                type: 'long',
            },
            'process.elf.sections.virtual_size': {
                dashed_name: 'process-elf-sections-virtual-size',
                description: 'ELF Section List virtual size.',
                flat_name: 'process.elf.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual size.',
                type: 'long',
            },
            'process.elf.segments': {
                dashed_name: 'process-elf-segments',
                description: 'An array containing an object for each segment of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.segments.*`.',
                flat_name: 'process.elf.segments',
                level: 'extended',
                name: 'segments',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'ELF object segment list.',
                type: 'nested',
            },
            'process.elf.segments.sections': {
                dashed_name: 'process-elf-segments-sections',
                description: 'ELF object segment sections.',
                flat_name: 'process.elf.segments.sections',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.sections',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment sections.',
                type: 'keyword',
            },
            'process.elf.segments.type': {
                dashed_name: 'process-elf-segments-type',
                description: 'ELF object segment type.',
                flat_name: 'process.elf.segments.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment type.',
                type: 'keyword',
            },
            'process.elf.shared_libraries': {
                dashed_name: 'process-elf-shared-libraries',
                description: 'List of shared libraries used by this ELF object.',
                flat_name: 'process.elf.shared_libraries',
                ignore_above: 1024,
                level: 'extended',
                name: 'shared_libraries',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of shared libraries used by this ELF object.',
                type: 'keyword',
            },
            'process.elf.telfhash': {
                dashed_name: 'process-elf-telfhash',
                description: 'telfhash symbol hash for ELF file.',
                flat_name: 'process.elf.telfhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'telfhash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'telfhash hash for ELF file.',
                type: 'keyword',
            },
            'process.end': {
                dashed_name: 'process-end',
                description: 'The time the process ended.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.end',
                level: 'extended',
                name: 'end',
                normalize: [],
                short: 'The time the process ended.',
                type: 'date',
            },
            'process.entity_id': {
                dashed_name: 'process-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.entry_leader.args': {
                dashed_name: 'process-entry-leader-args',
                description: 'Array of process arguments, starting with the absolute path to the executable.\nMay be filtered to protect sensitive information.',
                example: '["/usr/bin/ssh", "-l", "user", "10.0.0.16"]',
                flat_name: 'process.entry_leader.args',
                ignore_above: 1024,
                level: 'extended',
                name: 'args',
                normalize: ['array'],
                original_fieldset: 'process',
                short: 'Array of process arguments.',
                type: 'keyword',
            },
            'process.entry_leader.args_count': {
                dashed_name: 'process-entry-leader-args-count',
                description: 'Length of the process.args array.\nThis field can be useful for querying or performing bucket analysis on how many arguments were provided to start a process. More arguments may be an indication of suspicious activity.',
                example: 4,
                flat_name: 'process.entry_leader.args_count',
                level: 'extended',
                name: 'args_count',
                normalize: [],
                original_fieldset: 'process',
                short: 'Length of the process.args array.',
                type: 'long',
            },
            'process.entry_leader.attested_groups.name': {
                dashed_name: 'process-entry-leader-attested-groups-name',
                description: 'Name of the group.',
                flat_name: 'process.entry_leader.attested_groups.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.entry_leader.attested_user.id': {
                dashed_name: 'process-entry-leader-attested-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.entry_leader.attested_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.entry_leader.attested_user.name': {
                dashed_name: 'process-entry-leader-attested-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.entry_leader.attested_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.attested_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.entry_leader.command_line': {
                dashed_name: 'process-entry-leader-command-line',
                description: 'Full command line that started the process, including the absolute path to the executable, and all arguments.\nSome arguments may be filtered to protect sensitive information.',
                example: '/usr/bin/ssh -l user 10.0.0.16',
                flat_name: 'process.entry_leader.command_line',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.command_line.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'command_line',
                normalize: [],
                original_fieldset: 'process',
                short: 'Full command line that started the process.',
                type: 'wildcard',
            },
            'process.entry_leader.entity_id': {
                dashed_name: 'process-entry-leader-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.entry_leader.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.entry_leader.entry_meta.source.ip': {
                dashed_name: 'process-entry-leader-entry-meta-source-ip',
                description: 'IP address of the source (IPv4 or IPv6).',
                flat_name: 'process.entry_leader.entry_meta.source.ip',
                level: 'core',
                name: 'ip',
                normalize: [],
                original_fieldset: 'source',
                short: 'IP address of the source.',
                type: 'ip',
            },
            'process.entry_leader.entry_meta.type': {
                dashed_name: 'process-entry-leader-entry-meta-type',
                description: 'The entry type for the entry session leader. Values include: init(e.g systemd), sshd, ssm, kubelet, teleport, terminal, console\nNote: This field is only set on process.session_leader.',
                flat_name: 'process.entry_leader.entry_meta.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'entry_meta.type',
                normalize: [],
                original_fieldset: 'process',
                short: 'The entry type for the entry session leader.',
                type: 'keyword',
            },
            'process.entry_leader.executable': {
                dashed_name: 'process-entry-leader-executable',
                description: 'Absolute path to the process executable.',
                example: '/usr/bin/ssh',
                flat_name: 'process.entry_leader.executable',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.executable.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'executable',
                normalize: [],
                original_fieldset: 'process',
                short: 'Absolute path to the process executable.',
                type: 'keyword',
            },
            'process.entry_leader.group.id': {
                dashed_name: 'process-entry-leader-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.entry_leader.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.entry_leader.group.name': {
                dashed_name: 'process-entry-leader-group-name',
                description: 'Name of the group.',
                flat_name: 'process.entry_leader.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.entry_leader.interactive': {
                dashed_name: 'process-entry-leader-interactive',
                description: 'Whether the process is connected to an interactive shell.\nProcess interactivity is inferred from the processes file descriptors. If the character device for the controlling tty is the same as stdin and stderr for the process, the process is considered interactive.\nNote: A non-interactive process can belong to an interactive session and is simply one that does not have open file descriptors reading the controlling TTY on FD 0 (stdin) or writing to the controlling TTY on FD 2 (stderr). A backgrounded process is still considered interactive if stdin and stderr are connected to the controlling TTY.',
                example: true,
                flat_name: 'process.entry_leader.interactive',
                level: 'extended',
                name: 'interactive',
                normalize: [],
                original_fieldset: 'process',
                short: 'Whether the process is connected to an interactive shell.',
                type: 'boolean',
            },
            'process.entry_leader.name': {
                dashed_name: 'process-entry-leader-name',
                description: 'Process name.\nSometimes called program name or similar.',
                example: 'ssh',
                flat_name: 'process.entry_leader.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process name.',
                type: 'keyword',
            },
            'process.entry_leader.parent.entity_id': {
                dashed_name: 'process-entry-leader-parent-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.entry_leader.parent.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.entry_leader.parent.pid': {
                dashed_name: 'process-entry-leader-parent-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.entry_leader.parent.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.entry_leader.parent.session_leader.entity_id': {
                dashed_name: 'process-entry-leader-parent-session-leader-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.entry_leader.parent.session_leader.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.entry_leader.parent.session_leader.pid': {
                dashed_name: 'process-entry-leader-parent-session-leader-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.entry_leader.parent.session_leader.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.entry_leader.parent.session_leader.start': {
                dashed_name: 'process-entry-leader-parent-session-leader-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.entry_leader.parent.session_leader.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.entry_leader.parent.session_leader.vpid': {
                dashed_name: 'process-entry-leader-parent-session-leader-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.entry_leader.parent.session_leader.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.entry_leader.parent.start': {
                dashed_name: 'process-entry-leader-parent-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.entry_leader.parent.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.entry_leader.parent.vpid': {
                dashed_name: 'process-entry-leader-parent-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.entry_leader.parent.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.entry_leader.pid': {
                dashed_name: 'process-entry-leader-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.entry_leader.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.entry_leader.real_group.id': {
                dashed_name: 'process-entry-leader-real-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.entry_leader.real_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.entry_leader.real_group.name': {
                dashed_name: 'process-entry-leader-real-group-name',
                description: 'Name of the group.',
                flat_name: 'process.entry_leader.real_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.entry_leader.real_user.id': {
                dashed_name: 'process-entry-leader-real-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.entry_leader.real_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.entry_leader.real_user.name': {
                dashed_name: 'process-entry-leader-real-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.entry_leader.real_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.real_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.entry_leader.same_as_process': {
                dashed_name: 'process-entry-leader-same-as-process',
                description: "This boolean is used to identify if a leader process is the same as the top level process.\nFor example, if `process.group_leader.same_as_process = true`, it means the process event in question is the leader of its process group. Details under `process.*` like `pid` would be the same under `process.group_leader.*` The same applies for both `process.session_leader` and `process.entry_leader`.\nThis field exists to the benefit of EQL and other rule engines since it's not possible to compare equality between two fields in a single document. e.g `process.entity_id` = `process.group_leader.entity_id` (top level process is the process group leader) OR `process.entity_id` = `process.entry_leader.entity_id` (top level process is the entry session leader)\nInstead these rules could be written like: `process.group_leader.same_as_process: true` OR `process.entry_leader.same_as_process: true`\nNote: This field is only set on `process.entry_leader`, `process.session_leader` and `process.group_leader`.",
                example: true,
                flat_name: 'process.entry_leader.same_as_process',
                level: 'extended',
                name: 'same_as_process',
                normalize: [],
                original_fieldset: 'process',
                short: 'This boolean is used to identify if a leader process is the same as the top level process.',
                type: 'boolean',
            },
            'process.entry_leader.saved_group.id': {
                dashed_name: 'process-entry-leader-saved-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.entry_leader.saved_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.entry_leader.saved_group.name': {
                dashed_name: 'process-entry-leader-saved-group-name',
                description: 'Name of the group.',
                flat_name: 'process.entry_leader.saved_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.entry_leader.saved_user.id': {
                dashed_name: 'process-entry-leader-saved-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.entry_leader.saved_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.entry_leader.saved_user.name': {
                dashed_name: 'process-entry-leader-saved-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.entry_leader.saved_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.saved_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.entry_leader.start': {
                dashed_name: 'process-entry-leader-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.entry_leader.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.entry_leader.supplemental_groups.id': {
                dashed_name: 'process-entry-leader-supplemental-groups-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.entry_leader.supplemental_groups.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.entry_leader.supplemental_groups.name': {
                dashed_name: 'process-entry-leader-supplemental-groups-name',
                description: 'Name of the group.',
                flat_name: 'process.entry_leader.supplemental_groups.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.entry_leader.tty': {
                dashed_name: 'process-entry-leader-tty',
                description: 'Information about the controlling TTY device. If set, the process belongs to an interactive session.',
                flat_name: 'process.entry_leader.tty',
                level: 'extended',
                name: 'tty',
                normalize: [],
                original_fieldset: 'process',
                short: 'Information about the controlling TTY device.',
                type: 'object',
            },
            'process.entry_leader.tty.char_device.major': {
                dashed_name: 'process-entry-leader-tty-char-device-major',
                description: 'The major number identifies the driver associated with the device. The character device\'s major and minor numbers can be algorithmically combined to produce the more familiar terminal identifiers such as "ttyS0" and "pts/0". For more details, please refer to the Linux kernel documentation.',
                example: 4,
                flat_name: 'process.entry_leader.tty.char_device.major',
                level: 'extended',
                name: 'tty.char_device.major',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's major number.",
                type: 'long',
            },
            'process.entry_leader.tty.char_device.minor': {
                dashed_name: 'process-entry-leader-tty-char-device-minor',
                description: 'The minor number is used only by the driver specified by the major number; other parts of the kernel don’t use it, and merely pass it along to the driver. It is common for a driver to control several devices; the minor number provides a way for the driver to differentiate among them.',
                example: 1,
                flat_name: 'process.entry_leader.tty.char_device.minor',
                level: 'extended',
                name: 'tty.char_device.minor',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's minor number.",
                type: 'long',
            },
            'process.entry_leader.user.id': {
                dashed_name: 'process-entry-leader-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.entry_leader.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.entry_leader.user.name': {
                dashed_name: 'process-entry-leader-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.entry_leader.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.entry_leader.vpid': {
                dashed_name: 'process-entry-leader-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.entry_leader.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.entry_leader.working_directory': {
                dashed_name: 'process-entry-leader-working-directory',
                description: 'The working directory of the process.',
                example: '/home/alice',
                flat_name: 'process.entry_leader.working_directory',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.entry_leader.working_directory.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'working_directory',
                normalize: [],
                original_fieldset: 'process',
                short: 'The working directory of the process.',
                type: 'keyword',
            },
            'process.env_vars': {
                dashed_name: 'process-env-vars',
                description: 'Array of environment variable bindings. Captured from a snapshot of the environment at the time of execution.\nMay be filtered to protect sensitive information.',
                example: '["PATH=/usr/local/bin:/usr/bin", "USER=ubuntu"]',
                flat_name: 'process.env_vars',
                ignore_above: 1024,
                level: 'extended',
                name: 'env_vars',
                normalize: ['array'],
                short: 'Array of environment variable bindings.',
                type: 'keyword',
            },
            'process.executable': {
                dashed_name: 'process-executable',
                description: 'Absolute path to the process executable.',
                example: '/usr/bin/ssh',
                flat_name: 'process.executable',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.executable.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'executable',
                normalize: [],
                short: 'Absolute path to the process executable.',
                type: 'keyword',
            },
            'process.exit_code': {
                dashed_name: 'process-exit-code',
                description: 'The exit code of the process, if this is a termination event.\nThe field should be absent if there is no exit code for the event (e.g. process start).',
                example: 137,
                flat_name: 'process.exit_code',
                level: 'extended',
                name: 'exit_code',
                normalize: [],
                short: 'The exit code of the process.',
                type: 'long',
            },
            'process.group_leader.args': {
                dashed_name: 'process-group-leader-args',
                description: 'Array of process arguments, starting with the absolute path to the executable.\nMay be filtered to protect sensitive information.',
                example: '["/usr/bin/ssh", "-l", "user", "10.0.0.16"]',
                flat_name: 'process.group_leader.args',
                ignore_above: 1024,
                level: 'extended',
                name: 'args',
                normalize: ['array'],
                original_fieldset: 'process',
                short: 'Array of process arguments.',
                type: 'keyword',
            },
            'process.group_leader.args_count': {
                dashed_name: 'process-group-leader-args-count',
                description: 'Length of the process.args array.\nThis field can be useful for querying or performing bucket analysis on how many arguments were provided to start a process. More arguments may be an indication of suspicious activity.',
                example: 4,
                flat_name: 'process.group_leader.args_count',
                level: 'extended',
                name: 'args_count',
                normalize: [],
                original_fieldset: 'process',
                short: 'Length of the process.args array.',
                type: 'long',
            },
            'process.group_leader.command_line': {
                dashed_name: 'process-group-leader-command-line',
                description: 'Full command line that started the process, including the absolute path to the executable, and all arguments.\nSome arguments may be filtered to protect sensitive information.',
                example: '/usr/bin/ssh -l user 10.0.0.16',
                flat_name: 'process.group_leader.command_line',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.group_leader.command_line.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'command_line',
                normalize: [],
                original_fieldset: 'process',
                short: 'Full command line that started the process.',
                type: 'wildcard',
            },
            'process.group_leader.entity_id': {
                dashed_name: 'process-group-leader-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.group_leader.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.group_leader.executable': {
                dashed_name: 'process-group-leader-executable',
                description: 'Absolute path to the process executable.',
                example: '/usr/bin/ssh',
                flat_name: 'process.group_leader.executable',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.group_leader.executable.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'executable',
                normalize: [],
                original_fieldset: 'process',
                short: 'Absolute path to the process executable.',
                type: 'keyword',
            },
            'process.group_leader.group.id': {
                dashed_name: 'process-group-leader-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.group_leader.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.group_leader.group.name': {
                dashed_name: 'process-group-leader-group-name',
                description: 'Name of the group.',
                flat_name: 'process.group_leader.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.group_leader.interactive': {
                dashed_name: 'process-group-leader-interactive',
                description: 'Whether the process is connected to an interactive shell.\nProcess interactivity is inferred from the processes file descriptors. If the character device for the controlling tty is the same as stdin and stderr for the process, the process is considered interactive.\nNote: A non-interactive process can belong to an interactive session and is simply one that does not have open file descriptors reading the controlling TTY on FD 0 (stdin) or writing to the controlling TTY on FD 2 (stderr). A backgrounded process is still considered interactive if stdin and stderr are connected to the controlling TTY.',
                example: true,
                flat_name: 'process.group_leader.interactive',
                level: 'extended',
                name: 'interactive',
                normalize: [],
                original_fieldset: 'process',
                short: 'Whether the process is connected to an interactive shell.',
                type: 'boolean',
            },
            'process.group_leader.name': {
                dashed_name: 'process-group-leader-name',
                description: 'Process name.\nSometimes called program name or similar.',
                example: 'ssh',
                flat_name: 'process.group_leader.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.group_leader.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process name.',
                type: 'keyword',
            },
            'process.group_leader.pid': {
                dashed_name: 'process-group-leader-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.group_leader.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.group_leader.real_group.id': {
                dashed_name: 'process-group-leader-real-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.group_leader.real_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.group_leader.real_group.name': {
                dashed_name: 'process-group-leader-real-group-name',
                description: 'Name of the group.',
                flat_name: 'process.group_leader.real_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.group_leader.real_user.id': {
                dashed_name: 'process-group-leader-real-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.group_leader.real_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.group_leader.real_user.name': {
                dashed_name: 'process-group-leader-real-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.group_leader.real_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.group_leader.real_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.group_leader.same_as_process': {
                dashed_name: 'process-group-leader-same-as-process',
                description: "This boolean is used to identify if a leader process is the same as the top level process.\nFor example, if `process.group_leader.same_as_process = true`, it means the process event in question is the leader of its process group. Details under `process.*` like `pid` would be the same under `process.group_leader.*` The same applies for both `process.session_leader` and `process.entry_leader`.\nThis field exists to the benefit of EQL and other rule engines since it's not possible to compare equality between two fields in a single document. e.g `process.entity_id` = `process.group_leader.entity_id` (top level process is the process group leader) OR `process.entity_id` = `process.entry_leader.entity_id` (top level process is the entry session leader)\nInstead these rules could be written like: `process.group_leader.same_as_process: true` OR `process.entry_leader.same_as_process: true`\nNote: This field is only set on `process.entry_leader`, `process.session_leader` and `process.group_leader`.",
                example: true,
                flat_name: 'process.group_leader.same_as_process',
                level: 'extended',
                name: 'same_as_process',
                normalize: [],
                original_fieldset: 'process',
                short: 'This boolean is used to identify if a leader process is the same as the top level process.',
                type: 'boolean',
            },
            'process.group_leader.saved_group.id': {
                dashed_name: 'process-group-leader-saved-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.group_leader.saved_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.group_leader.saved_group.name': {
                dashed_name: 'process-group-leader-saved-group-name',
                description: 'Name of the group.',
                flat_name: 'process.group_leader.saved_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.group_leader.saved_user.id': {
                dashed_name: 'process-group-leader-saved-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.group_leader.saved_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.group_leader.saved_user.name': {
                dashed_name: 'process-group-leader-saved-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.group_leader.saved_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.group_leader.saved_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.group_leader.start': {
                dashed_name: 'process-group-leader-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.group_leader.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.group_leader.supplemental_groups.id': {
                dashed_name: 'process-group-leader-supplemental-groups-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.group_leader.supplemental_groups.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.group_leader.supplemental_groups.name': {
                dashed_name: 'process-group-leader-supplemental-groups-name',
                description: 'Name of the group.',
                flat_name: 'process.group_leader.supplemental_groups.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.group_leader.tty': {
                dashed_name: 'process-group-leader-tty',
                description: 'Information about the controlling TTY device. If set, the process belongs to an interactive session.',
                flat_name: 'process.group_leader.tty',
                level: 'extended',
                name: 'tty',
                normalize: [],
                original_fieldset: 'process',
                short: 'Information about the controlling TTY device.',
                type: 'object',
            },
            'process.group_leader.tty.char_device.major': {
                dashed_name: 'process-group-leader-tty-char-device-major',
                description: 'The major number identifies the driver associated with the device. The character device\'s major and minor numbers can be algorithmically combined to produce the more familiar terminal identifiers such as "ttyS0" and "pts/0". For more details, please refer to the Linux kernel documentation.',
                example: 4,
                flat_name: 'process.group_leader.tty.char_device.major',
                level: 'extended',
                name: 'tty.char_device.major',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's major number.",
                type: 'long',
            },
            'process.group_leader.tty.char_device.minor': {
                dashed_name: 'process-group-leader-tty-char-device-minor',
                description: 'The minor number is used only by the driver specified by the major number; other parts of the kernel don’t use it, and merely pass it along to the driver. It is common for a driver to control several devices; the minor number provides a way for the driver to differentiate among them.',
                example: 1,
                flat_name: 'process.group_leader.tty.char_device.minor',
                level: 'extended',
                name: 'tty.char_device.minor',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's minor number.",
                type: 'long',
            },
            'process.group_leader.user.id': {
                dashed_name: 'process-group-leader-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.group_leader.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.group_leader.user.name': {
                dashed_name: 'process-group-leader-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.group_leader.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.group_leader.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.group_leader.vpid': {
                dashed_name: 'process-group-leader-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.group_leader.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.group_leader.working_directory': {
                dashed_name: 'process-group-leader-working-directory',
                description: 'The working directory of the process.',
                example: '/home/alice',
                flat_name: 'process.group_leader.working_directory',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.group_leader.working_directory.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'working_directory',
                normalize: [],
                original_fieldset: 'process',
                short: 'The working directory of the process.',
                type: 'keyword',
            },
            'process.hash.md5': {
                dashed_name: 'process-hash-md5',
                description: 'MD5 hash.',
                flat_name: 'process.hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                original_fieldset: 'hash',
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'process.hash.sha1': {
                dashed_name: 'process-hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'process.hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'process.hash.sha256': {
                dashed_name: 'process-hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'process.hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'process.hash.sha384': {
                dashed_name: 'process-hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'process.hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'process.hash.sha512': {
                dashed_name: 'process-hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'process.hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'process.hash.ssdeep': {
                dashed_name: 'process-hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'process.hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'process.hash.tlsh': {
                dashed_name: 'process-hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'process.hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                original_fieldset: 'hash',
                short: 'TLSH hash.',
                type: 'keyword',
            },
            'process.interactive': {
                dashed_name: 'process-interactive',
                description: 'Whether the process is connected to an interactive shell.\nProcess interactivity is inferred from the processes file descriptors. If the character device for the controlling tty is the same as stdin and stderr for the process, the process is considered interactive.\nNote: A non-interactive process can belong to an interactive session and is simply one that does not have open file descriptors reading the controlling TTY on FD 0 (stdin) or writing to the controlling TTY on FD 2 (stderr). A backgrounded process is still considered interactive if stdin and stderr are connected to the controlling TTY.',
                example: true,
                flat_name: 'process.interactive',
                level: 'extended',
                name: 'interactive',
                normalize: [],
                short: 'Whether the process is connected to an interactive shell.',
                type: 'boolean',
            },
            'process.io': {
                dashed_name: 'process-io',
                description: 'A chunk of input or output (IO) from a single process.\nThis field only appears on the top level process object, which is the process that wrote the output or read the input.',
                flat_name: 'process.io',
                level: 'extended',
                name: 'io',
                normalize: [],
                short: 'A chunk of input or output (IO) from a single process.',
                type: 'object',
            },
            'process.io.bytes_skipped': {
                dashed_name: 'process-io-bytes-skipped',
                description: 'An array of byte offsets and lengths denoting where IO data has been skipped.',
                flat_name: 'process.io.bytes_skipped',
                level: 'extended',
                name: 'io.bytes_skipped',
                normalize: ['array'],
                short: 'An array of byte offsets and lengths denoting where IO data has been skipped.',
                type: 'object',
            },
            'process.io.bytes_skipped.length': {
                dashed_name: 'process-io-bytes-skipped-length',
                description: 'The length of bytes skipped.',
                flat_name: 'process.io.bytes_skipped.length',
                level: 'extended',
                name: 'io.bytes_skipped.length',
                normalize: [],
                short: 'The length of bytes skipped.',
                type: 'long',
            },
            'process.io.bytes_skipped.offset': {
                dashed_name: 'process-io-bytes-skipped-offset',
                description: "The byte offset into this event's io.text (or io.bytes in the future) where length bytes were skipped.",
                flat_name: 'process.io.bytes_skipped.offset',
                level: 'extended',
                name: 'io.bytes_skipped.offset',
                normalize: [],
                short: "The byte offset into this event's io.text (or io.bytes in the future) where length bytes were skipped.",
                type: 'long',
            },
            'process.io.max_bytes_per_process_exceeded': {
                dashed_name: 'process-io-max-bytes-per-process-exceeded',
                description: 'If true, the process producing the output has exceeded the max_kilobytes_per_process configuration setting.',
                flat_name: 'process.io.max_bytes_per_process_exceeded',
                level: 'extended',
                name: 'io.max_bytes_per_process_exceeded',
                normalize: [],
                short: 'If true, the process producing the output has exceeded the max_kilobytes_per_process configuration setting.',
                type: 'boolean',
            },
            'process.io.text': {
                dashed_name: 'process-io-text',
                description: 'A chunk of output or input sanitized to UTF-8.\nBest efforts are made to ensure complete lines are captured in these events. Assumptions should NOT be made that multiple lines will appear in the same event. TTY output may contain terminal control codes such as for cursor movement, so some string queries may not match due to terminal codes inserted between characters of a word.',
                flat_name: 'process.io.text',
                level: 'extended',
                name: 'io.text',
                normalize: [],
                short: 'A chunk of output or input sanitized to UTF-8.',
                type: 'wildcard',
            },
            'process.io.total_bytes_captured': {
                dashed_name: 'process-io-total-bytes-captured',
                description: 'The total number of bytes captured in this event.',
                flat_name: 'process.io.total_bytes_captured',
                level: 'extended',
                name: 'io.total_bytes_captured',
                normalize: [],
                short: 'The total number of bytes captured in this event.',
                type: 'long',
            },
            'process.io.total_bytes_skipped': {
                dashed_name: 'process-io-total-bytes-skipped',
                description: 'The total number of bytes that were not captured due to implementation restrictions such as buffer size limits. Implementors should strive to ensure this value is always zero',
                flat_name: 'process.io.total_bytes_skipped',
                level: 'extended',
                name: 'io.total_bytes_skipped',
                normalize: [],
                short: 'The total number of bytes that were not captured due to implementation restrictions such as buffer size limits.',
                type: 'long',
            },
            'process.io.type': {
                dashed_name: 'process-io-type',
                description: "The type of object on which the IO action (read or write) was taken.\nCurrently only 'tty' is supported. Other types may be added in the future for 'file' and 'socket' support.",
                flat_name: 'process.io.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'io.type',
                normalize: [],
                short: 'The type of object on which the IO action (read or write) was taken.',
                type: 'keyword',
            },
            'process.macho.go_import_hash': {
                dashed_name: 'process-macho-go-import-hash',
                description: 'A hash of the Go language imports in a Mach-O file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'process.macho.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the Go language imports in a Mach-O file.',
                type: 'keyword',
            },
            'process.macho.go_imports': {
                dashed_name: 'process-macho-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'process.macho.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'macho',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'process.macho.go_imports_names_entropy': {
                dashed_name: 'process-macho-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.macho.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.macho.go_imports_names_var_entropy': {
                dashed_name: 'process-macho-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.macho.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.macho.go_stripped': {
                dashed_name: 'process-macho-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'process.macho.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'process.macho.import_hash': {
                dashed_name: 'process-macho-import-hash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for symhash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'process.macho.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
            'process.macho.imports': {
                dashed_name: 'process-macho-imports',
                description: 'List of imported element names and types.',
                flat_name: 'process.macho.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'macho',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'process.macho.imports_names_entropy': {
                dashed_name: 'process-macho-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.macho.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.macho.imports_names_var_entropy': {
                dashed_name: 'process-macho-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.macho.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.macho.sections': {
                dashed_name: 'process-macho-sections',
                description: 'An array containing an object for each section of the Mach-O file.\nThe keys that should be present in these objects are defined by sub-fields underneath `macho.sections.*`.',
                flat_name: 'process.macho.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'macho',
                short: 'Section information of the Mach-O file.',
                type: 'nested',
            },
            'process.macho.sections.entropy': {
                dashed_name: 'process-macho-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'process.macho.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.macho.sections.name': {
                dashed_name: 'process-macho-sections-name',
                description: 'Mach-O Section List name.',
                flat_name: 'process.macho.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List name.',
                type: 'keyword',
            },
            'process.macho.sections.physical_size': {
                dashed_name: 'process-macho-sections-physical-size',
                description: 'Mach-O Section List physical size.',
                flat_name: 'process.macho.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List physical size.',
                type: 'long',
            },
            'process.macho.sections.var_entropy': {
                dashed_name: 'process-macho-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'process.macho.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.macho.sections.virtual_size': {
                dashed_name: 'process-macho-sections-virtual-size',
                description: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'process.macho.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'process.macho.symhash': {
                dashed_name: 'process-macho-symhash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a Mach-O implementation of the Windows PE imphash',
                example: 'd3ccf195b62a9279c3c19af1080497ec',
                flat_name: 'process.macho.symhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'symhash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
            'process.name': {
                dashed_name: 'process-name',
                description: 'Process name.\nSometimes called program name or similar.',
                example: 'ssh',
                flat_name: 'process.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                short: 'Process name.',
                type: 'keyword',
            },
            'process.parent.args': {
                dashed_name: 'process-parent-args',
                description: 'Array of process arguments, starting with the absolute path to the executable.\nMay be filtered to protect sensitive information.',
                example: '["/usr/bin/ssh", "-l", "user", "10.0.0.16"]',
                flat_name: 'process.parent.args',
                ignore_above: 1024,
                level: 'extended',
                name: 'args',
                normalize: ['array'],
                original_fieldset: 'process',
                short: 'Array of process arguments.',
                type: 'keyword',
            },
            'process.parent.args_count': {
                dashed_name: 'process-parent-args-count',
                description: 'Length of the process.args array.\nThis field can be useful for querying or performing bucket analysis on how many arguments were provided to start a process. More arguments may be an indication of suspicious activity.',
                example: 4,
                flat_name: 'process.parent.args_count',
                level: 'extended',
                name: 'args_count',
                normalize: [],
                original_fieldset: 'process',
                short: 'Length of the process.args array.',
                type: 'long',
            },
            'process.parent.code_signature.digest_algorithm': {
                dashed_name: 'process-parent-code-signature-digest-algorithm',
                description: 'The hashing algorithm used to sign the process.\nThis value can distinguish signatures when a file is signed multiple times by the same signer but with a different digest algorithm.',
                example: 'sha256',
                flat_name: 'process.parent.code_signature.digest_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'digest_algorithm',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Hashing algorithm used to sign the process.',
                type: 'keyword',
            },
            'process.parent.code_signature.exists': {
                dashed_name: 'process-parent-code-signature-exists',
                description: 'Boolean to capture if a signature is present.',
                example: 'true',
                flat_name: 'process.parent.code_signature.exists',
                level: 'core',
                name: 'exists',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if a signature is present.',
                type: 'boolean',
            },
            'process.parent.code_signature.signing_id': {
                dashed_name: 'process-parent-code-signature-signing-id',
                description: 'The identifier used to sign the process.\nThis is used to identify the application manufactured by a software vendor. The field is relevant to Apple *OS only.',
                example: 'com.apple.xpc.proxy',
                flat_name: 'process.parent.code_signature.signing_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'signing_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The identifier used to sign the process.',
                type: 'keyword',
            },
            'process.parent.code_signature.status': {
                dashed_name: 'process-parent-code-signature-status',
                description: 'Additional information about the certificate status.\nThis is useful for logging cryptographic errors with the certificate validity or trust status. Leave unpopulated if the validity or trust of the certificate was unchecked.',
                example: 'ERROR_UNTRUSTED_ROOT',
                flat_name: 'process.parent.code_signature.status',
                ignore_above: 1024,
                level: 'extended',
                name: 'status',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Additional information about the certificate status.',
                type: 'keyword',
            },
            'process.parent.code_signature.subject_name': {
                dashed_name: 'process-parent-code-signature-subject-name',
                description: 'Subject name of the code signer',
                example: 'Microsoft Corporation',
                flat_name: 'process.parent.code_signature.subject_name',
                ignore_above: 1024,
                level: 'core',
                name: 'subject_name',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Subject name of the code signer',
                type: 'keyword',
            },
            'process.parent.code_signature.team_id': {
                dashed_name: 'process-parent-code-signature-team-id',
                description: 'The team identifier used to sign the process.\nThis is used to identify the team or vendor of a software product. The field is relevant to Apple *OS only.',
                example: 'EQHXZ8M8AV',
                flat_name: 'process.parent.code_signature.team_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'team_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The team identifier used to sign the process.',
                type: 'keyword',
            },
            'process.parent.code_signature.timestamp': {
                dashed_name: 'process-parent-code-signature-timestamp',
                description: 'Date and time when the code signature was generated and signed.',
                example: '2021-01-01T12:10:30Z',
                flat_name: 'process.parent.code_signature.timestamp',
                level: 'extended',
                name: 'timestamp',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'When the signature was generated and signed.',
                type: 'date',
            },
            'process.parent.code_signature.trusted': {
                dashed_name: 'process-parent-code-signature-trusted',
                description: 'Stores the trust status of the certificate chain.\nValidating the trust of the certificate chain may be complicated, and this field should only be populated by tools that actively check the status.',
                example: 'true',
                flat_name: 'process.parent.code_signature.trusted',
                level: 'extended',
                name: 'trusted',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Stores the trust status of the certificate chain.',
                type: 'boolean',
            },
            'process.parent.code_signature.valid': {
                dashed_name: 'process-parent-code-signature-valid',
                description: 'Boolean to capture if the digital signature is verified against the binary content.\nLeave unpopulated if a certificate was unchecked.',
                example: 'true',
                flat_name: 'process.parent.code_signature.valid',
                level: 'extended',
                name: 'valid',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if the digital signature is verified against the binary content.',
                type: 'boolean',
            },
            'process.parent.command_line': {
                dashed_name: 'process-parent-command-line',
                description: 'Full command line that started the process, including the absolute path to the executable, and all arguments.\nSome arguments may be filtered to protect sensitive information.',
                example: '/usr/bin/ssh -l user 10.0.0.16',
                flat_name: 'process.parent.command_line',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.parent.command_line.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'command_line',
                normalize: [],
                original_fieldset: 'process',
                short: 'Full command line that started the process.',
                type: 'wildcard',
            },
            'process.parent.elf.architecture': {
                dashed_name: 'process-parent-elf-architecture',
                description: 'Machine architecture of the ELF file.',
                example: 'x86-64',
                flat_name: 'process.parent.elf.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Machine architecture of the ELF file.',
                type: 'keyword',
            },
            'process.parent.elf.byte_order': {
                dashed_name: 'process-parent-elf-byte-order',
                description: 'Byte sequence of ELF file.',
                example: 'Little Endian',
                flat_name: 'process.parent.elf.byte_order',
                ignore_above: 1024,
                level: 'extended',
                name: 'byte_order',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Byte sequence of ELF file.',
                type: 'keyword',
            },
            'process.parent.elf.cpu_type': {
                dashed_name: 'process-parent-elf-cpu-type',
                description: 'CPU type of the ELF file.',
                example: 'Intel',
                flat_name: 'process.parent.elf.cpu_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'cpu_type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'CPU type of the ELF file.',
                type: 'keyword',
            },
            'process.parent.elf.creation_date': {
                dashed_name: 'process-parent-elf-creation-date',
                description: "Extracted when possible from the file's metadata. Indicates when it was built or compiled. It can also be faked by malware creators.",
                flat_name: 'process.parent.elf.creation_date',
                level: 'extended',
                name: 'creation_date',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Build or compile date.',
                type: 'date',
            },
            'process.parent.elf.exports': {
                dashed_name: 'process-parent-elf-exports',
                description: 'List of exported element names and types.',
                flat_name: 'process.parent.elf.exports',
                level: 'extended',
                name: 'exports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of exported element names and types.',
                type: 'flattened',
            },
            'process.parent.elf.go_import_hash': {
                dashed_name: 'process-parent-elf-go-import-hash',
                description: 'A hash of the Go language imports in an ELF file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'process.parent.elf.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the Go language imports in an ELF file.',
                type: 'keyword',
            },
            'process.parent.elf.go_imports': {
                dashed_name: 'process-parent-elf-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'process.parent.elf.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'elf',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'process.parent.elf.go_imports_names_entropy': {
                dashed_name: 'process-parent-elf-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.parent.elf.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.parent.elf.go_imports_names_var_entropy': {
                dashed_name: 'process-parent-elf-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.parent.elf.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.parent.elf.go_stripped': {
                dashed_name: 'process-parent-elf-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'process.parent.elf.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'process.parent.elf.header.abi_version': {
                dashed_name: 'process-parent-elf-header-abi-version',
                description: 'Version of the ELF Application Binary Interface (ABI).',
                flat_name: 'process.parent.elf.header.abi_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.abi_version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF Application Binary Interface (ABI).',
                type: 'keyword',
            },
            'process.parent.elf.header.class': {
                dashed_name: 'process-parent-elf-header-class',
                description: 'Header class of the ELF file.',
                flat_name: 'process.parent.elf.header.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.class',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header class of the ELF file.',
                type: 'keyword',
            },
            'process.parent.elf.header.data': {
                dashed_name: 'process-parent-elf-header-data',
                description: 'Data table of the ELF header.',
                flat_name: 'process.parent.elf.header.data',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.data',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Data table of the ELF header.',
                type: 'keyword',
            },
            'process.parent.elf.header.entrypoint': {
                dashed_name: 'process-parent-elf-header-entrypoint',
                description: 'Header entrypoint of the ELF file.',
                flat_name: 'process.parent.elf.header.entrypoint',
                format: 'string',
                level: 'extended',
                name: 'header.entrypoint',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header entrypoint of the ELF file.',
                type: 'long',
            },
            'process.parent.elf.header.object_version': {
                dashed_name: 'process-parent-elf-header-object-version',
                description: '"0x1" for original ELF files.',
                flat_name: 'process.parent.elf.header.object_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.object_version',
                normalize: [],
                original_fieldset: 'elf',
                short: '"0x1" for original ELF files.',
                type: 'keyword',
            },
            'process.parent.elf.header.os_abi': {
                dashed_name: 'process-parent-elf-header-os-abi',
                description: 'Application Binary Interface (ABI) of the Linux OS.',
                flat_name: 'process.parent.elf.header.os_abi',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.os_abi',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Application Binary Interface (ABI) of the Linux OS.',
                type: 'keyword',
            },
            'process.parent.elf.header.type': {
                dashed_name: 'process-parent-elf-header-type',
                description: 'Header type of the ELF file.',
                flat_name: 'process.parent.elf.header.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header type of the ELF file.',
                type: 'keyword',
            },
            'process.parent.elf.header.version': {
                dashed_name: 'process-parent-elf-header-version',
                description: 'Version of the ELF header.',
                flat_name: 'process.parent.elf.header.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF header.',
                type: 'keyword',
            },
            'process.parent.elf.import_hash': {
                dashed_name: 'process-parent-elf-import-hash',
                description: 'A hash of the imports in an ELF file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is an ELF implementation of the Windows PE imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'process.parent.elf.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the imports in an ELF file.',
                type: 'keyword',
            },
            'process.parent.elf.imports': {
                dashed_name: 'process-parent-elf-imports',
                description: 'List of imported element names and types.',
                flat_name: 'process.parent.elf.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'process.parent.elf.imports_names_entropy': {
                dashed_name: 'process-parent-elf-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.parent.elf.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.parent.elf.imports_names_var_entropy': {
                dashed_name: 'process-parent-elf-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.parent.elf.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.parent.elf.sections': {
                dashed_name: 'process-parent-elf-sections',
                description: 'An array containing an object for each section of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.sections.*`.',
                flat_name: 'process.parent.elf.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'Section information of the ELF file.',
                type: 'nested',
            },
            'process.parent.elf.sections.chi2': {
                dashed_name: 'process-parent-elf-sections-chi2',
                description: 'Chi-square probability distribution of the section.',
                flat_name: 'process.parent.elf.sections.chi2',
                format: 'number',
                level: 'extended',
                name: 'sections.chi2',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Chi-square probability distribution of the section.',
                type: 'long',
            },
            'process.parent.elf.sections.entropy': {
                dashed_name: 'process-parent-elf-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'process.parent.elf.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.parent.elf.sections.flags': {
                dashed_name: 'process-parent-elf-sections-flags',
                description: 'ELF Section List flags.',
                flat_name: 'process.parent.elf.sections.flags',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.flags',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List flags.',
                type: 'keyword',
            },
            'process.parent.elf.sections.name': {
                dashed_name: 'process-parent-elf-sections-name',
                description: 'ELF Section List name.',
                flat_name: 'process.parent.elf.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List name.',
                type: 'keyword',
            },
            'process.parent.elf.sections.physical_offset': {
                dashed_name: 'process-parent-elf-sections-physical-offset',
                description: 'ELF Section List offset.',
                flat_name: 'process.parent.elf.sections.physical_offset',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.physical_offset',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List offset.',
                type: 'keyword',
            },
            'process.parent.elf.sections.physical_size': {
                dashed_name: 'process-parent-elf-sections-physical-size',
                description: 'ELF Section List physical size.',
                flat_name: 'process.parent.elf.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List physical size.',
                type: 'long',
            },
            'process.parent.elf.sections.type': {
                dashed_name: 'process-parent-elf-sections-type',
                description: 'ELF Section List type.',
                flat_name: 'process.parent.elf.sections.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List type.',
                type: 'keyword',
            },
            'process.parent.elf.sections.var_entropy': {
                dashed_name: 'process-parent-elf-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'process.parent.elf.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.parent.elf.sections.virtual_address': {
                dashed_name: 'process-parent-elf-sections-virtual-address',
                description: 'ELF Section List virtual address.',
                flat_name: 'process.parent.elf.sections.virtual_address',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_address',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual address.',
                type: 'long',
            },
            'process.parent.elf.sections.virtual_size': {
                dashed_name: 'process-parent-elf-sections-virtual-size',
                description: 'ELF Section List virtual size.',
                flat_name: 'process.parent.elf.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual size.',
                type: 'long',
            },
            'process.parent.elf.segments': {
                dashed_name: 'process-parent-elf-segments',
                description: 'An array containing an object for each segment of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.segments.*`.',
                flat_name: 'process.parent.elf.segments',
                level: 'extended',
                name: 'segments',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'ELF object segment list.',
                type: 'nested',
            },
            'process.parent.elf.segments.sections': {
                dashed_name: 'process-parent-elf-segments-sections',
                description: 'ELF object segment sections.',
                flat_name: 'process.parent.elf.segments.sections',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.sections',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment sections.',
                type: 'keyword',
            },
            'process.parent.elf.segments.type': {
                dashed_name: 'process-parent-elf-segments-type',
                description: 'ELF object segment type.',
                flat_name: 'process.parent.elf.segments.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment type.',
                type: 'keyword',
            },
            'process.parent.elf.shared_libraries': {
                dashed_name: 'process-parent-elf-shared-libraries',
                description: 'List of shared libraries used by this ELF object.',
                flat_name: 'process.parent.elf.shared_libraries',
                ignore_above: 1024,
                level: 'extended',
                name: 'shared_libraries',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of shared libraries used by this ELF object.',
                type: 'keyword',
            },
            'process.parent.elf.telfhash': {
                dashed_name: 'process-parent-elf-telfhash',
                description: 'telfhash symbol hash for ELF file.',
                flat_name: 'process.parent.elf.telfhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'telfhash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'telfhash hash for ELF file.',
                type: 'keyword',
            },
            'process.parent.end': {
                dashed_name: 'process-parent-end',
                description: 'The time the process ended.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.parent.end',
                level: 'extended',
                name: 'end',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process ended.',
                type: 'date',
            },
            'process.parent.entity_id': {
                dashed_name: 'process-parent-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.parent.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.parent.executable': {
                dashed_name: 'process-parent-executable',
                description: 'Absolute path to the process executable.',
                example: '/usr/bin/ssh',
                flat_name: 'process.parent.executable',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.parent.executable.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'executable',
                normalize: [],
                original_fieldset: 'process',
                short: 'Absolute path to the process executable.',
                type: 'keyword',
            },
            'process.parent.exit_code': {
                dashed_name: 'process-parent-exit-code',
                description: 'The exit code of the process, if this is a termination event.\nThe field should be absent if there is no exit code for the event (e.g. process start).',
                example: 137,
                flat_name: 'process.parent.exit_code',
                level: 'extended',
                name: 'exit_code',
                normalize: [],
                original_fieldset: 'process',
                short: 'The exit code of the process.',
                type: 'long',
            },
            'process.parent.group.id': {
                dashed_name: 'process-parent-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.parent.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.parent.group.name': {
                dashed_name: 'process-parent-group-name',
                description: 'Name of the group.',
                flat_name: 'process.parent.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.parent.group_leader.entity_id': {
                dashed_name: 'process-parent-group-leader-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.parent.group_leader.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.parent.group_leader.pid': {
                dashed_name: 'process-parent-group-leader-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.parent.group_leader.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.parent.group_leader.start': {
                dashed_name: 'process-parent-group-leader-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.parent.group_leader.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.parent.group_leader.vpid': {
                dashed_name: 'process-parent-group-leader-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.parent.group_leader.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.parent.hash.md5': {
                dashed_name: 'process-parent-hash-md5',
                description: 'MD5 hash.',
                flat_name: 'process.parent.hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                original_fieldset: 'hash',
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'process.parent.hash.sha1': {
                dashed_name: 'process-parent-hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'process.parent.hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'process.parent.hash.sha256': {
                dashed_name: 'process-parent-hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'process.parent.hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'process.parent.hash.sha384': {
                dashed_name: 'process-parent-hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'process.parent.hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'process.parent.hash.sha512': {
                dashed_name: 'process-parent-hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'process.parent.hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'process.parent.hash.ssdeep': {
                dashed_name: 'process-parent-hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'process.parent.hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'process.parent.hash.tlsh': {
                dashed_name: 'process-parent-hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'process.parent.hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                original_fieldset: 'hash',
                short: 'TLSH hash.',
                type: 'keyword',
            },
            'process.parent.interactive': {
                dashed_name: 'process-parent-interactive',
                description: 'Whether the process is connected to an interactive shell.\nProcess interactivity is inferred from the processes file descriptors. If the character device for the controlling tty is the same as stdin and stderr for the process, the process is considered interactive.\nNote: A non-interactive process can belong to an interactive session and is simply one that does not have open file descriptors reading the controlling TTY on FD 0 (stdin) or writing to the controlling TTY on FD 2 (stderr). A backgrounded process is still considered interactive if stdin and stderr are connected to the controlling TTY.',
                example: true,
                flat_name: 'process.parent.interactive',
                level: 'extended',
                name: 'interactive',
                normalize: [],
                original_fieldset: 'process',
                short: 'Whether the process is connected to an interactive shell.',
                type: 'boolean',
            },
            'process.parent.macho.go_import_hash': {
                dashed_name: 'process-parent-macho-go-import-hash',
                description: 'A hash of the Go language imports in a Mach-O file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'process.parent.macho.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the Go language imports in a Mach-O file.',
                type: 'keyword',
            },
            'process.parent.macho.go_imports': {
                dashed_name: 'process-parent-macho-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'process.parent.macho.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'macho',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'process.parent.macho.go_imports_names_entropy': {
                dashed_name: 'process-parent-macho-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.parent.macho.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.parent.macho.go_imports_names_var_entropy': {
                dashed_name: 'process-parent-macho-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.parent.macho.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.parent.macho.go_stripped': {
                dashed_name: 'process-parent-macho-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'process.parent.macho.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'process.parent.macho.import_hash': {
                dashed_name: 'process-parent-macho-import-hash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for symhash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'process.parent.macho.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
            'process.parent.macho.imports': {
                dashed_name: 'process-parent-macho-imports',
                description: 'List of imported element names and types.',
                flat_name: 'process.parent.macho.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'macho',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'process.parent.macho.imports_names_entropy': {
                dashed_name: 'process-parent-macho-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.parent.macho.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.parent.macho.imports_names_var_entropy': {
                dashed_name: 'process-parent-macho-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.parent.macho.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.parent.macho.sections': {
                dashed_name: 'process-parent-macho-sections',
                description: 'An array containing an object for each section of the Mach-O file.\nThe keys that should be present in these objects are defined by sub-fields underneath `macho.sections.*`.',
                flat_name: 'process.parent.macho.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'macho',
                short: 'Section information of the Mach-O file.',
                type: 'nested',
            },
            'process.parent.macho.sections.entropy': {
                dashed_name: 'process-parent-macho-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'process.parent.macho.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.parent.macho.sections.name': {
                dashed_name: 'process-parent-macho-sections-name',
                description: 'Mach-O Section List name.',
                flat_name: 'process.parent.macho.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List name.',
                type: 'keyword',
            },
            'process.parent.macho.sections.physical_size': {
                dashed_name: 'process-parent-macho-sections-physical-size',
                description: 'Mach-O Section List physical size.',
                flat_name: 'process.parent.macho.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List physical size.',
                type: 'long',
            },
            'process.parent.macho.sections.var_entropy': {
                dashed_name: 'process-parent-macho-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'process.parent.macho.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.parent.macho.sections.virtual_size': {
                dashed_name: 'process-parent-macho-sections-virtual-size',
                description: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'process.parent.macho.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'macho',
                short: 'Mach-O Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'process.parent.macho.symhash': {
                dashed_name: 'process-parent-macho-symhash',
                description: 'A hash of the imports in a Mach-O file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a Mach-O implementation of the Windows PE imphash',
                example: 'd3ccf195b62a9279c3c19af1080497ec',
                flat_name: 'process.parent.macho.symhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'symhash',
                normalize: [],
                original_fieldset: 'macho',
                short: 'A hash of the imports in a Mach-O file.',
                type: 'keyword',
            },
            'process.parent.name': {
                dashed_name: 'process-parent-name',
                description: 'Process name.\nSometimes called program name or similar.',
                example: 'ssh',
                flat_name: 'process.parent.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.parent.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process name.',
                type: 'keyword',
            },
            'process.parent.pe.architecture': {
                dashed_name: 'process-parent-pe-architecture',
                description: 'CPU architecture target for the file.',
                example: 'x64',
                flat_name: 'process.parent.pe.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'pe',
                short: 'CPU architecture target for the file.',
                type: 'keyword',
            },
            'process.parent.pe.company': {
                dashed_name: 'process-parent-pe-company',
                description: 'Internal company name of the file, provided at compile-time.',
                example: 'Microsoft Corporation',
                flat_name: 'process.parent.pe.company',
                ignore_above: 1024,
                level: 'extended',
                name: 'company',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal company name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.parent.pe.description': {
                dashed_name: 'process-parent-pe-description',
                description: 'Internal description of the file, provided at compile-time.',
                example: 'Paint',
                flat_name: 'process.parent.pe.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal description of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.parent.pe.file_version': {
                dashed_name: 'process-parent-pe-file-version',
                description: 'Internal version of the file, provided at compile-time.',
                example: '6.3.9600.17415',
                flat_name: 'process.parent.pe.file_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'file_version',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Process name.',
                type: 'keyword',
            },
            'process.parent.pe.go_import_hash': {
                dashed_name: 'process-parent-pe-go-import-hash',
                description: 'A hash of the Go language imports in a PE file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'process.parent.pe.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the Go language imports in a PE file.',
                type: 'keyword',
            },
            'process.parent.pe.go_imports': {
                dashed_name: 'process-parent-pe-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'process.parent.pe.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'pe',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'process.parent.pe.go_imports_names_entropy': {
                dashed_name: 'process-parent-pe-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.parent.pe.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.parent.pe.go_imports_names_var_entropy': {
                dashed_name: 'process-parent-pe-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.parent.pe.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.parent.pe.go_stripped': {
                dashed_name: 'process-parent-pe-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'process.parent.pe.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'process.parent.pe.imphash': {
                dashed_name: 'process-parent-pe-imphash',
                description: 'A hash of the imports in a PE file. An imphash -- or import hash -- can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nLearn more at https://www.fireeye.com/blog/threat-research/2014/01/tracking-malware-import-hashing.html.',
                example: '0c6803c4e922103c4dca5963aad36ddf',
                flat_name: 'process.parent.pe.imphash',
                ignore_above: 1024,
                level: 'extended',
                name: 'imphash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'process.parent.pe.import_hash': {
                dashed_name: 'process-parent-pe-import-hash',
                description: 'A hash of the imports in a PE file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'process.parent.pe.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'process.parent.pe.imports': {
                dashed_name: 'process-parent-pe-imports',
                description: 'List of imported element names and types.',
                flat_name: 'process.parent.pe.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'process.parent.pe.imports_names_entropy': {
                dashed_name: 'process-parent-pe-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.parent.pe.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.parent.pe.imports_names_var_entropy': {
                dashed_name: 'process-parent-pe-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.parent.pe.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.parent.pe.original_file_name': {
                dashed_name: 'process-parent-pe-original-file-name',
                description: 'Internal name of the file, provided at compile-time.',
                example: 'MSPAINT.EXE',
                flat_name: 'process.parent.pe.original_file_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'original_file_name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.parent.pe.pehash': {
                dashed_name: 'process-parent-pe-pehash',
                description: 'A hash of the PE header and data from one or more PE sections. An pehash can be used to cluster files by transforming structural information about a file into a hash value.\nLearn more at https://www.usenix.org/legacy/events/leet09/tech/full_papers/wicherski/wicherski_html/index.html.',
                example: '73ff189b63cd6be375a7ff25179a38d347651975',
                flat_name: 'process.parent.pe.pehash',
                ignore_above: 1024,
                level: 'extended',
                name: 'pehash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the PE header and data from one or more PE sections.',
                type: 'keyword',
            },
            'process.parent.pe.product': {
                dashed_name: 'process-parent-pe-product',
                description: 'Internal product name of the file, provided at compile-time.',
                example: 'Microsoft® Windows® Operating System',
                flat_name: 'process.parent.pe.product',
                ignore_above: 1024,
                level: 'extended',
                name: 'product',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal product name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.parent.pe.sections': {
                dashed_name: 'process-parent-pe-sections',
                description: 'An array containing an object for each section of the PE file.\nThe keys that should be present in these objects are defined by sub-fields underneath `pe.sections.*`.',
                flat_name: 'process.parent.pe.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'Section information of the PE file.',
                type: 'nested',
            },
            'process.parent.pe.sections.entropy': {
                dashed_name: 'process-parent-pe-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'process.parent.pe.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.parent.pe.sections.name': {
                dashed_name: 'process-parent-pe-sections-name',
                description: 'PE Section List name.',
                flat_name: 'process.parent.pe.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List name.',
                type: 'keyword',
            },
            'process.parent.pe.sections.physical_size': {
                dashed_name: 'process-parent-pe-sections-physical-size',
                description: 'PE Section List physical size.',
                flat_name: 'process.parent.pe.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List physical size.',
                type: 'long',
            },
            'process.parent.pe.sections.var_entropy': {
                dashed_name: 'process-parent-pe-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'process.parent.pe.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.parent.pe.sections.virtual_size': {
                dashed_name: 'process-parent-pe-sections-virtual-size',
                description: 'PE Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'process.parent.pe.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'process.parent.pgid': {
                dashed_name: 'process-parent-pgid',
                description: 'Deprecated for removal in next major version release. This field is superseded by `process.group_leader.pid`.\nIdentifier of the group of processes the process belongs to.',
                flat_name: 'process.parent.pgid',
                format: 'string',
                level: 'extended',
                name: 'pgid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Deprecated identifier of the group of processes the process belongs to.',
                type: 'long',
            },
            'process.parent.pid': {
                dashed_name: 'process-parent-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.parent.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.parent.real_group.id': {
                dashed_name: 'process-parent-real-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.parent.real_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.parent.real_group.name': {
                dashed_name: 'process-parent-real-group-name',
                description: 'Name of the group.',
                flat_name: 'process.parent.real_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.parent.real_user.id': {
                dashed_name: 'process-parent-real-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.parent.real_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.parent.real_user.name': {
                dashed_name: 'process-parent-real-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.parent.real_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.parent.real_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.parent.saved_group.id': {
                dashed_name: 'process-parent-saved-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.parent.saved_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.parent.saved_group.name': {
                dashed_name: 'process-parent-saved-group-name',
                description: 'Name of the group.',
                flat_name: 'process.parent.saved_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.parent.saved_user.id': {
                dashed_name: 'process-parent-saved-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.parent.saved_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.parent.saved_user.name': {
                dashed_name: 'process-parent-saved-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.parent.saved_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.parent.saved_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.parent.start': {
                dashed_name: 'process-parent-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.parent.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.parent.supplemental_groups.id': {
                dashed_name: 'process-parent-supplemental-groups-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.parent.supplemental_groups.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.parent.supplemental_groups.name': {
                dashed_name: 'process-parent-supplemental-groups-name',
                description: 'Name of the group.',
                flat_name: 'process.parent.supplemental_groups.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.parent.thread.capabilities.effective': {
                dashed_name: 'process-parent-thread-capabilities-effective',
                description: 'This is the set of capabilities used by the kernel to perform permission checks for the thread.',
                example: '["CAP_BPF", "CAP_SYS_ADMIN"]',
                flat_name: 'process.parent.thread.capabilities.effective',
                ignore_above: 1024,
                level: 'extended',
                name: 'thread.capabilities.effective',
                normalize: ['array'],
                original_fieldset: 'process',
                pattern: '^(CAP_[A-Z_]+|\\d+)$',
                short: 'Array of capabilities used for permission checks.',
                type: 'keyword',
            },
            'process.parent.thread.capabilities.permitted': {
                dashed_name: 'process-parent-thread-capabilities-permitted',
                description: 'This is a limiting superset for the effective capabilities that the thread may assume.',
                example: '["CAP_BPF", "CAP_SYS_ADMIN"]',
                flat_name: 'process.parent.thread.capabilities.permitted',
                ignore_above: 1024,
                level: 'extended',
                name: 'thread.capabilities.permitted',
                normalize: ['array'],
                original_fieldset: 'process',
                pattern: '^(CAP_[A-Z_]+|\\d+)$',
                short: 'Array of capabilities a thread could assume.',
                type: 'keyword',
            },
            'process.parent.thread.id': {
                dashed_name: 'process-parent-thread-id',
                description: 'Thread ID.',
                example: 4242,
                flat_name: 'process.parent.thread.id',
                format: 'string',
                level: 'extended',
                name: 'thread.id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Thread ID.',
                type: 'long',
            },
            'process.parent.thread.name': {
                dashed_name: 'process-parent-thread-name',
                description: 'Thread name.',
                example: 'thread-0',
                flat_name: 'process.parent.thread.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'thread.name',
                normalize: [],
                original_fieldset: 'process',
                short: 'Thread name.',
                type: 'keyword',
            },
            'process.parent.title': {
                dashed_name: 'process-parent-title',
                description: 'Process title.\nThe proctitle, some times the same as process name. Can also be different: for example a browser setting its title to the web page currently opened.',
                flat_name: 'process.parent.title',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.parent.title.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'title',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process title.',
                type: 'keyword',
            },
            'process.parent.tty': {
                dashed_name: 'process-parent-tty',
                description: 'Information about the controlling TTY device. If set, the process belongs to an interactive session.',
                flat_name: 'process.parent.tty',
                level: 'extended',
                name: 'tty',
                normalize: [],
                original_fieldset: 'process',
                short: 'Information about the controlling TTY device.',
                type: 'object',
            },
            'process.parent.tty.char_device.major': {
                dashed_name: 'process-parent-tty-char-device-major',
                description: 'The major number identifies the driver associated with the device. The character device\'s major and minor numbers can be algorithmically combined to produce the more familiar terminal identifiers such as "ttyS0" and "pts/0". For more details, please refer to the Linux kernel documentation.',
                example: 4,
                flat_name: 'process.parent.tty.char_device.major',
                level: 'extended',
                name: 'tty.char_device.major',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's major number.",
                type: 'long',
            },
            'process.parent.tty.char_device.minor': {
                dashed_name: 'process-parent-tty-char-device-minor',
                description: 'The minor number is used only by the driver specified by the major number; other parts of the kernel don’t use it, and merely pass it along to the driver. It is common for a driver to control several devices; the minor number provides a way for the driver to differentiate among them.',
                example: 1,
                flat_name: 'process.parent.tty.char_device.minor',
                level: 'extended',
                name: 'tty.char_device.minor',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's minor number.",
                type: 'long',
            },
            'process.parent.uptime': {
                dashed_name: 'process-parent-uptime',
                description: 'Seconds the process has been up.',
                example: 1325,
                flat_name: 'process.parent.uptime',
                level: 'extended',
                name: 'uptime',
                normalize: [],
                original_fieldset: 'process',
                short: 'Seconds the process has been up.',
                type: 'long',
            },
            'process.parent.user.id': {
                dashed_name: 'process-parent-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.parent.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.parent.user.name': {
                dashed_name: 'process-parent-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.parent.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.parent.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.parent.vpid': {
                dashed_name: 'process-parent-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.parent.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.parent.working_directory': {
                dashed_name: 'process-parent-working-directory',
                description: 'The working directory of the process.',
                example: '/home/alice',
                flat_name: 'process.parent.working_directory',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.parent.working_directory.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'working_directory',
                normalize: [],
                original_fieldset: 'process',
                short: 'The working directory of the process.',
                type: 'keyword',
            },
            'process.pe.architecture': {
                dashed_name: 'process-pe-architecture',
                description: 'CPU architecture target for the file.',
                example: 'x64',
                flat_name: 'process.pe.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'pe',
                short: 'CPU architecture target for the file.',
                type: 'keyword',
            },
            'process.pe.company': {
                dashed_name: 'process-pe-company',
                description: 'Internal company name of the file, provided at compile-time.',
                example: 'Microsoft Corporation',
                flat_name: 'process.pe.company',
                ignore_above: 1024,
                level: 'extended',
                name: 'company',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal company name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.pe.description': {
                dashed_name: 'process-pe-description',
                description: 'Internal description of the file, provided at compile-time.',
                example: 'Paint',
                flat_name: 'process.pe.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal description of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.pe.file_version': {
                dashed_name: 'process-pe-file-version',
                description: 'Internal version of the file, provided at compile-time.',
                example: '6.3.9600.17415',
                flat_name: 'process.pe.file_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'file_version',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Process name.',
                type: 'keyword',
            },
            'process.pe.go_import_hash': {
                dashed_name: 'process-pe-go-import-hash',
                description: 'A hash of the Go language imports in a PE file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'process.pe.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the Go language imports in a PE file.',
                type: 'keyword',
            },
            'process.pe.go_imports': {
                dashed_name: 'process-pe-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'process.pe.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'pe',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'process.pe.go_imports_names_entropy': {
                dashed_name: 'process-pe-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.pe.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.pe.go_imports_names_var_entropy': {
                dashed_name: 'process-pe-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'process.pe.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'process.pe.go_stripped': {
                dashed_name: 'process-pe-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'process.pe.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'process.pe.imphash': {
                dashed_name: 'process-pe-imphash',
                description: 'A hash of the imports in a PE file. An imphash -- or import hash -- can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nLearn more at https://www.fireeye.com/blog/threat-research/2014/01/tracking-malware-import-hashing.html.',
                example: '0c6803c4e922103c4dca5963aad36ddf',
                flat_name: 'process.pe.imphash',
                ignore_above: 1024,
                level: 'extended',
                name: 'imphash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'process.pe.import_hash': {
                dashed_name: 'process-pe-import-hash',
                description: 'A hash of the imports in a PE file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'process.pe.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'process.pe.imports': {
                dashed_name: 'process-pe-imports',
                description: 'List of imported element names and types.',
                flat_name: 'process.pe.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'process.pe.imports_names_entropy': {
                dashed_name: 'process-pe-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.pe.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.pe.imports_names_var_entropy': {
                dashed_name: 'process-pe-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'process.pe.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'process.pe.original_file_name': {
                dashed_name: 'process-pe-original-file-name',
                description: 'Internal name of the file, provided at compile-time.',
                example: 'MSPAINT.EXE',
                flat_name: 'process.pe.original_file_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'original_file_name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.pe.pehash': {
                dashed_name: 'process-pe-pehash',
                description: 'A hash of the PE header and data from one or more PE sections. An pehash can be used to cluster files by transforming structural information about a file into a hash value.\nLearn more at https://www.usenix.org/legacy/events/leet09/tech/full_papers/wicherski/wicherski_html/index.html.',
                example: '73ff189b63cd6be375a7ff25179a38d347651975',
                flat_name: 'process.pe.pehash',
                ignore_above: 1024,
                level: 'extended',
                name: 'pehash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the PE header and data from one or more PE sections.',
                type: 'keyword',
            },
            'process.pe.product': {
                dashed_name: 'process-pe-product',
                description: 'Internal product name of the file, provided at compile-time.',
                example: 'Microsoft® Windows® Operating System',
                flat_name: 'process.pe.product',
                ignore_above: 1024,
                level: 'extended',
                name: 'product',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal product name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'process.pe.sections': {
                dashed_name: 'process-pe-sections',
                description: 'An array containing an object for each section of the PE file.\nThe keys that should be present in these objects are defined by sub-fields underneath `pe.sections.*`.',
                flat_name: 'process.pe.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'Section information of the PE file.',
                type: 'nested',
            },
            'process.pe.sections.entropy': {
                dashed_name: 'process-pe-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'process.pe.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.pe.sections.name': {
                dashed_name: 'process-pe-sections-name',
                description: 'PE Section List name.',
                flat_name: 'process.pe.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List name.',
                type: 'keyword',
            },
            'process.pe.sections.physical_size': {
                dashed_name: 'process-pe-sections-physical-size',
                description: 'PE Section List physical size.',
                flat_name: 'process.pe.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List physical size.',
                type: 'long',
            },
            'process.pe.sections.var_entropy': {
                dashed_name: 'process-pe-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'process.pe.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'process.pe.sections.virtual_size': {
                dashed_name: 'process-pe-sections-virtual-size',
                description: 'PE Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'process.pe.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'process.pgid': {
                dashed_name: 'process-pgid',
                description: 'Deprecated for removal in next major version release. This field is superseded by `process.group_leader.pid`.\nIdentifier of the group of processes the process belongs to.',
                flat_name: 'process.pgid',
                format: 'string',
                level: 'extended',
                name: 'pgid',
                normalize: [],
                short: 'Deprecated identifier of the group of processes the process belongs to.',
                type: 'long',
            },
            'process.pid': {
                dashed_name: 'process-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                short: 'Process id.',
                type: 'long',
            },
            'process.previous.args': {
                dashed_name: 'process-previous-args',
                description: 'Array of process arguments, starting with the absolute path to the executable.\nMay be filtered to protect sensitive information.',
                example: '["/usr/bin/ssh", "-l", "user", "10.0.0.16"]',
                flat_name: 'process.previous.args',
                ignore_above: 1024,
                level: 'extended',
                name: 'args',
                normalize: ['array'],
                original_fieldset: 'process',
                short: 'Array of process arguments.',
                type: 'keyword',
            },
            'process.previous.args_count': {
                dashed_name: 'process-previous-args-count',
                description: 'Length of the process.args array.\nThis field can be useful for querying or performing bucket analysis on how many arguments were provided to start a process. More arguments may be an indication of suspicious activity.',
                example: 4,
                flat_name: 'process.previous.args_count',
                level: 'extended',
                name: 'args_count',
                normalize: [],
                original_fieldset: 'process',
                short: 'Length of the process.args array.',
                type: 'long',
            },
            'process.previous.executable': {
                dashed_name: 'process-previous-executable',
                description: 'Absolute path to the process executable.',
                example: '/usr/bin/ssh',
                flat_name: 'process.previous.executable',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.previous.executable.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'executable',
                normalize: [],
                original_fieldset: 'process',
                short: 'Absolute path to the process executable.',
                type: 'keyword',
            },
            'process.real_group.id': {
                dashed_name: 'process-real-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.real_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.real_group.name': {
                dashed_name: 'process-real-group-name',
                description: 'Name of the group.',
                flat_name: 'process.real_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.real_user.id': {
                dashed_name: 'process-real-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.real_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.real_user.name': {
                dashed_name: 'process-real-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.real_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.real_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.saved_group.id': {
                dashed_name: 'process-saved-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.saved_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.saved_group.name': {
                dashed_name: 'process-saved-group-name',
                description: 'Name of the group.',
                flat_name: 'process.saved_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.saved_user.id': {
                dashed_name: 'process-saved-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.saved_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.saved_user.name': {
                dashed_name: 'process-saved-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.saved_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.saved_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.session_leader.args': {
                dashed_name: 'process-session-leader-args',
                description: 'Array of process arguments, starting with the absolute path to the executable.\nMay be filtered to protect sensitive information.',
                example: '["/usr/bin/ssh", "-l", "user", "10.0.0.16"]',
                flat_name: 'process.session_leader.args',
                ignore_above: 1024,
                level: 'extended',
                name: 'args',
                normalize: ['array'],
                original_fieldset: 'process',
                short: 'Array of process arguments.',
                type: 'keyword',
            },
            'process.session_leader.args_count': {
                dashed_name: 'process-session-leader-args-count',
                description: 'Length of the process.args array.\nThis field can be useful for querying or performing bucket analysis on how many arguments were provided to start a process. More arguments may be an indication of suspicious activity.',
                example: 4,
                flat_name: 'process.session_leader.args_count',
                level: 'extended',
                name: 'args_count',
                normalize: [],
                original_fieldset: 'process',
                short: 'Length of the process.args array.',
                type: 'long',
            },
            'process.session_leader.command_line': {
                dashed_name: 'process-session-leader-command-line',
                description: 'Full command line that started the process, including the absolute path to the executable, and all arguments.\nSome arguments may be filtered to protect sensitive information.',
                example: '/usr/bin/ssh -l user 10.0.0.16',
                flat_name: 'process.session_leader.command_line',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.session_leader.command_line.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'command_line',
                normalize: [],
                original_fieldset: 'process',
                short: 'Full command line that started the process.',
                type: 'wildcard',
            },
            'process.session_leader.entity_id': {
                dashed_name: 'process-session-leader-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.session_leader.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.session_leader.executable': {
                dashed_name: 'process-session-leader-executable',
                description: 'Absolute path to the process executable.',
                example: '/usr/bin/ssh',
                flat_name: 'process.session_leader.executable',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.session_leader.executable.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'executable',
                normalize: [],
                original_fieldset: 'process',
                short: 'Absolute path to the process executable.',
                type: 'keyword',
            },
            'process.session_leader.group.id': {
                dashed_name: 'process-session-leader-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.session_leader.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.session_leader.group.name': {
                dashed_name: 'process-session-leader-group-name',
                description: 'Name of the group.',
                flat_name: 'process.session_leader.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.session_leader.interactive': {
                dashed_name: 'process-session-leader-interactive',
                description: 'Whether the process is connected to an interactive shell.\nProcess interactivity is inferred from the processes file descriptors. If the character device for the controlling tty is the same as stdin and stderr for the process, the process is considered interactive.\nNote: A non-interactive process can belong to an interactive session and is simply one that does not have open file descriptors reading the controlling TTY on FD 0 (stdin) or writing to the controlling TTY on FD 2 (stderr). A backgrounded process is still considered interactive if stdin and stderr are connected to the controlling TTY.',
                example: true,
                flat_name: 'process.session_leader.interactive',
                level: 'extended',
                name: 'interactive',
                normalize: [],
                original_fieldset: 'process',
                short: 'Whether the process is connected to an interactive shell.',
                type: 'boolean',
            },
            'process.session_leader.name': {
                dashed_name: 'process-session-leader-name',
                description: 'Process name.\nSometimes called program name or similar.',
                example: 'ssh',
                flat_name: 'process.session_leader.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.session_leader.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process name.',
                type: 'keyword',
            },
            'process.session_leader.parent.entity_id': {
                dashed_name: 'process-session-leader-parent-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.session_leader.parent.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.session_leader.parent.pid': {
                dashed_name: 'process-session-leader-parent-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.session_leader.parent.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.session_leader.parent.session_leader.entity_id': {
                dashed_name: 'process-session-leader-parent-session-leader-entity-id',
                description: 'Unique identifier for the process.\nThe implementation of this is specified by the data source, but some examples of what could be used here are a process-generated UUID, Sysmon Process GUIDs, or a hash of some uniquely identifying components of a process.\nConstructing a globally unique identifier is a common practice to mitigate PID reuse as well as to identify a specific process over time, across multiple monitored hosts.',
                example: 'c2c455d9f99375d',
                flat_name: 'process.session_leader.parent.session_leader.entity_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'entity_id',
                normalize: [],
                original_fieldset: 'process',
                short: 'Unique identifier for the process.',
                type: 'keyword',
            },
            'process.session_leader.parent.session_leader.pid': {
                dashed_name: 'process-session-leader-parent-session-leader-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.session_leader.parent.session_leader.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.session_leader.parent.session_leader.start': {
                dashed_name: 'process-session-leader-parent-session-leader-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.session_leader.parent.session_leader.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.session_leader.parent.session_leader.vpid': {
                dashed_name: 'process-session-leader-parent-session-leader-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.session_leader.parent.session_leader.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.session_leader.parent.start': {
                dashed_name: 'process-session-leader-parent-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.session_leader.parent.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.session_leader.parent.vpid': {
                dashed_name: 'process-session-leader-parent-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.session_leader.parent.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.session_leader.pid': {
                dashed_name: 'process-session-leader-pid',
                description: 'Process id.',
                example: 4242,
                flat_name: 'process.session_leader.pid',
                format: 'string',
                level: 'core',
                name: 'pid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Process id.',
                type: 'long',
            },
            'process.session_leader.real_group.id': {
                dashed_name: 'process-session-leader-real-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.session_leader.real_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.session_leader.real_group.name': {
                dashed_name: 'process-session-leader-real-group-name',
                description: 'Name of the group.',
                flat_name: 'process.session_leader.real_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.session_leader.real_user.id': {
                dashed_name: 'process-session-leader-real-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.session_leader.real_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.session_leader.real_user.name': {
                dashed_name: 'process-session-leader-real-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.session_leader.real_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.session_leader.real_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.session_leader.same_as_process': {
                dashed_name: 'process-session-leader-same-as-process',
                description: "This boolean is used to identify if a leader process is the same as the top level process.\nFor example, if `process.group_leader.same_as_process = true`, it means the process event in question is the leader of its process group. Details under `process.*` like `pid` would be the same under `process.group_leader.*` The same applies for both `process.session_leader` and `process.entry_leader`.\nThis field exists to the benefit of EQL and other rule engines since it's not possible to compare equality between two fields in a single document. e.g `process.entity_id` = `process.group_leader.entity_id` (top level process is the process group leader) OR `process.entity_id` = `process.entry_leader.entity_id` (top level process is the entry session leader)\nInstead these rules could be written like: `process.group_leader.same_as_process: true` OR `process.entry_leader.same_as_process: true`\nNote: This field is only set on `process.entry_leader`, `process.session_leader` and `process.group_leader`.",
                example: true,
                flat_name: 'process.session_leader.same_as_process',
                level: 'extended',
                name: 'same_as_process',
                normalize: [],
                original_fieldset: 'process',
                short: 'This boolean is used to identify if a leader process is the same as the top level process.',
                type: 'boolean',
            },
            'process.session_leader.saved_group.id': {
                dashed_name: 'process-session-leader-saved-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.session_leader.saved_group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.session_leader.saved_group.name': {
                dashed_name: 'process-session-leader-saved-group-name',
                description: 'Name of the group.',
                flat_name: 'process.session_leader.saved_group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.session_leader.saved_user.id': {
                dashed_name: 'process-session-leader-saved-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.session_leader.saved_user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.session_leader.saved_user.name': {
                dashed_name: 'process-session-leader-saved-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.session_leader.saved_user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.session_leader.saved_user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.session_leader.start': {
                dashed_name: 'process-session-leader-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.session_leader.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                original_fieldset: 'process',
                short: 'The time the process started.',
                type: 'date',
            },
            'process.session_leader.supplemental_groups.id': {
                dashed_name: 'process-session-leader-supplemental-groups-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.session_leader.supplemental_groups.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.session_leader.supplemental_groups.name': {
                dashed_name: 'process-session-leader-supplemental-groups-name',
                description: 'Name of the group.',
                flat_name: 'process.session_leader.supplemental_groups.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.session_leader.tty': {
                dashed_name: 'process-session-leader-tty',
                description: 'Information about the controlling TTY device. If set, the process belongs to an interactive session.',
                flat_name: 'process.session_leader.tty',
                level: 'extended',
                name: 'tty',
                normalize: [],
                original_fieldset: 'process',
                short: 'Information about the controlling TTY device.',
                type: 'object',
            },
            'process.session_leader.tty.char_device.major': {
                dashed_name: 'process-session-leader-tty-char-device-major',
                description: 'The major number identifies the driver associated with the device. The character device\'s major and minor numbers can be algorithmically combined to produce the more familiar terminal identifiers such as "ttyS0" and "pts/0". For more details, please refer to the Linux kernel documentation.',
                example: 4,
                flat_name: 'process.session_leader.tty.char_device.major',
                level: 'extended',
                name: 'tty.char_device.major',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's major number.",
                type: 'long',
            },
            'process.session_leader.tty.char_device.minor': {
                dashed_name: 'process-session-leader-tty-char-device-minor',
                description: 'The minor number is used only by the driver specified by the major number; other parts of the kernel don’t use it, and merely pass it along to the driver. It is common for a driver to control several devices; the minor number provides a way for the driver to differentiate among them.',
                example: 1,
                flat_name: 'process.session_leader.tty.char_device.minor',
                level: 'extended',
                name: 'tty.char_device.minor',
                normalize: [],
                original_fieldset: 'process',
                short: "The TTY character device's minor number.",
                type: 'long',
            },
            'process.session_leader.user.id': {
                dashed_name: 'process-session-leader-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.session_leader.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.session_leader.user.name': {
                dashed_name: 'process-session-leader-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.session_leader.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.session_leader.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.session_leader.vpid': {
                dashed_name: 'process-session-leader-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.session_leader.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                original_fieldset: 'process',
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.session_leader.working_directory': {
                dashed_name: 'process-session-leader-working-directory',
                description: 'The working directory of the process.',
                example: '/home/alice',
                flat_name: 'process.session_leader.working_directory',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.session_leader.working_directory.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'working_directory',
                normalize: [],
                original_fieldset: 'process',
                short: 'The working directory of the process.',
                type: 'keyword',
            },
            'process.start': {
                dashed_name: 'process-start',
                description: 'The time the process started.',
                example: '2016-05-23T08:05:34.853Z',
                flat_name: 'process.start',
                level: 'extended',
                name: 'start',
                normalize: [],
                short: 'The time the process started.',
                type: 'date',
            },
            'process.supplemental_groups.id': {
                dashed_name: 'process-supplemental-groups-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'process.supplemental_groups.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'process.supplemental_groups.name': {
                dashed_name: 'process-supplemental-groups-name',
                description: 'Name of the group.',
                flat_name: 'process.supplemental_groups.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'process.thread.capabilities.effective': {
                dashed_name: 'process-thread-capabilities-effective',
                description: 'This is the set of capabilities used by the kernel to perform permission checks for the thread.',
                example: '["CAP_BPF", "CAP_SYS_ADMIN"]',
                flat_name: 'process.thread.capabilities.effective',
                ignore_above: 1024,
                level: 'extended',
                name: 'thread.capabilities.effective',
                normalize: ['array'],
                pattern: '^(CAP_[A-Z_]+|\\d+)$',
                short: 'Array of capabilities used for permission checks.',
                type: 'keyword',
            },
            'process.thread.capabilities.permitted': {
                dashed_name: 'process-thread-capabilities-permitted',
                description: 'This is a limiting superset for the effective capabilities that the thread may assume.',
                example: '["CAP_BPF", "CAP_SYS_ADMIN"]',
                flat_name: 'process.thread.capabilities.permitted',
                ignore_above: 1024,
                level: 'extended',
                name: 'thread.capabilities.permitted',
                normalize: ['array'],
                pattern: '^(CAP_[A-Z_]+|\\d+)$',
                short: 'Array of capabilities a thread could assume.',
                type: 'keyword',
            },
            'process.thread.id': {
                dashed_name: 'process-thread-id',
                description: 'Thread ID.',
                example: 4242,
                flat_name: 'process.thread.id',
                format: 'string',
                level: 'extended',
                name: 'thread.id',
                normalize: [],
                short: 'Thread ID.',
                type: 'long',
            },
            'process.thread.name': {
                dashed_name: 'process-thread-name',
                description: 'Thread name.',
                example: 'thread-0',
                flat_name: 'process.thread.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'thread.name',
                normalize: [],
                short: 'Thread name.',
                type: 'keyword',
            },
            'process.title': {
                dashed_name: 'process-title',
                description: 'Process title.\nThe proctitle, some times the same as process name. Can also be different: for example a browser setting its title to the web page currently opened.',
                flat_name: 'process.title',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.title.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'title',
                normalize: [],
                short: 'Process title.',
                type: 'keyword',
            },
            'process.tty': {
                dashed_name: 'process-tty',
                description: 'Information about the controlling TTY device. If set, the process belongs to an interactive session.',
                flat_name: 'process.tty',
                level: 'extended',
                name: 'tty',
                normalize: [],
                short: 'Information about the controlling TTY device.',
                type: 'object',
            },
            'process.tty.char_device.major': {
                dashed_name: 'process-tty-char-device-major',
                description: 'The major number identifies the driver associated with the device. The character device\'s major and minor numbers can be algorithmically combined to produce the more familiar terminal identifiers such as "ttyS0" and "pts/0". For more details, please refer to the Linux kernel documentation.',
                example: 4,
                flat_name: 'process.tty.char_device.major',
                level: 'extended',
                name: 'tty.char_device.major',
                normalize: [],
                short: "The TTY character device's major number.",
                type: 'long',
            },
            'process.tty.char_device.minor': {
                dashed_name: 'process-tty-char-device-minor',
                description: 'The minor number is used only by the driver specified by the major number; other parts of the kernel don’t use it, and merely pass it along to the driver. It is common for a driver to control several devices; the minor number provides a way for the driver to differentiate among them.',
                example: 1,
                flat_name: 'process.tty.char_device.minor',
                level: 'extended',
                name: 'tty.char_device.minor',
                normalize: [],
                short: "The TTY character device's minor number.",
                type: 'long',
            },
            'process.tty.columns': {
                dashed_name: 'process-tty-columns',
                description: "The number of character columns per line. e.g terminal width\nTerminal sizes can change, so this value reflects the maximum value for a given IO event. i.e. where event.action = 'text_output'",
                example: 80,
                flat_name: 'process.tty.columns',
                level: 'extended',
                name: 'tty.columns',
                normalize: [],
                short: 'The number of character columns per line. e.g terminal width',
                type: 'long',
            },
            'process.tty.rows': {
                dashed_name: 'process-tty-rows',
                description: "The number of character rows in the terminal. e.g terminal height\nTerminal sizes can change, so this value reflects the maximum value for a given IO event. i.e. where event.action = 'text_output'",
                example: 24,
                flat_name: 'process.tty.rows',
                level: 'extended',
                name: 'tty.rows',
                normalize: [],
                short: 'The number of character rows in the terminal. e.g terminal height',
                type: 'long',
            },
            'process.uptime': {
                dashed_name: 'process-uptime',
                description: 'Seconds the process has been up.',
                example: 1325,
                flat_name: 'process.uptime',
                level: 'extended',
                name: 'uptime',
                normalize: [],
                short: 'Seconds the process has been up.',
                type: 'long',
            },
            'process.user.id': {
                dashed_name: 'process-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'process.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'process.user.name': {
                dashed_name: 'process-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'process.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'process.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'process.vpid': {
                dashed_name: 'process-vpid',
                description: 'Virtual process id.\nThe process id within a pid namespace. This is not necessarily unique across all processes on the host but it is unique within the process namespace that the process exists within.',
                example: 4242,
                flat_name: 'process.vpid',
                format: 'string',
                level: 'core',
                name: 'vpid',
                normalize: [],
                short: 'Virtual process id.',
                type: 'long',
            },
            'process.working_directory': {
                dashed_name: 'process-working-directory',
                description: 'The working directory of the process.',
                example: '/home/alice',
                flat_name: 'process.working_directory',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'process.working_directory.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'working_directory',
                normalize: [],
                short: 'The working directory of the process.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'process',
        nestings: [
            'process.attested_groups',
            'process.attested_user',
            'process.code_signature',
            'process.elf',
            'process.entry_leader',
            'process.entry_leader.parent',
            'process.entry_leader.parent.session_leader',
            'process.entry_meta.source',
            'process.group',
            'process.group_leader',
            'process.hash',
            'process.macho',
            'process.parent',
            'process.parent.group_leader',
            'process.pe',
            'process.previous',
            'process.real_group',
            'process.real_user',
            'process.saved_group',
            'process.saved_user',
            'process.session_leader',
            'process.session_leader.parent',
            'process.session_leader.parent.session_leader',
            'process.supplemental_groups',
            'process.user',
        ],
        prefix: 'process.',
        reusable: {
            expected: [
                {
                    as: 'parent',
                    at: 'process',
                    full: 'process.parent',
                    short_override: 'Information about the parent process.',
                },
                {
                    as: 'entry_leader',
                    at: 'process',
                    full: 'process.entry_leader',
                    short_override: 'First process from terminal or remote access via SSH, SSM, etc OR a service directly started by the init process.',
                },
                {
                    as: 'session_leader',
                    at: 'process',
                    full: 'process.session_leader',
                    short_override: 'Often the same as entry_leader. When it differs, it represents a session started within another session. e.g. using tmux',
                },
                {
                    as: 'group_leader',
                    at: 'process',
                    full: 'process.group_leader',
                    short_override: 'Information about the process group leader. In some cases this may be the same as the top level process.',
                },
                {
                    as: 'group_leader',
                    at: 'process.parent',
                    full: 'process.parent.group_leader',
                    short_override: "Information about the parent's process group leader. Only pid, start and entity_id fields are set.",
                },
                {
                    as: 'parent',
                    at: 'process.entry_leader',
                    full: 'process.entry_leader.parent',
                    short_override: "Information about the entry leader's parent process. Only pid, start and entity_id fields are set.",
                },
                {
                    as: 'parent',
                    at: 'process.session_leader',
                    full: 'process.session_leader.parent',
                    short_override: "Information about the session leader's parent process. Only pid, start and entity_id fields are set.",
                },
                {
                    as: 'session_leader',
                    at: 'process.entry_leader.parent',
                    full: 'process.entry_leader.parent.session_leader',
                    short_override: 'Information about the parent session of the entry leader. Only pid, start and entity_id fields are set.',
                },
                {
                    as: 'session_leader',
                    at: 'process.session_leader.parent',
                    full: 'process.session_leader.parent.session_leader',
                    short_override: 'Information about the parent session of the session leader. Only pid, start and entity_id fields are set.',
                },
                {
                    as: 'previous',
                    at: 'process',
                    full: 'process.previous',
                    normalize: ['array'],
                    short_override: 'An array of previous executions for the process, including the initial fork. Only executable and args are set.',
                },
            ],
            top_level: true,
        },
        reused_here: [
            {
                full: 'process.group',
                schema_name: 'group',
                short: 'The effective group (egid).',
            },
            {
                full: 'process.real_group',
                schema_name: 'group',
                short: 'The real group (rgid).',
            },
            {
                full: 'process.saved_group',
                schema_name: 'group',
                short: 'The saved group (sgid).',
            },
            {
                full: 'process.supplemental_groups',
                normalize: ['array'],
                schema_name: 'group',
                short: 'An array of supplemental groups.',
            },
            {
                beta: 'Reusing the `group` fields in this location is currently considered beta.',
                full: 'process.attested_groups',
                normalize: ['array'],
                schema_name: 'group',
                short: 'The externally attested groups based on an external source such as the Kube API.',
            },
            {
                full: 'process.hash',
                schema_name: 'hash',
                short: 'Hashes, usually file hashes.',
            },
            {
                full: 'process.pe',
                schema_name: 'pe',
                short: 'These fields contain Windows Portable Executable (PE) metadata.',
            },
            {
                full: 'process.code_signature',
                schema_name: 'code_signature',
                short: 'These fields contain information about binary code signatures.',
            },
            {
                beta: 'This field reuse is beta and subject to change.',
                full: 'process.elf',
                schema_name: 'elf',
                short: 'These fields contain Linux Executable Linkable Format (ELF) metadata.',
            },
            {
                beta: 'This field reuse is beta and subject to change.',
                full: 'process.macho',
                schema_name: 'macho',
                short: 'These fields contain Mac OS Mach Object file format (Mach-O) metadata.',
            },
            {
                full: 'process.entry_meta.source',
                schema_name: 'source',
                short: 'Remote client information such as ip, port and geo location.',
            },
            {
                full: 'process.user',
                schema_name: 'user',
                short: 'The effective user (euid).',
            },
            {
                full: 'process.saved_user',
                schema_name: 'user',
                short: 'The saved user (suid).',
            },
            {
                full: 'process.real_user',
                schema_name: 'user',
                short: 'The real user (ruid). Identifies the real owner of the process.',
            },
            {
                beta: 'Reusing the `user` fields in this location is currently considered beta.',
                full: 'process.attested_user',
                schema_name: 'user',
                short: 'The externally attested user based on an external source such as the Kube API.',
            },
            {
                full: 'process.parent',
                schema_name: 'process',
                short: 'Information about the parent process.',
            },
            {
                full: 'process.entry_leader',
                schema_name: 'process',
                short: 'First process from terminal or remote access via SSH, SSM, etc OR a service directly started by the init process.',
            },
            {
                full: 'process.session_leader',
                schema_name: 'process',
                short: 'Often the same as entry_leader. When it differs, it represents a session started within another session. e.g. using tmux',
            },
            {
                full: 'process.group_leader',
                schema_name: 'process',
                short: 'Information about the process group leader. In some cases this may be the same as the top level process.',
            },
            {
                full: 'process.parent.group_leader',
                schema_name: 'process',
                short: "Information about the parent's process group leader. Only pid, start and entity_id fields are set.",
            },
            {
                full: 'process.entry_leader.parent',
                schema_name: 'process',
                short: "Information about the entry leader's parent process. Only pid, start and entity_id fields are set.",
            },
            {
                full: 'process.session_leader.parent',
                schema_name: 'process',
                short: "Information about the session leader's parent process. Only pid, start and entity_id fields are set.",
            },
            {
                full: 'process.entry_leader.parent.session_leader',
                schema_name: 'process',
                short: 'Information about the parent session of the entry leader. Only pid, start and entity_id fields are set.',
            },
            {
                full: 'process.session_leader.parent.session_leader',
                schema_name: 'process',
                short: 'Information about the parent session of the session leader. Only pid, start and entity_id fields are set.',
            },
            {
                full: 'process.previous',
                normalize: ['array'],
                schema_name: 'process',
                short: 'An array of previous executions for the process, including the initial fork. Only executable and args are set.',
            },
        ],
        short: 'These fields contain information about a process.',
        title: 'Process',
        type: 'group',
    },
    registry: {
        description: 'Fields related to Windows Registry operations.',
        fields: {
            'registry.data.bytes': {
                dashed_name: 'registry-data-bytes',
                description: 'Original bytes written with base64 encoding.\nFor Windows registry operations, such as SetValueEx and RegQueryValueEx, this corresponds to the data pointed by `lp_data`. This is optional but provides better recoverability and should be populated for REG_BINARY encoded values.',
                example: 'ZQBuAC0AVQBTAAAAZQBuAAAAAAA=',
                flat_name: 'registry.data.bytes',
                ignore_above: 1024,
                level: 'extended',
                name: 'data.bytes',
                normalize: [],
                short: 'Original bytes written with base64 encoding.',
                type: 'keyword',
            },
            'registry.data.strings': {
                dashed_name: 'registry-data-strings',
                description: 'Content when writing string types.\nPopulated as an array when writing string data to the registry. For single string registry types (REG_SZ, REG_EXPAND_SZ), this should be an array with one string. For sequences of string with REG_MULTI_SZ, this array will be variable length. For numeric data, such as REG_DWORD and REG_QWORD, this should be populated with the decimal representation (e.g `"1"`).',
                example: '["C:\\rta\\red_ttp\\bin\\myapp.exe"]',
                flat_name: 'registry.data.strings',
                level: 'core',
                name: 'data.strings',
                normalize: ['array'],
                short: 'List of strings representing what was written to the registry.',
                type: 'wildcard',
            },
            'registry.data.type': {
                dashed_name: 'registry-data-type',
                description: 'Standard registry type for encoding contents',
                example: 'REG_SZ',
                flat_name: 'registry.data.type',
                ignore_above: 1024,
                level: 'core',
                name: 'data.type',
                normalize: [],
                short: 'Standard registry type for encoding contents',
                type: 'keyword',
            },
            'registry.hive': {
                dashed_name: 'registry-hive',
                description: 'Abbreviated name for the hive.',
                example: 'HKLM',
                flat_name: 'registry.hive',
                ignore_above: 1024,
                level: 'core',
                name: 'hive',
                normalize: [],
                short: 'Abbreviated name for the hive.',
                type: 'keyword',
            },
            'registry.key': {
                dashed_name: 'registry-key',
                description: 'Hive-relative path of keys.',
                example: 'SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Image File Execution Options\\winword.exe',
                flat_name: 'registry.key',
                ignore_above: 1024,
                level: 'core',
                name: 'key',
                normalize: [],
                short: 'Hive-relative path of keys.',
                type: 'keyword',
            },
            'registry.path': {
                dashed_name: 'registry-path',
                description: 'Full path, including hive, key and value',
                example: 'HKLM\\SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Image File Execution Options\\winword.exe\\Debugger',
                flat_name: 'registry.path',
                ignore_above: 1024,
                level: 'core',
                name: 'path',
                normalize: [],
                short: 'Full path, including hive, key and value',
                type: 'keyword',
            },
            'registry.value': {
                dashed_name: 'registry-value',
                description: 'Name of the value written.',
                example: 'Debugger',
                flat_name: 'registry.value',
                ignore_above: 1024,
                level: 'core',
                name: 'value',
                normalize: [],
                short: 'Name of the value written.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'registry',
        prefix: 'registry.',
        reusable: {
            expected: [
                {
                    as: 'registry',
                    at: 'threat.indicator',
                    full: 'threat.indicator.registry',
                },
                {
                    as: 'registry',
                    at: 'threat.enrichments.indicator',
                    full: 'threat.enrichments.indicator.registry',
                },
            ],
            top_level: true,
        },
        short: 'Fields related to Windows Registry operations.',
        title: 'Registry',
        type: 'group',
    },
    related: {
        description: 'This field set is meant to facilitate pivoting around a piece of data.\nSome pieces of information can be seen in many places in an ECS event. To facilitate searching for them, store an array of all seen values to their corresponding field in `related.`.\nA concrete example is IP addresses, which can be under host, observer, source, destination, client, server, and network.forwarded_ip. If you append all IPs to `related.ip`, you can then search for a given IP trivially, no matter where it appeared, by querying `related.ip:192.0.2.15`.',
        fields: {
            'related.hash': {
                dashed_name: 'related-hash',
                description: "All the hashes seen on your event. Populating this field, then using it to search for hashes can help in situations where you're unsure what the hash algorithm is (and therefore which key name to search).",
                flat_name: 'related.hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'hash',
                normalize: ['array'],
                short: 'All the hashes seen on your event.',
                type: 'keyword',
            },
            'related.hosts': {
                dashed_name: 'related-hosts',
                description: 'All hostnames or other host identifiers seen on your event. Example identifiers include FQDNs, domain names, workstation names, or aliases.',
                flat_name: 'related.hosts',
                ignore_above: 1024,
                level: 'extended',
                name: 'hosts',
                normalize: ['array'],
                short: 'All the host identifiers seen on your event.',
                type: 'keyword',
            },
            'related.ip': {
                dashed_name: 'related-ip',
                description: 'All of the IPs seen on your event.',
                flat_name: 'related.ip',
                level: 'extended',
                name: 'ip',
                normalize: ['array'],
                short: 'All of the IPs seen on your event.',
                type: 'ip',
            },
            'related.user': {
                dashed_name: 'related-user',
                description: 'All the user names or other user identifiers seen on the event.',
                flat_name: 'related.user',
                ignore_above: 1024,
                level: 'extended',
                name: 'user',
                normalize: ['array'],
                short: 'All the user names or other user identifiers seen on the event.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'related',
        prefix: 'related.',
        short: 'Fields meant to facilitate pivoting around a piece of data.',
        title: 'Related',
        type: 'group',
    },
    risk: {
        beta: 'These fields are in beta and are subject to change.',
        description: 'Fields for describing risk score and risk level of entities such as hosts and users.  These fields are not allowed to be nested under `event.*`. Please continue to use  `event.risk_score` and `event.risk_score_norm` for event risk.',
        fields: {
            'risk.calculated_level': {
                dashed_name: 'risk-calculated-level',
                description: 'A risk classification level calculated by an internal system as part of entity analytics and entity risk scoring.',
                example: 'High',
                flat_name: 'risk.calculated_level',
                ignore_above: 1024,
                level: 'extended',
                name: 'calculated_level',
                normalize: [],
                short: 'A risk classification level calculated by an internal system as part of entity analytics and entity risk scoring.',
                type: 'keyword',
            },
            'risk.calculated_score': {
                dashed_name: 'risk-calculated-score',
                description: 'A risk classification score calculated by an internal system as part of entity analytics and entity risk scoring.',
                example: 880.73,
                flat_name: 'risk.calculated_score',
                level: 'extended',
                name: 'calculated_score',
                normalize: [],
                short: 'A risk classification score calculated by an internal system as part of entity analytics and entity risk scoring.',
                type: 'float',
            },
            'risk.calculated_score_norm': {
                dashed_name: 'risk-calculated-score-norm',
                description: 'A risk classification score calculated by an internal system as part of entity analytics and entity risk scoring, and normalized to a range of 0 to 100.',
                example: 88.73,
                flat_name: 'risk.calculated_score_norm',
                level: 'extended',
                name: 'calculated_score_norm',
                normalize: [],
                short: 'A normalized risk score calculated by an internal system.',
                type: 'float',
            },
            'risk.static_level': {
                dashed_name: 'risk-static-level',
                description: 'A risk classification level obtained from outside the system, such as from some external Threat Intelligence Platform.',
                example: 'High',
                flat_name: 'risk.static_level',
                ignore_above: 1024,
                level: 'extended',
                name: 'static_level',
                normalize: [],
                short: 'A risk classification level obtained from outside the system, such as from some external Threat Intelligence Platform.',
                type: 'keyword',
            },
            'risk.static_score': {
                dashed_name: 'risk-static-score',
                description: 'A risk classification score obtained from outside the system, such as from some external Threat Intelligence Platform.',
                example: 830,
                flat_name: 'risk.static_score',
                level: 'extended',
                name: 'static_score',
                normalize: [],
                short: 'A risk classification score obtained from outside the system, such as from some external Threat Intelligence Platform.',
                type: 'float',
            },
            'risk.static_score_norm': {
                dashed_name: 'risk-static-score-norm',
                description: 'A risk classification score obtained from outside the system, such as from some external Threat Intelligence Platform, and normalized to a range of 0 to 100.',
                example: 83,
                flat_name: 'risk.static_score_norm',
                level: 'extended',
                name: 'static_score_norm',
                normalize: [],
                short: 'A normalized risk score calculated by an external system.',
                type: 'float',
            },
        },
        group: 2,
        name: 'risk',
        prefix: 'risk.',
        reusable: {
            expected: [
                { as: 'risk', at: 'host', full: 'host.risk' },
                { as: 'risk', at: 'user', full: 'user.risk' },
            ],
            top_level: false,
        },
        short: 'Fields for describing risk score and level.',
        title: 'Risk information',
        type: 'group',
    },
    rule: {
        description: 'Rule fields are used to capture the specifics of any observer or agent rules that generate alerts or other notable events.\nExamples of data sources that would populate the rule fields include: network admission control platforms, network or host IDS/IPS, network firewalls, web application firewalls, url filters, endpoint detection and response (EDR) systems, etc.',
        fields: {
            'rule.author': {
                dashed_name: 'rule-author',
                description: 'Name, organization, or pseudonym of the author or authors who created the rule used to generate this event.',
                example: '["Star-Lord"]',
                flat_name: 'rule.author',
                ignore_above: 1024,
                level: 'extended',
                name: 'author',
                normalize: ['array'],
                short: 'Rule author',
                type: 'keyword',
            },
            'rule.category': {
                dashed_name: 'rule-category',
                description: 'A categorization value keyword used by the entity using the rule for detection of this event.',
                example: 'Attempted Information Leak',
                flat_name: 'rule.category',
                ignore_above: 1024,
                level: 'extended',
                name: 'category',
                normalize: [],
                short: 'Rule category',
                type: 'keyword',
            },
            'rule.description': {
                dashed_name: 'rule-description',
                description: 'The description of the rule generating the event.',
                example: 'Block requests to public DNS over HTTPS / TLS protocols',
                flat_name: 'rule.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                short: 'Rule description',
                type: 'keyword',
            },
            'rule.id': {
                dashed_name: 'rule-id',
                description: 'A rule ID that is unique within the scope of an agent, observer, or other entity using the rule for detection of this event.',
                example: 101,
                flat_name: 'rule.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                short: 'Rule ID',
                type: 'keyword',
            },
            'rule.license': {
                dashed_name: 'rule-license',
                description: 'Name of the license under which the rule used to generate this event is made available.',
                example: 'Apache 2.0',
                flat_name: 'rule.license',
                ignore_above: 1024,
                level: 'extended',
                name: 'license',
                normalize: [],
                short: 'Rule license',
                type: 'keyword',
            },
            'rule.name': {
                dashed_name: 'rule-name',
                description: 'The name of the rule or signature generating the event.',
                example: 'BLOCK_DNS_over_TLS',
                flat_name: 'rule.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                short: 'Rule name',
                type: 'keyword',
            },
            'rule.reference': {
                dashed_name: 'rule-reference',
                description: "Reference URL to additional information about the rule used to generate this event.\nThe URL can point to the vendor's documentation about the rule. If that's not available, it can also be a link to a more general page describing this type of alert.",
                example: 'https://en.wikipedia.org/wiki/DNS_over_TLS',
                flat_name: 'rule.reference',
                ignore_above: 1024,
                level: 'extended',
                name: 'reference',
                normalize: [],
                short: 'Rule reference URL',
                type: 'keyword',
            },
            'rule.ruleset': {
                dashed_name: 'rule-ruleset',
                description: 'Name of the ruleset, policy, group, or parent category in which the rule used to generate this event is a member.',
                example: 'Standard_Protocol_Filters',
                flat_name: 'rule.ruleset',
                ignore_above: 1024,
                level: 'extended',
                name: 'ruleset',
                normalize: [],
                short: 'Rule ruleset',
                type: 'keyword',
            },
            'rule.uuid': {
                dashed_name: 'rule-uuid',
                description: 'A rule ID that is unique within the scope of a set or group of agents, observers, or other entities using the rule for detection of this event.',
                example: 1100110011,
                flat_name: 'rule.uuid',
                ignore_above: 1024,
                level: 'extended',
                name: 'uuid',
                normalize: [],
                short: 'Rule UUID',
                type: 'keyword',
            },
            'rule.version': {
                dashed_name: 'rule-version',
                description: 'The version / revision of the rule being used for analysis.',
                example: 1.1,
                flat_name: 'rule.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'version',
                normalize: [],
                short: 'Rule version',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'rule',
        prefix: 'rule.',
        short: 'Fields to capture details about rules used to generate alerts or other notable events.',
        title: 'Rule',
        type: 'group',
    },
    server: {
        description: 'A Server is defined as the responder in a network connection for events regarding sessions, connections, or bidirectional flow records.\nFor TCP events, the server is the receiver of the initial SYN packet(s) of the TCP connection. For other protocols, the server is generally the responder in the network transaction. Some systems actually use the term "responder" to refer the server in TCP connections. The server fields describe details about the system acting as the server in the network event. Server fields are usually populated in conjunction with client fields. Server fields are generally not populated for packet-level events.\nClient / server representations can add semantic context to an exchange, which is helpful to visualize the data in certain situations. If your context falls in that category, you should still ensure that source and destination are filled appropriately.',
        fields: {
            'server.address': {
                dashed_name: 'server-address',
                description: 'Some event server addresses are defined ambiguously. The event will sometimes list an IP, a domain or a unix socket.  You should always store the raw address in the `.address` field.\nThen it should be duplicated to `.ip` or `.domain`, depending on which one it is.',
                flat_name: 'server.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'address',
                normalize: [],
                short: 'Server network address.',
                type: 'keyword',
            },
            'server.as.number': {
                dashed_name: 'server-as-number',
                description: 'Unique number allocated to the autonomous system. The autonomous system number (ASN) uniquely identifies each network on the Internet.',
                example: 15169,
                flat_name: 'server.as.number',
                level: 'extended',
                name: 'number',
                normalize: [],
                original_fieldset: 'as',
                short: 'Unique number allocated to the autonomous system.',
                type: 'long',
            },
            'server.as.organization.name': {
                dashed_name: 'server-as-organization-name',
                description: 'Organization name.',
                example: 'Google LLC',
                flat_name: 'server.as.organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'server.as.organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'organization.name',
                normalize: [],
                original_fieldset: 'as',
                short: 'Organization name.',
                type: 'keyword',
            },
            'server.bytes': {
                dashed_name: 'server-bytes',
                description: 'Bytes sent from the server to the client.',
                example: 184,
                flat_name: 'server.bytes',
                format: 'bytes',
                level: 'core',
                name: 'bytes',
                normalize: [],
                short: 'Bytes sent from the server to the client.',
                type: 'long',
            },
            'server.domain': {
                dashed_name: 'server-domain',
                description: 'The domain name of the server system.\nThis value may be a host name, a fully qualified domain name, or another host naming format. The value may derive from the original event or be added from enrichment.',
                example: 'foo.example.com',
                flat_name: 'server.domain',
                ignore_above: 1024,
                level: 'core',
                name: 'domain',
                normalize: [],
                short: 'The domain name of the server.',
                type: 'keyword',
            },
            'server.geo.city_name': {
                dashed_name: 'server-geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'server.geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'City name.',
                type: 'keyword',
            },
            'server.geo.continent_code': {
                dashed_name: 'server-geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'server.geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Continent code.',
                type: 'keyword',
            },
            'server.geo.continent_name': {
                dashed_name: 'server-geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'server.geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'server.geo.country_iso_code': {
                dashed_name: 'server-geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'server.geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'server.geo.country_name': {
                dashed_name: 'server-geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'server.geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country name.',
                type: 'keyword',
            },
            'server.geo.location': {
                dashed_name: 'server-geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'server.geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'server.geo.name': {
                dashed_name: 'server-geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'server.geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'server.geo.postal_code': {
                dashed_name: 'server-geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'server.geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Postal code.',
                type: 'keyword',
            },
            'server.geo.region_iso_code': {
                dashed_name: 'server-geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'server.geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'server.geo.region_name': {
                dashed_name: 'server-geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'server.geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region name.',
                type: 'keyword',
            },
            'server.geo.timezone': {
                dashed_name: 'server-geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'server.geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Time zone.',
                type: 'keyword',
            },
            'server.ip': {
                dashed_name: 'server-ip',
                description: 'IP address of the server (IPv4 or IPv6).',
                flat_name: 'server.ip',
                level: 'core',
                name: 'ip',
                normalize: [],
                short: 'IP address of the server.',
                type: 'ip',
            },
            'server.mac': {
                dashed_name: 'server-mac',
                description: 'MAC address of the server.\nThe notation format from RFC 7042 is suggested: Each octet (that is, 8-bit byte) is represented by two [uppercase] hexadecimal digits giving the value of the octet as an unsigned integer. Successive octets are separated by a hyphen.',
                example: '00-00-5E-00-53-23',
                flat_name: 'server.mac',
                ignore_above: 1024,
                level: 'core',
                name: 'mac',
                normalize: [],
                pattern: '^[A-F0-9]{2}(-[A-F0-9]{2}){5,}$',
                short: 'MAC address of the server.',
                type: 'keyword',
            },
            'server.nat.ip': {
                dashed_name: 'server-nat-ip',
                description: 'Translated ip of destination based NAT sessions (e.g. internet to private DMZ)\nTypically used with load balancers, firewalls, or routers.',
                flat_name: 'server.nat.ip',
                level: 'extended',
                name: 'nat.ip',
                normalize: [],
                short: 'Server NAT ip',
                type: 'ip',
            },
            'server.nat.port': {
                dashed_name: 'server-nat-port',
                description: 'Translated port of destination based NAT sessions (e.g. internet to private DMZ)\nTypically used with load balancers, firewalls, or routers.',
                flat_name: 'server.nat.port',
                format: 'string',
                level: 'extended',
                name: 'nat.port',
                normalize: [],
                short: 'Server NAT port',
                type: 'long',
            },
            'server.packets': {
                dashed_name: 'server-packets',
                description: 'Packets sent from the server to the client.',
                example: 12,
                flat_name: 'server.packets',
                level: 'core',
                name: 'packets',
                normalize: [],
                short: 'Packets sent from the server to the client.',
                type: 'long',
            },
            'server.port': {
                dashed_name: 'server-port',
                description: 'Port of the server.',
                flat_name: 'server.port',
                format: 'string',
                level: 'core',
                name: 'port',
                normalize: [],
                short: 'Port of the server.',
                type: 'long',
            },
            'server.registered_domain': {
                dashed_name: 'server-registered-domain',
                description: 'The highest registered server domain, stripped of the subdomain.\nFor example, the registered domain for "foo.example.com" is "example.com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last two labels will not work well for TLDs such as "co.uk".',
                example: 'example.com',
                flat_name: 'server.registered_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'registered_domain',
                normalize: [],
                short: 'The highest registered server domain, stripped of the subdomain.',
                type: 'keyword',
            },
            'server.subdomain': {
                dashed_name: 'server-subdomain',
                description: 'The subdomain portion of a fully qualified domain name includes all of the names except the host name under the registered_domain.  In a partially qualified domain, or if the the qualification level of the full name cannot be determined, subdomain contains all of the names below the registered domain.\nFor example the subdomain portion of "www.east.mydomain.co.uk" is "east". If the domain has multiple levels of subdomain, such as "sub2.sub1.example.com", the subdomain field should contain "sub2.sub1", with no trailing period.',
                example: 'east',
                flat_name: 'server.subdomain',
                ignore_above: 1024,
                level: 'extended',
                name: 'subdomain',
                normalize: [],
                short: 'The subdomain of the domain.',
                type: 'keyword',
            },
            'server.top_level_domain': {
                dashed_name: 'server-top-level-domain',
                description: 'The effective top level domain (eTLD), also known as the domain suffix, is the last part of the domain name. For example, the top level domain for example.com is "com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last label will not work well for effective TLDs such as "co.uk".',
                example: 'co.uk',
                flat_name: 'server.top_level_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'top_level_domain',
                normalize: [],
                short: 'The effective top level domain (com, org, net, co.uk).',
                type: 'keyword',
            },
            'server.user.domain': {
                dashed_name: 'server-user-domain',
                description: 'Name of the directory the user is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'server.user.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'user',
                short: 'Name of the directory the user is a member of.',
                type: 'keyword',
            },
            'server.user.email': {
                dashed_name: 'server-user-email',
                description: 'User email address.',
                flat_name: 'server.user.email',
                ignore_above: 1024,
                level: 'extended',
                name: 'email',
                normalize: [],
                original_fieldset: 'user',
                short: 'User email address.',
                type: 'keyword',
            },
            'server.user.full_name': {
                dashed_name: 'server-user-full-name',
                description: "User's full name, if available.",
                example: 'Albert Einstein',
                flat_name: 'server.user.full_name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'server.user.full_name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'full_name',
                normalize: [],
                original_fieldset: 'user',
                short: "User's full name, if available.",
                type: 'keyword',
            },
            'server.user.group.domain': {
                dashed_name: 'server-user-group-domain',
                description: 'Name of the directory the group is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'server.user.group.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the directory the group is a member of.',
                type: 'keyword',
            },
            'server.user.group.id': {
                dashed_name: 'server-user-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'server.user.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'server.user.group.name': {
                dashed_name: 'server-user-group-name',
                description: 'Name of the group.',
                flat_name: 'server.user.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'server.user.hash': {
                dashed_name: 'server-user-hash',
                description: 'Unique user hash to correlate information for a user in anonymized form.\nUseful if `user.id` or `user.name` contain confidential information and cannot be used.',
                flat_name: 'server.user.hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'hash',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique user hash to correlate information for a user in anonymized form.',
                type: 'keyword',
            },
            'server.user.id': {
                dashed_name: 'server-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'server.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'server.user.name': {
                dashed_name: 'server-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'server.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'server.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'server.user.roles': {
                dashed_name: 'server-user-roles',
                description: 'Array of user roles at the time of the event.',
                example: '["kibana_admin", "reporting_user"]',
                flat_name: 'server.user.roles',
                ignore_above: 1024,
                level: 'extended',
                name: 'roles',
                normalize: ['array'],
                original_fieldset: 'user',
                short: 'Array of user roles at the time of the event.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'server',
        nestings: ['server.as', 'server.geo', 'server.user'],
        prefix: 'server.',
        reused_here: [
            {
                full: 'server.as',
                schema_name: 'as',
                short: 'Fields describing an Autonomous System (Internet routing prefix).',
            },
            {
                full: 'server.geo',
                schema_name: 'geo',
                short: 'Fields describing a location.',
            },
            {
                full: 'server.user',
                schema_name: 'user',
                short: 'Fields to describe the user relevant to the event.',
            },
        ],
        short: 'Fields about the server side of a network connection, used with client.',
        title: 'Server',
        type: 'group',
    },
    service: {
        description: 'The service fields describe the service for or from which the data was collected.\nThese fields help you find and correlate logs for a specific service and version.',
        fields: {
            'service.address': {
                dashed_name: 'service-address',
                description: 'Address where data about this service was collected from.\nThis should be a URI, network address (ipv4:port or [ipv6]:port) or a resource path (sockets).',
                example: '172.26.0.2:5432',
                flat_name: 'service.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'address',
                normalize: [],
                short: 'Address of this service.',
                type: 'keyword',
            },
            'service.environment': {
                beta: 'This field is beta and subject to change.',
                dashed_name: 'service-environment',
                description: 'Identifies the environment where the service is running.\nIf the same service runs in different environments (production, staging, QA, development, etc.), the environment can identify other instances of the same service. Can also group services and applications from the same environment.',
                example: 'production',
                flat_name: 'service.environment',
                ignore_above: 1024,
                level: 'extended',
                name: 'environment',
                normalize: [],
                short: 'Environment of the service.',
                type: 'keyword',
            },
            'service.ephemeral_id': {
                dashed_name: 'service-ephemeral-id',
                description: 'Ephemeral identifier of this service (if one exists).\nThis id normally changes across restarts, but `service.id` does not.',
                example: '8a4f500f',
                flat_name: 'service.ephemeral_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'ephemeral_id',
                normalize: [],
                short: 'Ephemeral identifier of this service.',
                type: 'keyword',
            },
            'service.id': {
                dashed_name: 'service-id',
                description: 'Unique identifier of the running service. If the service is comprised of many nodes, the `service.id` should be the same for all nodes.\nThis id should uniquely identify the service. This makes it possible to correlate logs and metrics for one specific service, no matter which particular node emitted the event.\nNote that if you need to see the events from one specific host of the service, you should filter on that `host.name` or `host.id` instead.',
                example: 'd37e5ebfe0ae6c4972dbe9f0174a1637bb8247f6',
                flat_name: 'service.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                short: 'Unique identifier of the running service.',
                type: 'keyword',
            },
            'service.name': {
                dashed_name: 'service-name',
                description: 'Name of the service data is collected from.\nThe name of the service is normally user given. This allows for distributed services that run on multiple hosts to correlate the related instances based on the name.\nIn the case of Elasticsearch the `service.name` could contain the cluster name. For Beats the `service.name` is by default a copy of the `service.type` field if no name is specified.',
                example: 'elasticsearch-metrics',
                flat_name: 'service.name',
                ignore_above: 1024,
                level: 'core',
                name: 'name',
                normalize: [],
                short: 'Name of the service.',
                type: 'keyword',
            },
            'service.node.name': {
                dashed_name: 'service-node-name',
                description: "Name of a service node.\nThis allows for two nodes of the same service running on the same host to be differentiated. Therefore, `service.node.name` should typically be unique across nodes of a given service.\nIn the case of Elasticsearch, the `service.node.name` could contain the unique node name within the Elasticsearch cluster. In cases where the service doesn't have the concept of a node name, the host name or container name can be used to distinguish running instances that make up this service. If those do not provide uniqueness (e.g. multiple instances of the service running on the same host) - the node name can be manually set.",
                example: 'instance-0000000016',
                flat_name: 'service.node.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.name',
                normalize: [],
                short: 'Name of the service node.',
                type: 'keyword',
            },
            'service.node.role': {
                dashed_name: 'service-node-role',
                description: 'Deprecated for removal in next major version release. This field will be superseded by `node.roles`.\nRole of a service node.\nThis allows for distinction between different running roles of the same service.\nIn the case of Kibana, the `service.node.role` could be `ui` or `background_tasks`.\nIn the case of Elasticsearch, the `service.node.role` could be `master` or `data`.\nOther services could use this to distinguish between a `web` and `worker` role running as part of the service.',
                example: 'background_tasks',
                flat_name: 'service.node.role',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.role',
                normalize: [],
                short: 'Deprecated role (singular) of the service node.',
                type: 'keyword',
            },
            'service.node.roles': {
                dashed_name: 'service-node-roles',
                description: 'Roles of a service node.\nThis allows for distinction between different running roles of the same service.\nIn the case of Kibana, the `service.node.role` could be `ui` or `background_tasks` or both.\nIn the case of Elasticsearch, the `service.node.role` could be `master` or `data` or both.\nOther services could use this to distinguish between a `web` and `worker` role running as part of the service.',
                example: '["ui", "background_tasks"]',
                flat_name: 'service.node.roles',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.roles',
                normalize: ['array'],
                short: 'Roles of the service node.',
                type: 'keyword',
            },
            'service.origin.address': {
                dashed_name: 'service-origin-address',
                description: 'Address where data about this service was collected from.\nThis should be a URI, network address (ipv4:port or [ipv6]:port) or a resource path (sockets).',
                example: '172.26.0.2:5432',
                flat_name: 'service.origin.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'address',
                normalize: [],
                original_fieldset: 'service',
                short: 'Address of this service.',
                type: 'keyword',
            },
            'service.origin.environment': {
                beta: 'This field is beta and subject to change.',
                dashed_name: 'service-origin-environment',
                description: 'Identifies the environment where the service is running.\nIf the same service runs in different environments (production, staging, QA, development, etc.), the environment can identify other instances of the same service. Can also group services and applications from the same environment.',
                example: 'production',
                flat_name: 'service.origin.environment',
                ignore_above: 1024,
                level: 'extended',
                name: 'environment',
                normalize: [],
                original_fieldset: 'service',
                short: 'Environment of the service.',
                type: 'keyword',
            },
            'service.origin.ephemeral_id': {
                dashed_name: 'service-origin-ephemeral-id',
                description: 'Ephemeral identifier of this service (if one exists).\nThis id normally changes across restarts, but `service.id` does not.',
                example: '8a4f500f',
                flat_name: 'service.origin.ephemeral_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'ephemeral_id',
                normalize: [],
                original_fieldset: 'service',
                short: 'Ephemeral identifier of this service.',
                type: 'keyword',
            },
            'service.origin.id': {
                dashed_name: 'service-origin-id',
                description: 'Unique identifier of the running service. If the service is comprised of many nodes, the `service.id` should be the same for all nodes.\nThis id should uniquely identify the service. This makes it possible to correlate logs and metrics for one specific service, no matter which particular node emitted the event.\nNote that if you need to see the events from one specific host of the service, you should filter on that `host.name` or `host.id` instead.',
                example: 'd37e5ebfe0ae6c4972dbe9f0174a1637bb8247f6',
                flat_name: 'service.origin.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'service',
                short: 'Unique identifier of the running service.',
                type: 'keyword',
            },
            'service.origin.name': {
                dashed_name: 'service-origin-name',
                description: 'Name of the service data is collected from.\nThe name of the service is normally user given. This allows for distributed services that run on multiple hosts to correlate the related instances based on the name.\nIn the case of Elasticsearch the `service.name` could contain the cluster name. For Beats the `service.name` is by default a copy of the `service.type` field if no name is specified.',
                example: 'elasticsearch-metrics',
                flat_name: 'service.origin.name',
                ignore_above: 1024,
                level: 'core',
                name: 'name',
                normalize: [],
                original_fieldset: 'service',
                short: 'Name of the service.',
                type: 'keyword',
            },
            'service.origin.node.name': {
                dashed_name: 'service-origin-node-name',
                description: "Name of a service node.\nThis allows for two nodes of the same service running on the same host to be differentiated. Therefore, `service.node.name` should typically be unique across nodes of a given service.\nIn the case of Elasticsearch, the `service.node.name` could contain the unique node name within the Elasticsearch cluster. In cases where the service doesn't have the concept of a node name, the host name or container name can be used to distinguish running instances that make up this service. If those do not provide uniqueness (e.g. multiple instances of the service running on the same host) - the node name can be manually set.",
                example: 'instance-0000000016',
                flat_name: 'service.origin.node.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.name',
                normalize: [],
                original_fieldset: 'service',
                short: 'Name of the service node.',
                type: 'keyword',
            },
            'service.origin.node.role': {
                dashed_name: 'service-origin-node-role',
                description: 'Deprecated for removal in next major version release. This field will be superseded by `node.roles`.\nRole of a service node.\nThis allows for distinction between different running roles of the same service.\nIn the case of Kibana, the `service.node.role` could be `ui` or `background_tasks`.\nIn the case of Elasticsearch, the `service.node.role` could be `master` or `data`.\nOther services could use this to distinguish between a `web` and `worker` role running as part of the service.',
                example: 'background_tasks',
                flat_name: 'service.origin.node.role',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.role',
                normalize: [],
                original_fieldset: 'service',
                short: 'Deprecated role (singular) of the service node.',
                type: 'keyword',
            },
            'service.origin.node.roles': {
                dashed_name: 'service-origin-node-roles',
                description: 'Roles of a service node.\nThis allows for distinction between different running roles of the same service.\nIn the case of Kibana, the `service.node.role` could be `ui` or `background_tasks` or both.\nIn the case of Elasticsearch, the `service.node.role` could be `master` or `data` or both.\nOther services could use this to distinguish between a `web` and `worker` role running as part of the service.',
                example: '["ui", "background_tasks"]',
                flat_name: 'service.origin.node.roles',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.roles',
                normalize: ['array'],
                original_fieldset: 'service',
                short: 'Roles of the service node.',
                type: 'keyword',
            },
            'service.origin.state': {
                dashed_name: 'service-origin-state',
                description: 'Current state of the service.',
                flat_name: 'service.origin.state',
                ignore_above: 1024,
                level: 'core',
                name: 'state',
                normalize: [],
                original_fieldset: 'service',
                short: 'Current state of the service.',
                type: 'keyword',
            },
            'service.origin.type': {
                dashed_name: 'service-origin-type',
                description: 'The type of the service data is collected from.\nThe type can be used to group and correlate logs and metrics from one service type.\nExample: If logs or metrics are collected from Elasticsearch, `service.type` would be `elasticsearch`.',
                example: 'elasticsearch',
                flat_name: 'service.origin.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: [],
                original_fieldset: 'service',
                short: 'The type of the service.',
                type: 'keyword',
            },
            'service.origin.version': {
                dashed_name: 'service-origin-version',
                description: 'Version of the service the data was collected from.\nThis allows to look at a data set only for a specific version of a service.',
                example: '3.2.4',
                flat_name: 'service.origin.version',
                ignore_above: 1024,
                level: 'core',
                name: 'version',
                normalize: [],
                original_fieldset: 'service',
                short: 'Version of the service.',
                type: 'keyword',
            },
            'service.state': {
                dashed_name: 'service-state',
                description: 'Current state of the service.',
                flat_name: 'service.state',
                ignore_above: 1024,
                level: 'core',
                name: 'state',
                normalize: [],
                short: 'Current state of the service.',
                type: 'keyword',
            },
            'service.target.address': {
                dashed_name: 'service-target-address',
                description: 'Address where data about this service was collected from.\nThis should be a URI, network address (ipv4:port or [ipv6]:port) or a resource path (sockets).',
                example: '172.26.0.2:5432',
                flat_name: 'service.target.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'address',
                normalize: [],
                original_fieldset: 'service',
                short: 'Address of this service.',
                type: 'keyword',
            },
            'service.target.environment': {
                beta: 'This field is beta and subject to change.',
                dashed_name: 'service-target-environment',
                description: 'Identifies the environment where the service is running.\nIf the same service runs in different environments (production, staging, QA, development, etc.), the environment can identify other instances of the same service. Can also group services and applications from the same environment.',
                example: 'production',
                flat_name: 'service.target.environment',
                ignore_above: 1024,
                level: 'extended',
                name: 'environment',
                normalize: [],
                original_fieldset: 'service',
                short: 'Environment of the service.',
                type: 'keyword',
            },
            'service.target.ephemeral_id': {
                dashed_name: 'service-target-ephemeral-id',
                description: 'Ephemeral identifier of this service (if one exists).\nThis id normally changes across restarts, but `service.id` does not.',
                example: '8a4f500f',
                flat_name: 'service.target.ephemeral_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'ephemeral_id',
                normalize: [],
                original_fieldset: 'service',
                short: 'Ephemeral identifier of this service.',
                type: 'keyword',
            },
            'service.target.id': {
                dashed_name: 'service-target-id',
                description: 'Unique identifier of the running service. If the service is comprised of many nodes, the `service.id` should be the same for all nodes.\nThis id should uniquely identify the service. This makes it possible to correlate logs and metrics for one specific service, no matter which particular node emitted the event.\nNote that if you need to see the events from one specific host of the service, you should filter on that `host.name` or `host.id` instead.',
                example: 'd37e5ebfe0ae6c4972dbe9f0174a1637bb8247f6',
                flat_name: 'service.target.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'service',
                short: 'Unique identifier of the running service.',
                type: 'keyword',
            },
            'service.target.name': {
                dashed_name: 'service-target-name',
                description: 'Name of the service data is collected from.\nThe name of the service is normally user given. This allows for distributed services that run on multiple hosts to correlate the related instances based on the name.\nIn the case of Elasticsearch the `service.name` could contain the cluster name. For Beats the `service.name` is by default a copy of the `service.type` field if no name is specified.',
                example: 'elasticsearch-metrics',
                flat_name: 'service.target.name',
                ignore_above: 1024,
                level: 'core',
                name: 'name',
                normalize: [],
                original_fieldset: 'service',
                short: 'Name of the service.',
                type: 'keyword',
            },
            'service.target.node.name': {
                dashed_name: 'service-target-node-name',
                description: "Name of a service node.\nThis allows for two nodes of the same service running on the same host to be differentiated. Therefore, `service.node.name` should typically be unique across nodes of a given service.\nIn the case of Elasticsearch, the `service.node.name` could contain the unique node name within the Elasticsearch cluster. In cases where the service doesn't have the concept of a node name, the host name or container name can be used to distinguish running instances that make up this service. If those do not provide uniqueness (e.g. multiple instances of the service running on the same host) - the node name can be manually set.",
                example: 'instance-0000000016',
                flat_name: 'service.target.node.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.name',
                normalize: [],
                original_fieldset: 'service',
                short: 'Name of the service node.',
                type: 'keyword',
            },
            'service.target.node.role': {
                dashed_name: 'service-target-node-role',
                description: 'Deprecated for removal in next major version release. This field will be superseded by `node.roles`.\nRole of a service node.\nThis allows for distinction between different running roles of the same service.\nIn the case of Kibana, the `service.node.role` could be `ui` or `background_tasks`.\nIn the case of Elasticsearch, the `service.node.role` could be `master` or `data`.\nOther services could use this to distinguish between a `web` and `worker` role running as part of the service.',
                example: 'background_tasks',
                flat_name: 'service.target.node.role',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.role',
                normalize: [],
                original_fieldset: 'service',
                short: 'Deprecated role (singular) of the service node.',
                type: 'keyword',
            },
            'service.target.node.roles': {
                dashed_name: 'service-target-node-roles',
                description: 'Roles of a service node.\nThis allows for distinction between different running roles of the same service.\nIn the case of Kibana, the `service.node.role` could be `ui` or `background_tasks` or both.\nIn the case of Elasticsearch, the `service.node.role` could be `master` or `data` or both.\nOther services could use this to distinguish between a `web` and `worker` role running as part of the service.',
                example: '["ui", "background_tasks"]',
                flat_name: 'service.target.node.roles',
                ignore_above: 1024,
                level: 'extended',
                name: 'node.roles',
                normalize: ['array'],
                original_fieldset: 'service',
                short: 'Roles of the service node.',
                type: 'keyword',
            },
            'service.target.state': {
                dashed_name: 'service-target-state',
                description: 'Current state of the service.',
                flat_name: 'service.target.state',
                ignore_above: 1024,
                level: 'core',
                name: 'state',
                normalize: [],
                original_fieldset: 'service',
                short: 'Current state of the service.',
                type: 'keyword',
            },
            'service.target.type': {
                dashed_name: 'service-target-type',
                description: 'The type of the service data is collected from.\nThe type can be used to group and correlate logs and metrics from one service type.\nExample: If logs or metrics are collected from Elasticsearch, `service.type` would be `elasticsearch`.',
                example: 'elasticsearch',
                flat_name: 'service.target.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: [],
                original_fieldset: 'service',
                short: 'The type of the service.',
                type: 'keyword',
            },
            'service.target.version': {
                dashed_name: 'service-target-version',
                description: 'Version of the service the data was collected from.\nThis allows to look at a data set only for a specific version of a service.',
                example: '3.2.4',
                flat_name: 'service.target.version',
                ignore_above: 1024,
                level: 'core',
                name: 'version',
                normalize: [],
                original_fieldset: 'service',
                short: 'Version of the service.',
                type: 'keyword',
            },
            'service.type': {
                dashed_name: 'service-type',
                description: 'The type of the service data is collected from.\nThe type can be used to group and correlate logs and metrics from one service type.\nExample: If logs or metrics are collected from Elasticsearch, `service.type` would be `elasticsearch`.',
                example: 'elasticsearch',
                flat_name: 'service.type',
                ignore_above: 1024,
                level: 'core',
                name: 'type',
                normalize: [],
                short: 'The type of the service.',
                type: 'keyword',
            },
            'service.version': {
                dashed_name: 'service-version',
                description: 'Version of the service the data was collected from.\nThis allows to look at a data set only for a specific version of a service.',
                example: '3.2.4',
                flat_name: 'service.version',
                ignore_above: 1024,
                level: 'core',
                name: 'version',
                normalize: [],
                short: 'Version of the service.',
                type: 'keyword',
            },
        },
        footnote: 'The service fields may be self-nested under service.origin.* and service.target.* to describe origin or target services in the context of incoming or outgoing requests, respectively. However, the fieldsets service.origin.* and service.target.* must not be confused with the root service fieldset that is used to describe the actual service under observation. The fieldset service.origin.* may only be used in the context of incoming requests or events to describe the originating service of the request. The fieldset service.target.* may only be used in the context of outgoing requests or events to describe the target service of the request.',
        group: 2,
        name: 'service',
        nestings: ['service.origin', 'service.target'],
        prefix: 'service.',
        reusable: {
            expected: [
                {
                    as: 'origin',
                    at: 'service',
                    beta: 'Reusing the `service` fields in this location is currently considered beta.',
                    full: 'service.origin',
                    short_override: 'Describes the origin service in case of an incoming request or event.',
                },
                {
                    as: 'target',
                    at: 'service',
                    beta: 'Reusing the `service` fields in this location is currently considered beta.',
                    full: 'service.target',
                    short_override: 'Describes the target service in case of an outgoing request or event.',
                },
            ],
            top_level: true,
        },
        reused_here: [
            {
                beta: 'Reusing the `service` fields in this location is currently considered beta.',
                full: 'service.origin',
                schema_name: 'service',
                short: 'Describes the origin service in case of an incoming request or event.',
            },
            {
                beta: 'Reusing the `service` fields in this location is currently considered beta.',
                full: 'service.target',
                schema_name: 'service',
                short: 'Describes the target service in case of an outgoing request or event.',
            },
        ],
        short: 'Fields describing the service for or from which the data was collected.',
        title: 'Service',
        type: 'group',
    },
    source: {
        description: 'Source fields capture details about the sender of a network exchange/packet. These fields are populated from a network event, packet, or other event containing details of a network transaction.\nSource fields are usually populated in conjunction with destination fields. The source and destination fields are considered the baseline and should always be filled if an event contains source and destination details from a network transaction. If the event also contains identification of the client and server roles, then the client and server fields should also be populated.',
        fields: {
            'source.address': {
                dashed_name: 'source-address',
                description: 'Some event source addresses are defined ambiguously. The event will sometimes list an IP, a domain or a unix socket.  You should always store the raw address in the `.address` field.\nThen it should be duplicated to `.ip` or `.domain`, depending on which one it is.',
                flat_name: 'source.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'address',
                normalize: [],
                short: 'Source network address.',
                type: 'keyword',
            },
            'source.as.number': {
                dashed_name: 'source-as-number',
                description: 'Unique number allocated to the autonomous system. The autonomous system number (ASN) uniquely identifies each network on the Internet.',
                example: 15169,
                flat_name: 'source.as.number',
                level: 'extended',
                name: 'number',
                normalize: [],
                original_fieldset: 'as',
                short: 'Unique number allocated to the autonomous system.',
                type: 'long',
            },
            'source.as.organization.name': {
                dashed_name: 'source-as-organization-name',
                description: 'Organization name.',
                example: 'Google LLC',
                flat_name: 'source.as.organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'source.as.organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'organization.name',
                normalize: [],
                original_fieldset: 'as',
                short: 'Organization name.',
                type: 'keyword',
            },
            'source.bytes': {
                dashed_name: 'source-bytes',
                description: 'Bytes sent from the source to the destination.',
                example: 184,
                flat_name: 'source.bytes',
                format: 'bytes',
                level: 'core',
                name: 'bytes',
                normalize: [],
                short: 'Bytes sent from the source to the destination.',
                type: 'long',
            },
            'source.domain': {
                dashed_name: 'source-domain',
                description: 'The domain name of the source system.\nThis value may be a host name, a fully qualified domain name, or another host naming format. The value may derive from the original event or be added from enrichment.',
                example: 'foo.example.com',
                flat_name: 'source.domain',
                ignore_above: 1024,
                level: 'core',
                name: 'domain',
                normalize: [],
                short: 'The domain name of the source.',
                type: 'keyword',
            },
            'source.geo.city_name': {
                dashed_name: 'source-geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'source.geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'City name.',
                type: 'keyword',
            },
            'source.geo.continent_code': {
                dashed_name: 'source-geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'source.geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Continent code.',
                type: 'keyword',
            },
            'source.geo.continent_name': {
                dashed_name: 'source-geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'source.geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'source.geo.country_iso_code': {
                dashed_name: 'source-geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'source.geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'source.geo.country_name': {
                dashed_name: 'source-geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'source.geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country name.',
                type: 'keyword',
            },
            'source.geo.location': {
                dashed_name: 'source-geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'source.geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'source.geo.name': {
                dashed_name: 'source-geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'source.geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'source.geo.postal_code': {
                dashed_name: 'source-geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'source.geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Postal code.',
                type: 'keyword',
            },
            'source.geo.region_iso_code': {
                dashed_name: 'source-geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'source.geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'source.geo.region_name': {
                dashed_name: 'source-geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'source.geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region name.',
                type: 'keyword',
            },
            'source.geo.timezone': {
                dashed_name: 'source-geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'source.geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Time zone.',
                type: 'keyword',
            },
            'source.ip': {
                dashed_name: 'source-ip',
                description: 'IP address of the source (IPv4 or IPv6).',
                flat_name: 'source.ip',
                level: 'core',
                name: 'ip',
                normalize: [],
                short: 'IP address of the source.',
                type: 'ip',
            },
            'source.mac': {
                dashed_name: 'source-mac',
                description: 'MAC address of the source.\nThe notation format from RFC 7042 is suggested: Each octet (that is, 8-bit byte) is represented by two [uppercase] hexadecimal digits giving the value of the octet as an unsigned integer. Successive octets are separated by a hyphen.',
                example: '00-00-5E-00-53-23',
                flat_name: 'source.mac',
                ignore_above: 1024,
                level: 'core',
                name: 'mac',
                normalize: [],
                pattern: '^[A-F0-9]{2}(-[A-F0-9]{2}){5,}$',
                short: 'MAC address of the source.',
                type: 'keyword',
            },
            'source.nat.ip': {
                dashed_name: 'source-nat-ip',
                description: 'Translated ip of source based NAT sessions (e.g. internal client to internet)\nTypically connections traversing load balancers, firewalls, or routers.',
                flat_name: 'source.nat.ip',
                level: 'extended',
                name: 'nat.ip',
                normalize: [],
                short: 'Source NAT ip',
                type: 'ip',
            },
            'source.nat.port': {
                dashed_name: 'source-nat-port',
                description: 'Translated port of source based NAT sessions. (e.g. internal client to internet)\nTypically used with load balancers, firewalls, or routers.',
                flat_name: 'source.nat.port',
                format: 'string',
                level: 'extended',
                name: 'nat.port',
                normalize: [],
                short: 'Source NAT port',
                type: 'long',
            },
            'source.packets': {
                dashed_name: 'source-packets',
                description: 'Packets sent from the source to the destination.',
                example: 12,
                flat_name: 'source.packets',
                level: 'core',
                name: 'packets',
                normalize: [],
                short: 'Packets sent from the source to the destination.',
                type: 'long',
            },
            'source.port': {
                dashed_name: 'source-port',
                description: 'Port of the source.',
                flat_name: 'source.port',
                format: 'string',
                level: 'core',
                name: 'port',
                normalize: [],
                short: 'Port of the source.',
                type: 'long',
            },
            'source.registered_domain': {
                dashed_name: 'source-registered-domain',
                description: 'The highest registered source domain, stripped of the subdomain.\nFor example, the registered domain for "foo.example.com" is "example.com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last two labels will not work well for TLDs such as "co.uk".',
                example: 'example.com',
                flat_name: 'source.registered_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'registered_domain',
                normalize: [],
                short: 'The highest registered source domain, stripped of the subdomain.',
                type: 'keyword',
            },
            'source.subdomain': {
                dashed_name: 'source-subdomain',
                description: 'The subdomain portion of a fully qualified domain name includes all of the names except the host name under the registered_domain.  In a partially qualified domain, or if the the qualification level of the full name cannot be determined, subdomain contains all of the names below the registered domain.\nFor example the subdomain portion of "www.east.mydomain.co.uk" is "east". If the domain has multiple levels of subdomain, such as "sub2.sub1.example.com", the subdomain field should contain "sub2.sub1", with no trailing period.',
                example: 'east',
                flat_name: 'source.subdomain',
                ignore_above: 1024,
                level: 'extended',
                name: 'subdomain',
                normalize: [],
                short: 'The subdomain of the domain.',
                type: 'keyword',
            },
            'source.top_level_domain': {
                dashed_name: 'source-top-level-domain',
                description: 'The effective top level domain (eTLD), also known as the domain suffix, is the last part of the domain name. For example, the top level domain for example.com is "com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last label will not work well for effective TLDs such as "co.uk".',
                example: 'co.uk',
                flat_name: 'source.top_level_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'top_level_domain',
                normalize: [],
                short: 'The effective top level domain (com, org, net, co.uk).',
                type: 'keyword',
            },
            'source.user.domain': {
                dashed_name: 'source-user-domain',
                description: 'Name of the directory the user is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'source.user.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'user',
                short: 'Name of the directory the user is a member of.',
                type: 'keyword',
            },
            'source.user.email': {
                dashed_name: 'source-user-email',
                description: 'User email address.',
                flat_name: 'source.user.email',
                ignore_above: 1024,
                level: 'extended',
                name: 'email',
                normalize: [],
                original_fieldset: 'user',
                short: 'User email address.',
                type: 'keyword',
            },
            'source.user.full_name': {
                dashed_name: 'source-user-full-name',
                description: "User's full name, if available.",
                example: 'Albert Einstein',
                flat_name: 'source.user.full_name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'source.user.full_name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'full_name',
                normalize: [],
                original_fieldset: 'user',
                short: "User's full name, if available.",
                type: 'keyword',
            },
            'source.user.group.domain': {
                dashed_name: 'source-user-group-domain',
                description: 'Name of the directory the group is a member of.\nFor example, an LDAP or Active Directory domain name.',
                flat_name: 'source.user.group.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the directory the group is a member of.',
                type: 'keyword',
            },
            'source.user.group.id': {
                dashed_name: 'source-user-group-id',
                description: 'Unique identifier for the group on the system/platform.',
                flat_name: 'source.user.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'id',
                normalize: [],
                original_fieldset: 'group',
                short: 'Unique identifier for the group on the system/platform.',
                type: 'keyword',
            },
            'source.user.group.name': {
                dashed_name: 'source-user-group-name',
                description: 'Name of the group.',
                flat_name: 'source.user.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'group',
                short: 'Name of the group.',
                type: 'keyword',
            },
            'source.user.hash': {
                dashed_name: 'source-user-hash',
                description: 'Unique user hash to correlate information for a user in anonymized form.\nUseful if `user.id` or `user.name` contain confidential information and cannot be used.',
                flat_name: 'source.user.hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'hash',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique user hash to correlate information for a user in anonymized form.',
                type: 'keyword',
            },
            'source.user.id': {
                dashed_name: 'source-user-id',
                description: 'Unique identifier of the user.',
                example: 'S-1-5-21-202424912787-2692429404-2351956786-1000',
                flat_name: 'source.user.id',
                ignore_above: 1024,
                level: 'core',
                name: 'id',
                normalize: [],
                original_fieldset: 'user',
                short: 'Unique identifier of the user.',
                type: 'keyword',
            },
            'source.user.name': {
                dashed_name: 'source-user-name',
                description: 'Short name or login of the user.',
                example: 'a.einstein',
                flat_name: 'source.user.name',
                ignore_above: 1024,
                level: 'core',
                multi_fields: [
                    {
                        flat_name: 'source.user.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'name',
                normalize: [],
                original_fieldset: 'user',
                short: 'Short name or login of the user.',
                type: 'keyword',
            },
            'source.user.roles': {
                dashed_name: 'source-user-roles',
                description: 'Array of user roles at the time of the event.',
                example: '["kibana_admin", "reporting_user"]',
                flat_name: 'source.user.roles',
                ignore_above: 1024,
                level: 'extended',
                name: 'roles',
                normalize: ['array'],
                original_fieldset: 'user',
                short: 'Array of user roles at the time of the event.',
                type: 'keyword',
            },
        },
        group: 2,
        name: 'source',
        nestings: ['source.as', 'source.geo', 'source.user'],
        prefix: 'source.',
        reusable: {
            expected: [
                {
                    as: 'source',
                    at: 'process.entry_meta',
                    full: 'process.entry_meta.source',
                    short_override: 'Remote client information such as ip, port and geo location.',
                },
            ],
            top_level: true,
        },
        reused_here: [
            {
                full: 'source.as',
                schema_name: 'as',
                short: 'Fields describing an Autonomous System (Internet routing prefix).',
            },
            {
                full: 'source.geo',
                schema_name: 'geo',
                short: 'Fields describing a location.',
            },
            {
                full: 'source.user',
                schema_name: 'user',
                short: 'Fields to describe the user relevant to the event.',
            },
        ],
        short: 'Fields about the source side of a network connection, used with destination.',
        title: 'Source',
        type: 'group',
    },
    threat: {
        description: 'Fields to classify events and alerts according to a threat taxonomy such as the MITRE ATT&CK® framework.\nThese fields are for users to classify alerts from all of their sources (e.g. IDS, NGFW, etc.) within a common taxonomy. The threat.tactic.* fields are meant to capture the high level category of the threat (e.g. "impact"). The threat.technique.* fields are meant to capture which kind of approach is used by this detected threat, to accomplish the goal (e.g. "endpoint denial of service").',
        fields: {
            'threat.enrichments': {
                dashed_name: 'threat-enrichments',
                description: 'A list of associated indicators objects enriching the event, and the context of that association/enrichment.',
                flat_name: 'threat.enrichments',
                level: 'extended',
                name: 'enrichments',
                normalize: ['array'],
                short: 'List of objects containing indicators enriching the event.',
                type: 'nested',
            },
            'threat.enrichments.indicator': {
                dashed_name: 'threat-enrichments-indicator',
                description: 'Object containing associated indicators enriching the event.',
                flat_name: 'threat.enrichments.indicator',
                level: 'extended',
                name: 'enrichments.indicator',
                normalize: [],
                short: 'Object containing indicators enriching the event.',
                type: 'object',
            },
            'threat.enrichments.indicator.as.number': {
                dashed_name: 'threat-enrichments-indicator-as-number',
                description: 'Unique number allocated to the autonomous system. The autonomous system number (ASN) uniquely identifies each network on the Internet.',
                example: 15169,
                flat_name: 'threat.enrichments.indicator.as.number',
                level: 'extended',
                name: 'number',
                normalize: [],
                original_fieldset: 'as',
                short: 'Unique number allocated to the autonomous system.',
                type: 'long',
            },
            'threat.enrichments.indicator.as.organization.name': {
                dashed_name: 'threat-enrichments-indicator-as-organization-name',
                description: 'Organization name.',
                example: 'Google LLC',
                flat_name: 'threat.enrichments.indicator.as.organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'threat.enrichments.indicator.as.organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'organization.name',
                normalize: [],
                original_fieldset: 'as',
                short: 'Organization name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.confidence': {
                dashed_name: 'threat-enrichments-indicator-confidence',
                description: 'Identifies the vendor-neutral confidence rating using the None/Low/Medium/High scale defined in Appendix A of the STIX 2.1 framework. Vendor-specific confidence scales may be added as custom fields.',
                example: 'Medium',
                expected_values: ['Not Specified', 'None', 'Low', 'Medium', 'High'],
                flat_name: 'threat.enrichments.indicator.confidence',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.confidence',
                normalize: [],
                short: 'Indicator confidence rating',
                type: 'keyword',
            },
            'threat.enrichments.indicator.description': {
                dashed_name: 'threat-enrichments-indicator-description',
                description: 'Describes the type of action conducted by the threat.',
                example: 'IP x.x.x.x was observed delivering the Angler EK.',
                flat_name: 'threat.enrichments.indicator.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.description',
                normalize: [],
                short: 'Indicator description',
                type: 'keyword',
            },
            'threat.enrichments.indicator.email.address': {
                dashed_name: 'threat-enrichments-indicator-email-address',
                description: 'Identifies a threat indicator as an email address (irrespective of direction).',
                example: 'phish@example.com',
                flat_name: 'threat.enrichments.indicator.email.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.email.address',
                normalize: [],
                short: 'Indicator email address',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.accessed': {
                dashed_name: 'threat-enrichments-indicator-file-accessed',
                description: 'Last time the file was accessed.\nNote that not all filesystems keep track of access time.',
                flat_name: 'threat.enrichments.indicator.file.accessed',
                level: 'extended',
                name: 'accessed',
                normalize: [],
                original_fieldset: 'file',
                short: 'Last time the file was accessed.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.attributes': {
                dashed_name: 'threat-enrichments-indicator-file-attributes',
                description: "Array of file attributes.\nAttributes names will vary by platform. Here's a non-exhaustive list of values that are expected in this field: archive, compressed, directory, encrypted, execute, hidden, read, readonly, system, write.",
                example: '["readonly", "system"]',
                flat_name: 'threat.enrichments.indicator.file.attributes',
                ignore_above: 1024,
                level: 'extended',
                name: 'attributes',
                normalize: ['array'],
                original_fieldset: 'file',
                short: 'Array of file attributes.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.code_signature.digest_algorithm': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-digest-algorithm',
                description: 'The hashing algorithm used to sign the process.\nThis value can distinguish signatures when a file is signed multiple times by the same signer but with a different digest algorithm.',
                example: 'sha256',
                flat_name: 'threat.enrichments.indicator.file.code_signature.digest_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'digest_algorithm',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Hashing algorithm used to sign the process.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.code_signature.exists': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-exists',
                description: 'Boolean to capture if a signature is present.',
                example: 'true',
                flat_name: 'threat.enrichments.indicator.file.code_signature.exists',
                level: 'core',
                name: 'exists',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if a signature is present.',
                type: 'boolean',
            },
            'threat.enrichments.indicator.file.code_signature.signing_id': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-signing-id',
                description: 'The identifier used to sign the process.\nThis is used to identify the application manufactured by a software vendor. The field is relevant to Apple *OS only.',
                example: 'com.apple.xpc.proxy',
                flat_name: 'threat.enrichments.indicator.file.code_signature.signing_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'signing_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The identifier used to sign the process.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.code_signature.status': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-status',
                description: 'Additional information about the certificate status.\nThis is useful for logging cryptographic errors with the certificate validity or trust status. Leave unpopulated if the validity or trust of the certificate was unchecked.',
                example: 'ERROR_UNTRUSTED_ROOT',
                flat_name: 'threat.enrichments.indicator.file.code_signature.status',
                ignore_above: 1024,
                level: 'extended',
                name: 'status',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Additional information about the certificate status.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.code_signature.subject_name': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-subject-name',
                description: 'Subject name of the code signer',
                example: 'Microsoft Corporation',
                flat_name: 'threat.enrichments.indicator.file.code_signature.subject_name',
                ignore_above: 1024,
                level: 'core',
                name: 'subject_name',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Subject name of the code signer',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.code_signature.team_id': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-team-id',
                description: 'The team identifier used to sign the process.\nThis is used to identify the team or vendor of a software product. The field is relevant to Apple *OS only.',
                example: 'EQHXZ8M8AV',
                flat_name: 'threat.enrichments.indicator.file.code_signature.team_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'team_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The team identifier used to sign the process.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.code_signature.timestamp': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-timestamp',
                description: 'Date and time when the code signature was generated and signed.',
                example: '2021-01-01T12:10:30Z',
                flat_name: 'threat.enrichments.indicator.file.code_signature.timestamp',
                level: 'extended',
                name: 'timestamp',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'When the signature was generated and signed.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.code_signature.trusted': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-trusted',
                description: 'Stores the trust status of the certificate chain.\nValidating the trust of the certificate chain may be complicated, and this field should only be populated by tools that actively check the status.',
                example: 'true',
                flat_name: 'threat.enrichments.indicator.file.code_signature.trusted',
                level: 'extended',
                name: 'trusted',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Stores the trust status of the certificate chain.',
                type: 'boolean',
            },
            'threat.enrichments.indicator.file.code_signature.valid': {
                dashed_name: 'threat-enrichments-indicator-file-code-signature-valid',
                description: 'Boolean to capture if the digital signature is verified against the binary content.\nLeave unpopulated if a certificate was unchecked.',
                example: 'true',
                flat_name: 'threat.enrichments.indicator.file.code_signature.valid',
                level: 'extended',
                name: 'valid',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if the digital signature is verified against the binary content.',
                type: 'boolean',
            },
            'threat.enrichments.indicator.file.created': {
                dashed_name: 'threat-enrichments-indicator-file-created',
                description: 'File creation time.\nNote that not all filesystems store the creation time.',
                flat_name: 'threat.enrichments.indicator.file.created',
                level: 'extended',
                name: 'created',
                normalize: [],
                original_fieldset: 'file',
                short: 'File creation time.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.ctime': {
                dashed_name: 'threat-enrichments-indicator-file-ctime',
                description: 'Last time the file attributes or metadata changed.\nNote that changes to the file content will update `mtime`. This implies `ctime` will be adjusted at the same time, since `mtime` is an attribute of the file.',
                flat_name: 'threat.enrichments.indicator.file.ctime',
                level: 'extended',
                name: 'ctime',
                normalize: [],
                original_fieldset: 'file',
                short: 'Last time the file attributes or metadata changed.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.device': {
                dashed_name: 'threat-enrichments-indicator-file-device',
                description: 'Device that is the source of the file.',
                example: 'sda',
                flat_name: 'threat.enrichments.indicator.file.device',
                ignore_above: 1024,
                level: 'extended',
                name: 'device',
                normalize: [],
                original_fieldset: 'file',
                short: 'Device that is the source of the file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.directory': {
                dashed_name: 'threat-enrichments-indicator-file-directory',
                description: 'Directory where the file is located. It should include the drive letter, when appropriate.',
                example: '/home/alice',
                flat_name: 'threat.enrichments.indicator.file.directory',
                ignore_above: 1024,
                level: 'extended',
                name: 'directory',
                normalize: [],
                original_fieldset: 'file',
                short: 'Directory where the file is located.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.drive_letter': {
                dashed_name: 'threat-enrichments-indicator-file-drive-letter',
                description: 'Drive letter where the file is located. This field is only relevant on Windows.\nThe value should be uppercase, and not include the colon.',
                example: 'C',
                flat_name: 'threat.enrichments.indicator.file.drive_letter',
                ignore_above: 1,
                level: 'extended',
                name: 'drive_letter',
                normalize: [],
                original_fieldset: 'file',
                short: 'Drive letter where the file is located.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.architecture': {
                dashed_name: 'threat-enrichments-indicator-file-elf-architecture',
                description: 'Machine architecture of the ELF file.',
                example: 'x86-64',
                flat_name: 'threat.enrichments.indicator.file.elf.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Machine architecture of the ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.byte_order': {
                dashed_name: 'threat-enrichments-indicator-file-elf-byte-order',
                description: 'Byte sequence of ELF file.',
                example: 'Little Endian',
                flat_name: 'threat.enrichments.indicator.file.elf.byte_order',
                ignore_above: 1024,
                level: 'extended',
                name: 'byte_order',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Byte sequence of ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.cpu_type': {
                dashed_name: 'threat-enrichments-indicator-file-elf-cpu-type',
                description: 'CPU type of the ELF file.',
                example: 'Intel',
                flat_name: 'threat.enrichments.indicator.file.elf.cpu_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'cpu_type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'CPU type of the ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.creation_date': {
                dashed_name: 'threat-enrichments-indicator-file-elf-creation-date',
                description: "Extracted when possible from the file's metadata. Indicates when it was built or compiled. It can also be faked by malware creators.",
                flat_name: 'threat.enrichments.indicator.file.elf.creation_date',
                level: 'extended',
                name: 'creation_date',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Build or compile date.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.elf.exports': {
                dashed_name: 'threat-enrichments-indicator-file-elf-exports',
                description: 'List of exported element names and types.',
                flat_name: 'threat.enrichments.indicator.file.elf.exports',
                level: 'extended',
                name: 'exports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of exported element names and types.',
                type: 'flattened',
            },
            'threat.enrichments.indicator.file.elf.go_import_hash': {
                dashed_name: 'threat-enrichments-indicator-file-elf-go-import-hash',
                description: 'A hash of the Go language imports in an ELF file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'threat.enrichments.indicator.file.elf.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the Go language imports in an ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.go_imports': {
                dashed_name: 'threat-enrichments-indicator-file-elf-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'threat.enrichments.indicator.file.elf.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'elf',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'threat.enrichments.indicator.file.elf.go_imports_names_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-elf-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'threat.enrichments.indicator.file.elf.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.go_imports_names_var_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-elf-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'threat.enrichments.indicator.file.elf.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.go_stripped': {
                dashed_name: 'threat-enrichments-indicator-file-elf-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'threat.enrichments.indicator.file.elf.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'threat.enrichments.indicator.file.elf.header.abi_version': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-abi-version',
                description: 'Version of the ELF Application Binary Interface (ABI).',
                flat_name: 'threat.enrichments.indicator.file.elf.header.abi_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.abi_version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF Application Binary Interface (ABI).',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.header.class': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-class',
                description: 'Header class of the ELF file.',
                flat_name: 'threat.enrichments.indicator.file.elf.header.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.class',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header class of the ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.header.data': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-data',
                description: 'Data table of the ELF header.',
                flat_name: 'threat.enrichments.indicator.file.elf.header.data',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.data',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Data table of the ELF header.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.header.entrypoint': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-entrypoint',
                description: 'Header entrypoint of the ELF file.',
                flat_name: 'threat.enrichments.indicator.file.elf.header.entrypoint',
                format: 'string',
                level: 'extended',
                name: 'header.entrypoint',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header entrypoint of the ELF file.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.header.object_version': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-object-version',
                description: '"0x1" for original ELF files.',
                flat_name: 'threat.enrichments.indicator.file.elf.header.object_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.object_version',
                normalize: [],
                original_fieldset: 'elf',
                short: '"0x1" for original ELF files.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.header.os_abi': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-os-abi',
                description: 'Application Binary Interface (ABI) of the Linux OS.',
                flat_name: 'threat.enrichments.indicator.file.elf.header.os_abi',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.os_abi',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Application Binary Interface (ABI) of the Linux OS.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.header.type': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-type',
                description: 'Header type of the ELF file.',
                flat_name: 'threat.enrichments.indicator.file.elf.header.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header type of the ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.header.version': {
                dashed_name: 'threat-enrichments-indicator-file-elf-header-version',
                description: 'Version of the ELF header.',
                flat_name: 'threat.enrichments.indicator.file.elf.header.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF header.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.import_hash': {
                dashed_name: 'threat-enrichments-indicator-file-elf-import-hash',
                description: 'A hash of the imports in an ELF file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is an ELF implementation of the Windows PE imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'threat.enrichments.indicator.file.elf.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the imports in an ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.imports': {
                dashed_name: 'threat-enrichments-indicator-file-elf-imports',
                description: 'List of imported element names and types.',
                flat_name: 'threat.enrichments.indicator.file.elf.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'threat.enrichments.indicator.file.elf.imports_names_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-elf-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'threat.enrichments.indicator.file.elf.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.imports_names_var_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-elf-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'threat.enrichments.indicator.file.elf.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.sections': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections',
                description: 'An array containing an object for each section of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.sections.*`.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'Section information of the ELF file.',
                type: 'nested',
            },
            'threat.enrichments.indicator.file.elf.sections.chi2': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-chi2',
                description: 'Chi-square probability distribution of the section.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.chi2',
                format: 'number',
                level: 'extended',
                name: 'sections.chi2',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Chi-square probability distribution of the section.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.sections.entropy': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.sections.flags': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-flags',
                description: 'ELF Section List flags.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.flags',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.flags',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List flags.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.sections.name': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-name',
                description: 'ELF Section List name.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.sections.physical_offset': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-physical-offset',
                description: 'ELF Section List offset.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.physical_offset',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.physical_offset',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List offset.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.sections.physical_size': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-physical-size',
                description: 'ELF Section List physical size.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List physical size.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.sections.type': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-type',
                description: 'ELF Section List type.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List type.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.sections.var_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.sections.virtual_address': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-virtual-address',
                description: 'ELF Section List virtual address.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.virtual_address',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_address',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual address.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.sections.virtual_size': {
                dashed_name: 'threat-enrichments-indicator-file-elf-sections-virtual-size',
                description: 'ELF Section List virtual size.',
                flat_name: 'threat.enrichments.indicator.file.elf.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual size.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.elf.segments': {
                dashed_name: 'threat-enrichments-indicator-file-elf-segments',
                description: 'An array containing an object for each segment of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.segments.*`.',
                flat_name: 'threat.enrichments.indicator.file.elf.segments',
                level: 'extended',
                name: 'segments',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'ELF object segment list.',
                type: 'nested',
            },
            'threat.enrichments.indicator.file.elf.segments.sections': {
                dashed_name: 'threat-enrichments-indicator-file-elf-segments-sections',
                description: 'ELF object segment sections.',
                flat_name: 'threat.enrichments.indicator.file.elf.segments.sections',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.sections',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment sections.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.segments.type': {
                dashed_name: 'threat-enrichments-indicator-file-elf-segments-type',
                description: 'ELF object segment type.',
                flat_name: 'threat.enrichments.indicator.file.elf.segments.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment type.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.shared_libraries': {
                dashed_name: 'threat-enrichments-indicator-file-elf-shared-libraries',
                description: 'List of shared libraries used by this ELF object.',
                flat_name: 'threat.enrichments.indicator.file.elf.shared_libraries',
                ignore_above: 1024,
                level: 'extended',
                name: 'shared_libraries',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of shared libraries used by this ELF object.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.elf.telfhash': {
                dashed_name: 'threat-enrichments-indicator-file-elf-telfhash',
                description: 'telfhash symbol hash for ELF file.',
                flat_name: 'threat.enrichments.indicator.file.elf.telfhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'telfhash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'telfhash hash for ELF file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.extension': {
                dashed_name: 'threat-enrichments-indicator-file-extension',
                description: 'File extension, excluding the leading dot.\nNote that when the file name has multiple extensions (example.tar.gz), only the last one should be captured ("gz", not "tar.gz").',
                example: 'png',
                flat_name: 'threat.enrichments.indicator.file.extension',
                ignore_above: 1024,
                level: 'extended',
                name: 'extension',
                normalize: [],
                original_fieldset: 'file',
                short: 'File extension, excluding the leading dot.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.fork_name': {
                dashed_name: 'threat-enrichments-indicator-file-fork-name',
                description: 'A fork is additional data associated with a filesystem object.\nOn Linux, a resource fork is used to store additional data with a filesystem object. A file always has at least one fork for the data portion, and additional forks may exist.\nOn NTFS, this is analogous to an Alternate Data Stream (ADS), and the default data stream for a file is just called $DATA. Zone.Identifier is commonly used by Windows to track contents downloaded from the Internet. An ADS is typically of the form: `C:\\path\\to\\filename.extension:some_fork_name`, and `some_fork_name` is the value that should populate `fork_name`. `filename.extension` should populate `file.name`, and `extension` should populate `file.extension`. The full path, `file.path`, will include the fork name.',
                example: 'Zone.Identifer',
                flat_name: 'threat.enrichments.indicator.file.fork_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'fork_name',
                normalize: [],
                original_fieldset: 'file',
                short: 'A fork is additional data associated with a filesystem object.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.gid': {
                dashed_name: 'threat-enrichments-indicator-file-gid',
                description: 'Primary group ID (GID) of the file.',
                example: '1001',
                flat_name: 'threat.enrichments.indicator.file.gid',
                ignore_above: 1024,
                level: 'extended',
                name: 'gid',
                normalize: [],
                original_fieldset: 'file',
                short: 'Primary group ID (GID) of the file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.group': {
                dashed_name: 'threat-enrichments-indicator-file-group',
                description: 'Primary group name of the file.',
                example: 'alice',
                flat_name: 'threat.enrichments.indicator.file.group',
                ignore_above: 1024,
                level: 'extended',
                name: 'group',
                normalize: [],
                original_fieldset: 'file',
                short: 'Primary group name of the file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.hash.md5': {
                dashed_name: 'threat-enrichments-indicator-file-hash-md5',
                description: 'MD5 hash.',
                flat_name: 'threat.enrichments.indicator.file.hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                original_fieldset: 'hash',
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.hash.sha1': {
                dashed_name: 'threat-enrichments-indicator-file-hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'threat.enrichments.indicator.file.hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.hash.sha256': {
                dashed_name: 'threat-enrichments-indicator-file-hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'threat.enrichments.indicator.file.hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.hash.sha384': {
                dashed_name: 'threat-enrichments-indicator-file-hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'threat.enrichments.indicator.file.hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.hash.sha512': {
                dashed_name: 'threat-enrichments-indicator-file-hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'threat.enrichments.indicator.file.hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.hash.ssdeep': {
                dashed_name: 'threat-enrichments-indicator-file-hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'threat.enrichments.indicator.file.hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.hash.tlsh': {
                dashed_name: 'threat-enrichments-indicator-file-hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'threat.enrichments.indicator.file.hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                original_fieldset: 'hash',
                short: 'TLSH hash.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.inode': {
                dashed_name: 'threat-enrichments-indicator-file-inode',
                description: 'Inode representing the file in the filesystem.',
                example: '256383',
                flat_name: 'threat.enrichments.indicator.file.inode',
                ignore_above: 1024,
                level: 'extended',
                name: 'inode',
                normalize: [],
                original_fieldset: 'file',
                short: 'Inode representing the file in the filesystem.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.mime_type': {
                dashed_name: 'threat-enrichments-indicator-file-mime-type',
                description: 'MIME type should identify the format of the file or stream of bytes using https://www.iana.org/assignments/media-types/media-types.xhtml[IANA official types], where possible. When more than one type is applicable, the most specific type should be used.',
                flat_name: 'threat.enrichments.indicator.file.mime_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'mime_type',
                normalize: [],
                original_fieldset: 'file',
                short: 'Media type of file, document, or arrangement of bytes.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.mode': {
                dashed_name: 'threat-enrichments-indicator-file-mode',
                description: 'Mode of the file in octal representation.',
                example: '0640',
                flat_name: 'threat.enrichments.indicator.file.mode',
                ignore_above: 1024,
                level: 'extended',
                name: 'mode',
                normalize: [],
                original_fieldset: 'file',
                short: 'Mode of the file in octal representation.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.mtime': {
                dashed_name: 'threat-enrichments-indicator-file-mtime',
                description: 'Last time the file content was modified.',
                flat_name: 'threat.enrichments.indicator.file.mtime',
                level: 'extended',
                name: 'mtime',
                normalize: [],
                original_fieldset: 'file',
                short: 'Last time the file content was modified.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.name': {
                dashed_name: 'threat-enrichments-indicator-file-name',
                description: 'Name of the file including the extension, without the directory.',
                example: 'example.png',
                flat_name: 'threat.enrichments.indicator.file.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'file',
                short: 'Name of the file including the extension, without the directory.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.owner': {
                dashed_name: 'threat-enrichments-indicator-file-owner',
                description: "File owner's username.",
                example: 'alice',
                flat_name: 'threat.enrichments.indicator.file.owner',
                ignore_above: 1024,
                level: 'extended',
                name: 'owner',
                normalize: [],
                original_fieldset: 'file',
                short: "File owner's username.",
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.path': {
                dashed_name: 'threat-enrichments-indicator-file-path',
                description: 'Full path to the file, including the file name. It should include the drive letter, when appropriate.',
                example: '/home/alice/example.png',
                flat_name: 'threat.enrichments.indicator.file.path',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'threat.enrichments.indicator.file.path.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'path',
                normalize: [],
                original_fieldset: 'file',
                short: 'Full path to the file, including the file name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.architecture': {
                dashed_name: 'threat-enrichments-indicator-file-pe-architecture',
                description: 'CPU architecture target for the file.',
                example: 'x64',
                flat_name: 'threat.enrichments.indicator.file.pe.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'pe',
                short: 'CPU architecture target for the file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.company': {
                dashed_name: 'threat-enrichments-indicator-file-pe-company',
                description: 'Internal company name of the file, provided at compile-time.',
                example: 'Microsoft Corporation',
                flat_name: 'threat.enrichments.indicator.file.pe.company',
                ignore_above: 1024,
                level: 'extended',
                name: 'company',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal company name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.description': {
                dashed_name: 'threat-enrichments-indicator-file-pe-description',
                description: 'Internal description of the file, provided at compile-time.',
                example: 'Paint',
                flat_name: 'threat.enrichments.indicator.file.pe.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'description',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal description of the file, provided at compile-time.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.file_version': {
                dashed_name: 'threat-enrichments-indicator-file-pe-file-version',
                description: 'Internal version of the file, provided at compile-time.',
                example: '6.3.9600.17415',
                flat_name: 'threat.enrichments.indicator.file.pe.file_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'file_version',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Process name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.go_import_hash': {
                dashed_name: 'threat-enrichments-indicator-file-pe-go-import-hash',
                description: 'A hash of the Go language imports in a PE file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'threat.enrichments.indicator.file.pe.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the Go language imports in a PE file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.go_imports': {
                dashed_name: 'threat-enrichments-indicator-file-pe-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'threat.enrichments.indicator.file.pe.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'pe',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'threat.enrichments.indicator.file.pe.go_imports_names_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-pe-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'threat.enrichments.indicator.file.pe.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.pe.go_imports_names_var_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-pe-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'threat.enrichments.indicator.file.pe.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.pe.go_stripped': {
                dashed_name: 'threat-enrichments-indicator-file-pe-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'threat.enrichments.indicator.file.pe.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'threat.enrichments.indicator.file.pe.imphash': {
                dashed_name: 'threat-enrichments-indicator-file-pe-imphash',
                description: 'A hash of the imports in a PE file. An imphash -- or import hash -- can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nLearn more at https://www.fireeye.com/blog/threat-research/2014/01/tracking-malware-import-hashing.html.',
                example: '0c6803c4e922103c4dca5963aad36ddf',
                flat_name: 'threat.enrichments.indicator.file.pe.imphash',
                ignore_above: 1024,
                level: 'extended',
                name: 'imphash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.import_hash': {
                dashed_name: 'threat-enrichments-indicator-file-pe-import-hash',
                description: 'A hash of the imports in a PE file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is a synonym for imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'threat.enrichments.indicator.file.pe.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the imports in a PE file.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.imports': {
                dashed_name: 'threat-enrichments-indicator-file-pe-imports',
                description: 'List of imported element names and types.',
                flat_name: 'threat.enrichments.indicator.file.pe.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'threat.enrichments.indicator.file.pe.imports_names_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-pe-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'threat.enrichments.indicator.file.pe.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.pe.imports_names_var_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-pe-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'threat.enrichments.indicator.file.pe.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.pe.original_file_name': {
                dashed_name: 'threat-enrichments-indicator-file-pe-original-file-name',
                description: 'Internal name of the file, provided at compile-time.',
                example: 'MSPAINT.EXE',
                flat_name: 'threat.enrichments.indicator.file.pe.original_file_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'original_file_name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.pehash': {
                dashed_name: 'threat-enrichments-indicator-file-pe-pehash',
                description: 'A hash of the PE header and data from one or more PE sections. An pehash can be used to cluster files by transforming structural information about a file into a hash value.\nLearn more at https://www.usenix.org/legacy/events/leet09/tech/full_papers/wicherski/wicherski_html/index.html.',
                example: '73ff189b63cd6be375a7ff25179a38d347651975',
                flat_name: 'threat.enrichments.indicator.file.pe.pehash',
                ignore_above: 1024,
                level: 'extended',
                name: 'pehash',
                normalize: [],
                original_fieldset: 'pe',
                short: 'A hash of the PE header and data from one or more PE sections.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.product': {
                dashed_name: 'threat-enrichments-indicator-file-pe-product',
                description: 'Internal product name of the file, provided at compile-time.',
                example: 'Microsoft® Windows® Operating System',
                flat_name: 'threat.enrichments.indicator.file.pe.product',
                ignore_above: 1024,
                level: 'extended',
                name: 'product',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Internal product name of the file, provided at compile-time.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.sections': {
                dashed_name: 'threat-enrichments-indicator-file-pe-sections',
                description: 'An array containing an object for each section of the PE file.\nThe keys that should be present in these objects are defined by sub-fields underneath `pe.sections.*`.',
                flat_name: 'threat.enrichments.indicator.file.pe.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'pe',
                short: 'Section information of the PE file.',
                type: 'nested',
            },
            'threat.enrichments.indicator.file.pe.sections.entropy': {
                dashed_name: 'threat-enrichments-indicator-file-pe-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'threat.enrichments.indicator.file.pe.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.pe.sections.name': {
                dashed_name: 'threat-enrichments-indicator-file-pe-sections-name',
                description: 'PE Section List name.',
                flat_name: 'threat.enrichments.indicator.file.pe.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.pe.sections.physical_size': {
                dashed_name: 'threat-enrichments-indicator-file-pe-sections-physical-size',
                description: 'PE Section List physical size.',
                flat_name: 'threat.enrichments.indicator.file.pe.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List physical size.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.pe.sections.var_entropy': {
                dashed_name: 'threat-enrichments-indicator-file-pe-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'threat.enrichments.indicator.file.pe.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'pe',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.pe.sections.virtual_size': {
                dashed_name: 'threat-enrichments-indicator-file-pe-sections-virtual-size',
                description: 'PE Section List virtual size. This is always the same as `physical_size`.',
                flat_name: 'threat.enrichments.indicator.file.pe.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'pe',
                short: 'PE Section List virtual size. This is always the same as `physical_size`.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.size': {
                dashed_name: 'threat-enrichments-indicator-file-size',
                description: 'File size in bytes.\nOnly relevant when `file.type` is "file".',
                example: 16384,
                flat_name: 'threat.enrichments.indicator.file.size',
                level: 'extended',
                name: 'size',
                normalize: [],
                original_fieldset: 'file',
                short: 'File size in bytes.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.target_path': {
                dashed_name: 'threat-enrichments-indicator-file-target-path',
                description: 'Target path for symlinks.',
                flat_name: 'threat.enrichments.indicator.file.target_path',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'threat.enrichments.indicator.file.target_path.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'target_path',
                normalize: [],
                original_fieldset: 'file',
                short: 'Target path for symlinks.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.type': {
                dashed_name: 'threat-enrichments-indicator-file-type',
                description: 'File type (file, dir, or symlink).',
                example: 'file',
                flat_name: 'threat.enrichments.indicator.file.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'type',
                normalize: [],
                original_fieldset: 'file',
                short: 'File type (file, dir, or symlink).',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.uid': {
                dashed_name: 'threat-enrichments-indicator-file-uid',
                description: 'The user ID (UID) or security identifier (SID) of the file owner.',
                example: '1001',
                flat_name: 'threat.enrichments.indicator.file.uid',
                ignore_above: 1024,
                level: 'extended',
                name: 'uid',
                normalize: [],
                original_fieldset: 'file',
                short: 'The user ID (UID) or security identifier (SID) of the file owner.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.alternative_names': {
                dashed_name: 'threat-enrichments-indicator-file-x509-alternative-names',
                description: 'List of subject alternative names (SAN). Name types vary by certificate authority and certificate type but commonly contain IP addresses, DNS names (and wildcards), and email addresses.',
                example: '*.elastic.co',
                flat_name: 'threat.enrichments.indicator.file.x509.alternative_names',
                ignore_above: 1024,
                level: 'extended',
                name: 'alternative_names',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of subject alternative names (SAN).',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.issuer.common_name': {
                dashed_name: 'threat-enrichments-indicator-file-x509-issuer-common-name',
                description: 'List of common name (CN) of issuing certificate authority.',
                example: 'Example SHA2 High Assurance Server CA',
                flat_name: 'threat.enrichments.indicator.file.x509.issuer.common_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.common_name',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of common name (CN) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.issuer.country': {
                dashed_name: 'threat-enrichments-indicator-file-x509-issuer-country',
                description: 'List of country \\(C) codes',
                example: 'US',
                flat_name: 'threat.enrichments.indicator.file.x509.issuer.country',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.country',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of country \\(C) codes',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.issuer.distinguished_name': {
                dashed_name: 'threat-enrichments-indicator-file-x509-issuer-distinguished-name',
                description: 'Distinguished name (DN) of issuing certificate authority.',
                example: 'C=US, O=Example Inc, OU=www.example.com, CN=Example SHA2 High Assurance Server CA',
                flat_name: 'threat.enrichments.indicator.file.x509.issuer.distinguished_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.distinguished_name',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Distinguished name (DN) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.issuer.locality': {
                dashed_name: 'threat-enrichments-indicator-file-x509-issuer-locality',
                description: 'List of locality names (L)',
                example: 'Mountain View',
                flat_name: 'threat.enrichments.indicator.file.x509.issuer.locality',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.locality',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of locality names (L)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.issuer.organization': {
                dashed_name: 'threat-enrichments-indicator-file-x509-issuer-organization',
                description: 'List of organizations (O) of issuing certificate authority.',
                example: 'Example Inc',
                flat_name: 'threat.enrichments.indicator.file.x509.issuer.organization',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.organization',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizations (O) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.issuer.organizational_unit': {
                dashed_name: 'threat-enrichments-indicator-file-x509-issuer-organizational-unit',
                description: 'List of organizational units (OU) of issuing certificate authority.',
                example: 'www.example.com',
                flat_name: 'threat.enrichments.indicator.file.x509.issuer.organizational_unit',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.organizational_unit',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizational units (OU) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.issuer.state_or_province': {
                dashed_name: 'threat-enrichments-indicator-file-x509-issuer-state-or-province',
                description: 'List of state or province names (ST, S, or P)',
                example: 'California',
                flat_name: 'threat.enrichments.indicator.file.x509.issuer.state_or_province',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.state_or_province',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of state or province names (ST, S, or P)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.not_after': {
                dashed_name: 'threat-enrichments-indicator-file-x509-not-after',
                description: 'Time at which the certificate is no longer considered valid.',
                example: '2020-07-16T03:15:39Z',
                flat_name: 'threat.enrichments.indicator.file.x509.not_after',
                level: 'extended',
                name: 'not_after',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Time at which the certificate is no longer considered valid.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.x509.not_before': {
                dashed_name: 'threat-enrichments-indicator-file-x509-not-before',
                description: 'Time at which the certificate is first considered valid.',
                example: '2019-08-16T01:40:25Z',
                flat_name: 'threat.enrichments.indicator.file.x509.not_before',
                level: 'extended',
                name: 'not_before',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Time at which the certificate is first considered valid.',
                type: 'date',
            },
            'threat.enrichments.indicator.file.x509.public_key_algorithm': {
                dashed_name: 'threat-enrichments-indicator-file-x509-public-key-algorithm',
                description: 'Algorithm used to generate the public key.',
                example: 'RSA',
                flat_name: 'threat.enrichments.indicator.file.x509.public_key_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'public_key_algorithm',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Algorithm used to generate the public key.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.public_key_curve': {
                dashed_name: 'threat-enrichments-indicator-file-x509-public-key-curve',
                description: 'The curve used by the elliptic curve public key algorithm. This is algorithm specific.',
                example: 'nistp521',
                flat_name: 'threat.enrichments.indicator.file.x509.public_key_curve',
                ignore_above: 1024,
                level: 'extended',
                name: 'public_key_curve',
                normalize: [],
                original_fieldset: 'x509',
                short: 'The curve used by the elliptic curve public key algorithm. This is algorithm specific.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.public_key_exponent': {
                dashed_name: 'threat-enrichments-indicator-file-x509-public-key-exponent',
                description: 'Exponent used to derive the public key. This is algorithm specific.',
                doc_values: false,
                example: 65537,
                flat_name: 'threat.enrichments.indicator.file.x509.public_key_exponent',
                index: false,
                level: 'extended',
                name: 'public_key_exponent',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Exponent used to derive the public key. This is algorithm specific.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.x509.public_key_size': {
                dashed_name: 'threat-enrichments-indicator-file-x509-public-key-size',
                description: 'The size of the public key space in bits.',
                example: 2048,
                flat_name: 'threat.enrichments.indicator.file.x509.public_key_size',
                level: 'extended',
                name: 'public_key_size',
                normalize: [],
                original_fieldset: 'x509',
                short: 'The size of the public key space in bits.',
                type: 'long',
            },
            'threat.enrichments.indicator.file.x509.serial_number': {
                dashed_name: 'threat-enrichments-indicator-file-x509-serial-number',
                description: 'Unique serial number issued by the certificate authority. For consistency, if this value is alphanumeric, it should be formatted without colons and uppercase characters.',
                example: '55FBB9C7DEBF09809D12CCAA',
                flat_name: 'threat.enrichments.indicator.file.x509.serial_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'serial_number',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Unique serial number issued by the certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.signature_algorithm': {
                dashed_name: 'threat-enrichments-indicator-file-x509-signature-algorithm',
                description: 'Identifier for certificate signature algorithm. We recommend using names found in Go Lang Crypto library. See https://github.com/golang/go/blob/go1.14/src/crypto/x509/x509.go#L337-L353.',
                example: 'SHA256-RSA',
                flat_name: 'threat.enrichments.indicator.file.x509.signature_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'signature_algorithm',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Identifier for certificate signature algorithm.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.subject.common_name': {
                dashed_name: 'threat-enrichments-indicator-file-x509-subject-common-name',
                description: 'List of common names (CN) of subject.',
                example: 'shared.global.example.net',
                flat_name: 'threat.enrichments.indicator.file.x509.subject.common_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.common_name',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of common names (CN) of subject.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.subject.country': {
                dashed_name: 'threat-enrichments-indicator-file-x509-subject-country',
                description: 'List of country \\(C) code',
                example: 'US',
                flat_name: 'threat.enrichments.indicator.file.x509.subject.country',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.country',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of country \\(C) code',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.subject.distinguished_name': {
                dashed_name: 'threat-enrichments-indicator-file-x509-subject-distinguished-name',
                description: 'Distinguished name (DN) of the certificate subject entity.',
                example: 'C=US, ST=California, L=San Francisco, O=Example, Inc., CN=shared.global.example.net',
                flat_name: 'threat.enrichments.indicator.file.x509.subject.distinguished_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.distinguished_name',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Distinguished name (DN) of the certificate subject entity.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.subject.locality': {
                dashed_name: 'threat-enrichments-indicator-file-x509-subject-locality',
                description: 'List of locality names (L)',
                example: 'San Francisco',
                flat_name: 'threat.enrichments.indicator.file.x509.subject.locality',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.locality',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of locality names (L)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.subject.organization': {
                dashed_name: 'threat-enrichments-indicator-file-x509-subject-organization',
                description: 'List of organizations (O) of subject.',
                example: 'Example, Inc.',
                flat_name: 'threat.enrichments.indicator.file.x509.subject.organization',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.organization',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizations (O) of subject.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.subject.organizational_unit': {
                dashed_name: 'threat-enrichments-indicator-file-x509-subject-organizational-unit',
                description: 'List of organizational units (OU) of subject.',
                flat_name: 'threat.enrichments.indicator.file.x509.subject.organizational_unit',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.organizational_unit',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizational units (OU) of subject.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.subject.state_or_province': {
                dashed_name: 'threat-enrichments-indicator-file-x509-subject-state-or-province',
                description: 'List of state or province names (ST, S, or P)',
                example: 'California',
                flat_name: 'threat.enrichments.indicator.file.x509.subject.state_or_province',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.state_or_province',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of state or province names (ST, S, or P)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.file.x509.version_number': {
                dashed_name: 'threat-enrichments-indicator-file-x509-version-number',
                description: 'Version of x509 format.',
                example: 3,
                flat_name: 'threat.enrichments.indicator.file.x509.version_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'version_number',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Version of x509 format.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.first_seen': {
                dashed_name: 'threat-enrichments-indicator-first-seen',
                description: 'The date and time when intelligence source first reported sighting this indicator.',
                example: '2020-11-05T17:25:47.000Z',
                flat_name: 'threat.enrichments.indicator.first_seen',
                level: 'extended',
                name: 'enrichments.indicator.first_seen',
                normalize: [],
                short: 'Date/time indicator was first reported.',
                type: 'date',
            },
            'threat.enrichments.indicator.geo.city_name': {
                dashed_name: 'threat-enrichments-indicator-geo-city-name',
                description: 'City name.',
                example: 'Montreal',
                flat_name: 'threat.enrichments.indicator.geo.city_name',
                ignore_above: 1024,
                level: 'core',
                name: 'city_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'City name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.continent_code': {
                dashed_name: 'threat-enrichments-indicator-geo-continent-code',
                description: "Two-letter code representing continent's name.",
                example: 'NA',
                flat_name: 'threat.enrichments.indicator.geo.continent_code',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Continent code.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.continent_name': {
                dashed_name: 'threat-enrichments-indicator-geo-continent-name',
                description: 'Name of the continent.',
                example: 'North America',
                flat_name: 'threat.enrichments.indicator.geo.continent_name',
                ignore_above: 1024,
                level: 'core',
                name: 'continent_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Name of the continent.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.country_iso_code': {
                dashed_name: 'threat-enrichments-indicator-geo-country-iso-code',
                description: 'Country ISO code.',
                example: 'CA',
                flat_name: 'threat.enrichments.indicator.geo.country_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'country_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country ISO code.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.country_name': {
                dashed_name: 'threat-enrichments-indicator-geo-country-name',
                description: 'Country name.',
                example: 'Canada',
                flat_name: 'threat.enrichments.indicator.geo.country_name',
                ignore_above: 1024,
                level: 'core',
                name: 'country_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Country name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.location': {
                dashed_name: 'threat-enrichments-indicator-geo-location',
                description: 'Longitude and latitude.',
                example: '{ "lon": -73.614830, "lat": 45.505918 }',
                flat_name: 'threat.enrichments.indicator.geo.location',
                level: 'core',
                name: 'location',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Longitude and latitude.',
                type: 'geo_point',
            },
            'threat.enrichments.indicator.geo.name': {
                dashed_name: 'threat-enrichments-indicator-geo-name',
                description: 'User-defined description of a location, at the level of granularity they care about.\nCould be the name of their data centers, the floor number, if this describes a local physical entity, city names.\nNot typically used in automated geolocation.',
                example: 'boston-dc',
                flat_name: 'threat.enrichments.indicator.geo.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'User-defined description of a location.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.postal_code': {
                dashed_name: 'threat-enrichments-indicator-geo-postal-code',
                description: 'Postal code associated with the location.\nValues appropriate for this field may also be known as a postcode or ZIP code and will vary widely from country to country.',
                example: 94040,
                flat_name: 'threat.enrichments.indicator.geo.postal_code',
                ignore_above: 1024,
                level: 'core',
                name: 'postal_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Postal code.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.region_iso_code': {
                dashed_name: 'threat-enrichments-indicator-geo-region-iso-code',
                description: 'Region ISO code.',
                example: 'CA-QC',
                flat_name: 'threat.enrichments.indicator.geo.region_iso_code',
                ignore_above: 1024,
                level: 'core',
                name: 'region_iso_code',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region ISO code.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.region_name': {
                dashed_name: 'threat-enrichments-indicator-geo-region-name',
                description: 'Region name.',
                example: 'Quebec',
                flat_name: 'threat.enrichments.indicator.geo.region_name',
                ignore_above: 1024,
                level: 'core',
                name: 'region_name',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Region name.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.geo.timezone': {
                dashed_name: 'threat-enrichments-indicator-geo-timezone',
                description: 'The time zone of the location, such as IANA time zone name.',
                example: 'America/Argentina/Buenos_Aires',
                flat_name: 'threat.enrichments.indicator.geo.timezone',
                ignore_above: 1024,
                level: 'core',
                name: 'timezone',
                normalize: [],
                original_fieldset: 'geo',
                short: 'Time zone.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.ip': {
                dashed_name: 'threat-enrichments-indicator-ip',
                description: 'Identifies a threat indicator as an IP address (irrespective of direction).',
                example: '1.2.3.4',
                flat_name: 'threat.enrichments.indicator.ip',
                level: 'extended',
                name: 'enrichments.indicator.ip',
                normalize: [],
                short: 'Indicator IP address',
                type: 'ip',
            },
            'threat.enrichments.indicator.last_seen': {
                dashed_name: 'threat-enrichments-indicator-last-seen',
                description: 'The date and time when intelligence source last reported sighting this indicator.',
                example: '2020-11-05T17:25:47.000Z',
                flat_name: 'threat.enrichments.indicator.last_seen',
                level: 'extended',
                name: 'enrichments.indicator.last_seen',
                normalize: [],
                short: 'Date/time indicator was last reported.',
                type: 'date',
            },
            'threat.enrichments.indicator.marking.tlp': {
                dashed_name: 'threat-enrichments-indicator-marking-tlp',
                description: 'Traffic Light Protocol sharing markings.',
                example: 'CLEAR',
                expected_values: [
                    'WHITE',
                    'CLEAR',
                    'GREEN',
                    'AMBER',
                    'AMBER+STRICT',
                    'RED',
                ],
                flat_name: 'threat.enrichments.indicator.marking.tlp',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.marking.tlp',
                normalize: [],
                short: 'Indicator TLP marking',
                type: 'keyword',
            },
            'threat.enrichments.indicator.marking.tlp_version': {
                dashed_name: 'threat-enrichments-indicator-marking-tlp-version',
                description: 'Traffic Light Protocol version.',
                example: 2,
                flat_name: 'threat.enrichments.indicator.marking.tlp_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.marking.tlp_version',
                normalize: [],
                short: 'Indicator TLP version',
                type: 'keyword',
            },
            'threat.enrichments.indicator.modified_at': {
                dashed_name: 'threat-enrichments-indicator-modified-at',
                description: 'The date and time when intelligence source last modified information for this indicator.',
                example: '2020-11-05T17:25:47.000Z',
                flat_name: 'threat.enrichments.indicator.modified_at',
                level: 'extended',
                name: 'enrichments.indicator.modified_at',
                normalize: [],
                short: 'Date/time indicator was last updated.',
                type: 'date',
            },
            'threat.enrichments.indicator.name': {
                dashed_name: 'threat-enrichments-indicator-name',
                description: 'The display name indicator in an UI friendly format\nURL, IP address, email address, registry key, port number, hash value, or other relevant name can serve as the display name.',
                example: '5.2.75.227',
                flat_name: 'threat.enrichments.indicator.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.name',
                normalize: [],
                short: 'Indicator display name',
                type: 'keyword',
            },
            'threat.enrichments.indicator.port': {
                dashed_name: 'threat-enrichments-indicator-port',
                description: 'Identifies a threat indicator as a port number (irrespective of direction).',
                example: 443,
                flat_name: 'threat.enrichments.indicator.port',
                level: 'extended',
                name: 'enrichments.indicator.port',
                normalize: [],
                short: 'Indicator port',
                type: 'long',
            },
            'threat.enrichments.indicator.provider': {
                dashed_name: 'threat-enrichments-indicator-provider',
                description: "The name of the indicator's provider.",
                example: 'lrz_urlhaus',
                flat_name: 'threat.enrichments.indicator.provider',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.provider',
                normalize: [],
                short: 'Indicator provider',
                type: 'keyword',
            },
            'threat.enrichments.indicator.reference': {
                dashed_name: 'threat-enrichments-indicator-reference',
                description: 'Reference URL linking to additional information about this indicator.',
                example: 'https://system.example.com/indicator/0001234',
                flat_name: 'threat.enrichments.indicator.reference',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.reference',
                normalize: [],
                short: 'Indicator reference URL',
                type: 'keyword',
            },
            'threat.enrichments.indicator.registry.data.bytes': {
                dashed_name: 'threat-enrichments-indicator-registry-data-bytes',
                description: 'Original bytes written with base64 encoding.\nFor Windows registry operations, such as SetValueEx and RegQueryValueEx, this corresponds to the data pointed by `lp_data`. This is optional but provides better recoverability and should be populated for REG_BINARY encoded values.',
                example: 'ZQBuAC0AVQBTAAAAZQBuAAAAAAA=',
                flat_name: 'threat.enrichments.indicator.registry.data.bytes',
                ignore_above: 1024,
                level: 'extended',
                name: 'data.bytes',
                normalize: [],
                original_fieldset: 'registry',
                short: 'Original bytes written with base64 encoding.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.registry.data.strings': {
                dashed_name: 'threat-enrichments-indicator-registry-data-strings',
                description: 'Content when writing string types.\nPopulated as an array when writing string data to the registry. For single string registry types (REG_SZ, REG_EXPAND_SZ), this should be an array with one string. For sequences of string with REG_MULTI_SZ, this array will be variable length. For numeric data, such as REG_DWORD and REG_QWORD, this should be populated with the decimal representation (e.g `"1"`).',
                example: '["C:\\rta\\red_ttp\\bin\\myapp.exe"]',
                flat_name: 'threat.enrichments.indicator.registry.data.strings',
                level: 'core',
                name: 'data.strings',
                normalize: ['array'],
                original_fieldset: 'registry',
                short: 'List of strings representing what was written to the registry.',
                type: 'wildcard',
            },
            'threat.enrichments.indicator.registry.data.type': {
                dashed_name: 'threat-enrichments-indicator-registry-data-type',
                description: 'Standard registry type for encoding contents',
                example: 'REG_SZ',
                flat_name: 'threat.enrichments.indicator.registry.data.type',
                ignore_above: 1024,
                level: 'core',
                name: 'data.type',
                normalize: [],
                original_fieldset: 'registry',
                short: 'Standard registry type for encoding contents',
                type: 'keyword',
            },
            'threat.enrichments.indicator.registry.hive': {
                dashed_name: 'threat-enrichments-indicator-registry-hive',
                description: 'Abbreviated name for the hive.',
                example: 'HKLM',
                flat_name: 'threat.enrichments.indicator.registry.hive',
                ignore_above: 1024,
                level: 'core',
                name: 'hive',
                normalize: [],
                original_fieldset: 'registry',
                short: 'Abbreviated name for the hive.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.registry.key': {
                dashed_name: 'threat-enrichments-indicator-registry-key',
                description: 'Hive-relative path of keys.',
                example: 'SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Image File Execution Options\\winword.exe',
                flat_name: 'threat.enrichments.indicator.registry.key',
                ignore_above: 1024,
                level: 'core',
                name: 'key',
                normalize: [],
                original_fieldset: 'registry',
                short: 'Hive-relative path of keys.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.registry.path': {
                dashed_name: 'threat-enrichments-indicator-registry-path',
                description: 'Full path, including hive, key and value',
                example: 'HKLM\\SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Image File Execution Options\\winword.exe\\Debugger',
                flat_name: 'threat.enrichments.indicator.registry.path',
                ignore_above: 1024,
                level: 'core',
                name: 'path',
                normalize: [],
                original_fieldset: 'registry',
                short: 'Full path, including hive, key and value',
                type: 'keyword',
            },
            'threat.enrichments.indicator.registry.value': {
                dashed_name: 'threat-enrichments-indicator-registry-value',
                description: 'Name of the value written.',
                example: 'Debugger',
                flat_name: 'threat.enrichments.indicator.registry.value',
                ignore_above: 1024,
                level: 'core',
                name: 'value',
                normalize: [],
                original_fieldset: 'registry',
                short: 'Name of the value written.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.scanner_stats': {
                dashed_name: 'threat-enrichments-indicator-scanner-stats',
                description: 'Count of AV/EDR vendors that successfully detected malicious file or URL.',
                example: 4,
                flat_name: 'threat.enrichments.indicator.scanner_stats',
                level: 'extended',
                name: 'enrichments.indicator.scanner_stats',
                normalize: [],
                short: 'Scanner statistics',
                type: 'long',
            },
            'threat.enrichments.indicator.sightings': {
                dashed_name: 'threat-enrichments-indicator-sightings',
                description: 'Number of times this indicator was observed conducting threat activity.',
                example: 20,
                flat_name: 'threat.enrichments.indicator.sightings',
                level: 'extended',
                name: 'enrichments.indicator.sightings',
                normalize: [],
                short: 'Number of times indicator observed',
                type: 'long',
            },
            'threat.enrichments.indicator.type': {
                dashed_name: 'threat-enrichments-indicator-type',
                description: 'Type of indicator as represented by Cyber Observable in STIX 2.0.',
                example: 'ipv4-addr',
                expected_values: [
                    'autonomous-system',
                    'artifact',
                    'directory',
                    'domain-name',
                    'email-addr',
                    'file',
                    'ipv4-addr',
                    'ipv6-addr',
                    'mac-addr',
                    'mutex',
                    'port',
                    'process',
                    'software',
                    'url',
                    'user-account',
                    'windows-registry-key',
                    'x509-certificate',
                ],
                flat_name: 'threat.enrichments.indicator.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.indicator.type',
                normalize: [],
                short: 'Type of indicator',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.domain': {
                dashed_name: 'threat-enrichments-indicator-url-domain',
                description: 'Domain of the url, such as "www.elastic.co".\nIn some cases a URL may refer to an IP and/or port directly, without a domain name. In this case, the IP address would go to the `domain` field.\nIf the URL contains a literal IPv6 address enclosed by `[` and `]` (IETF RFC 2732), the `[` and `]` characters should also be captured in the `domain` field.',
                example: 'www.elastic.co',
                flat_name: 'threat.enrichments.indicator.url.domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'domain',
                normalize: [],
                original_fieldset: 'url',
                short: 'Domain of the url.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.extension': {
                dashed_name: 'threat-enrichments-indicator-url-extension',
                description: 'The field contains the file extension from the original request url, excluding the leading dot.\nThe file extension is only set if it exists, as not every url has a file extension.\nThe leading period must not be included. For example, the value must be "png", not ".png".\nNote that when the file name has multiple extensions (example.tar.gz), only the last one should be captured ("gz", not "tar.gz").',
                example: 'png',
                flat_name: 'threat.enrichments.indicator.url.extension',
                ignore_above: 1024,
                level: 'extended',
                name: 'extension',
                normalize: [],
                original_fieldset: 'url',
                short: 'File extension from the request url, excluding the leading dot.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.fragment': {
                dashed_name: 'threat-enrichments-indicator-url-fragment',
                description: 'Portion of the url after the `#`, such as "top".\nThe `#` is not part of the fragment.',
                flat_name: 'threat.enrichments.indicator.url.fragment',
                ignore_above: 1024,
                level: 'extended',
                name: 'fragment',
                normalize: [],
                original_fieldset: 'url',
                short: 'Portion of the url after the `#`.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.full': {
                dashed_name: 'threat-enrichments-indicator-url-full',
                description: 'If full URLs are important to your use case, they should be stored in `url.full`, whether this field is reconstructed or present in the event source.',
                example: 'https://www.elastic.co:443/search?q=elasticsearch#top',
                flat_name: 'threat.enrichments.indicator.url.full',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'threat.enrichments.indicator.url.full.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'full',
                normalize: [],
                original_fieldset: 'url',
                short: 'Full unparsed URL.',
                type: 'wildcard',
            },
            'threat.enrichments.indicator.url.original': {
                dashed_name: 'threat-enrichments-indicator-url-original',
                description: 'Unmodified original url as seen in the event source.\nNote that in network monitoring, the observed URL may be a full URL, whereas in access logs, the URL is often just represented as a path.\nThis field is meant to represent the URL as it was observed, complete or not.',
                example: 'https://www.elastic.co:443/search?q=elasticsearch#top or /search?q=elasticsearch',
                flat_name: 'threat.enrichments.indicator.url.original',
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'threat.enrichments.indicator.url.original.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'original',
                normalize: [],
                original_fieldset: 'url',
                short: 'Unmodified original url as seen in the event source.',
                type: 'wildcard',
            },
            'threat.enrichments.indicator.url.password': {
                dashed_name: 'threat-enrichments-indicator-url-password',
                description: 'Password of the request.',
                flat_name: 'threat.enrichments.indicator.url.password',
                ignore_above: 1024,
                level: 'extended',
                name: 'password',
                normalize: [],
                original_fieldset: 'url',
                short: 'Password of the request.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.path': {
                dashed_name: 'threat-enrichments-indicator-url-path',
                description: 'Path of the request, such as "/search".',
                flat_name: 'threat.enrichments.indicator.url.path',
                level: 'extended',
                name: 'path',
                normalize: [],
                original_fieldset: 'url',
                short: 'Path of the request, such as "/search".',
                type: 'wildcard',
            },
            'threat.enrichments.indicator.url.port': {
                dashed_name: 'threat-enrichments-indicator-url-port',
                description: 'Port of the request, such as 443.',
                example: 443,
                flat_name: 'threat.enrichments.indicator.url.port',
                format: 'string',
                level: 'extended',
                name: 'port',
                normalize: [],
                original_fieldset: 'url',
                short: 'Port of the request, such as 443.',
                type: 'long',
            },
            'threat.enrichments.indicator.url.query': {
                dashed_name: 'threat-enrichments-indicator-url-query',
                description: 'The query field describes the query string of the request, such as "q=elasticsearch".\nThe `?` is excluded from the query string. If a URL contains no `?`, there is no query field. If there is a `?` but no query, the query field exists with an empty string. The `exists` query can be used to differentiate between the two cases.',
                flat_name: 'threat.enrichments.indicator.url.query',
                ignore_above: 1024,
                level: 'extended',
                name: 'query',
                normalize: [],
                original_fieldset: 'url',
                short: 'Query string of the request.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.registered_domain': {
                dashed_name: 'threat-enrichments-indicator-url-registered-domain',
                description: 'The highest registered url domain, stripped of the subdomain.\nFor example, the registered domain for "foo.example.com" is "example.com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last two labels will not work well for TLDs such as "co.uk".',
                example: 'example.com',
                flat_name: 'threat.enrichments.indicator.url.registered_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'registered_domain',
                normalize: [],
                original_fieldset: 'url',
                short: 'The highest registered url domain, stripped of the subdomain.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.scheme': {
                dashed_name: 'threat-enrichments-indicator-url-scheme',
                description: 'Scheme of the request, such as "https".\nNote: The `:` is not part of the scheme.',
                example: 'https',
                flat_name: 'threat.enrichments.indicator.url.scheme',
                ignore_above: 1024,
                level: 'extended',
                name: 'scheme',
                normalize: [],
                original_fieldset: 'url',
                short: 'Scheme of the url.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.subdomain': {
                dashed_name: 'threat-enrichments-indicator-url-subdomain',
                description: 'The subdomain portion of a fully qualified domain name includes all of the names except the host name under the registered_domain.  In a partially qualified domain, or if the the qualification level of the full name cannot be determined, subdomain contains all of the names below the registered domain.\nFor example the subdomain portion of "www.east.mydomain.co.uk" is "east". If the domain has multiple levels of subdomain, such as "sub2.sub1.example.com", the subdomain field should contain "sub2.sub1", with no trailing period.',
                example: 'east',
                flat_name: 'threat.enrichments.indicator.url.subdomain',
                ignore_above: 1024,
                level: 'extended',
                name: 'subdomain',
                normalize: [],
                original_fieldset: 'url',
                short: 'The subdomain of the domain.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.top_level_domain': {
                dashed_name: 'threat-enrichments-indicator-url-top-level-domain',
                description: 'The effective top level domain (eTLD), also known as the domain suffix, is the last part of the domain name. For example, the top level domain for example.com is "com".\nThis value can be determined precisely with a list like the public suffix list (http://publicsuffix.org). Trying to approximate this by simply taking the last label will not work well for effective TLDs such as "co.uk".',
                example: 'co.uk',
                flat_name: 'threat.enrichments.indicator.url.top_level_domain',
                ignore_above: 1024,
                level: 'extended',
                name: 'top_level_domain',
                normalize: [],
                original_fieldset: 'url',
                short: 'The effective top level domain (com, org, net, co.uk).',
                type: 'keyword',
            },
            'threat.enrichments.indicator.url.username': {
                dashed_name: 'threat-enrichments-indicator-url-username',
                description: 'Username of the request.',
                flat_name: 'threat.enrichments.indicator.url.username',
                ignore_above: 1024,
                level: 'extended',
                name: 'username',
                normalize: [],
                original_fieldset: 'url',
                short: 'Username of the request.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.alternative_names': {
                dashed_name: 'threat-enrichments-indicator-x509-alternative-names',
                description: 'List of subject alternative names (SAN). Name types vary by certificate authority and certificate type but commonly contain IP addresses, DNS names (and wildcards), and email addresses.',
                example: '*.elastic.co',
                flat_name: 'threat.enrichments.indicator.x509.alternative_names',
                ignore_above: 1024,
                level: 'extended',
                name: 'alternative_names',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of subject alternative names (SAN).',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.issuer.common_name': {
                dashed_name: 'threat-enrichments-indicator-x509-issuer-common-name',
                description: 'List of common name (CN) of issuing certificate authority.',
                example: 'Example SHA2 High Assurance Server CA',
                flat_name: 'threat.enrichments.indicator.x509.issuer.common_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.common_name',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of common name (CN) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.issuer.country': {
                dashed_name: 'threat-enrichments-indicator-x509-issuer-country',
                description: 'List of country \\(C) codes',
                example: 'US',
                flat_name: 'threat.enrichments.indicator.x509.issuer.country',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.country',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of country \\(C) codes',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.issuer.distinguished_name': {
                dashed_name: 'threat-enrichments-indicator-x509-issuer-distinguished-name',
                description: 'Distinguished name (DN) of issuing certificate authority.',
                example: 'C=US, O=Example Inc, OU=www.example.com, CN=Example SHA2 High Assurance Server CA',
                flat_name: 'threat.enrichments.indicator.x509.issuer.distinguished_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.distinguished_name',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Distinguished name (DN) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.issuer.locality': {
                dashed_name: 'threat-enrichments-indicator-x509-issuer-locality',
                description: 'List of locality names (L)',
                example: 'Mountain View',
                flat_name: 'threat.enrichments.indicator.x509.issuer.locality',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.locality',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of locality names (L)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.issuer.organization': {
                dashed_name: 'threat-enrichments-indicator-x509-issuer-organization',
                description: 'List of organizations (O) of issuing certificate authority.',
                example: 'Example Inc',
                flat_name: 'threat.enrichments.indicator.x509.issuer.organization',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.organization',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizations (O) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.issuer.organizational_unit': {
                dashed_name: 'threat-enrichments-indicator-x509-issuer-organizational-unit',
                description: 'List of organizational units (OU) of issuing certificate authority.',
                example: 'www.example.com',
                flat_name: 'threat.enrichments.indicator.x509.issuer.organizational_unit',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.organizational_unit',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizational units (OU) of issuing certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.issuer.state_or_province': {
                dashed_name: 'threat-enrichments-indicator-x509-issuer-state-or-province',
                description: 'List of state or province names (ST, S, or P)',
                example: 'California',
                flat_name: 'threat.enrichments.indicator.x509.issuer.state_or_province',
                ignore_above: 1024,
                level: 'extended',
                name: 'issuer.state_or_province',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of state or province names (ST, S, or P)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.not_after': {
                dashed_name: 'threat-enrichments-indicator-x509-not-after',
                description: 'Time at which the certificate is no longer considered valid.',
                example: '2020-07-16T03:15:39Z',
                flat_name: 'threat.enrichments.indicator.x509.not_after',
                level: 'extended',
                name: 'not_after',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Time at which the certificate is no longer considered valid.',
                type: 'date',
            },
            'threat.enrichments.indicator.x509.not_before': {
                dashed_name: 'threat-enrichments-indicator-x509-not-before',
                description: 'Time at which the certificate is first considered valid.',
                example: '2019-08-16T01:40:25Z',
                flat_name: 'threat.enrichments.indicator.x509.not_before',
                level: 'extended',
                name: 'not_before',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Time at which the certificate is first considered valid.',
                type: 'date',
            },
            'threat.enrichments.indicator.x509.public_key_algorithm': {
                dashed_name: 'threat-enrichments-indicator-x509-public-key-algorithm',
                description: 'Algorithm used to generate the public key.',
                example: 'RSA',
                flat_name: 'threat.enrichments.indicator.x509.public_key_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'public_key_algorithm',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Algorithm used to generate the public key.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.public_key_curve': {
                dashed_name: 'threat-enrichments-indicator-x509-public-key-curve',
                description: 'The curve used by the elliptic curve public key algorithm. This is algorithm specific.',
                example: 'nistp521',
                flat_name: 'threat.enrichments.indicator.x509.public_key_curve',
                ignore_above: 1024,
                level: 'extended',
                name: 'public_key_curve',
                normalize: [],
                original_fieldset: 'x509',
                short: 'The curve used by the elliptic curve public key algorithm. This is algorithm specific.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.public_key_exponent': {
                dashed_name: 'threat-enrichments-indicator-x509-public-key-exponent',
                description: 'Exponent used to derive the public key. This is algorithm specific.',
                doc_values: false,
                example: 65537,
                flat_name: 'threat.enrichments.indicator.x509.public_key_exponent',
                index: false,
                level: 'extended',
                name: 'public_key_exponent',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Exponent used to derive the public key. This is algorithm specific.',
                type: 'long',
            },
            'threat.enrichments.indicator.x509.public_key_size': {
                dashed_name: 'threat-enrichments-indicator-x509-public-key-size',
                description: 'The size of the public key space in bits.',
                example: 2048,
                flat_name: 'threat.enrichments.indicator.x509.public_key_size',
                level: 'extended',
                name: 'public_key_size',
                normalize: [],
                original_fieldset: 'x509',
                short: 'The size of the public key space in bits.',
                type: 'long',
            },
            'threat.enrichments.indicator.x509.serial_number': {
                dashed_name: 'threat-enrichments-indicator-x509-serial-number',
                description: 'Unique serial number issued by the certificate authority. For consistency, if this value is alphanumeric, it should be formatted without colons and uppercase characters.',
                example: '55FBB9C7DEBF09809D12CCAA',
                flat_name: 'threat.enrichments.indicator.x509.serial_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'serial_number',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Unique serial number issued by the certificate authority.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.signature_algorithm': {
                dashed_name: 'threat-enrichments-indicator-x509-signature-algorithm',
                description: 'Identifier for certificate signature algorithm. We recommend using names found in Go Lang Crypto library. See https://github.com/golang/go/blob/go1.14/src/crypto/x509/x509.go#L337-L353.',
                example: 'SHA256-RSA',
                flat_name: 'threat.enrichments.indicator.x509.signature_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'signature_algorithm',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Identifier for certificate signature algorithm.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.subject.common_name': {
                dashed_name: 'threat-enrichments-indicator-x509-subject-common-name',
                description: 'List of common names (CN) of subject.',
                example: 'shared.global.example.net',
                flat_name: 'threat.enrichments.indicator.x509.subject.common_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.common_name',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of common names (CN) of subject.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.subject.country': {
                dashed_name: 'threat-enrichments-indicator-x509-subject-country',
                description: 'List of country \\(C) code',
                example: 'US',
                flat_name: 'threat.enrichments.indicator.x509.subject.country',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.country',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of country \\(C) code',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.subject.distinguished_name': {
                dashed_name: 'threat-enrichments-indicator-x509-subject-distinguished-name',
                description: 'Distinguished name (DN) of the certificate subject entity.',
                example: 'C=US, ST=California, L=San Francisco, O=Example, Inc., CN=shared.global.example.net',
                flat_name: 'threat.enrichments.indicator.x509.subject.distinguished_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.distinguished_name',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Distinguished name (DN) of the certificate subject entity.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.subject.locality': {
                dashed_name: 'threat-enrichments-indicator-x509-subject-locality',
                description: 'List of locality names (L)',
                example: 'San Francisco',
                flat_name: 'threat.enrichments.indicator.x509.subject.locality',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.locality',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of locality names (L)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.subject.organization': {
                dashed_name: 'threat-enrichments-indicator-x509-subject-organization',
                description: 'List of organizations (O) of subject.',
                example: 'Example, Inc.',
                flat_name: 'threat.enrichments.indicator.x509.subject.organization',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.organization',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizations (O) of subject.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.subject.organizational_unit': {
                dashed_name: 'threat-enrichments-indicator-x509-subject-organizational-unit',
                description: 'List of organizational units (OU) of subject.',
                flat_name: 'threat.enrichments.indicator.x509.subject.organizational_unit',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.organizational_unit',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of organizational units (OU) of subject.',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.subject.state_or_province': {
                dashed_name: 'threat-enrichments-indicator-x509-subject-state-or-province',
                description: 'List of state or province names (ST, S, or P)',
                example: 'California',
                flat_name: 'threat.enrichments.indicator.x509.subject.state_or_province',
                ignore_above: 1024,
                level: 'extended',
                name: 'subject.state_or_province',
                normalize: ['array'],
                original_fieldset: 'x509',
                short: 'List of state or province names (ST, S, or P)',
                type: 'keyword',
            },
            'threat.enrichments.indicator.x509.version_number': {
                dashed_name: 'threat-enrichments-indicator-x509-version-number',
                description: 'Version of x509 format.',
                example: 3,
                flat_name: 'threat.enrichments.indicator.x509.version_number',
                ignore_above: 1024,
                level: 'extended',
                name: 'version_number',
                normalize: [],
                original_fieldset: 'x509',
                short: 'Version of x509 format.',
                type: 'keyword',
            },
            'threat.enrichments.matched.atomic': {
                dashed_name: 'threat-enrichments-matched-atomic',
                description: 'Identifies the atomic indicator value that matched a local environment endpoint or network event.',
                example: 'bad-domain.com',
                flat_name: 'threat.enrichments.matched.atomic',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.matched.atomic',
                normalize: [],
                short: 'Matched indicator value',
                type: 'keyword',
            },
            'threat.enrichments.matched.field': {
                dashed_name: 'threat-enrichments-matched-field',
                description: 'Identifies the field of the atomic indicator that matched a local environment endpoint or network event.',
                example: 'file.hash.sha256',
                flat_name: 'threat.enrichments.matched.field',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.matched.field',
                normalize: [],
                short: 'Matched indicator field',
                type: 'keyword',
            },
            'threat.enrichments.matched.id': {
                dashed_name: 'threat-enrichments-matched-id',
                description: 'Identifies the _id of the indicator document enriching the event.',
                example: 'ff93aee5-86a1-4a61-b0e6-0cdc313d01b5',
                flat_name: 'threat.enrichments.matched.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.matched.id',
                normalize: [],
                short: 'Matched indicator identifier',
                type: 'keyword',
            },
            'threat.enrichments.matched.index': {
                dashed_name: 'threat-enrichments-matched-index',
                description: 'Identifies the _index of the indicator document enriching the event.',
                example: 'filebeat-8.0.0-2021.05.23-000011',
                flat_name: 'threat.enrichments.matched.index',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.matched.index',
                normalize: [],
                short: 'Matched indicator index',
                type: 'keyword',
            },
            'threat.enrichments.matched.occurred': {
                dashed_name: 'threat-enrichments-matched-occurred',
                description: 'Indicates when the indicator match was generated',
                example: '2021-10-05T17:00:58.326Z',
                flat_name: 'threat.enrichments.matched.occurred',
                level: 'extended',
                name: 'enrichments.matched.occurred',
                normalize: [],
                short: 'Date of match',
                type: 'date',
            },
            'threat.enrichments.matched.type': {
                dashed_name: 'threat-enrichments-matched-type',
                description: 'Identifies the type of match that caused the event to be enriched with the given indicator',
                example: 'indicator_match_rule',
                flat_name: 'threat.enrichments.matched.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'enrichments.matched.type',
                normalize: [],
                short: 'Type of indicator match',
                type: 'keyword',
            },
            'threat.feed.dashboard_id': {
                dashed_name: 'threat-feed-dashboard-id',
                description: 'The saved object ID of the dashboard belonging to the threat feed for displaying dashboard links to threat feeds in Kibana.',
                example: '5ba16340-72e6-11eb-a3e3-b3cc7c78a70f',
                flat_name: 'threat.feed.dashboard_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'feed.dashboard_id',
                normalize: [],
                short: 'Feed dashboard ID.',
                type: 'keyword',
            },
            'threat.feed.description': {
                dashed_name: 'threat-feed-description',
                description: 'Description of the threat feed in a UI friendly format.',
                example: 'Threat feed from the AlienVault Open Threat eXchange network.',
                flat_name: 'threat.feed.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'feed.description',
                normalize: [],
                short: 'Description of the threat feed.',
                type: 'keyword',
            },
            'threat.feed.name': {
                dashed_name: 'threat-feed-name',
                description: 'The name of the threat feed in UI friendly format.',
                example: 'AlienVault OTX',
                flat_name: 'threat.feed.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'feed.name',
                normalize: [],
                short: 'Name of the threat feed.',
                type: 'keyword',
            },
            'threat.feed.reference': {
                dashed_name: 'threat-feed-reference',
                description: 'Reference information for the threat feed in a UI friendly format.',
                example: 'https://otx.alienvault.com',
                flat_name: 'threat.feed.reference',
                ignore_above: 1024,
                level: 'extended',
                name: 'feed.reference',
                normalize: [],
                short: 'Reference for the threat feed.',
                type: 'keyword',
            },
            'threat.framework': {
                dashed_name: 'threat-framework',
                description: 'Name of the threat framework used to further categorize and classify the tactic and technique of the reported threat. Framework classification can be provided by detecting systems, evaluated at ingest time, or retrospectively tagged to events.',
                example: 'MITRE ATT&CK',
                flat_name: 'threat.framework',
                ignore_above: 1024,
                level: 'extended',
                name: 'framework',
                normalize: [],
                short: 'Threat classification framework.',
                type: 'keyword',
            },
            'threat.group.alias': {
                dashed_name: 'threat-group-alias',
                description: 'The alias(es) of the group for a set of related intrusion activity that are tracked by a common name in the security community.\nWhile not required, you can use a MITRE ATT&CK® group alias(es).',
                example: '[ "Magecart Group 6" ]',
                flat_name: 'threat.group.alias',
                ignore_above: 1024,
                level: 'extended',
                name: 'group.alias',
                normalize: ['array'],
                short: 'Alias of the group.',
                type: 'keyword',
            },
            'threat.group.id': {
                dashed_name: 'threat-group-id',
                description: 'The id of the group for a set of related intrusion activity that are tracked by a common name in the security community.\nWhile not required, you can use a MITRE ATT&CK® group id.',
                example: 'G0037',
                flat_name: 'threat.group.id',
                ignore_above: 1024,
                level: 'extended',
                name: 'group.id',
                normalize: [],
                short: 'ID of the group.',
                type: 'keyword',
            },
            'threat.group.name': {
                dashed_name: 'threat-group-name',
                description: 'The name of the group for a set of related intrusion activity that are tracked by a common name in the security community.\nWhile not required, you can use a MITRE ATT&CK® group name.',
                example: 'FIN6',
                flat_name: 'threat.group.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'group.name',
                normalize: [],
                short: 'Name of the group.',
                type: 'keyword',
            },
            'threat.group.reference': {
                dashed_name: 'threat-group-reference',
                description: 'The reference URL of the group for a set of related intrusion activity that are tracked by a common name in the security community.\nWhile not required, you can use a MITRE ATT&CK® group reference URL.',
                example: 'https://attack.mitre.org/groups/G0037/',
                flat_name: 'threat.group.reference',
                ignore_above: 1024,
                level: 'extended',
                name: 'group.reference',
                normalize: [],
                short: 'Reference URL of the group.',
                type: 'keyword',
            },
            'threat.indicator.as.number': {
                dashed_name: 'threat-indicator-as-number',
                description: 'Unique number allocated to the autonomous system. The autonomous system number (ASN) uniquely identifies each network on the Internet.',
                example: 15169,
                flat_name: 'threat.indicator.as.number',
                level: 'extended',
                name: 'number',
                normalize: [],
                original_fieldset: 'as',
                short: 'Unique number allocated to the autonomous system.',
                type: 'long',
            },
            'threat.indicator.as.organization.name': {
                dashed_name: 'threat-indicator-as-organization-name',
                description: 'Organization name.',
                example: 'Google LLC',
                flat_name: 'threat.indicator.as.organization.name',
                ignore_above: 1024,
                level: 'extended',
                multi_fields: [
                    {
                        flat_name: 'threat.indicator.as.organization.name.text',
                        name: 'text',
                        type: 'match_only_text',
                    },
                ],
                name: 'organization.name',
                normalize: [],
                original_fieldset: 'as',
                short: 'Organization name.',
                type: 'keyword',
            },
            'threat.indicator.confidence': {
                dashed_name: 'threat-indicator-confidence',
                description: 'Identifies the vendor-neutral confidence rating using the None/Low/Medium/High scale defined in Appendix A of the STIX 2.1 framework. Vendor-specific confidence scales may be added as custom fields.',
                example: 'Medium',
                expected_values: ['Not Specified', 'None', 'Low', 'Medium', 'High'],
                flat_name: 'threat.indicator.confidence',
                ignore_above: 1024,
                level: 'extended',
                name: 'indicator.confidence',
                normalize: [],
                short: 'Indicator confidence rating',
                type: 'keyword',
            },
            'threat.indicator.description': {
                dashed_name: 'threat-indicator-description',
                description: 'Describes the type of action conducted by the threat.',
                example: 'IP x.x.x.x was observed delivering the Angler EK.',
                flat_name: 'threat.indicator.description',
                ignore_above: 1024,
                level: 'extended',
                name: 'indicator.description',
                normalize: [],
                short: 'Indicator description',
                type: 'keyword',
            },
            'threat.indicator.email.address': {
                dashed_name: 'threat-indicator-email-address',
                description: 'Identifies a threat indicator as an email address (irrespective of direction).',
                example: 'phish@example.com',
                flat_name: 'threat.indicator.email.address',
                ignore_above: 1024,
                level: 'extended',
                name: 'indicator.email.address',
                normalize: [],
                short: 'Indicator email address',
                type: 'keyword',
            },
            'threat.indicator.file.accessed': {
                dashed_name: 'threat-indicator-file-accessed',
                description: 'Last time the file was accessed.\nNote that not all filesystems keep track of access time.',
                flat_name: 'threat.indicator.file.accessed',
                level: 'extended',
                name: 'accessed',
                normalize: [],
                original_fieldset: 'file',
                short: 'Last time the file was accessed.',
                type: 'date',
            },
            'threat.indicator.file.attributes': {
                dashed_name: 'threat-indicator-file-attributes',
                description: "Array of file attributes.\nAttributes names will vary by platform. Here's a non-exhaustive list of values that are expected in this field: archive, compressed, directory, encrypted, execute, hidden, read, readonly, system, write.",
                example: '["readonly", "system"]',
                flat_name: 'threat.indicator.file.attributes',
                ignore_above: 1024,
                level: 'extended',
                name: 'attributes',
                normalize: ['array'],
                original_fieldset: 'file',
                short: 'Array of file attributes.',
                type: 'keyword',
            },
            'threat.indicator.file.code_signature.digest_algorithm': {
                dashed_name: 'threat-indicator-file-code-signature-digest-algorithm',
                description: 'The hashing algorithm used to sign the process.\nThis value can distinguish signatures when a file is signed multiple times by the same signer but with a different digest algorithm.',
                example: 'sha256',
                flat_name: 'threat.indicator.file.code_signature.digest_algorithm',
                ignore_above: 1024,
                level: 'extended',
                name: 'digest_algorithm',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Hashing algorithm used to sign the process.',
                type: 'keyword',
            },
            'threat.indicator.file.code_signature.exists': {
                dashed_name: 'threat-indicator-file-code-signature-exists',
                description: 'Boolean to capture if a signature is present.',
                example: 'true',
                flat_name: 'threat.indicator.file.code_signature.exists',
                level: 'core',
                name: 'exists',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if a signature is present.',
                type: 'boolean',
            },
            'threat.indicator.file.code_signature.signing_id': {
                dashed_name: 'threat-indicator-file-code-signature-signing-id',
                description: 'The identifier used to sign the process.\nThis is used to identify the application manufactured by a software vendor. The field is relevant to Apple *OS only.',
                example: 'com.apple.xpc.proxy',
                flat_name: 'threat.indicator.file.code_signature.signing_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'signing_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The identifier used to sign the process.',
                type: 'keyword',
            },
            'threat.indicator.file.code_signature.status': {
                dashed_name: 'threat-indicator-file-code-signature-status',
                description: 'Additional information about the certificate status.\nThis is useful for logging cryptographic errors with the certificate validity or trust status. Leave unpopulated if the validity or trust of the certificate was unchecked.',
                example: 'ERROR_UNTRUSTED_ROOT',
                flat_name: 'threat.indicator.file.code_signature.status',
                ignore_above: 1024,
                level: 'extended',
                name: 'status',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Additional information about the certificate status.',
                type: 'keyword',
            },
            'threat.indicator.file.code_signature.subject_name': {
                dashed_name: 'threat-indicator-file-code-signature-subject-name',
                description: 'Subject name of the code signer',
                example: 'Microsoft Corporation',
                flat_name: 'threat.indicator.file.code_signature.subject_name',
                ignore_above: 1024,
                level: 'core',
                name: 'subject_name',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Subject name of the code signer',
                type: 'keyword',
            },
            'threat.indicator.file.code_signature.team_id': {
                dashed_name: 'threat-indicator-file-code-signature-team-id',
                description: 'The team identifier used to sign the process.\nThis is used to identify the team or vendor of a software product. The field is relevant to Apple *OS only.',
                example: 'EQHXZ8M8AV',
                flat_name: 'threat.indicator.file.code_signature.team_id',
                ignore_above: 1024,
                level: 'extended',
                name: 'team_id',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'The team identifier used to sign the process.',
                type: 'keyword',
            },
            'threat.indicator.file.code_signature.timestamp': {
                dashed_name: 'threat-indicator-file-code-signature-timestamp',
                description: 'Date and time when the code signature was generated and signed.',
                example: '2021-01-01T12:10:30Z',
                flat_name: 'threat.indicator.file.code_signature.timestamp',
                level: 'extended',
                name: 'timestamp',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'When the signature was generated and signed.',
                type: 'date',
            },
            'threat.indicator.file.code_signature.trusted': {
                dashed_name: 'threat-indicator-file-code-signature-trusted',
                description: 'Stores the trust status of the certificate chain.\nValidating the trust of the certificate chain may be complicated, and this field should only be populated by tools that actively check the status.',
                example: 'true',
                flat_name: 'threat.indicator.file.code_signature.trusted',
                level: 'extended',
                name: 'trusted',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Stores the trust status of the certificate chain.',
                type: 'boolean',
            },
            'threat.indicator.file.code_signature.valid': {
                dashed_name: 'threat-indicator-file-code-signature-valid',
                description: 'Boolean to capture if the digital signature is verified against the binary content.\nLeave unpopulated if a certificate was unchecked.',
                example: 'true',
                flat_name: 'threat.indicator.file.code_signature.valid',
                level: 'extended',
                name: 'valid',
                normalize: [],
                original_fieldset: 'code_signature',
                short: 'Boolean to capture if the digital signature is verified against the binary content.',
                type: 'boolean',
            },
            'threat.indicator.file.created': {
                dashed_name: 'threat-indicator-file-created',
                description: 'File creation time.\nNote that not all filesystems store the creation time.',
                flat_name: 'threat.indicator.file.created',
                level: 'extended',
                name: 'created',
                normalize: [],
                original_fieldset: 'file',
                short: 'File creation time.',
                type: 'date',
            },
            'threat.indicator.file.ctime': {
                dashed_name: 'threat-indicator-file-ctime',
                description: 'Last time the file attributes or metadata changed.\nNote that changes to the file content will update `mtime`. This implies `ctime` will be adjusted at the same time, since `mtime` is an attribute of the file.',
                flat_name: 'threat.indicator.file.ctime',
                level: 'extended',
                name: 'ctime',
                normalize: [],
                original_fieldset: 'file',
                short: 'Last time the file attributes or metadata changed.',
                type: 'date',
            },
            'threat.indicator.file.device': {
                dashed_name: 'threat-indicator-file-device',
                description: 'Device that is the source of the file.',
                example: 'sda',
                flat_name: 'threat.indicator.file.device',
                ignore_above: 1024,
                level: 'extended',
                name: 'device',
                normalize: [],
                original_fieldset: 'file',
                short: 'Device that is the source of the file.',
                type: 'keyword',
            },
            'threat.indicator.file.directory': {
                dashed_name: 'threat-indicator-file-directory',
                description: 'Directory where the file is located. It should include the drive letter, when appropriate.',
                example: '/home/alice',
                flat_name: 'threat.indicator.file.directory',
                ignore_above: 1024,
                level: 'extended',
                name: 'directory',
                normalize: [],
                original_fieldset: 'file',
                short: 'Directory where the file is located.',
                type: 'keyword',
            },
            'threat.indicator.file.drive_letter': {
                dashed_name: 'threat-indicator-file-drive-letter',
                description: 'Drive letter where the file is located. This field is only relevant on Windows.\nThe value should be uppercase, and not include the colon.',
                example: 'C',
                flat_name: 'threat.indicator.file.drive_letter',
                ignore_above: 1,
                level: 'extended',
                name: 'drive_letter',
                normalize: [],
                original_fieldset: 'file',
                short: 'Drive letter where the file is located.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.architecture': {
                dashed_name: 'threat-indicator-file-elf-architecture',
                description: 'Machine architecture of the ELF file.',
                example: 'x86-64',
                flat_name: 'threat.indicator.file.elf.architecture',
                ignore_above: 1024,
                level: 'extended',
                name: 'architecture',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Machine architecture of the ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.byte_order': {
                dashed_name: 'threat-indicator-file-elf-byte-order',
                description: 'Byte sequence of ELF file.',
                example: 'Little Endian',
                flat_name: 'threat.indicator.file.elf.byte_order',
                ignore_above: 1024,
                level: 'extended',
                name: 'byte_order',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Byte sequence of ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.cpu_type': {
                dashed_name: 'threat-indicator-file-elf-cpu-type',
                description: 'CPU type of the ELF file.',
                example: 'Intel',
                flat_name: 'threat.indicator.file.elf.cpu_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'cpu_type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'CPU type of the ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.creation_date': {
                dashed_name: 'threat-indicator-file-elf-creation-date',
                description: "Extracted when possible from the file's metadata. Indicates when it was built or compiled. It can also be faked by malware creators.",
                flat_name: 'threat.indicator.file.elf.creation_date',
                level: 'extended',
                name: 'creation_date',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Build or compile date.',
                type: 'date',
            },
            'threat.indicator.file.elf.exports': {
                dashed_name: 'threat-indicator-file-elf-exports',
                description: 'List of exported element names and types.',
                flat_name: 'threat.indicator.file.elf.exports',
                level: 'extended',
                name: 'exports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of exported element names and types.',
                type: 'flattened',
            },
            'threat.indicator.file.elf.go_import_hash': {
                dashed_name: 'threat-indicator-file-elf-go-import-hash',
                description: 'A hash of the Go language imports in an ELF file excluding standard library imports. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThe algorithm used to calculate the Go symbol hash and a reference implementation are available [here](https://github.com/elastic/toutoumomoma).',
                example: '10bddcb4cee42080f76c88d9ff964491',
                flat_name: 'threat.indicator.file.elf.go_import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'go_import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the Go language imports in an ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.go_imports': {
                dashed_name: 'threat-indicator-file-elf-go-imports',
                description: 'List of imported Go language element names and types.',
                flat_name: 'threat.indicator.file.elf.go_imports',
                level: 'extended',
                name: 'go_imports',
                normalize: [],
                original_fieldset: 'elf',
                short: 'List of imported Go language element names and types.',
                type: 'flattened',
            },
            'threat.indicator.file.elf.go_imports_names_entropy': {
                dashed_name: 'threat-indicator-file-elf-go-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of Go imports.',
                flat_name: 'threat.indicator.file.elf.go_imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'threat.indicator.file.elf.go_imports_names_var_entropy': {
                dashed_name: 'threat-indicator-file-elf-go-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of Go imports.',
                flat_name: 'threat.indicator.file.elf.go_imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'go_imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of Go imports.',
                type: 'long',
            },
            'threat.indicator.file.elf.go_stripped': {
                dashed_name: 'threat-indicator-file-elf-go-stripped',
                description: 'Set to true if the file is a Go executable that has had its symbols stripped or obfuscated and false if an unobfuscated Go executable.',
                flat_name: 'threat.indicator.file.elf.go_stripped',
                level: 'extended',
                name: 'go_stripped',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Whether the file is a stripped or obfuscated Go executable.',
                type: 'boolean',
            },
            'threat.indicator.file.elf.header.abi_version': {
                dashed_name: 'threat-indicator-file-elf-header-abi-version',
                description: 'Version of the ELF Application Binary Interface (ABI).',
                flat_name: 'threat.indicator.file.elf.header.abi_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.abi_version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF Application Binary Interface (ABI).',
                type: 'keyword',
            },
            'threat.indicator.file.elf.header.class': {
                dashed_name: 'threat-indicator-file-elf-header-class',
                description: 'Header class of the ELF file.',
                flat_name: 'threat.indicator.file.elf.header.class',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.class',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header class of the ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.header.data': {
                dashed_name: 'threat-indicator-file-elf-header-data',
                description: 'Data table of the ELF header.',
                flat_name: 'threat.indicator.file.elf.header.data',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.data',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Data table of the ELF header.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.header.entrypoint': {
                dashed_name: 'threat-indicator-file-elf-header-entrypoint',
                description: 'Header entrypoint of the ELF file.',
                flat_name: 'threat.indicator.file.elf.header.entrypoint',
                format: 'string',
                level: 'extended',
                name: 'header.entrypoint',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header entrypoint of the ELF file.',
                type: 'long',
            },
            'threat.indicator.file.elf.header.object_version': {
                dashed_name: 'threat-indicator-file-elf-header-object-version',
                description: '"0x1" for original ELF files.',
                flat_name: 'threat.indicator.file.elf.header.object_version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.object_version',
                normalize: [],
                original_fieldset: 'elf',
                short: '"0x1" for original ELF files.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.header.os_abi': {
                dashed_name: 'threat-indicator-file-elf-header-os-abi',
                description: 'Application Binary Interface (ABI) of the Linux OS.',
                flat_name: 'threat.indicator.file.elf.header.os_abi',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.os_abi',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Application Binary Interface (ABI) of the Linux OS.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.header.type': {
                dashed_name: 'threat-indicator-file-elf-header-type',
                description: 'Header type of the ELF file.',
                flat_name: 'threat.indicator.file.elf.header.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Header type of the ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.header.version': {
                dashed_name: 'threat-indicator-file-elf-header-version',
                description: 'Version of the ELF header.',
                flat_name: 'threat.indicator.file.elf.header.version',
                ignore_above: 1024,
                level: 'extended',
                name: 'header.version',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Version of the ELF header.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.import_hash': {
                dashed_name: 'threat-indicator-file-elf-import-hash',
                description: 'A hash of the imports in an ELF file. An import hash can be used to fingerprint binaries even after recompilation or other code-level transformations have occurred, which would change more traditional hash values.\nThis is an ELF implementation of the Windows PE imphash.',
                example: 'd41d8cd98f00b204e9800998ecf8427e',
                flat_name: 'threat.indicator.file.elf.import_hash',
                ignore_above: 1024,
                level: 'extended',
                name: 'import_hash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'A hash of the imports in an ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.imports': {
                dashed_name: 'threat-indicator-file-elf-imports',
                description: 'List of imported element names and types.',
                flat_name: 'threat.indicator.file.elf.imports',
                level: 'extended',
                name: 'imports',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of imported element names and types.',
                type: 'flattened',
            },
            'threat.indicator.file.elf.imports_names_entropy': {
                dashed_name: 'threat-indicator-file-elf-imports-names-entropy',
                description: 'Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'threat.indicator.file.elf.imports_names_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'threat.indicator.file.elf.imports_names_var_entropy': {
                dashed_name: 'threat-indicator-file-elf-imports-names-var-entropy',
                description: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                flat_name: 'threat.indicator.file.elf.imports_names_var_entropy',
                format: 'number',
                level: 'extended',
                name: 'imports_names_var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the list of imported element names and types.',
                type: 'long',
            },
            'threat.indicator.file.elf.sections': {
                dashed_name: 'threat-indicator-file-elf-sections',
                description: 'An array containing an object for each section of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.sections.*`.',
                flat_name: 'threat.indicator.file.elf.sections',
                level: 'extended',
                name: 'sections',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'Section information of the ELF file.',
                type: 'nested',
            },
            'threat.indicator.file.elf.sections.chi2': {
                dashed_name: 'threat-indicator-file-elf-sections-chi2',
                description: 'Chi-square probability distribution of the section.',
                flat_name: 'threat.indicator.file.elf.sections.chi2',
                format: 'number',
                level: 'extended',
                name: 'sections.chi2',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Chi-square probability distribution of the section.',
                type: 'long',
            },
            'threat.indicator.file.elf.sections.entropy': {
                dashed_name: 'threat-indicator-file-elf-sections-entropy',
                description: 'Shannon entropy calculation from the section.',
                flat_name: 'threat.indicator.file.elf.sections.entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Shannon entropy calculation from the section.',
                type: 'long',
            },
            'threat.indicator.file.elf.sections.flags': {
                dashed_name: 'threat-indicator-file-elf-sections-flags',
                description: 'ELF Section List flags.',
                flat_name: 'threat.indicator.file.elf.sections.flags',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.flags',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List flags.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.sections.name': {
                dashed_name: 'threat-indicator-file-elf-sections-name',
                description: 'ELF Section List name.',
                flat_name: 'threat.indicator.file.elf.sections.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.name',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List name.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.sections.physical_offset': {
                dashed_name: 'threat-indicator-file-elf-sections-physical-offset',
                description: 'ELF Section List offset.',
                flat_name: 'threat.indicator.file.elf.sections.physical_offset',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.physical_offset',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List offset.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.sections.physical_size': {
                dashed_name: 'threat-indicator-file-elf-sections-physical-size',
                description: 'ELF Section List physical size.',
                flat_name: 'threat.indicator.file.elf.sections.physical_size',
                format: 'bytes',
                level: 'extended',
                name: 'sections.physical_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List physical size.',
                type: 'long',
            },
            'threat.indicator.file.elf.sections.type': {
                dashed_name: 'threat-indicator-file-elf-sections-type',
                description: 'ELF Section List type.',
                flat_name: 'threat.indicator.file.elf.sections.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'sections.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List type.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.sections.var_entropy': {
                dashed_name: 'threat-indicator-file-elf-sections-var-entropy',
                description: 'Variance for Shannon entropy calculation from the section.',
                flat_name: 'threat.indicator.file.elf.sections.var_entropy',
                format: 'number',
                level: 'extended',
                name: 'sections.var_entropy',
                normalize: [],
                original_fieldset: 'elf',
                short: 'Variance for Shannon entropy calculation from the section.',
                type: 'long',
            },
            'threat.indicator.file.elf.sections.virtual_address': {
                dashed_name: 'threat-indicator-file-elf-sections-virtual-address',
                description: 'ELF Section List virtual address.',
                flat_name: 'threat.indicator.file.elf.sections.virtual_address',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_address',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual address.',
                type: 'long',
            },
            'threat.indicator.file.elf.sections.virtual_size': {
                dashed_name: 'threat-indicator-file-elf-sections-virtual-size',
                description: 'ELF Section List virtual size.',
                flat_name: 'threat.indicator.file.elf.sections.virtual_size',
                format: 'string',
                level: 'extended',
                name: 'sections.virtual_size',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF Section List virtual size.',
                type: 'long',
            },
            'threat.indicator.file.elf.segments': {
                dashed_name: 'threat-indicator-file-elf-segments',
                description: 'An array containing an object for each segment of the ELF file.\nThe keys that should be present in these objects are defined by sub-fields underneath `elf.segments.*`.',
                flat_name: 'threat.indicator.file.elf.segments',
                level: 'extended',
                name: 'segments',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'ELF object segment list.',
                type: 'nested',
            },
            'threat.indicator.file.elf.segments.sections': {
                dashed_name: 'threat-indicator-file-elf-segments-sections',
                description: 'ELF object segment sections.',
                flat_name: 'threat.indicator.file.elf.segments.sections',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.sections',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment sections.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.segments.type': {
                dashed_name: 'threat-indicator-file-elf-segments-type',
                description: 'ELF object segment type.',
                flat_name: 'threat.indicator.file.elf.segments.type',
                ignore_above: 1024,
                level: 'extended',
                name: 'segments.type',
                normalize: [],
                original_fieldset: 'elf',
                short: 'ELF object segment type.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.shared_libraries': {
                dashed_name: 'threat-indicator-file-elf-shared-libraries',
                description: 'List of shared libraries used by this ELF object.',
                flat_name: 'threat.indicator.file.elf.shared_libraries',
                ignore_above: 1024,
                level: 'extended',
                name: 'shared_libraries',
                normalize: ['array'],
                original_fieldset: 'elf',
                short: 'List of shared libraries used by this ELF object.',
                type: 'keyword',
            },
            'threat.indicator.file.elf.telfhash': {
                dashed_name: 'threat-indicator-file-elf-telfhash',
                description: 'telfhash symbol hash for ELF file.',
                flat_name: 'threat.indicator.file.elf.telfhash',
                ignore_above: 1024,
                level: 'extended',
                name: 'telfhash',
                normalize: [],
                original_fieldset: 'elf',
                short: 'telfhash hash for ELF file.',
                type: 'keyword',
            },
            'threat.indicator.file.extension': {
                dashed_name: 'threat-indicator-file-extension',
                description: 'File extension, excluding the leading dot.\nNote that when the file name has multiple extensions (example.tar.gz), only the last one should be captured ("gz", not "tar.gz").',
                example: 'png',
                flat_name: 'threat.indicator.file.extension',
                ignore_above: 1024,
                level: 'extended',
                name: 'extension',
                normalize: [],
                original_fieldset: 'file',
                short: 'File extension, excluding the leading dot.',
                type: 'keyword',
            },
            'threat.indicator.file.fork_name': {
                dashed_name: 'threat-indicator-file-fork-name',
                description: 'A fork is additional data associated with a filesystem object.\nOn Linux, a resource fork is used to store additional data with a filesystem object. A file always has at least one fork for the data portion, and additional forks may exist.\nOn NTFS, this is analogous to an Alternate Data Stream (ADS), and the default data stream for a file is just called $DATA. Zone.Identifier is commonly used by Windows to track contents downloaded from the Internet. An ADS is typically of the form: `C:\\path\\to\\filename.extension:some_fork_name`, and `some_fork_name` is the value that should populate `fork_name`. `filename.extension` should populate `file.name`, and `extension` should populate `file.extension`. The full path, `file.path`, will include the fork name.',
                example: 'Zone.Identifer',
                flat_name: 'threat.indicator.file.fork_name',
                ignore_above: 1024,
                level: 'extended',
                name: 'fork_name',
                normalize: [],
                original_fieldset: 'file',
                short: 'A fork is additional data associated with a filesystem object.',
                type: 'keyword',
            },
            'threat.indicator.file.gid': {
                dashed_name: 'threat-indicator-file-gid',
                description: 'Primary group ID (GID) of the file.',
                example: '1001',
                flat_name: 'threat.indicator.file.gid',
                ignore_above: 1024,
                level: 'extended',
                name: 'gid',
                normalize: [],
                original_fieldset: 'file',
                short: 'Primary group ID (GID) of the file.',
                type: 'keyword',
            },
            'threat.indicator.file.group': {
                dashed_name: 'threat-indicator-file-group',
                description: 'Primary group name of the file.',
                example: 'alice',
                flat_name: 'threat.indicator.file.group',
                ignore_above: 1024,
                level: 'extended',
                name: 'group',
                normalize: [],
                original_fieldset: 'file',
                short: 'Primary group name of the file.',
                type: 'keyword',
            },
            'threat.indicator.file.hash.md5': {
                dashed_name: 'threat-indicator-file-hash-md5',
                description: 'MD5 hash.',
                flat_name: 'threat.indicator.file.hash.md5',
                ignore_above: 1024,
                level: 'extended',
                name: 'md5',
                normalize: [],
                original_fieldset: 'hash',
                short: 'MD5 hash.',
                type: 'keyword',
            },
            'threat.indicator.file.hash.sha1': {
                dashed_name: 'threat-indicator-file-hash-sha1',
                description: 'SHA1 hash.',
                flat_name: 'threat.indicator.file.hash.sha1',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha1',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA1 hash.',
                type: 'keyword',
            },
            'threat.indicator.file.hash.sha256': {
                dashed_name: 'threat-indicator-file-hash-sha256',
                description: 'SHA256 hash.',
                flat_name: 'threat.indicator.file.hash.sha256',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha256',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA256 hash.',
                type: 'keyword',
            },
            'threat.indicator.file.hash.sha384': {
                dashed_name: 'threat-indicator-file-hash-sha384',
                description: 'SHA384 hash.',
                flat_name: 'threat.indicator.file.hash.sha384',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha384',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA384 hash.',
                type: 'keyword',
            },
            'threat.indicator.file.hash.sha512': {
                dashed_name: 'threat-indicator-file-hash-sha512',
                description: 'SHA512 hash.',
                flat_name: 'threat.indicator.file.hash.sha512',
                ignore_above: 1024,
                level: 'extended',
                name: 'sha512',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SHA512 hash.',
                type: 'keyword',
            },
            'threat.indicator.file.hash.ssdeep': {
                dashed_name: 'threat-indicator-file-hash-ssdeep',
                description: 'SSDEEP hash.',
                flat_name: 'threat.indicator.file.hash.ssdeep',
                ignore_above: 1024,
                level: 'extended',
                name: 'ssdeep',
                normalize: [],
                original_fieldset: 'hash',
                short: 'SSDEEP hash.',
                type: 'keyword',
            },
            'threat.indicator.file.hash.tlsh': {
                dashed_name: 'threat-indicator-file-hash-tlsh',
                description: 'TLSH hash.',
                flat_name: 'threat.indicator.file.hash.tlsh',
                ignore_above: 1024,
                level: 'extended',
                name: 'tlsh',
                normalize: [],
                original_fieldset: 'hash',
                short: 'TLSH hash.',
                type: 'keyword',
            },
            'threat.indicator.file.inode': {
                dashed_name: 'threat-indicator-file-inode',
                description: 'Inode representing the file in the filesystem.',
                example: '256383',
                flat_name: 'threat.indicator.file.inode',
                ignore_above: 1024,
                level: 'extended',
                name: 'inode',
                normalize: [],
                original_fieldset: 'file',
                short: 'Inode representing the file in the filesystem.',
                type: 'keyword',
            },
            'threat.indicator.file.mime_type': {
                dashed_name: 'threat-indicator-file-mime-type',
                description: 'MIME type should identify the format of the file or stream of bytes using https://www.iana.org/assignments/media-types/media-types.xhtml[IANA official types], where possible. When more than one type is applicable, the most specific type should be used.',
                flat_name: 'threat.indicator.file.mime_type',
                ignore_above: 1024,
                level: 'extended',
                name: 'mime_type',
                normalize: [],
                original_fieldset: 'file',
                short: 'Media type of file, document, or arrangement of bytes.',
                type: 'keyword',
            },
            'threat.indicator.file.mode': {
                dashed_name: 'threat-indicator-file-mode',
                description: 'Mode of the file in octal representation.',
                example: '0640',
                flat_name: 'threat.indicator.file.mode',
                ignore_above: 1024,
                level: 'extended',
                name: 'mode',
                normalize: [],
                original_fieldset: 'file',
                short: 'Mode of the file in octal representation.',
                type: 'keyword',
            },
            'threat.indicator.file.mtime': {
                dashed_name: 'threat-indicator-file-mtime',
                description: 'Last time the file content was modified.',
                flat_name: 'threat.indicator.file.mtime',
                level: 'extended',
                name: 'mtime',
                normalize: [],
                original_fieldset: 'file',
                short: 'Last time the file content was modified.',
                type: 'date',
            },
            'threat.indicator.file.name': {
                dashed_name: 'threat-indicator-file-name',
                description: 'Name of the file including the extension, without the directory.',
                example: 'example.png',
                flat_name: 'threat.indicator.file.name',
                ignore_above: 1024,
                level: 'extended',
                name: 'name',
                normalize: [],
                original_fieldset: 'file',
                short: 'Name of the file including the extension, without the directory.',
                type: 'keyword',
            },
            'threat.indicator.file.owner': {
                dashed_name: 'threat-indicator-file-owner',
                description: "File owner's username.",
                example: 'alice',
                flat_name: 'threat.indicator.file.owner',
                ignore_above: 1024,
                level: 'extended',
                name: 'owner',
                normalize: [],
                original_fieldset: 'file',
                short: "File owner's username.",
                type: 'keyword',
            },
            'threat.indicator.file.path': {
                dashed_name