/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.allocator;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.allocator.DesiredBalance;
import org.elasticsearch.cluster.routing.allocation.allocator.ShardAssignment;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.FrequencyCappedAction;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.time.TimeProvider;
import org.elasticsearch.common.util.FeatureFlag;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;

public class UndesiredAllocationsTracker {
    private static final Logger logger = LogManager.getLogger(UndesiredAllocationsTracker.class);
    private static final TimeValue FIVE_MINUTES = TimeValue.timeValueMinutes((long)5L);
    private static final FeatureFlag UNDESIRED_ALLOCATION_TRACKER_ENABLED = new FeatureFlag("undesired_allocation_tracker");
    public static final Setting<TimeValue> UNDESIRED_ALLOCATION_DURATION_LOG_THRESHOLD_SETTING = Setting.timeSetting("cluster.routing.allocation.desired_balance.undesired_duration_logging.threshold", FIVE_MINUTES, TimeValue.ONE_MINUTE, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> UNDESIRED_ALLOCATION_DURATION_LOG_INTERVAL_SETTING = Setting.timeSetting("cluster.routing.allocation.desired_balance.undesired_duration_logging.interval", FIVE_MINUTES, TimeValue.ONE_MINUTE, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Integer> MAX_UNDESIRED_ALLOCATIONS_TO_TRACK = Setting.intSetting("cluster.routing.allocation.desired_balance.undesired_duration_logging.max_to_track", UNDESIRED_ALLOCATION_TRACKER_ENABLED.isEnabled() ? 10 : 0, 0, 100, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final TimeProvider timeProvider;
    private final LinkedHashMap<String, UndesiredAllocation> undesiredAllocations = new LinkedHashMap();
    private final FrequencyCappedAction undesiredAllocationDurationLogInterval;
    private volatile TimeValue undesiredAllocationDurationLoggingThreshold;
    private volatile int maxUndesiredAllocationsToTrack;

    UndesiredAllocationsTracker(ClusterSettings clusterSettings, TimeProvider timeProvider) {
        this.timeProvider = timeProvider;
        this.undesiredAllocationDurationLogInterval = new FrequencyCappedAction(timeProvider::relativeTimeInMillis, TimeValue.ZERO);
        clusterSettings.initializeAndWatch(UNDESIRED_ALLOCATION_DURATION_LOG_INTERVAL_SETTING, this.undesiredAllocationDurationLogInterval::setMinInterval);
        clusterSettings.initializeAndWatch(UNDESIRED_ALLOCATION_DURATION_LOG_THRESHOLD_SETTING, value -> {
            this.undesiredAllocationDurationLoggingThreshold = value;
        });
        clusterSettings.initializeAndWatch(MAX_UNDESIRED_ALLOCATIONS_TO_TRACK, value -> {
            this.maxUndesiredAllocationsToTrack = value;
        });
    }

    public void trackUndesiredAllocation(ShardRouting shardRouting) {
        String allocationId;
        assert (!shardRouting.unassigned()) : "Shouldn't record unassigned shards as undesired allocations";
        if (this.undesiredAllocations.size() < this.maxUndesiredAllocationsToTrack && !this.undesiredAllocations.containsKey(allocationId = shardRouting.allocationId().getId())) {
            this.undesiredAllocations.put(allocationId, new UndesiredAllocation(shardRouting.shardId(), this.timeProvider.relativeTimeInMillis()));
        }
    }

    public void removeTracking(ShardRouting shardRouting) {
        assert (!shardRouting.unassigned()) : "Shouldn't remove tracking of unassigned shards";
        this.undesiredAllocations.remove(shardRouting.allocationId().getId());
    }

    public void cleanup(RoutingNodes routingNodes) {
        this.undesiredAllocations.entrySet().removeIf(e -> {
            UndesiredAllocation undesiredAllocation = (UndesiredAllocation)e.getValue();
            String allocationId = (String)e.getKey();
            return routingNodes.getByAllocationId(undesiredAllocation.shardId(), allocationId) == null;
        });
        this.shrinkIfOversized();
    }

    public void clear() {
        this.undesiredAllocations.clear();
    }

    public void maybeLogUndesiredShardsWarning(RoutingNodes routingNodes, RoutingAllocation routingAllocation, DesiredBalance desiredBalance) {
        long earliestUndesiredTimestamp;
        long currentTimeMillis = this.timeProvider.relativeTimeInMillis();
        if (!this.undesiredAllocations.isEmpty() && (earliestUndesiredTimestamp = ((UndesiredAllocation)this.undesiredAllocations.firstEntry().getValue()).undesiredSince()) < currentTimeMillis && currentTimeMillis - earliestUndesiredTimestamp > this.undesiredAllocationDurationLoggingThreshold.millis()) {
            this.undesiredAllocationDurationLogInterval.maybeExecute(() -> this.logDecisionsForUndesiredShardsOverThreshold(routingNodes, routingAllocation, desiredBalance));
        }
    }

    private void logDecisionsForUndesiredShardsOverThreshold(RoutingNodes routingNodes, RoutingAllocation routingAllocation, DesiredBalance desiredBalance) {
        long currentTimeMillis = this.timeProvider.relativeTimeInMillis();
        long loggingThresholdTimestamp = currentTimeMillis - this.undesiredAllocationDurationLoggingThreshold.millis();
        for (Map.Entry<String, UndesiredAllocation> allocation : this.undesiredAllocations.entrySet()) {
            UndesiredAllocation undesiredAllocation = allocation.getValue();
            String allocationId = allocation.getKey();
            if (undesiredAllocation.undesiredSince() >= loggingThresholdTimestamp) continue;
            ShardRouting shardRouting = routingNodes.getByAllocationId(undesiredAllocation.shardId(), allocationId);
            if (shardRouting != null) {
                this.logUndesiredShardDetails(shardRouting, TimeValue.timeValueMillis((long)(currentTimeMillis - undesiredAllocation.undesiredSince())), routingNodes, routingAllocation, desiredBalance);
                continue;
            }
            assert (false) : String.valueOf(undesiredAllocation) + " for allocationID " + allocationId + " was not cleaned up";
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void logUndesiredShardDetails(ShardRouting shardRouting, TimeValue undesiredDuration, RoutingNodes routingNodes, RoutingAllocation allocation, DesiredBalance desiredBalance) {
        RoutingAllocation.DebugMode originalDebugMode = allocation.getDebugMode();
        allocation.setDebugMode(RoutingAllocation.DebugMode.EXCLUDE_YES_DECISIONS);
        try {
            ShardAssignment assignment = desiredBalance.getAssignment(shardRouting.shardId());
            logger.warn("Shard {} has been in an undesired allocation for {}", new Object[]{shardRouting.shardId(), undesiredDuration});
            for (String nodeId : assignment.nodeIds()) {
                Decision decision = allocation.deciders().canAllocate(shardRouting, routingNodes.node(nodeId), allocation);
                logger.warn("Shard {} allocation decision for node [{}]: {}", new Object[]{shardRouting.shardId(), nodeId, decision});
            }
        }
        finally {
            allocation.setDebugMode(originalDebugMode);
        }
    }

    private void shrinkIfOversized() {
        if (this.undesiredAllocations.size() > this.maxUndesiredAllocationsToTrack) {
            Set newestExcessAllocationIds = this.undesiredAllocations.entrySet().stream().sorted((a, b) -> Long.compare(((UndesiredAllocation)b.getValue()).undesiredSince(), ((UndesiredAllocation)a.getValue()).undesiredSince())).limit(this.undesiredAllocations.size() - this.maxUndesiredAllocationsToTrack).map(Map.Entry::getKey).collect(Collectors.toSet());
            this.undesiredAllocations.keySet().removeAll(newestExcessAllocationIds);
        }
    }

    Map<String, UndesiredAllocation> getUndesiredAllocations() {
        return Map.copyOf(this.undesiredAllocations);
    }

    record UndesiredAllocation(ShardId shardId, long undesiredSince) {
    }
}

