/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.health.node;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.health.metadata.HealthMetadata;
import org.elasticsearch.health.node.UpdateHealthInfoCacheAction;
import org.elasticsearch.health.node.action.HealthNodeNotDiscoveredException;
import org.elasticsearch.health.node.selection.HealthNode;
import org.elasticsearch.health.node.selection.HealthNodeTaskExecutor;
import org.elasticsearch.health.node.tracker.HealthTracker;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.NodeNotConnectedException;

public class LocalHealthMonitor
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(LocalHealthMonitor.class);
    public static final Setting<TimeValue> POLL_INTERVAL_SETTING = Setting.timeSetting("health.reporting.local.monitor.interval", TimeValue.timeValueSeconds(30L), TimeValue.timeValueSeconds(10L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final Client client;
    private volatile TimeValue monitorInterval;
    private volatile boolean enabled;
    private volatile boolean prerequisitesFulfilled;
    private final List<HealthTracker<?>> healthTrackers;
    private final AtomicReference<String> lastSeenHealthNode = new AtomicReference();
    private volatile Monitoring monitoring;
    private final AtomicBoolean inFlightRequest = new AtomicBoolean(false);

    private LocalHealthMonitor(Settings settings, ClusterService clusterService, ThreadPool threadPool, Client client, List<HealthTracker<?>> healthTrackers) {
        this.threadPool = threadPool;
        this.monitorInterval = POLL_INTERVAL_SETTING.get(settings);
        this.enabled = HealthNodeTaskExecutor.ENABLED_SETTING.get(settings);
        this.clusterService = clusterService;
        this.client = client;
        this.healthTrackers = healthTrackers;
    }

    public static LocalHealthMonitor create(Settings settings, ClusterService clusterService, ThreadPool threadPool, Client client, List<HealthTracker<?>> healthTrackers) {
        LocalHealthMonitor localHealthMonitor = new LocalHealthMonitor(settings, clusterService, threadPool, client, healthTrackers);
        localHealthMonitor.registerListeners();
        return localHealthMonitor;
    }

    private void registerListeners() {
        ClusterSettings clusterSettings = this.clusterService.getClusterSettings();
        clusterSettings.addSettingsUpdateConsumer(POLL_INTERVAL_SETTING, this::setMonitorInterval);
        clusterSettings.addSettingsUpdateConsumer(HealthNodeTaskExecutor.ENABLED_SETTING, this::setEnabled);
        this.clusterService.addListener(this);
    }

    void setMonitorInterval(TimeValue monitorInterval) {
        this.monitorInterval = monitorInterval;
        this.stopMonitoring();
        this.startMonitoringIfNecessary();
    }

    void setEnabled(boolean enabled) {
        this.enabled = enabled;
        if (enabled) {
            this.startMonitoringIfNecessary();
        } else {
            this.stopMonitoring();
        }
    }

    private void stopMonitoring() {
        Monitoring currentMonitoring = this.monitoring;
        if (currentMonitoring != null) {
            currentMonitoring.cancel();
        }
    }

    private void startMonitoringIfNecessary() {
        if (this.prerequisitesFulfilled && this.enabled) {
            if (!this.isMonitorRunning()) {
                this.monitoring = new Monitoring(this.monitorInterval, this.threadPool, this.healthTrackers, this.clusterService, this.client, this.inFlightRequest);
                this.monitoring.start();
                logger.debug("Local health monitoring started {}", (Object)this.monitoring);
            } else {
                logger.trace("Local health monitoring already started {}, skipping", (Object)this.monitoring);
            }
        }
    }

    private boolean isMonitorRunning() {
        Monitoring scheduled = this.monitoring;
        return scheduled != null && !scheduled.isCancelled();
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        DiscoveryNode currentHealthNode = HealthNode.findHealthNode(event.state());
        DiscoveryNode currentMasterNode = event.state().nodes().getMasterNode();
        boolean healthNodeChanged = this.hasHealthNodeChanged(currentHealthNode, event);
        boolean masterNodeChanged = LocalHealthMonitor.hasMasterNodeChanged(currentMasterNode, event);
        if (healthNodeChanged || masterNodeChanged) {
            this.lastSeenHealthNode.set(currentHealthNode == null ? null : currentHealthNode.getId());
            if (logger.isDebugEnabled()) {
                String reason = healthNodeChanged && masterNodeChanged ? "the master node and the health node" : (healthNodeChanged ? "the health node" : "the master node");
                logger.debug("Resetting the health monitoring because {} changed, current health node is {}.", (Object)reason, currentHealthNode == null ? null : Strings.format("[%s][%s]", currentHealthNode.getName(), currentHealthNode.getId()));
            }
        }
        boolean bl = this.prerequisitesFulfilled = event.state().clusterRecovered() && HealthMetadata.getFromClusterState(event.state()) != null && currentHealthNode != null && currentMasterNode != null;
        if (!this.prerequisitesFulfilled || healthNodeChanged || masterNodeChanged) {
            this.stopMonitoring();
        }
        if (this.prerequisitesFulfilled) {
            this.startMonitoringIfNecessary();
        }
    }

    private static boolean hasMasterNodeChanged(DiscoveryNode currentMasterNode, ClusterChangedEvent event) {
        DiscoveryNode previousMasterNode = event.previousState().nodes().getMasterNode();
        if (currentMasterNode == null || previousMasterNode == null) {
            return currentMasterNode != previousMasterNode;
        }
        return !previousMasterNode.getEphemeralId().equals(currentMasterNode.getEphemeralId());
    }

    private boolean hasHealthNodeChanged(DiscoveryNode currentHealthNode, ClusterChangedEvent event) {
        DiscoveryNode previousHealthNode = HealthNode.findHealthNode(event.previousState());
        return !Objects.equals(this.lastSeenHealthNode.get(), currentHealthNode == null ? null : currentHealthNode.getId()) || !Objects.equals(previousHealthNode, currentHealthNode);
    }

    static class Monitoring
    implements Runnable,
    Scheduler.Cancellable {
        private final TimeValue interval;
        private final Executor executor;
        private final ThreadPool threadPool;
        private final ClusterService clusterService;
        private final Client client;
        private final List<HealthTracker<?>> healthTrackers;
        private final AtomicBoolean inFlightRequest;
        private volatile boolean cancelled = false;
        private volatile boolean fistRun = true;
        private volatile Scheduler.ScheduledCancellable scheduledRun;

        private Monitoring(TimeValue interval, ThreadPool threadPool, List<HealthTracker<?>> healthTrackers, ClusterService clusterService, Client client, AtomicBoolean inFlightRequest) {
            this.interval = interval;
            this.threadPool = threadPool;
            this.executor = threadPool.executor("management");
            this.clusterService = clusterService;
            this.healthTrackers = healthTrackers;
            this.client = client;
            this.inFlightRequest = inFlightRequest;
        }

        public void start() {
            this.scheduledRun = this.threadPool.schedule(this, TimeValue.ZERO, this.executor);
        }

        @Override
        public boolean cancel() {
            if (this.cancelled) {
                return false;
            }
            this.cancelled = true;
            Scheduler.ScheduledCancellable scheduledRun = this.scheduledRun;
            if (scheduledRun != null) {
                scheduledRun.cancel();
            }
            return true;
        }

        @Override
        public boolean isCancelled() {
            return this.cancelled;
        }

        @Override
        public void run() {
            if (this.cancelled) {
                return;
            }
            if (!this.inFlightRequest.compareAndSet(false, true)) {
                logger.debug("Not allowed to send health info update request due to in-flight request, will try again.");
                this.scheduleNextRunIfNecessary();
                return;
            }
            try {
                List<HealthTracker<?>> changedHealthTrackers;
                if (this.fistRun) {
                    this.healthTrackers.forEach(HealthTracker::reset);
                    this.fistRun = false;
                }
                if ((changedHealthTrackers = this.getChangedHealthTrackers()).isEmpty()) {
                    this.releaseAndScheduleNextRun();
                    return;
                }
                UpdateHealthInfoCacheAction.Request.Builder builder = new UpdateHealthInfoCacheAction.Request.Builder().nodeId(this.clusterService.localNode().getId());
                changedHealthTrackers.forEach(changedHealthTracker -> changedHealthTracker.addToRequestBuilder(builder));
                ActionListener<AcknowledgedResponse> listener = ActionListener.wrap(response -> {}, e -> {
                    if (e.getCause() instanceof NodeNotConnectedException || e.getCause() instanceof HealthNodeNotDiscoveredException) {
                        logger.debug("Failed to connect to the health node [{}], will try again.", (Object)e.getCause().getMessage());
                    } else {
                        logger.debug(() -> Strings.format("Failed to send health info to health node, will try again.", new Object[0]), (Throwable)e);
                    }
                    changedHealthTrackers.forEach(HealthTracker::reset);
                });
                this.client.execute(UpdateHealthInfoCacheAction.INSTANCE, builder.build(), ActionListener.runAfter(listener, new RunOnce(this::releaseAndScheduleNextRun)));
            }
            catch (Exception e2) {
                logger.warn(() -> Strings.format("Failed to run scheduled health monitoring on thread pool [%s]", this.executor), (Throwable)e2);
                this.healthTrackers.forEach(HealthTracker::reset);
                this.releaseAndScheduleNextRun();
            }
        }

        private List<HealthTracker<?>> getChangedHealthTrackers() {
            HealthMetadata healthMetadata = HealthMetadata.getFromClusterState(this.clusterService.state());
            if (healthMetadata == null) {
                return List.of();
            }
            return this.healthTrackers.stream().filter(HealthTracker::checkHealthChanged).toList();
        }

        private void releaseAndScheduleNextRun() {
            this.inFlightRequest.set(false);
            this.scheduleNextRunIfNecessary();
        }

        private void scheduleNextRunIfNecessary() {
            if (this.cancelled) {
                return;
            }
            try {
                this.scheduledRun = this.threadPool.schedule(this, this.interval, this.executor);
            }
            catch (EsRejectedExecutionException e) {
                logger.debug(() -> Strings.format("Scheduled health monitoring was rejected on thread pool [%s]", this.executor), (Throwable)e);
            }
        }

        public String toString() {
            return "Monitoring{interval=" + String.valueOf(this.interval) + ", cancelled=" + this.cancelled + "}";
        }
    }
}

