"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StatusRuleExecutor = void 0;
exports.getConfigsByIds = getConfigsByIds;
exports.getDoesMonitorMeetLocationThreshold = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _common = require("@kbn/observability-plugin/common");
var _saved_objects = require("../../../common/types/saved_objects");
var _monitor_config_repository = require("../../services/monitor_config_repository");
var _filter_monitors = require("./queries/filter_monitors");
var _common2 = require("../common");
var _message_utils = require("./message_utils");
var _query_monitor_status_alert = require("./queries/query_monitor_status_alert");
var _get_step_information = require("./queries/get_step_information");
var _common3 = require("../../routes/common");
var _process_monitors = require("../../saved_objects/synthetics_monitor/process_monitors");
var _status_rule = require("../../../common/rules/status_rule");
var _runtime_types = require("../../../common/runtime_types");
var _monitor_management = require("../../../common/constants/monitor_management");
var _action_variables = require("../action_variables");
var _synthetics_alerts = require("../../../common/constants/synthetics_alerts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_RECOVERY_STRATEGY = 'conditionNotMet';
class StatusRuleExecutor {
  constructor(esClient, server, syntheticsMonitorClient, options) {
    (0, _defineProperty2.default)(this, "previousStartedAt", void 0);
    (0, _defineProperty2.default)(this, "params", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "server", void 0);
    (0, _defineProperty2.default)(this, "syntheticsMonitorClient", void 0);
    (0, _defineProperty2.default)(this, "monitors", []);
    (0, _defineProperty2.default)(this, "hasCustomCondition", void 0);
    (0, _defineProperty2.default)(this, "monitorLocationsMap", void 0);
    // monitorId: locationIds
    (0, _defineProperty2.default)(this, "dateFormat", void 0);
    (0, _defineProperty2.default)(this, "tz", void 0);
    (0, _defineProperty2.default)(this, "options", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "ruleName", void 0);
    (0, _defineProperty2.default)(this, "monitorConfigRepository", void 0);
    (0, _defineProperty2.default)(this, "getRange", maxPeriod => {
      const {
        numberOfChecks,
        useLatestChecks,
        timeWindow
      } = (0, _status_rule.getConditionType)(this.params.condition);
      if (useLatestChecks) {
        const from = (0, _moment.default)().subtract(maxPeriod * numberOfChecks, 'milliseconds').subtract(5, 'minutes').toISOString();
        this.debug(`Using range from ${from} to now, diff of ${(0, _moment.default)().diff(from, 'minutes')} minutes`);
        return {
          from,
          to: 'now'
        };
      }

      // `timeWindow` is guaranteed to be defined here.
      const {
        unit,
        size
      } = timeWindow;
      const from = (0, _moment.default)().subtract(size, unit).toISOString();
      this.debug(`Using range from ${from} to now, diff of ${(0, _moment.default)().diff(from, 'minutes')} minutes`);
      return {
        from,
        to: 'now'
      };
    });
    (0, _defineProperty2.default)(this, "handlePendingMonitorAlert", async ({
      pendingConfigs
    }) => {
      var _this$params$conditio;
      if ((_this$params$conditio = this.params.condition) !== null && _this$params$conditio !== void 0 && _this$params$conditio.alertOnNoData) {
        var _this$params$conditio2;
        if ((_this$params$conditio2 = this.params.condition) !== null && _this$params$conditio2 !== void 0 && _this$params$conditio2.groupBy && this.params.condition.groupBy !== 'locationId') {
          await this.schedulePendingAlertPerConfigId({
            pendingConfigs
          });
        } else {
          await this.schedulePendingAlertPerConfigIdPerLocation({
            pendingConfigs
          });
        }
      }
    });
    (0, _defineProperty2.default)(this, "handleDownMonitorThresholdAlert", async ({
      downConfigs
    }) => {
      var _this$params, _this$params$conditio3, _this$params2, _this$params2$conditi, _this$params$conditio4, _this$params3, _this$params3$conditi;
      const {
        useTimeWindow,
        useLatestChecks,
        downThreshold,
        locationsThreshold
      } = (0, _status_rule.getConditionType)((_this$params = this.params) === null || _this$params === void 0 ? void 0 : _this$params.condition);
      const recoveryStrategy = (_this$params$conditio3 = (_this$params2 = this.params) === null || _this$params2 === void 0 ? void 0 : (_this$params2$conditi = _this$params2.condition) === null || _this$params2$conditi === void 0 ? void 0 : _this$params2$conditi.recoveryStrategy) !== null && _this$params$conditio3 !== void 0 ? _this$params$conditio3 : DEFAULT_RECOVERY_STRATEGY;
      const groupBy = (_this$params$conditio4 = (_this$params3 = this.params) === null || _this$params3 === void 0 ? void 0 : (_this$params3$conditi = _this$params3.condition) === null || _this$params3$conditi === void 0 ? void 0 : _this$params3$conditi.groupBy) !== null && _this$params$conditio4 !== void 0 ? _this$params$conditio4 : 'locationId';
      if (groupBy === 'locationId' && locationsThreshold === 1) {
        for (const [idWithLocation, statusConfig] of Object.entries(downConfigs)) {
          var _statusConfig$latestP, _statusConfig$latestP2;
          // Skip scheduling if recoveryStrategy is 'firstUp' and latest ping is up
          if (recoveryStrategy === 'firstUp' && ((_statusConfig$latestP = (_statusConfig$latestP2 = statusConfig.latestPing.summary) === null || _statusConfig$latestP2 === void 0 ? void 0 : _statusConfig$latestP2.up) !== null && _statusConfig$latestP !== void 0 ? _statusConfig$latestP : 0) > 0) {
            continue;
          }
          const doesMonitorMeetLocationThreshold = getDoesMonitorMeetLocationThreshold({
            matchesByLocation: [statusConfig],
            locationsThreshold,
            downThreshold,
            useTimeWindow: useTimeWindow || false
          });
          if (doesMonitorMeetLocationThreshold) {
            var _statusConfig$latestP3;
            const alertId = idWithLocation;
            const monitorSummary = this.getMonitorDownSummary({
              statusConfig
            });
            await this.scheduleAlert({
              idWithLocation,
              alertId,
              monitorSummary,
              statusConfig,
              downThreshold,
              useLatestChecks,
              locationNames: [(_statusConfig$latestP3 = statusConfig.latestPing.observer.geo) === null || _statusConfig$latestP3 === void 0 ? void 0 : _statusConfig$latestP3.name],
              locationIds: [statusConfig.latestPing.observer.name]
            });
          }
        }
      } else {
        const downConfigsById = getConfigsByIds(downConfigs);
        for (const [configId, locationConfigs] of downConfigsById) {
          // If recoveryStrategy is 'firstUp', we only consider configs that are not up
          const configs = recoveryStrategy === 'firstUp' ? locationConfigs.filter(c => {
            var _c$latestPing$summary, _c$latestPing$summary2;
            return ((_c$latestPing$summary = (_c$latestPing$summary2 = c.latestPing.summary) === null || _c$latestPing$summary2 === void 0 ? void 0 : _c$latestPing$summary2.up) !== null && _c$latestPing$summary !== void 0 ? _c$latestPing$summary : 0) === 0;
          }) : locationConfigs;
          if (!configs.length) {
            continue;
          }
          const doesMonitorMeetLocationThreshold = getDoesMonitorMeetLocationThreshold({
            matchesByLocation: configs,
            locationsThreshold,
            downThreshold,
            useTimeWindow: useTimeWindow || false
          });
          if (doesMonitorMeetLocationThreshold) {
            const alertId = configId;
            const monitorSummary = this.getUngroupedDownSummary({
              statusConfigs: configs
            });
            await this.scheduleAlert({
              idWithLocation: configId,
              alertId,
              monitorSummary,
              downConfigs: configs.reduce((acc, conf) => ({
                ...acc,
                [`${conf.configId}-${conf.locationId}`]: conf
              }), {}),
              configId,
              downThreshold,
              useLatestChecks,
              locationNames: configs.map(c => {
                var _c$latestPing$observe;
                return (_c$latestPing$observe = c.latestPing.observer.geo) === null || _c$latestPing$observe === void 0 ? void 0 : _c$latestPing$observe.name;
              }),
              locationIds: configs.map(c => c.latestPing.observer.name)
            });
          }
        }
      }
    });
    (0, _defineProperty2.default)(this, "getRuleThresholdOverview", async () => {
      const data = await this.getConfigs({});
      return {
        ...data,
        monitors: this.monitors.map(monitor => ({
          id: monitor.id,
          name: monitor.attributes.name,
          type: monitor.attributes.type
        }))
      };
    });
    const {
      services,
      params,
      previousStartedAt,
      rule
    } = options;
    const {
      savedObjectsClient
    } = services;
    this.ruleName = rule.name;
    this.logger = server.logger;
    this.previousStartedAt = previousStartedAt;
    this.params = params;
    this.soClient = savedObjectsClient;
    this.esClient = esClient;
    this.monitorConfigRepository = new _monitor_config_repository.MonitorConfigRepository(savedObjectsClient, server.encryptedSavedObjects.getClient());
    this.server = server;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
    this.hasCustomCondition = !(0, _lodash.isEmpty)(this.params);
    this.monitorLocationsMap = {};
    this.options = options;
  }
  debug(message) {
    this.logger.debug(`[Status Rule Executor][${this.ruleName}] ${message}`);
  }
  async getStepInformationForBrowserMonitor(checkGroup, monitorType) {
    if (monitorType !== 'browser') {
      return null;
    }
    try {
      return await (0, _get_step_information.getStepInformation)(this.esClient, checkGroup, monitorType);
    } catch (error) {
      this.debug(`Failed to fetch step information for check group ${checkGroup}: ${error}`);
      return null;
    }
  }
  async init() {
    const {
      uiSettingsClient
    } = this.options.services;
    this.dateFormat = await uiSettingsClient.get('dateFormat');
    const timezone = await uiSettingsClient.get('dateFormat:tz');
    this.tz = timezone === 'Browser' ? 'UTC' : timezone;
    return await this.getMonitors();
  }
  async getMonitors() {
    const baseFilter = !this.hasCustomCondition ? `${_saved_objects.syntheticsMonitorAttributes}.${_monitor_management.AlertConfigKey.STATUS_ENABLED}: true` : '';
    const configIds = await (0, _filter_monitors.queryFilterMonitors)({
      spaceId: this.options.spaceId,
      esClient: this.esClient,
      ruleParams: this.params
    });
    if (this.params.kqlQuery && (0, _lodash.isEmpty)(configIds)) {
      this.debug(`No monitor found with the given KQL query ${this.params.kqlQuery}`);
      return (0, _process_monitors.processMonitors)([]);
    }
    const locationIds = await (0, _common3.parseLocationFilter)({
      savedObjectsClient: this.soClient,
      server: this.server,
      syntheticsMonitorClient: this.syntheticsMonitorClient
    }, this.params.locations);
    const {
      filtersStr
    } = (0, _common3.parseArrayFilters)({
      configIds,
      filter: baseFilter,
      tags: this.params.tags,
      locations: locationIds,
      monitorTypes: this.params.monitorTypes,
      monitorQueryIds: this.params.monitorIds,
      projects: this.params.projects
    });
    this.monitors = await this.monitorConfigRepository.getAll({
      filter: filtersStr
    });
    this.debug(`Found ${this.monitors.length} monitors for params ${JSON.stringify(this.params)} | parsed location filter is ${JSON.stringify(locationIds)} `);
    return (0, _process_monitors.processMonitors)(this.monitors);
  }
  async getConfigs({
    prevDownConfigs = {},
    prevPendingConfigs = {}
  }) {
    var _this$params4, _this$params$conditio5;
    const {
      enabledMonitorQueryIds,
      maxPeriod,
      monitorLocationIds,
      monitorsData
    } = await this.init();
    const range = this.getRange(maxPeriod);
    const {
      numberOfChecks
    } = (0, _status_rule.getConditionType)(this.params.condition);
    if (enabledMonitorQueryIds.length === 0) {
      const staleDownConfigs = this.markDeletedConfigs(prevDownConfigs);
      const stalePendingConfigs = this.markDeletedConfigs(prevPendingConfigs);
      return {
        downConfigs: {
          ...prevDownConfigs
        },
        upConfigs: {},
        staleDownConfigs,
        enabledMonitorQueryIds,
        pendingConfigs: {
          ...prevPendingConfigs
        },
        stalePendingConfigs,
        maxPeriod
      };
    }
    const queryLocations = (_this$params4 = this.params) === null || _this$params4 === void 0 ? void 0 : _this$params4.locations;

    // Account for locations filter
    const listOfLocationAfterFilter = queryLocations && queryLocations.length ? (0, _lodash.intersection)(monitorLocationIds, queryLocations) : monitorLocationIds;
    const currentStatus = await (0, _query_monitor_status_alert.queryMonitorStatusAlert)({
      esClient: this.esClient,
      monitorLocationIds: listOfLocationAfterFilter,
      range,
      monitorQueryIds: enabledMonitorQueryIds,
      numberOfChecks,
      includeRetests: (_this$params$conditio5 = this.params.condition) === null || _this$params$conditio5 === void 0 ? void 0 : _this$params$conditio5.includeRetests,
      monitorsData,
      monitors: this.monitors,
      logger: this.logger
    });
    const {
      downConfigs,
      upConfigs,
      pendingConfigs,
      configStats
    } = currentStatus;
    this.debug(`Found ${Object.keys(downConfigs).length} down configs, ${Object.keys(upConfigs).length} up configs and ${Object.keys(pendingConfigs).length} pending configs`);
    Object.entries(configStats).forEach(([configId, configStat]) => {
      var _this$monitors$find$a, _this$monitors$find;
      const name = (_this$monitors$find$a = (_this$monitors$find = this.monitors.find(m => m.id === configId)) === null || _this$monitors$find === void 0 ? void 0 : _this$monitors$find.attributes.name) !== null && _this$monitors$find$a !== void 0 ? _this$monitors$find$a : configId;
      this.debug(`Monitor: ${name} with id ${configId} has ${configStat.down} down check, ${configStat.up} up check and ${configStat.pending} pending check`);
    });
    const downConfigsToMarkAsStale = Object.keys(prevDownConfigs).reduce((acc, locId) => {
      if (!pendingConfigs[locId] && !upConfigs[locId] && !downConfigs[locId]) {
        acc[locId] = prevDownConfigs[locId];
      }
      return acc;
    }, {});
    const pendingConfigsToMarkAsStale = Object.keys(prevPendingConfigs).reduce((acc, locId) => {
      if (!pendingConfigs[locId] && !upConfigs[locId] && !downConfigs[locId]) {
        acc[locId] = prevPendingConfigs[locId];
      }
      return acc;
    }, {});
    const staleDownConfigs = this.markDeletedConfigs(downConfigsToMarkAsStale);
    const stalePendingConfigs = this.markDeletedConfigs(pendingConfigsToMarkAsStale);
    return {
      ...currentStatus,
      staleDownConfigs,
      stalePendingConfigs,
      maxPeriod
    };
  }
  markDeletedConfigs(configs) {
    const monitors = this.monitors;
    const staleConfigs = {};
    Object.keys(configs).forEach(locPlusId => {
      const config = configs[locPlusId];
      const monitor = monitors.find(m => {
        return m.id === config.configId || m.attributes[_runtime_types.ConfigKey.MONITOR_QUERY_ID] === config.monitorQueryId;
      });
      if (!monitor) {
        staleConfigs[locPlusId] = {
          ...config,
          isDeleted: true
        };
        delete configs[locPlusId];
      } else {
        const {
          locations
        } = monitor.attributes;
        const isLocationRemoved = !locations.some(l => l.id === config.locationId);
        if (isLocationRemoved) {
          staleConfigs[locPlusId] = {
            ...config,
            isLocationRemoved: true
          };
          delete configs[locPlusId];
        }
      }
    });
    return staleConfigs;
  }
  async schedulePendingAlertPerConfigIdPerLocation({
    pendingConfigs
  }) {
    for (const [idWithLocation, statusConfig] of Object.entries(pendingConfigs)) {
      const alertId = idWithLocation;
      const monitorSummary = this.getMonitorPendingSummary({
        statusConfig
      });
      await this.scheduleAlert({
        idWithLocation,
        alertId,
        monitorSummary,
        statusConfig,
        locationNames: [monitorSummary.locationName],
        locationIds: [statusConfig.locationId]
      });
    }
  }
  async schedulePendingAlertPerConfigId({
    pendingConfigs
  }) {
    const pendingConfigsById = getConfigsByIds(pendingConfigs);
    for (const [configId, configs] of pendingConfigsById) {
      const alertId = configId;
      const monitorSummary = this.getUngroupedPendingSummary({
        statusConfigs: configs
      });
      await this.scheduleAlert({
        idWithLocation: configId,
        alertId,
        monitorSummary,
        statusConfig: configs[0],
        locationNames: configs.map(({
          locationId,
          latestPing
        }) => (latestPing === null || latestPing === void 0 ? void 0 : latestPing.observer.geo.name) || locationId),
        locationIds: configs.map(({
          locationId
        }) => locationId)
      });
    }
  }
  getMonitorDownSummary({
    statusConfig
  }) {
    var _this$dateFormat, _this$tz;
    const {
      latestPing: ping,
      configId,
      locationId,
      checks
    } = statusConfig;
    return (0, _message_utils.getMonitorSummary)({
      monitorInfo: ping,
      reason: 'down',
      locationId: [locationId],
      configId,
      dateFormat: (_this$dateFormat = this.dateFormat) !== null && _this$dateFormat !== void 0 ? _this$dateFormat : 'Y-MM-DD HH:mm:ss',
      tz: (_this$tz = this.tz) !== null && _this$tz !== void 0 ? _this$tz : 'UTC',
      checks,
      params: this.params
    });
  }
  getMonitorPendingSummary({
    statusConfig
  }) {
    var _this$dateFormat2, _this$tz2;
    const {
      monitorInfo
    } = statusConfig;
    return (0, _message_utils.getMonitorSummary)({
      monitorInfo,
      reason: 'pending',
      locationId: [statusConfig.locationId],
      configId: statusConfig.configId,
      dateFormat: (_this$dateFormat2 = this.dateFormat) !== null && _this$dateFormat2 !== void 0 ? _this$dateFormat2 : 'Y-MM-DD HH:mm:ss',
      tz: (_this$tz2 = this.tz) !== null && _this$tz2 !== void 0 ? _this$tz2 : 'UTC',
      params: this.params
    });
  }
  getUngroupedDownSummary({
    statusConfigs
  }) {
    const sampleConfig = statusConfigs[0];
    const {
      latestPing: ping,
      configId,
      checks
    } = sampleConfig;
    const baseSummary = (0, _message_utils.getMonitorSummary)({
      monitorInfo: ping,
      reason: 'down',
      locationId: statusConfigs.map(c => c.latestPing.observer.name),
      configId,
      dateFormat: this.dateFormat,
      tz: this.tz,
      checks,
      params: this.params
    });
    baseSummary.reason = (0, _message_utils.getUngroupedReasonMessage)({
      statusConfigs,
      monitorName: baseSummary.monitorName,
      params: this.params,
      reason: 'down'
    });
    if (statusConfigs.length > 1) {
      baseSummary.locationNames = statusConfigs.map(c => {
        var _c$latestPing$observe2;
        return (_c$latestPing$observe2 = c.latestPing.observer.geo) === null || _c$latestPing$observe2 === void 0 ? void 0 : _c$latestPing$observe2.name;
      }).join(` ${_common2.AND_LABEL} `);
    }
    return baseSummary;
  }
  getUngroupedPendingSummary({
    statusConfigs
  }) {
    const sampleConfig = statusConfigs[0];
    const {
      configId,
      monitorInfo
    } = sampleConfig;
    const baseSummary = (0, _message_utils.getMonitorSummary)({
      monitorInfo,
      reason: 'pending',
      locationId: statusConfigs.map(({
        locationId
      }) => locationId),
      configId,
      dateFormat: this.dateFormat,
      tz: this.tz,
      params: this.params
    });
    baseSummary.reason = (0, _message_utils.getUngroupedReasonMessage)({
      statusConfigs,
      monitorName: baseSummary.monitorName,
      params: this.params,
      reason: 'pending'
    });
    return baseSummary;
  }
  async scheduleAlert(params) {
    var _ref;
    const {
      idWithLocation,
      alertId,
      monitorSummary,
      useLatestChecks = false,
      locationNames,
      locationIds
    } = params;
    const {
      configId
    } = 'statusConfig' in params ? params.statusConfig : params;
    const locationId = 'statusConfig' in params ? params.statusConfig.locationId : undefined;
    const {
      spaceId,
      startedAt
    } = this.options;
    const {
      alertsClient
    } = this.options.services;
    const {
      basePath
    } = this.server;
    if (!alertsClient) return;
    const {
      uuid: alertUuid,
      start
    } = alertsClient.report({
      id: alertId,
      actionGroup: _synthetics_alerts.MONITOR_STATUS.id
    });
    const errorStartedAt = (_ref = start !== null && start !== void 0 ? start : startedAt.toISOString()) !== null && _ref !== void 0 ? _ref : monitorSummary.timestamp;
    let relativeViewInAppUrl = '';
    if (monitorSummary.stateId) {
      // For ungrouped monitors with downConfigs, use the first location ID
      const effectiveLocationId = locationId || locationIds[0];
      if (effectiveLocationId) {
        relativeViewInAppUrl = (0, _common2.getRelativeViewInAppUrl)({
          configId,
          locationId: effectiveLocationId,
          stateId: monitorSummary.stateId
        });
      }
    }
    const grouping = {
      monitor: {
        id: monitorSummary.monitorId,
        config_id: monitorSummary.configId
      }
    };
    if (locationIds.length === 1) {
      grouping.location = {
        id: locationIds[0]
      };
    }
    if (monitorSummary.serviceName) {
      grouping.service = {
        name: monitorSummary.serviceName
      };
    }
    const context = {
      ...monitorSummary,
      idWithLocation,
      errorStartedAt,
      linkMessage: monitorSummary.stateId ? (0, _common2.getFullViewInAppMessage)(basePath, spaceId, relativeViewInAppUrl) : '',
      [_action_variables.VIEW_IN_APP_URL]: (0, _common2.getViewInAppUrl)(basePath, spaceId, relativeViewInAppUrl),
      [_action_variables.ALERT_DETAILS_URL]: (0, _common.getAlertDetailsUrl)(basePath, spaceId, alertUuid),
      grouping
    };

    // downThreshold and checks are only available for down alerts
    if ('downThreshold' in params) {
      context.downThreshold = params.downThreshold;
    }
    if ('statusConfig' in params && 'checks' in params.statusConfig) {
      context.checks = params.statusConfig.checks;
    } else if ('downConfigs' in params) {
      // For ungrouped alerts with downConfigs, get checks from the first config
      const configsArray = Object.values(params.downConfigs);
      if (configsArray.length > 0 && 'checks' in configsArray[0]) {
        context.checks = configsArray[0].checks;
      }
    }

    // Fetch step information for browser monitors synchronously before creating alert
    let failedStepInfo = '';
    let failedStepName;
    let failedStepNumber;
    // Get monitor type directly from saved object attributes
    const monitor = this.monitors.find(m => m.id === configId);
    const monitorType = monitor === null || monitor === void 0 ? void 0 : monitor.attributes.type;
    if (monitorType === 'browser') {
      let allConfigs = [];
      if ('statusConfig' in params) {
        allConfigs = [params.statusConfig];
      } else if ('downConfigs' in params) {
        allConfigs = getConfigsByIds(params.downConfigs).get(params.configId) || [];
      }
      const stepInfoPromises = allConfigs.map(async config => {
        if ('latestPing' in config && config.latestPing) {
          var _config$latestPing$mo;
          const checkGroup = (_config$latestPing$mo = config.latestPing.monitor) === null || _config$latestPing$mo === void 0 ? void 0 : _config$latestPing$mo.check_group;
          if (checkGroup) {
            try {
              return await this.getStepInformationForBrowserMonitor(checkGroup, 'browser');
            } catch (error) {
              this.debug(`Failed to fetch step information for alert ${alertId}: ${error}`);
            }
          }
        }
        return null;
      });
      const stepInfoResults = await Promise.all(stepInfoPromises);
      const validStepInfos = stepInfoResults.filter(info => info !== null);

      // Format step information for the alert body
      const formattedStepInfos = validStepInfos.map(info => (0, _message_utils.formatStepInformation)(info));
      failedStepInfo = formattedStepInfos.filter(info => info).join('\n');

      // Extract first step name and number for the email subject
      if (validStepInfos.length > 0) {
        const firstStep = validStepInfos[0];
        failedStepName = firstStep.stepName;
        failedStepNumber = firstStep.stepNumber;
      }
    }

    // Update monitor summary with step info
    const updatedMonitorSummary = {
      ...monitorSummary,
      failedStepInfo,
      failedStepName,
      failedStepNumber
    };
    const alertDocument = (0, _message_utils.getMonitorAlertDocument)(updatedMonitorSummary, locationNames, locationIds, useLatestChecks, 'downThreshold' in params ? params.downThreshold : 1, grouping);

    // Update context with step info if available
    const updatedContext = {
      ...context,
      ...(failedStepInfo ? {
        failedStepInfo
      } : {}),
      ...(failedStepName ? {
        failedStepName
      } : {}),
      ...(failedStepNumber !== undefined ? {
        failedStepNumber
      } : {})
    };
    alertsClient.setAlertData({
      id: alertId,
      payload: alertDocument,
      context: updatedContext
    });
  }
}
exports.StatusRuleExecutor = StatusRuleExecutor;
const getDoesMonitorMeetLocationThreshold = ({
  matchesByLocation,
  locationsThreshold,
  downThreshold,
  useTimeWindow
}) => {
  // for location based we need to make sure, monitor is down for the threshold for all locations
  const getMatchingLocationsWithDownThresholdWithXChecks = matches => {
    return matches.filter(config => {
      var _config$checks$downWi, _config$checks;
      return ((_config$checks$downWi = (_config$checks = config.checks) === null || _config$checks === void 0 ? void 0 : _config$checks.downWithinXChecks) !== null && _config$checks$downWi !== void 0 ? _config$checks$downWi : 1) >= downThreshold;
    });
  };
  const getMatchingLocationsWithDownThresholdWithinTimeWindow = matches => {
    return matches.filter(config => {
      var _config$checks$down, _config$checks2;
      return ((_config$checks$down = (_config$checks2 = config.checks) === null || _config$checks2 === void 0 ? void 0 : _config$checks2.down) !== null && _config$checks$down !== void 0 ? _config$checks$down : 1) >= downThreshold;
    });
  };
  if (useTimeWindow) {
    const matchingLocationsWithDownThreshold = getMatchingLocationsWithDownThresholdWithinTimeWindow(matchesByLocation);
    return matchingLocationsWithDownThreshold.length >= locationsThreshold;
  } else {
    const matchingLocationsWithDownThreshold = getMatchingLocationsWithDownThresholdWithXChecks(matchesByLocation);
    return matchingLocationsWithDownThreshold.length >= locationsThreshold;
  }
};
exports.getDoesMonitorMeetLocationThreshold = getDoesMonitorMeetLocationThreshold;
function getConfigsByIds(configs) {
  const configsById = new Map();
  Object.entries(configs).forEach(([_, config]) => {
    var _configsById$get;
    const {
      configId
    } = config;
    if (!configsById.has(configId)) {
      configsById.set(configId, []);
    }
    (_configsById$get = configsById.get(configId)) === null || _configsById$get === void 0 ? void 0 : _configsById$get.push(config);
  });
  return configsById;
}