"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.groupRoutes = void 0;
var _zod = require("@kbn/zod");
var _boom = require("@hapi/boom");
var _streamsSchema = require("@kbn/streams-schema");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _constants = require("../../../../common/constants");
var _create_server_route = require("../../create_server_route");
var _fields = require("../../../lib/streams/assets/fields");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const readGroupRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/_group 2023-10-31',
  options: {
    access: 'public',
    summary: 'Get group stream settings',
    description: 'Fetches the group settings of a group stream definition',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const definition = await streamsClient.getStream(name);
    if (_streamsSchema.Streams.GroupStream.Definition.is(definition)) {
      return {
        group: definition.group
      };
    }
    throw (0, _boom.badRequest)(`Stream is not a group stream`);
  }
});
const upsertGroupRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /api/streams/{name}/_group 2023-10-31',
  options: {
    access: 'public',
    description: 'Upserts the group settings of a group stream definition',
    summary: 'Upsert group stream settings',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      group: _streamsSchema.Group.right
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    context
  }) => {
    const {
      streamsClient,
      assetClient,
      attachmentClient
    } = await getScopedClients({
      request
    });
    const core = await context.core;
    const groupStreamsEnabled = await core.uiSettings.client.get(_managementSettingsIds.OBSERVABILITY_STREAMS_ENABLE_GROUP_STREAMS);
    if (!groupStreamsEnabled) {
      throw (0, _boom.badData)('Streams are not enabled for Group streams.');
    }
    const {
      name
    } = params.path;
    const {
      group
    } = params.body;
    const definition = await streamsClient.getStream(name);
    if (!_streamsSchema.Streams.GroupStream.Definition.is(definition)) {
      throw (0, _boom.badData)(`Cannot update group capabilities of non-group stream`);
    }
    const [assets, attachments] = await Promise.all([assetClient.getAssets(name), attachmentClient.getAttachments(name)]);
    const dashboards = attachments.filter(attachment => attachment.type === 'dashboard').map(attachment => attachment.id);
    const rules = attachments.filter(attachment => attachment.type === 'rule').map(attachment => attachment.id);
    const queries = assets.filter(asset => asset[_fields.ASSET_TYPE] === 'query').map(asset => asset.query);
    const {
      name: _name,
      ...stream
    } = definition;
    const upsertRequest = {
      dashboards,
      stream: {
        ...stream,
        group
      },
      queries,
      rules
    };
    return await streamsClient.upsertStream({
      request: upsertRequest,
      name
    });
  }
});
const groupRoutes = exports.groupRoutes = {
  ...readGroupRoute,
  ...upsertGroupRoute
};