"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readStreamRoute = exports.listStreamsRoute = exports.editStreamRoute = exports.deleteStreamRoute = exports.crudRoutes = void 0;
var _zod = require("@kbn/zod");
var _boom = require("@hapi/boom");
var _streamsSchema = require("@kbn/streams-schema");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _constants = require("../../../../common/constants");
var _create_server_route = require("../../create_server_route");
var _read_stream = require("./read_stream");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const readStreamRoute = exports.readStreamRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Get a stream',
    description: 'Fetches a stream definition and associated dashboards',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      assetClient,
      attachmentClient,
      streamsClient,
      scopedClusterClient
    } = await getScopedClients({
      request
    });
    const body = await (0, _read_stream.readStream)({
      name: params.path.name,
      assetClient,
      attachmentClient,
      scopedClusterClient,
      streamsClient
    });
    return body;
  }
});
const listStreamsRoute = exports.listStreamsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams 2023-10-31',
  options: {
    access: 'public',
    description: 'Fetches list of all streams',
    summary: 'Get stream list',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({}),
  handler: async ({
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    return {
      streams: await streamsClient.listStreams()
    };
  }
});
const editStreamRoute = exports.editStreamRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /api/streams/{name} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Create or update a stream',
    description: 'Creates or updates a stream definition. Classic streams can not be created through this API, only updated',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _streamsSchema.Streams.all.UpsertRequest.right
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    context
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    if (_streamsSchema.Streams.WiredStream.UpsertRequest.is(params.body) && !(await streamsClient.isStreamsEnabled())) {
      throw (0, _boom.badData)('Streams are not enabled for Wired streams.');
    }
    const core = await context.core;
    const groupStreamsEnabled = await core.uiSettings.client.get(_managementSettingsIds.OBSERVABILITY_STREAMS_ENABLE_GROUP_STREAMS);
    if (_streamsSchema.Streams.GroupStream.UpsertRequest.is(params.body) && !groupStreamsEnabled) {
      throw (0, _boom.badData)('Streams are not enabled for Group streams.');
    }
    return await streamsClient.upsertStream({
      request: params.body,
      name: params.path.name
    });
  }
});
const deleteStreamRoute = exports.deleteStreamRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'DELETE /api/streams/{name} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Delete a stream',
    description: 'Deletes a stream definition and the underlying data stream',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      streamsClient
    } = await getScopedClients({
      request
    });
    return await streamsClient.deleteStream(params.path.name);
  }
});
const crudRoutes = exports.crudRoutes = {
  ...readStreamRoute,
  ...listStreamsRoute,
  ...editStreamRoute,
  ...deleteStreamRoute
};