"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ruleRoutes = void 0;
var _zod = require("@kbn/zod");
var _constants = require("../../../common/constants");
var _create_server_route = require("../create_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const listRulesRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /api/streams/{name}/rules 2023-10-31',
  options: {
    access: 'public',
    summary: 'Get stream rules',
    description: 'Fetches all rules linked to a stream that are visible to the current user in the current space.',
    availability: {
      stability: 'experimental'
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  async handler({
    params,
    request,
    getScopedClients
  }) {
    const {
      attachmentClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    const {
      path: {
        name: streamName
      }
    } = params;
    function isRule(attachment) {
      return attachment.type === 'rule';
    }
    return {
      rules: (await attachmentClient.getAttachments(streamName, 'rule')).filter(isRule)
    };
  }
});
const linkRuleRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /api/streams/{name}/rules/{ruleId} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Link a rule to a stream',
    description: 'Links a rule to a stream. Noop if the rule is already linked to the stream.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      ruleId: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      attachmentClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      path: {
        ruleId,
        name: streamName
      }
    } = params;
    await streamsClient.ensureStream(streamName);
    await attachmentClient.linkAttachment(streamName, {
      id: ruleId,
      type: 'rule'
    });
    return {
      acknowledged: true
    };
  }
});
const unlinkRuleRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'DELETE /api/streams/{name}/rules/{ruleId} 2023-10-31',
  options: {
    access: 'public',
    summary: 'Unlink a rule from a stream',
    description: 'Unlinks a rule from a stream. Noop if the rule is not linked to the stream.',
    availability: {
      stability: 'experimental'
    }
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string(),
      ruleId: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      attachmentClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    const {
      path: {
        ruleId,
        name: streamName
      }
    } = params;
    await attachmentClient.unlinkAttachment(streamName, {
      id: ruleId,
      type: 'rule'
    });
    return {
      acknowledged: true
    };
  }
});
const suggestRulesRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/rules/_suggestions',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    query: _zod.z.object({
      query: _zod.z.string()
    }),
    body: _zod.z.object({
      tags: _zod.z.optional(_zod.z.array(_zod.z.string()))
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      attachmentClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    await streamsClient.ensureStream(params.path.name);
    const {
      query: {
        query
      },
      body: {
        tags
      }
    } = params;
    const suggestions = (await attachmentClient.getSuggestions({
      attachmentTypes: ['rule'],
      query,
      tags
    })).attachments;
    return {
      suggestions
    };
  }
});
const ruleRoutes = exports.ruleRoutes = {
  ...listRulesRoute,
  ...linkRuleRoute,
  ...unlinkRuleRoute,
  ...suggestRulesRoute
};