"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.simulateProcessorRoute = exports.processingGrokSuggestionRoute = exports.processingDissectSuggestionRoute = exports.processingDateSuggestionsRoute = exports.internalProcessingRoutes = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _zod = require("@kbn/zod");
var _streamlang = require("@kbn/streamlang");
var _rxjs = require("rxjs");
var _constants = require("../../../../../common/constants");
var _security_error = require("../../../../lib/streams/errors/security_error");
var _stream_crud = require("../../../../lib/streams/stream_crud");
var _create_server_route = require("../../../create_server_route");
var _simulation_handler = require("./simulation_handler");
var _date_suggestions_handler = require("./date_suggestions_handler");
var _grok_suggestions_handler = require("./grok_suggestions_handler");
var _dissect_suggestions_handler = require("./dissect_suggestions_handler");
var _get_request_abort_signal = require("../../../utils/get_request_abort_signal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const paramsSchema = _zod.z.object({
  path: _zod.z.object({
    name: _zod.z.string()
  }),
  body: _zod.z.object({
    processing: _streamlang.streamlangDSLSchema,
    documents: _zod.z.array(_streamsSchema.flattenRecord),
    detected_fields: _zod.z.array(_streamsSchema.namedFieldDefinitionConfigSchema).optional()
  })
});
const simulateProcessorRoute = exports.simulateProcessorRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/processing/_simulate',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: paramsSchema,
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient,
      fieldsMetadataClient
    } = await getScopedClients({
      request
    });
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name: params.path.name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${params.path.name}, insufficient privileges`);
    }
    return (0, _simulation_handler.simulateProcessing)({
      params,
      scopedClusterClient,
      streamsClient,
      fieldsMetadataClient
    });
  }
});
const processingGrokSuggestionRoute = exports.processingGrokSuggestionRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/processing/_suggestions/grok',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _grok_suggestions_handler.processingGrokSuggestionsSchema,
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const isAvailableForTier = server.core.pricing.isFeatureAvailable(_constants.STREAMS_TIERED_ML_FEATURE.id);
    if (!isAvailableForTier) {
      throw new _security_error.SecurityError('Cannot access API on the current pricing tier');
    }
    const {
      inferenceClient,
      scopedClusterClient,
      streamsClient,
      fieldsMetadataClient
    } = await getScopedClients({
      request
    });

    // Turn our promise into an Observable ServerSideEvent. The only reason we're streaming the
    // response here is to avoid timeout issues prevalent with long-running requests to LLMs.
    return (0, _rxjs.from)((0, _grok_suggestions_handler.handleProcessingGrokSuggestions)({
      params,
      inferenceClient,
      streamsClient,
      scopedClusterClient,
      fieldsMetadataClient,
      signal: (0, _get_request_abort_signal.getRequestAbortSignal)(request)
    })).pipe((0, _rxjs.map)(grokProcessor => ({
      grokProcessor,
      type: 'grok_suggestion'
    })));
  }
});
const processingDissectSuggestionRoute = exports.processingDissectSuggestionRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/processing/_suggestions/dissect',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _dissect_suggestions_handler.processingDissectSuggestionsSchema,
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const isAvailableForTier = server.core.pricing.isFeatureAvailable(_constants.STREAMS_TIERED_ML_FEATURE.id);
    if (!isAvailableForTier) {
      throw new _security_error.SecurityError('Cannot access API on the current pricing tier');
    }
    const {
      inferenceClient,
      scopedClusterClient,
      streamsClient,
      fieldsMetadataClient
    } = await getScopedClients({
      request
    });

    // Turn our promise into an Observable ServerSideEvent. The only reason we're streaming the
    // response here is to avoid timeout issues prevalent with long-running requests to LLMs.
    return (0, _rxjs.from)((0, _dissect_suggestions_handler.handleProcessingDissectSuggestions)({
      params,
      inferenceClient,
      streamsClient,
      scopedClusterClient,
      fieldsMetadataClient,
      signal: (0, _get_request_abort_signal.getRequestAbortSignal)(request)
    })).pipe((0, _rxjs.map)(dissectProcessor => ({
      dissectProcessor,
      type: 'dissect_suggestion'
    })));
  }
});
const processingDateSuggestionsRoute = exports.processingDateSuggestionsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/processing/_suggestions/date',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _date_suggestions_handler.processingDateSuggestionsSchema,
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const isAvailableForTier = server.core.pricing.isFeatureAvailable(_constants.STREAMS_TIERED_ML_FEATURE.id);
    if (!isAvailableForTier) {
      throw new _security_error.SecurityError('Cannot access API on the current pricing tier');
    }
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${name}, insufficient privileges`);
    }
    const {
      text_structure: hasTextStructurePrivileges
    } = await streamsClient.getPrivileges(name);
    if (!hasTextStructurePrivileges) {
      throw new _security_error.SecurityError(`Cannot access text structure capabilities, insufficient privileges`);
    }
    return (0, _date_suggestions_handler.handleProcessingDateSuggestions)({
      params,
      scopedClusterClient
    });
  }
});
const internalProcessingRoutes = exports.internalProcessingRoutes = {
  ...simulateProcessorRoute,
  ...processingGrokSuggestionRoute,
  ...processingDissectSuggestionRoute,
  ...processingDateSuggestionsRoute
};