"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callInferenceWithPrompt = callInferenceWithPrompt;
exports.determineOtelFieldNameUsage = determineOtelFieldNameUsage;
exports.fetchFieldMetadata = fetchFieldMetadata;
exports.normalizeFieldName = normalizeFieldName;
var _streamsSchema = require("@kbn/streams-schema");
var _otelSemanticConventions = require("@kbn/otel-semantic-conventions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Determines whether OTEL field names should be used for a given stream.
 * Returns true if the stream is a wired stream or matches the logs-*.otel-* pattern.
 */
async function determineOtelFieldNameUsage(streamsClient, streamName) {
  const stream = await streamsClient.getStream(streamName);
  const isWiredStream = _streamsSchema.Streams.WiredStream.Definition.is(stream);
  return isWiredStream || !!streamName.match(/^logs-.*\.otel-/);
}

/**
 * Calls the LLM inference API with the provided prompt and input data.
 * Returns the parsed tool call arguments from the response.
 */
async function callInferenceWithPrompt(inferenceClient, connectorId, prompt, sampleMessages, reviewFields, signal) {
  const response = await inferenceClient.prompt({
    connectorId,
    prompt,
    input: {
      sample_messages: sampleMessages,
      review_fields: JSON.stringify(reviewFields)
    },
    abortSignal: signal
  });

  // Access toolCalls from the response
  if (!('toolCalls' in response)) {
    throw new Error('Expected toolCalls in LLM response');
  }
  return response.toolCalls[0].function.arguments;
}

/**
 * Fetches ECS/OTEL field metadata for the provided field names.
 * Returns a dictionary mapping field names to their metadata.
 */
async function fetchFieldMetadata(fieldsMetadataClient, ecsFields) {
  return fieldsMetadataClient.find({
    fieldNames: ecsFields
  }).then(fieldsDictionary => fieldsDictionary.toPlain());
}

/**
 * Normalizes a field name by:
 * 1. Replacing @timestamp with custom.timestamp to avoid format issues
 * 2. Resolving to OTEL field name equivalent if needed
 */
function normalizeFieldName(ecsField, fieldMetadata, useOtelFieldNames) {
  var _fieldMetadata$name$o, _fieldMetadata$name;
  // @timestamp is a special case that we want to map to custom.timestamp - if we let it overwrite @timestamp it will most likely
  // fail because the format won't be right. In a follow-up we can extend the suggestion to also add a date format processor step
  // to map it back correctly.
  const name = ecsField.startsWith('@timestamp') ? ecsField.replace('@timestamp', 'custom.timestamp') : ecsField;

  // Make sure otel field names are translated/prefixed correctly
  return useOtelFieldNames ? (_fieldMetadata$name$o = (_fieldMetadata$name = fieldMetadata[name]) === null || _fieldMetadata$name === void 0 ? void 0 : _fieldMetadata$name.otel_equivalent) !== null && _fieldMetadata$name$o !== void 0 ? _fieldMetadata$name$o : (0, _otelSemanticConventions.prefixOTelField)(name) : name;
}