"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestPartitionsSchema = exports.suggestPartitionsRoute = void 0;
var _zod = require("@kbn/zod");
var _streamsSchema = require("@kbn/streams-schema");
var _streamsAi = require("@kbn/streams-ai");
var _rxjs = require("rxjs");
var _common = require("../../../../../common");
var _constants = require("../../../../../common/constants");
var _security_error = require("../../../../lib/streams/errors/security_error");
var _create_server_route = require("../../../create_server_route");
var _get_request_abort_signal = require("../../../utils/get_request_abort_signal");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const suggestPartitionsSchema = exports.suggestPartitionsSchema = _zod.z.object({
  path: _zod.z.object({
    name: _zod.z.string()
  }),
  body: _zod.z.object({
    connector_id: _zod.z.string(),
    start: _zod.z.number(),
    end: _zod.z.number()
  })
});
const suggestPartitionsRoute = exports.suggestPartitionsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/_suggest_partitions',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: suggestPartitionsSchema,
  handler: async ({
    params,
    request,
    getScopedClients,
    server,
    logger
  }) => {
    const isAvailableForTier = server.core.pricing.isFeatureAvailable(_common.STREAMS_TIERED_ML_FEATURE.id);
    if (!isAvailableForTier) {
      throw new _security_error.SecurityError('Cannot access API on the current pricing tier');
    }
    const {
      inferenceClient,
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const stream = await streamsClient.getStream(params.path.name);
    if (!_streamsSchema.Streams.ingest.all.Definition.is(stream)) {
      throw new Error(`Stream ${stream.name} is not a valid ingest stream`);
    }
    const partitionsPromise = (0, _streamsAi.partitionStream)({
      definition: stream,
      inferenceClient: inferenceClient.bindTo({
        connectorId: params.body.connector_id
      }),
      esClient: scopedClusterClient.asCurrentUser,
      logger,
      start: params.body.start,
      end: params.body.end,
      maxSteps: 1,
      // Longer reasoning seems to add unnecessary conditions (and latency), instead of improving accuracy, so we limit the steps.
      signal: (0, _get_request_abort_signal.getRequestAbortSignal)(request)
    });

    // Turn our promise into an Observable ServerSideEvent. The only reason we're streaming the
    // response here is to avoid timeout issues prevalent with long-running requests to LLMs.
    return (0, _rxjs.from)(partitionsPromise).pipe((0, _rxjs.map)(partitions => ({
      partitions,
      type: 'suggested_partitions'
    })));
  }
});