"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFailureStoreStatsRoute = exports.getFailureStoreDefaultRetentionRoute = exports.failureStoreRoutes = void 0;
var _zod = require("@kbn/zod");
var _stream_crud = require("../../../../lib/streams/stream_crud");
var _constants = require("../../../../../common/constants");
var _create_server_route = require("../../../create_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFailureStoreStatsRoute = exports.getFailureStoreStatsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/failure_store/stats',
  options: {
    access: 'internal',
    summary: 'Get failure store stats',
    description: 'Gets the failure store statistics for a stream'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const privileges = await streamsClient.getPrivileges(name);
    if (!privileges.manage_failure_store) {
      return {
        stats: null
      };
    }
    const stats = await (0, _stream_crud.getFailureStoreStats)({
      name,
      scopedClusterClient,
      isServerless: server.isServerless
    });
    return {
      stats
    };
  }
});
const getFailureStoreDefaultRetentionRoute = exports.getFailureStoreDefaultRetentionRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/failure_store/default_retention',
  options: {
    access: 'internal',
    summary: 'Get failure store default retention',
    description: 'Gets the default retention period for the failure store'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      scopedClusterClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const defaultRetention = await (0, _stream_crud.getFailureStoreDefaultRetention)({
      name,
      scopedClusterClient,
      isServerless: !!server.isServerless
    });
    return {
      default_retention: defaultRetention
    };
  }
});
const failureStoreRoutes = exports.failureStoreRoutes = {
  ...getFailureStoreStatsRoute,
  ...getFailureStoreDefaultRetentionRoute
};