"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectorsRoute = exports.connectorRoutes = void 0;
var _inferenceCommon = require("@kbn/inference-common");
var _constants = require("../../../../common/constants");
var _create_server_route = require("../../create_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INFERENCE_CONNECTOR_TYPE = '.inference';
async function inferenceEndpointExists(esClient, inferenceId) {
  try {
    const endpoints = await esClient.inference.get({
      inference_id: inferenceId
    });
    return endpoints.endpoints.some(endpoint => endpoint.inference_id === inferenceId);
  } catch (error) {
    return false;
  }
}
const getConnectorsRoute = exports.getConnectorsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/connectors',
  options: {
    access: 'internal',
    summary: 'Get GenAI connectors',
    description: 'Fetches all available GenAI connectors for AI features'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  handler: async ({
    request,
    getScopedClients,
    server
  }) => {
    const {
      scopedClusterClient
    } = await getScopedClients({
      request
    });

    // Get actions client with request
    const actionsClient = await server.actions.getActionsClientWithRequest(request);
    if (!actionsClient) {
      throw new Error('Actions client not available');
    }
    const connectors = await actionsClient.getAll();

    // Filter to only supported GenAI connector types
    const supportedConnectors = connectors.filter(connector => (0, _inferenceCommon.isSupportedConnectorType)(connector.actionTypeId));

    // Validate inference connectors have endpoints
    const validatedConnectors = await Promise.all(supportedConnectors.map(async connector => {
      if (connector.actionTypeId === INFERENCE_CONNECTOR_TYPE) {
        var _connector$config;
        const inferenceId = (_connector$config = connector.config) === null || _connector$config === void 0 ? void 0 : _connector$config.inferenceId;
        if (inferenceId) {
          const exists = await inferenceEndpointExists(scopedClusterClient.asCurrentUser, inferenceId);
          if (!exists) {
            return null;
          }
        }
      }
      return connector;
    }));
    const filteredConnectors = validatedConnectors.filter(connector => connector !== null);
    return {
      connectors: filteredConnectors
    };
  }
});
const connectorRoutes = exports.connectorRoutes = {
  ...getConnectorsRoute
};