"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.migrateOnRead = migrateOnRead;
var _lodash = require("lodash");
var _streamsSchema = require("@kbn/streams-schema");
var _hierarchy = require("@kbn/streams-schema/src/shared/hierarchy");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _streamlang = require("@kbn/streamlang");
var _migrate_to_streamlang_on_read = require("./migrate_to_streamlang_on_read");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function migrateOnRead(definition) {
  let migratedDefinition = definition;
  let hasBeenMigrated = false;
  // Add required description
  if (typeof migratedDefinition.description !== 'string') {
    migratedDefinition = {
      ...migratedDefinition,
      description: ''
    };
    hasBeenMigrated = true;
  }
  // Rename unwired to classic
  if ((0, _lodash.isObject)(migratedDefinition.ingest) && 'unwired' in migratedDefinition.ingest && (0, _lodash.isObject)(migratedDefinition.ingest.unwired)) {
    migratedDefinition = {
      ...migratedDefinition,
      ingest: {
        ...migratedDefinition.ingest,
        classic: {
          ...migratedDefinition.ingest.unwired
        }
      }
    };
    delete migratedDefinition.ingest.unwired;
    hasBeenMigrated = true;
  }

  // Migrate routing "if" Condition to Streamlang
  if ((0, _lodash.isObject)(migratedDefinition.ingest) && 'wired' in migratedDefinition.ingest && migratedDefinition.ingest.wired && typeof migratedDefinition.ingest.wired === 'object' && Array.isArray(migratedDefinition.ingest.wired.routing) && migratedDefinition.ingest.wired.routing.some(route => 'if' in route)) {
    migratedDefinition = (0, _migrate_to_streamlang_on_read.migrateRoutingIfConditionToStreamlang)(migratedDefinition);
    hasBeenMigrated = true;
  }

  // Migrate old flat processing array to Streamlang DSL
  if (migratedDefinition.ingest && typeof migratedDefinition.ingest === 'object' && Array.isArray(migratedDefinition.ingest.processing)) {
    migratedDefinition = (0, _migrate_to_streamlang_on_read.migrateOldProcessingArrayToStreamlang)(migratedDefinition);
    hasBeenMigrated = true;
  }

  // Migrate routing definitions to include status field
  if ((0, _lodash.isObject)(migratedDefinition.ingest) && 'wired' in migratedDefinition.ingest && (0, _lodash.isObject)(migratedDefinition.ingest.wired) && Array.isArray(migratedDefinition.ingest.wired.routing) && migratedDefinition.ingest.wired.routing.some(route => !('status' in route))) {
    const routings = (0, _lodash.get)(migratedDefinition, 'ingest.wired.routing', []);
    const migratedRouting = routings.map(route => {
      // If route doesn't have status field, add it based on the condition
      if (!('status' in route) && 'where' in route) {
        const isDisabledCondition = (0, _streamlang.isNeverCondition)(route.where);
        return {
          ...route,
          where: isDisabledCondition ? {
            always: {}
          } : route.where,
          status: isDisabledCondition ? 'disabled' : 'enabled'
        };
      }
      return route;
    });
    (0, _saferLodashSet.set)(migratedDefinition, 'ingest.wired.routing', migratedRouting);
    hasBeenMigrated = true;
  }

  // add settings
  if ((0, _lodash.isObject)(migratedDefinition.ingest) && !('settings' in migratedDefinition.ingest)) {
    (0, _saferLodashSet.set)(migratedDefinition, 'ingest.settings', {});
    hasBeenMigrated = true;
  }

  // Add metadata to Group stream if missing
  if ((0, _lodash.isObject)(migratedDefinition.group) && !('metadata' in migratedDefinition.group)) {
    migratedDefinition = {
      ...migratedDefinition,
      group: {
        ...migratedDefinition.group,
        metadata: {}
      }
    };
    hasBeenMigrated = true;
  }

  // Add tags to Group stream if missing
  if ((0, _lodash.isObject)(migratedDefinition.group) && !('tags' in migratedDefinition.group)) {
    migratedDefinition = {
      ...migratedDefinition,
      group: {
        ...migratedDefinition.group,
        tags: []
      }
    };
    hasBeenMigrated = true;
  }

  // Add failure_store to ingest streams if missing
  if ((0, _lodash.isObject)(migratedDefinition.ingest) && !('failure_store' in migratedDefinition.ingest)) {
    const streamName = migratedDefinition.name;
    if ('wired' in migratedDefinition.ingest && typeof streamName === 'string' && (0, _hierarchy.isRoot)(streamName)) {
      (0, _saferLodashSet.set)(migratedDefinition, 'ingest.failure_store', {
        lifecycle: {
          enabled: {
            data_retention: '30d'
          }
        }
      });
    } else {
      (0, _saferLodashSet.set)(migratedDefinition, 'ingest.failure_store', {
        inherit: {}
      });
    }
    hasBeenMigrated = true;
  }
  if (hasBeenMigrated) {
    _streamsSchema.Streams.all.Definition.asserts(migratedDefinition);
  }
  return migratedDefinition;
}