"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ClassicStream = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = _interopRequireWildcard(require("lodash"));
var _esErrors = require("@kbn/es-errors");
var _fields = require("@kbn/streams-schema/src/fields");
var _failure_store = require("@kbn/streams-schema/src/models/ingest/failure_store");
var _status_error = require("../../errors/status_error");
var _generate_ingest_pipeline = require("../../ingest_pipelines/generate_ingest_pipeline");
var _name = require("../../ingest_pipelines/name");
var _stream_crud = require("../../stream_crud");
var _stream_active_record = require("../stream_active_record/stream_active_record");
var _validate_fields = require("../../helpers/validate_fields");
var _validate_stream = require("../../helpers/validate_stream");
var _helpers = require("./helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ClassicStream extends _stream_active_record.StreamActiveRecord {
  constructor(definition, dependencies) {
    super(definition, dependencies);
    (0, _defineProperty2.default)(this, "_changes", {
      processing: false,
      field_overrides: false,
      lifecycle: false,
      failure_store: false,
      settings: false
    });
    (0, _defineProperty2.default)(this, "_effectiveSettings", void 0);
  }
  doClone() {
    return new ClassicStream((0, _lodash.cloneDeep)(this._definition), this.dependencies);
  }
  async doHandleUpsertChange(definition, desiredState, startingState) {
    var _startingState$get;
    if (definition.name !== this._definition.name) {
      return {
        cascadingChanges: [],
        changeStatus: this.changeStatus
      };
    }
    if (!_streamsSchema.Streams.ClassicStream.Definition.is(definition)) {
      throw new _status_error.StatusError('Cannot change stream types', 400);
    }
    this._definition = definition;
    const startingStateStreamDefinition = (_startingState$get = startingState.get(this._definition.name)) === null || _startingState$get === void 0 ? void 0 : _startingState$get.definition;
    if (startingStateStreamDefinition && !_streamsSchema.Streams.ClassicStream.Definition.is(startingStateStreamDefinition)) {
      throw new _status_error.StatusError('Unexpected starting state stream type', 400);
    }
    this._changes.processing = !startingStateStreamDefinition || !_lodash.default.isEqual(this._definition.ingest.processing, startingStateStreamDefinition.ingest.processing);
    this._changes.lifecycle = !startingStateStreamDefinition || !_lodash.default.isEqual(this._definition.ingest.lifecycle, startingStateStreamDefinition.ingest.lifecycle);
    this._changes.settings = !startingStateStreamDefinition || !_lodash.default.isEqual(await this.getEffectiveSettings(), this._definition.ingest.settings);
    this._changes.field_overrides = !startingStateStreamDefinition || !_lodash.default.isEqual(this._definition.ingest.classic.field_overrides, startingStateStreamDefinition.ingest.classic.field_overrides);
    this._changes.failure_store = !startingStateStreamDefinition || !_lodash.default.isEqual(this._definition.ingest.failure_store, startingStateStreamDefinition.ingest.failure_store);
    return {
      cascadingChanges: [],
      changeStatus: 'upserted'
    };
  }
  async doHandleDeleteChange(target, desiredState, startingState) {
    if (target !== this._definition.name) {
      return {
        cascadingChanges: [],
        changeStatus: this.changeStatus
      };
    }
    return {
      cascadingChanges: [],
      changeStatus: 'deleted'
    };
  }
  async doValidateUpsertion(desiredState, startingState) {
    if (this.dependencies.isServerless) {
      if ((0, _streamsSchema.isIlmLifecycle)(this.getLifecycle())) {
        return {
          isValid: false,
          errors: [new Error('Using ILM is not supported in Serverless')]
        };
      }
      if ((0, _failure_store.isDisabledLifecycleFailureStore)(this.getFailureStore())) {
        return {
          isValid: false,
          errors: [new Error('Disabling failure store lifecycle is not supported in Serverless')]
        };
      }
    }

    // Check for conflicts
    if (this._changes.lifecycle || this._changes.processing) {
      try {
        const dataStreamResult = await this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStream({
          name: this._definition.name
        });
        if (dataStreamResult.data_streams.length === 0) {
          // There is an index but no data stream
          return {
            isValid: false,
            errors: [new Error(`Cannot create Classic stream ${this.definition.name} due to existing index`)]
          };
        }
      } catch (error) {
        if ((0, _esErrors.isNotFoundError)(error)) {
          return {
            isValid: false,
            errors: [new Error(`Cannot create Classic stream ${this.definition.name} due to missing backing Data Stream`)]
          };
        }
        throw error;
      }
    }
    if (this._changes.field_overrides) {
      const response = await this.dependencies.scopedClusterClient.asCurrentUser.transport.request({
        method: 'PUT',
        path: `/_data_stream/${this._definition.name}/_mappings?dry_run=true`,
        body: {
          properties: this._definition.ingest.classic.field_overrides,
          _meta: {
            managed_by: 'streams'
          }
        }
      });
      if (response.data_streams.length === 0) {
        return {
          isValid: false,
          errors: [new Error(`Cannot create Classic stream ${this.definition.name} due to existing Data Stream mappings`)]
        };
      }
      if (response.data_streams[0].error) {
        return {
          isValid: false,
          errors: [new Error(`Cannot create Classic stream ${this.definition.name} due to error in Data Stream mappings: ${response.data_streams[0].error}`)]
        };
      }
    }
    (0, _validate_fields.validateClassicFields)(this._definition);
    (0, _validate_stream.validateBracketsInFieldNames)(this._definition);
    (0, _validate_stream.validateSettings)(this._definition, this.dependencies.isServerless);
    return {
      isValid: true,
      errors: []
    };
  }
  async doValidateDeletion(desiredState, startingState) {
    return {
      isValid: true,
      errors: []
    };
  }

  // The actions append_processor_to_ingest_pipeline and delete_processor_from_ingest_pipeline are unique to ClassicStreams
  // Because there is no guarantee that there is a dedicated index template and ingest pipeline for ClassicStreams
  // These actions are merged across ClassicStream instances as part of ExecutionPlan.plan()
  // This is to enable us to clean up any pipeline Streams creates when it is no longer needed
  async doDetermineCreateActions() {
    const actions = [];
    if (this._definition.ingest.processing.steps.length > 0) {
      actions.push(...(await this.createUpsertPipelineActions()));
    }
    if (!(0, _streamsSchema.isInheritLifecycle)(this.getLifecycle())) {
      actions.push({
        type: 'update_lifecycle',
        request: {
          name: this._definition.name,
          lifecycle: this.getLifecycle()
        }
      });
    }
    actions.push({
      type: 'update_ingest_settings',
      request: {
        name: this._definition.name,
        settings: (0, _helpers.formatSettings)(this._definition.ingest.settings, this.dependencies.isServerless)
      }
    });
    if ((0, _helpers.settingsUpdateRequiresRollover)(await this.getEffectiveSettings(), this._definition.ingest.settings)) {
      actions.push({
        type: 'rollover',
        request: {
          name: this._definition.name
        }
      });
    }
    if (!(0, _failure_store.isInheritFailureStore)(this._definition.ingest.failure_store)) {
      actions.push({
        type: 'update_failure_store',
        request: {
          name: this._definition.name,
          failure_store: this._definition.ingest.failure_store,
          definition: this._definition
        }
      });
    }
    if (this._definition.ingest.classic.field_overrides && (0, _fields.isMappingProperties)(this._definition.ingest.classic.field_overrides)) {
      actions.push({
        type: 'update_data_stream_mappings',
        request: {
          name: this._definition.name,
          mappings: this._definition.ingest.classic.field_overrides
        }
      });
    }
    actions.push({
      type: 'upsert_dot_streams_document',
      request: this._definition
    });
    return actions;
  }
  hasChangedLifecycle() {
    return this._changes.lifecycle;
  }
  getLifecycle() {
    return this._definition.ingest.lifecycle;
  }
  getFailureStore() {
    return this._definition.ingest.failure_store;
  }
  async doDetermineUpdateActions(desiredState, startingState, startingStateStream) {
    const actions = [];
    if (this._changes.processing && this._definition.ingest.processing.steps.length > 0) {
      actions.push(...(await this.createUpsertPipelineActions()));
    }
    if (this._changes.processing && this._definition.ingest.processing.steps.length === 0) {
      const streamManagedPipelineName = (0, _name.getProcessingPipelineName)(this._definition.name);
      actions.push({
        type: 'delete_ingest_pipeline',
        request: {
          name: streamManagedPipelineName
        }
      });
      const pipelineTargets = await this.getPipelineTargets();
      if (!pipelineTargets) {
        throw new _status_error.StatusError('Could not find pipeline targets', 500);
      }
      const {
        pipeline,
        template
      } = pipelineTargets;
      actions.push({
        type: 'delete_processor_from_ingest_pipeline',
        pipeline,
        template,
        dataStream: this._definition.name,
        referencePipeline: streamManagedPipelineName
      });
    }
    if (this._changes.lifecycle) {
      actions.push({
        type: 'update_lifecycle',
        request: {
          name: this._definition.name,
          lifecycle: this.getLifecycle()
        }
      });
    }
    if (this._changes.failure_store) {
      actions.push({
        type: 'update_failure_store',
        request: {
          name: this._definition.name,
          failure_store: this._definition.ingest.failure_store,
          definition: this._definition
        }
      });
    }
    if (this._changes.settings) {
      actions.push({
        type: 'update_ingest_settings',
        request: {
          name: this._definition.name,
          settings: (0, _helpers.formatSettings)(this._definition.ingest.settings, this.dependencies.isServerless)
        }
      });
      if ((0, _helpers.settingsUpdateRequiresRollover)(await this.getEffectiveSettings(), this._definition.ingest.settings)) {
        actions.push({
          type: 'rollover',
          request: {
            name: this._definition.name
          }
        });
      }
    }
    if (this._changes.field_overrides) {
      const mappings = this._definition.ingest.classic.field_overrides || {};
      if (!(0, _fields.isMappingProperties)(mappings)) {
        throw new Error('Field overrides must be a valid mapping properties object');
      }
      actions.push({
        type: 'update_data_stream_mappings',
        request: {
          name: this._definition.name,
          mappings
        }
      });
    }
    actions.push({
      type: 'upsert_dot_streams_document',
      request: this._definition
    });
    return actions;
  }
  async createUpsertPipelineActions() {
    const actions = [];
    actions.push({
      type: 'upsert_ingest_pipeline',
      stream: this._definition.name,
      request: {
        id: (0, _name.getProcessingPipelineName)(this._definition.name),
        ...(0, _generate_ingest_pipeline.generateClassicIngestPipelineBody)(this._definition)
      }
    });
    const streamManagedPipelineName = (0, _name.getProcessingPipelineName)(this._definition.name);
    const callStreamManagedPipelineProcessor = {
      pipeline: {
        name: streamManagedPipelineName,
        if: `ctx._index == '${this._definition.name}'`,
        ignore_missing_pipeline: true,
        description: "Call the stream's managed pipeline - do not change this manually but instead use the streams UI or API"
      }
    };
    const pipelineTargets = await this.getPipelineTargets();
    if (!pipelineTargets) {
      throw new _status_error.StatusError('Could not find pipeline targets', 500);
    }
    const {
      pipeline,
      template
    } = pipelineTargets;
    actions.push({
      type: 'append_processor_to_ingest_pipeline',
      pipeline,
      template,
      dataStream: this._definition.name,
      processor: callStreamManagedPipelineProcessor,
      referencePipeline: streamManagedPipelineName
    });
    return actions;
  }
  async doDetermineDeleteActions() {
    const actions = [{
      type: 'delete_datastream',
      request: {
        name: this._definition.name
      }
    }, {
      type: 'delete_dot_streams_document',
      request: {
        name: this._definition.name
      }
    }, {
      type: 'delete_queries',
      request: {
        name: this._definition.name
      }
    }, {
      type: 'unlink_assets',
      request: {
        name: this._definition.name
      }
    }, {
      type: 'unlink_features',
      request: {
        name: this._definition.name
      }
    }];
    if (this._definition.ingest.processing.steps.length > 0) {
      const streamManagedPipelineName = (0, _name.getProcessingPipelineName)(this._definition.name);
      actions.push({
        type: 'delete_ingest_pipeline',
        request: {
          name: streamManagedPipelineName
        }
      });
      const pipelineTargets = await this.getPipelineTargets();
      if (pipelineTargets) {
        const {
          pipeline,
          template
        } = pipelineTargets;
        actions.push({
          type: 'delete_processor_from_ingest_pipeline',
          pipeline,
          template,
          dataStream: this._definition.name,
          referencePipeline: streamManagedPipelineName
        });
      }
    }
    return actions;
  }
  async getPipelineTargets() {
    var _unmanagedAssets$inge;
    let dataStream;
    try {
      dataStream = await this.dependencies.streamsClient.getDataStream(this._definition.name);
    } catch (error) {
      if ((0, _esErrors.isNotFoundError)(error)) {
        return undefined;
      }
      throw error;
    }
    const unmanagedAssets = await (0, _stream_crud.getUnmanagedElasticsearchAssets)({
      dataStream,
      scopedClusterClient: this.dependencies.scopedClusterClient
    });
    return {
      pipeline: (_unmanagedAssets$inge = unmanagedAssets.ingestPipeline) !== null && _unmanagedAssets$inge !== void 0 ? _unmanagedAssets$inge : `${dataStream.template}-pipeline`,
      template: dataStream.template
    };
  }
  async getEffectiveSettings() {
    if (!this._effectiveSettings) {
      this._effectiveSettings = (0, _stream_crud.getDataStreamSettings)(await this.dependencies.scopedClusterClient.asCurrentUser.indices.getDataStreamSettings({
        name: this._definition.name
      }).then(res => res.data_streams[0]));
    }
    return this._effectiveSettings;
  }
}
exports.ClassicStream = ClassicStream;