"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateBracketsInFieldNames = validateBracketsInFieldNames;
exports.validateNoManualIngestPipelineUsage = validateNoManualIngestPipelineUsage;
exports.validateRootStreamChanges = validateRootStreamChanges;
exports.validateSettings = validateSettings;
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _streamlang = require("@kbn/streamlang");
var _malformed_stream_error = require("../errors/malformed_stream_error");
var _root_stream_immutability_error = require("../errors/root_stream_immutability_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Changes to mappings (fields) and processing rules are not allowed on the root stream.
 * Changes to routing rules are allowed.
 * Root stream cannot inherit a lifecycle.
 */
function validateRootStreamChanges(currentStreamDefinition, nextStreamDefinition) {
  const hasFieldChanges = !(0, _lodash.isEqual)(currentStreamDefinition.ingest.wired.fields, nextStreamDefinition.ingest.wired.fields);
  if (hasFieldChanges) {
    throw new _root_stream_immutability_error.RootStreamImmutabilityError('Root stream fields cannot be changed');
  }
  const hasProcessingChanges = !(0, _lodash.isEqual)(currentStreamDefinition.ingest.processing, nextStreamDefinition.ingest.processing);
  if (hasProcessingChanges) {
    throw new _root_stream_immutability_error.RootStreamImmutabilityError('Root stream processing rules cannot be changed');
  }
  if ((0, _streamsSchema.isInheritLifecycle)(nextStreamDefinition.ingest.lifecycle)) {
    throw new _malformed_stream_error.MalformedStreamError('Root stream cannot inherit lifecycle');
  }
  if ((0, _streamsSchema.isInheritFailureStore)(nextStreamDefinition.ingest.failure_store)) {
    throw new _malformed_stream_error.MalformedStreamError('Root stream cannot inherit failure store');
  }
}
function validateNoManualIngestPipelineUsage(steps) {
  for (const step of steps) {
    if ('action' in step && step.action === 'manual_ingest_pipeline') {
      throw new _malformed_stream_error.MalformedStreamError('Manual ingest pipelines are not allowed');
    }
    if ('where' in step && step.where && 'steps' in step.where) {
      validateNoManualIngestPipelineUsage(step.where.steps);
    }
  }
}
const INVALID_CHARS_REGEX = /[\[\]]/; // Checks for either '[' or ']'

function checkFieldName(fieldName) {
  if (INVALID_CHARS_REGEX.test(fieldName)) {
    throw new _malformed_stream_error.MalformedStreamError(`Invalid field name: [${fieldName}] contains illegal characters.`);
  }
}
function validateCondition(condition) {
  if ((0, _streamlang.isAndCondition)(condition)) {
    condition.and.forEach(validateCondition);
  } else if ((0, _streamlang.isOrCondition)(condition)) {
    condition.or.forEach(validateCondition);
  } else if ((0, _streamlang.isNotCondition)(condition)) {
    validateCondition(condition.not);
  } else if ((0, _streamlang.isFilterCondition)(condition)) {
    checkFieldName(condition.field);
  }
}
const actionStepValidators = {
  append: step => checkFieldName(step.to),
  convert: step => {
    checkFieldName(step.from);
    if ('to' in step && step.to) {
      checkFieldName(step.to);
    }
  },
  date: step => {
    checkFieldName(step.from);
    if ('to' in step && step.to) {
      checkFieldName(step.to);
    }
  },
  dissect: step => checkFieldName(step.from),
  grok: step => checkFieldName(step.from),
  rename: step => {
    checkFieldName(step.from);
    checkFieldName(step.to);
  },
  set: step => {
    checkFieldName(step.to);
    if (step.copy_from) {
      checkFieldName(step.copy_from);
    }
  },
  remove_by_prefix: step => checkFieldName(step.from),
  remove: step => checkFieldName(step.from),
  drop_document: _lodash.noop,
  // 'where' condition is already validated in validateSteps function
  replace: step => {
    checkFieldName(step.from);
    if ('to' in step && step.to) {
      checkFieldName(step.to);
    }
  },
  // fields referenced in manual ingest pipelines are not validated here because
  // the interface is Elasticsearch directly here, which has its own validation
  manual_ingest_pipeline: () => {}
};
function validateSteps(steps, isWithinWhereBlock = false) {
  for (const step of steps) {
    if ('where' in step && step.where && 'steps' in step.where) {
      validateCondition(step.where);
      // Nested steps are within a where block
      validateSteps(step.where.steps, true);
    } else if ((0, _streamlang.isActionBlock)(step)) {
      // Check if remove_by_prefix is being used within a where block
      if (step.action === 'remove_by_prefix' && isWithinWhereBlock) {
        throw new _malformed_stream_error.MalformedStreamError('remove_by_prefix processor cannot be used within a where block. Use it at the root level or use the remove processor with a condition instead.');
      }
      if ('where' in step && step.where) {
        validateCondition(step.where);
      }
      const validateStep = actionStepValidators[step.action];
      validateStep(step);
    }
  }
}
function validateBracketsInFieldNames(definition) {
  var _definition$ingest$pr;
  if (!definition.ingest) {
    return;
  }
  if (_streamsSchema.Streams.WiredStream.Definition.is(definition)) {
    if (definition.ingest.wired.fields) {
      for (const fieldName of Object.keys(definition.ingest.wired.fields)) {
        checkFieldName(fieldName);
      }
    }
    if (definition.ingest.wired.routing) {
      for (const rule of definition.ingest.wired.routing) {
        validateCondition(rule.where);
      }
    }
  } else if (_streamsSchema.Streams.ClassicStream.Definition.is(definition)) {
    if (definition.ingest.classic.field_overrides) {
      for (const fieldName of Object.keys(definition.ingest.classic.field_overrides)) {
        checkFieldName(fieldName);
      }
    }
  }
  if ((_definition$ingest$pr = definition.ingest.processing) !== null && _definition$ingest$pr !== void 0 && _definition$ingest$pr.steps) {
    validateSteps(definition.ingest.processing.steps);
  }
}
function validateSettings(definition, isServerless) {
  if (!isServerless) {
    return;
  }
  const serverlessAllowList = ['index.refresh_interval'];
  Object.keys(definition.ingest.settings).forEach(setting => {
    if (!serverlessAllowList.includes(setting)) {
      throw new Error(`Setting [${setting}] is not allowed in serverless`);
    }
  });
}