"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QueryClient = void 0;
var _boom = require("@hapi/boom");
var _streamsSchema = require("@kbn/streams-schema");
var _lodash = require("lodash");
var _pLimit = _interopRequireDefault(require("p-limit"));
var _asset_client = require("../asset_client");
var _fields = require("../fields");
var _query = require("./helpers/query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function hasBreakingChange(currentQuery, nextQuery) {
  return currentQuery.kql.query !== nextQuery.kql.query || !(0, _lodash.isEqual)(currentQuery.feature, nextQuery.feature);
}
function toQueryLink(query, stream) {
  return {
    'asset.uuid': (0, _asset_client.getAssetLinkUuid)(stream, {
      'asset.type': 'query',
      'asset.id': query.id
    }),
    'asset.type': 'query',
    'asset.id': query.id,
    query
  };
}
class QueryClient {
  constructor(dependencies, isSignificantEventsEnabled = false) {
    this.dependencies = dependencies;
    this.isSignificantEventsEnabled = isSignificantEventsEnabled;
  }
  async syncQueries(stream, queries) {
    if (!this.isSignificantEventsEnabled) {
      this.dependencies.logger.debug(`Skipping syncQueries for stream "${stream}" because significant events feature is disabled.`);
      return;
    }

    /**
     * This method is used to synchronize queries/rules for a stream.
     * It manages the rules associated with these queries, ensuring that any query breaking changes
     * are handled appropriately:
     * - If a query is new, it creates a new rule.
     * - If a query is updated with a breaking change, it removes the old rule and creates a new one.
     * - If a query is updated without a breaking change, it updates the existing rule.
     * - If a query is deleted, it removes the associated rule.
     */
    const {
      [stream]: currentQueryLinks
    } = await this.dependencies.assetClient.getAssetLinks([stream], ['query']);
    const currentIds = new Set(currentQueryLinks.map(link => link.query.id));
    const nextIds = new Set(queries.map(query => query.id));
    const nextQueriesToCreate = queries.filter(query => !currentIds.has(query.id)).map(query => toQueryLink(query, stream));
    const currentQueriesToDelete = currentQueryLinks.filter(link => !nextIds.has(link.query.id));
    const currentQueriesToDeleteBeforeUpdate = currentQueryLinks.filter(link => queries.some(query => query.id === link.query.id && hasBreakingChange(link.query, query)));
    const [nextQueriesUpdatedWithBreakingChange, nextQueriesUpdatedWithoutBreakingChange] = (0, _lodash.map)((0, _lodash.partition)(queries.filter(query => currentIds.has(query.id)), query => {
      const currentLink = currentQueryLinks.find(link => link.query.id === query.id);
      return hasBreakingChange(currentLink.query, query);
    }), partitionedQueries => partitionedQueries.map(query => toQueryLink(query, stream)));
    await this.uninstallQueries([...currentQueriesToDelete, ...currentQueriesToDeleteBeforeUpdate]);
    await this.installQueries([...nextQueriesToCreate, ...nextQueriesUpdatedWithBreakingChange], nextQueriesUpdatedWithoutBreakingChange, stream);
    await this.dependencies.assetClient.syncAssetList(stream, queries.map(query => ({
      [_fields.ASSET_ID]: query.id,
      [_fields.ASSET_TYPE]: 'query',
      query
    })), 'query');
  }
  async upsert(stream, query) {
    if (!this.isSignificantEventsEnabled) {
      this.dependencies.logger.debug(`Skipping upsert for stream "${stream}" because significant events feature is disabled.`);
      return;
    }
    await this.bulk(stream, [{
      index: query
    }]);
  }
  async delete(stream, queryId) {
    if (!this.isSignificantEventsEnabled) {
      this.dependencies.logger.debug(`Skipping delete for stream "${stream}" because significant events feature is disabled.`);
      return;
    }
    await this.bulk(stream, [{
      delete: {
        id: queryId
      }
    }]);
  }
  async deleteAll(stream) {
    if (!this.isSignificantEventsEnabled) {
      this.dependencies.logger.debug(`Skipping deleteAll for stream "${stream}" because significant events feature is disabled.`);
      return;
    }
    const {
      [stream]: currentQueryLinks
    } = await this.dependencies.assetClient.getAssetLinks([stream], ['query']);
    const queriesToDelete = currentQueryLinks.map(link => ({
      delete: {
        id: link.query.id
      }
    }));
    await this.bulk(stream, queriesToDelete);
  }
  async bulk(stream, operations) {
    if (!this.isSignificantEventsEnabled) {
      this.dependencies.logger.debug(`Skipping bulk update for stream "${stream}" because significant events feature is disabled.`);
      return;
    }
    const {
      [stream]: currentQueryLinks
    } = await this.dependencies.assetClient.getAssetLinks([stream], ['query']);
    const currentIds = new Set(currentQueryLinks.map(link => link.query.id));
    const indexOperationsMap = new Map(operations.filter(operation => operation.index).map(operation => [operation.index.id, operation.index]));
    const deleteOperationIds = new Set(operations.filter(operation => operation.delete).map(operation => operation.delete.id));
    const nextQueries = [...currentQueryLinks.filter(link => !deleteOperationIds.has(link.query.id)).map(link => {
      const update = indexOperationsMap.get(link.query.id);
      return update ? {
        ...link,
        query: update
      } : link;
    }), ...operations.filter(operation => operation.index && !currentIds.has(operation.index.id)).map(operation => toQueryLink(operation.index, stream))];
    await this.syncQueries(stream, nextQueries.map(link => link.query));
  }
  async installQueries(queriesToCreate, queriesToUpdate, stream) {
    const {
      rulesClient
    } = this.dependencies;
    const limiter = (0, _pLimit.default)(10);
    await Promise.all([...queriesToCreate.map(query => {
      return limiter(() => rulesClient.create(this.toCreateRuleParams(query, stream)).catch(error => {
        if ((0, _boom.isBoom)(error) && error.output.statusCode === 409) {
          return rulesClient.update(this.toUpdateRuleParams(query, stream));
        }
        throw error;
      }));
    }), ...queriesToUpdate.map(query => {
      return limiter(() => rulesClient.update(this.toUpdateRuleParams(query, stream)).catch(error => {
        if ((0, _boom.isBoom)(error) && error.output.statusCode === 404) {
          return rulesClient.create(this.toCreateRuleParams(query, stream));
        }
        throw error;
      }));
    })]);
  }
  async uninstallQueries(queries) {
    if (queries.length === 0) {
      return;
    }
    const {
      rulesClient
    } = this.dependencies;
    await rulesClient.bulkDeleteRules({
      ids: queries.map(_query.getRuleIdFromQueryLink),
      ignoreInternalRuleTypes: false
    }).catch(error => {
      if ((0, _boom.isBoom)(error) && error.output.statusCode === 400) {
        return;
      }
      throw error;
    });
  }
  toCreateRuleParams(query, stream) {
    const ruleId = (0, _query.getRuleIdFromQueryLink)(query);
    const esqlQuery = (0, _streamsSchema.buildEsqlQuery)([stream, `${stream}.*`], query.query, true);
    return {
      data: {
        name: query.query.title,
        consumer: 'streams',
        alertTypeId: 'streams.rules.esql',
        actions: [],
        params: {
          timestampField: '@timestamp',
          query: esqlQuery
        },
        enabled: true,
        tags: ['streams', stream],
        schedule: {
          interval: '1m'
        }
      },
      options: {
        id: ruleId
      }
    };
  }
  toUpdateRuleParams(query, stream) {
    const ruleId = (0, _query.getRuleIdFromQueryLink)(query);
    const esqlQuery = (0, _streamsSchema.buildEsqlQuery)([stream, `${stream}.*`], query.query, true);
    return {
      id: ruleId,
      data: {
        name: query.query.title,
        actions: [],
        params: {
          timestampField: '@timestamp',
          query: esqlQuery
        },
        tags: ['streams', stream],
        schedule: {
          interval: '1m'
        }
      }
    };
  }
}
exports.QueryClient = QueryClient;